/* NT object module analyzer */

/* Copyright 1994 Digital Equipment Corporation.  All rights reserved. */

/* Author: Paul S. Winalski */

#include <time.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#if VMS
#include "windef.h"
#include "ntimage.h"
#elif defined (__osf__)
#include "wntbasetypes.h"
#include "ntimage.h"
#else
#include <windows.h>
#endif

static IMAGE_FILE_HEADER file_header;
static IMAGE_OPTIONAL_HEADER opt_header;
static FILE *infile;
static FILE *outfile;
static PIMAGE_SECTION_HEADER sec_header;
struct sec_info_s {
	int is_comdat;
	int sec_sym_seen;
	int comdat_sym_seen;
	};
static struct sec_info_s *sec_info;
static PIMAGE_SYMBOL st;
static ULONG st_size;
static char *indent = "                      ";
static n_errors;
static char *stringtab;
static ULONG stringtab_size;
struct mem_seg {
    ULONG base;
    ULONG high;
    };
static struct mem_seg *seg;
static int n_segs;
static int is_alpha_system;
static int has_linenums;
static int first_sym_is_file;

#define two(x) x, x


/* Convert ASCII string to integer */

static int ascii_to_int(char *s)
{
int retval = 0;
int i;

for (i = 0;; i++)
    {
    if (s[i] == '\0')
	return i;  /* NUL encountered; end of string */

    if ((s[i] < '0') || (s[i] > '9'))
	return -1;  /* non-numeric character */

    i = i * 10 + (s[i] - '0');
    }

/* shouldn't ever get here, but if we do.... */

return -1;
}


/* Report the number of errors found */

static void report_errors(void)
{
fprintf(outfile, "\n");
if (n_errors)
    {
    fprintf(outfile, "Analysis complete; %d error%sfound\n", n_errors,
	n_errors > 1 ? "s " : " ");
    printf("Analysis complete; %d error%sfound\n", n_errors,
	n_errors > 1 ? "s " : " ");
    }
else
    fprintf(outfile, "Analysis complete; no errors found\n");
}


/* Seek to the specified file position and check for errors */

static int dfseek(FILE *f, int pos)
{
return (fseek(f, pos, 0) == 0);
}


/* Read the specified number of bytes and check for errors */

static int dfread(FILE *f, int bufsiz, void *buffer, char *funcname)
{
int n_read;

n_read = fread(buffer, 1, bufsiz, f);
if (ferror(f))
    {
    printf("??? fread failed in %s\n", funcname);
    fprintf(outfile, "??? fread failed in %s\n", funcname);
    n_errors++;
    report_errors();
    exit(0);
    }
else if (n_read != bufsiz)
    {
    fprintf(outfile, "??? fread returned %d bytes (%d expected)\n",
	n_read, bufsiz);
    n_errors++;
    return 0;
    }

return 1;
}


/* Format an unsigned longword value */

static char *flong(ULONG v)
{
static char b[32];

sprintf(b, "%8x (%d)", v, v);
return b;
}


/* Format an unsigned word value */

static char *fword(USHORT v)
{
static char b[32];

sprintf(b, "    %4x (%d)", v, v);
return b;
}


/* Format a signed word value */

static char *fsword(SHORT v)
{
static char b[32];

sprintf(b, "    %4x (%d)", (USHORT) v, v);
return b;
}


/* Format an unsigned byte value */

static char *fbyte(UCHAR v)
{
static char b[32];

sprintf(b, "      %2x (%d)", v, v);
return b;
}


/* Format a major/minor version value */

static char *fver(int major, int minor)
{
static char xb[16];
static char db[16];
static char b[32];

sprintf(xb, "%x.%x", major, minor);
sprintf(db, "%d.%d", major, minor);
if (strlen(xb) > 8)
    sprintf(b, "%s (%s)", xb, db);
else
    sprintf(b, "%8s (%s)", xb, db);

return b;
}


/* Process the NTCOFF file header
 *
 * IMPLICIT INPUTS:
 *
 *	infile		input file; positioned to the file header
 *
 * IMPLICIT OUTPUTS:
 *
 *	outfile		output file
 *	file_header	file header struct
 *
 * SIDE EFFECTS:
 *
 *	infile is positioned just after the file header
 */
static int process_file_header(void)
{
static char *funcname = "process_file_header";
int n_read;
struct tm *lt;
char *lts;
char timestring[26];
char *ms;
int ch;
int all;

if (!dfread(infile, sizeof(IMAGE_FILE_HEADER), &file_header, funcname))
    {
    fprintf(outfile, "??? error reading file header\n");
    return 0;
    }

if (file_header.TimeDateStamp == 0x01020304)
    lts = "cross-compiler value";
else if ((lt = localtime(&file_header.TimeDateStamp)) == NULL)
    {
    fprintf(outfile, "??? bad file header time stamp\n");
    lts = "bad time value";
    }
else
    {
    strcpy(timestring, asctime(lt));
    timestring[24] = 0;
    lts = &timestring[0];
    }

switch (file_header.Machine)
    {
    case IMAGE_FILE_MACHINE_UNKNOWN:
	ms = "unknown";
	break;

    case IMAGE_FILE_MACHINE_I386:
	ms = "Intel x86";
	break;

    case IMAGE_FILE_MACHINE_R3000:
	ms = "MIPS R3000";
	break;

    case IMAGE_FILE_MACHINE_R4000:
	ms = "MIPS R4000";
	break;

    case 0x183:
	is_alpha_system = 1;
	ms = "Alpha AXP; acc-compatible small structs";
	break;

    case 0x184:
	is_alpha_system = 1;
	ms = "Alpha AXP";
	break;

    default:
	fprintf(outfile, "??? bad file header machine value\n");
	n_errors++;
	ms = "*** invalid ***";
	break;
    }

fprintf(outfile, "Image File Header:\n");
fprintf(outfile, "%-22.22sMachine (%s)\n", fword(file_header.Machine), ms);
fprintf(outfile, "%-22.22sNumber of Sections\n",
    fword(file_header.NumberOfSections));
fprintf(outfile, "%-22.22sDate/Time Stamp (%s)\n",
    flong(file_header.TimeDateStamp), lts);
fprintf(outfile, "%-22.22sFile Pointer to Symbol Table\n",
    flong(file_header.PointerToSymbolTable));
fprintf(outfile, "%-22.22sNumber of Symbols\n",
    flong(file_header.NumberOfSymbols));
fprintf(outfile, "%-22.22sOptional Header Size\n",
    fword(file_header.SizeOfOptionalHeader));
fprintf(outfile, "%-22.22sCharacteristics:\n",
    fword(file_header.Characteristics));
ch = file_header.Characteristics;
if (ch & IMAGE_FILE_RELOCS_STRIPPED)
    fprintf(outfile, "%srelocation info stripped\n", indent);

if (ch & IMAGE_FILE_EXECUTABLE_IMAGE)
    fprintf(outfile, "%sexecutable\n", indent);

if (ch & IMAGE_FILE_LINE_NUMS_STRIPPED)
    fprintf(outfile, "%sline numbers stripped\n", indent);

if (ch & IMAGE_FILE_LOCAL_SYMS_STRIPPED)
    fprintf(outfile, "%slocal symbols stripped\n", indent);

if (ch & IMAGE_FILE_BYTES_REVERSED_LO)
    fprintf(outfile, "%sbytes reversed (low)\n", indent);

if (ch & IMAGE_FILE_32BIT_MACHINE)
    fprintf(outfile, "%s32-bit machine\n", indent);

if (ch & IMAGE_FILE_SYSTEM)
    fprintf(outfile, "%ssystem image\n", indent);

if (ch & IMAGE_FILE_DLL)
    fprintf(outfile, "%sDLL\n", indent);

if (ch & IMAGE_FILE_BYTES_REVERSED_HI)
    fprintf(outfile, "%sbytes reversed (high)\n", indent);

all = IMAGE_FILE_RELOCS_STRIPPED | IMAGE_FILE_EXECUTABLE_IMAGE |
	IMAGE_FILE_LINE_NUMS_STRIPPED | IMAGE_FILE_LOCAL_SYMS_STRIPPED |
	IMAGE_FILE_32BIT_MACHINE |
	IMAGE_FILE_BYTES_REVERSED_LO | IMAGE_FILE_SYSTEM |
	IMAGE_FILE_DLL | IMAGE_FILE_BYTES_REVERSED_HI;
if (ch & (~all))
    {
    fprintf(outfile, "\n??? one or more undefined characteristics bits set\n");
    n_errors++;
    }

return 1;
}


/* Format an Optional Header directory entry */

static char *fdir(IMAGE_DATA_DIRECTORY *d)
{
static char b[64];

sprintf(b, "%8x [%8x] (%d [%d]) ", d->VirtualAddress, d->Size,
    d->VirtualAddress, d->Size);
return b;
}


/* Process the NTCOFF optional header
 *
 * IMPLICIT INPUTS:
 *
 *	infile		input file; positioned to the optional header
 *	file_header	file header struct
 *
 * IMPLICIT OUTPUTS:
 *
 *	outfile		output file
 *	opt_header	optional header struct
 *
 * SIDE EFFECTS:
 *
 *	infile is positioned just after the optional header
 */
static int process_opt_header(void)
{
static char *funcname = "process_opt_header";
static char *subsys;

if ((file_header.SizeOfOptionalHeader != IMAGE_SIZEOF_STD_OPTIONAL_HEADER) &&
	(file_header.SizeOfOptionalHeader != IMAGE_SIZEOF_NT_OPTIONAL_HEADER))
    {
    fprintf(outfile,
	"??? bad optional header size of %d (expecting %d or %d)\n",
	file_header.SizeOfOptionalHeader, IMAGE_SIZEOF_STD_OPTIONAL_HEADER,
	IMAGE_SIZEOF_NT_OPTIONAL_HEADER);
    n_errors++;
    return 0;
    }

if (!dfread(infile, file_header.SizeOfOptionalHeader, &opt_header, funcname))
    {
    fprintf(outfile, "??? error reading optional header\n");
    return 0;
    }

if (file_header.SizeOfOptionalHeader > IMAGE_SIZEOF_STD_OPTIONAL_HEADER)
    {
    switch (opt_header.Subsystem)
	{
	case IMAGE_SUBSYSTEM_UNKNOWN:
	    subsys = "unknown";
	    break;

	case IMAGE_SUBSYSTEM_NATIVE:
	    subsys = "none required";
	    break;

	case IMAGE_SUBSYSTEM_WINDOWS_GUI:
	    subsys = "Windows GUI";
	    break;

	case IMAGE_SUBSYSTEM_WINDOWS_CUI:
	    subsys = "Windows character";
	    break;

	case IMAGE_SUBSYSTEM_OS2_CUI:
	    subsys = "OS/2 character";
	    break;

	case IMAGE_SUBSYSTEM_POSIX_CUI:
	    subsys = "POSIX character";
	    break;

	default:
	    subsys = "*** invalid ***";
	    fprintf(outfile, "??? bad optional header subsystem value\n");
	    n_errors++;
	    break;
	}
    }

fprintf(outfile, "Optional Header:\n");
fprintf(outfile, "%-22.22sMagic number\n", fword(opt_header.Magic));
fprintf(outfile, "%-22.22sLinker Version\n",
    fver(opt_header.MajorLinkerVersion, opt_header.MinorLinkerVersion));
fprintf(outfile, "%-22.22sSize of Code\n", flong(opt_header.SizeOfCode));
fprintf(outfile, "%-22.22sSize of Initialized Data\n",
    flong(opt_header.SizeOfInitializedData));
fprintf(outfile, "%-22.22sSize of Uninitialized Data\n",
    flong(opt_header.SizeOfUninitializedData));
fprintf(outfile, "%-22.22sEntry Point Address\n",
    flong(opt_header.AddressOfEntryPoint));
fprintf(outfile, "%-22.22sBase of Code\n", flong(opt_header.BaseOfCode));
fprintf(outfile, "%-22.22sBase of Data\n", flong(opt_header.BaseOfData));

if (file_header.SizeOfOptionalHeader > IMAGE_SIZEOF_STD_OPTIONAL_HEADER)
    {
    int dc = opt_header.DllCharacteristics;
    int lf = opt_header.LoaderFlags;

    fprintf(outfile, "%-22.22sImage Base\n", flong(opt_header.ImageBase));
    fprintf(outfile, "%-22.22sSection Alignment\n",
	flong(opt_header.SectionAlignment));
    fprintf(outfile, "%-22.22sFile alignment\n",
	flong(opt_header.FileAlignment));
    fprintf(outfile, "%-22.22sOS Version\n",
	fver(opt_header.MajorOperatingSystemVersion,
	    opt_header.MinorOperatingSystemVersion));
    fprintf(outfile, "%-22.22sImage Version\n",
	fver(opt_header.MajorImageVersion, opt_header.MinorImageVersion));
    fprintf(outfile, "%-22.22sSubsystem Version\n",
	fver(opt_header.MajorSubsystemVersion,
	    opt_header.MinorSubsystemVersion));
    fprintf(outfile, "%-22.22sSize of Image\n", flong(opt_header.SizeOfImage));
    fprintf(outfile, "%-22.22sSize of Headers\n",
	flong(opt_header.SizeOfHeaders));
    fprintf(outfile, "%-22.22sChecksum\n", flong(opt_header.CheckSum));
    fprintf(outfile, "%-22.22sSubsystem (%s)\n",
	fword(opt_header.Subsystem), subsys);
    fprintf(outfile, "%-22.22sDLL Characteristics:\n",
	fword(opt_header.DllCharacteristics));
    fprintf(outfile, "%-22.22sStack Reserve Size\n",
	flong(opt_header.SizeOfStackReserve));
    fprintf(outfile, "%-22.22sStack Commit Size\n",
	flong(opt_header.SizeOfStackCommit));
    fprintf(outfile, "%-22.22sHeap Reserve Size\n",
	flong(opt_header.SizeOfHeapReserve));
    fprintf(outfile, "%-22.22sHeap Commit Size\n",
	flong(opt_header.SizeOfHeapCommit));
    fprintf(outfile, "%-22.22sLoader Flags:\n",
	flong(opt_header.LoaderFlags));
    fprintf(outfile, "%-22.22sNumber of Directory RVA and Sizes\n",
	opt_header.NumberOfRvaAndSizes);
    fprintf(outfile, "%-46.46sExport Directory\n",
	fdir(&opt_header.DataDirectory[IMAGE_DIRECTORY_ENTRY_EXPORT]));
    fprintf(outfile, "%-46.46sImport Directory\n",
	fdir(&opt_header.DataDirectory[IMAGE_DIRECTORY_ENTRY_IMPORT]));
    fprintf(outfile, "%-46.46sResource Directory\n",
	fdir(&opt_header.DataDirectory[IMAGE_DIRECTORY_ENTRY_RESOURCE]));
    fprintf(outfile, "%-46.46sException Directory\n",
	fdir(&opt_header.DataDirectory[IMAGE_DIRECTORY_ENTRY_EXCEPTION]));
    fprintf(outfile, "%-46.46sSecurity Directory\n",
	fdir(&opt_header.DataDirectory[IMAGE_DIRECTORY_ENTRY_SECURITY]));
    fprintf(outfile, "%-46.46sBase Relocation Table\n",
	fdir(&opt_header.DataDirectory[IMAGE_DIRECTORY_ENTRY_BASERELOC]));
    fprintf(outfile, "%-46.46sDebug Directory\n",
	fdir(&opt_header.DataDirectory[IMAGE_DIRECTORY_ENTRY_DEBUG]));
    fprintf(outfile, "%-46.46sDescription String\n",
	fdir(&opt_header.DataDirectory[IMAGE_DIRECTORY_ENTRY_COPYRIGHT]));
    fprintf(outfile, "%-46.46sMachine-Specific Value\n",
	fdir(&opt_header.DataDirectory[IMAGE_DIRECTORY_ENTRY_GLOBALPTR]));
    fprintf(outfile, "%-46.46sThread Local Storage Directory\n",
	fdir(&opt_header.DataDirectory[IMAGE_DIRECTORY_ENTRY_TLS]));
    fprintf(outfile, "%-46.46sLoad Configuration Directory\n",
	fdir(&opt_header.DataDirectory[IMAGE_DIRECTORY_ENTRY_LOAD_CONFIG]));

    if (opt_header.DllCharacteristics)
	{
	fprintf(outfile,
	    "??? one or more undefined DLL characteristcs bits set\n");
	n_errors++;
	}

    if (opt_header.LoaderFlags)
	{
	fprintf(outfile, "??? one or more undefined Loader Flags bits set\n");
	n_errors++;
	}
    
    if (opt_header.Reserved1)
	{
	fprintf(outfile, "??? Reserved1 field has non-zero value %x\n",
	    opt_header.Reserved1);
	n_errors++;
	}

    if (opt_header.LoaderFlags)
	{
	fprintf(outfile,
	    "??? one or more undefined loader flags bits are set\n");
	n_errors++;
	}
    }

return 1;
}

/* Hex Dump Section Data */

static void dump_section_data(ULONG size, char *data, ULONG sect_base_va,
		ULONG sect_file_addr)
{
static char *trantable ="\
................\
................\
 !\"#$%&\'()*+,-./\
0123456789:;<=>?\
@ABCDEFGHIJKLMNO\
PQRSTUVWXYZ[\\]^_\
`abcdefghijklmno\
pqrstuvwxyz{|}~.\
................\
................\
................\
................\
................\
................\
................\
................";

ULONG offset = 0;		/* current offset within section */
ULONG va = sect_base_va;	/* current virtual address */
ULONG faddr = sect_file_addr;	/* current file address */
ULONG lw[4];
static char s[9];
static char t[9];
ULONG remaining = size;
unsigned char *p = (unsigned char *) data;

while (remaining > 16)
    {
    int i;

    memcpy(lw, p, 16);
    for (i = 0; i < 8; i++)
	s[i] = trantable[p[i]];

    for (i = 8; i < 16; i++)
	t[i - 8] = trantable[p[i]];

    fprintf(outfile, "%8.8x%8.8x %8.8x%8.8x %s %s  %8.8x %8.8x\n",
	lw[3], lw[2], lw[1], lw[0], s, t, offset, faddr);
    p += 16;
    offset += 16;
    va += 16;
    faddr += 16;
    remaining -= 16;
    }

if (remaining > 0)
    {
    int i;
    int j;
    int k;
    char st[81];

    memcpy(lw, p, remaining);
    if (remaining < 8)
	j = remaining;
    else
	j = 8;

    for (i = 0; i < j; i++)
	s[i] = trantable[p[i]];

    for (i = 8; i < remaining; i++)
	t[i - 8] = trantable[p[i]];

    sprintf(st, "%8.8x%8.8x %8.8x%8.8x %s %s  %8.8x %8.8x",
	lw[3], lw[2], lw[1], lw[0], s, t, offset, faddr);
    j = 16 - remaining;
    if (remaining >= 8)
	k = 1;
    else
	k = 0;

    memset(&st[0], ' ', j * 2 + k);  /* clear out unused hex digits */
    memset(&st[34 + remaining], ' ', j + k); /* clear out unused characters */
    fprintf(outfile, "%s\n", st);
    }
}


/* Produce formatted dump of a procedure descriptor section
 *
 * FORMAL PARAMETERS:
 *
 *	sn		(in) section number of section being dumped
 *
 *	size		(in) size of section in bytes
 *
 *	contents	(in) section contents
 *
 *	file_loc	(in) file address of section contents
 *
 * IMPLICIT INPUTS:
 *
 *	none
 *
 * IMPLICIT OUTPUTS:
 *
 *	none
 *
 * SIDE EFFECTS:
 *
 *	none
 */
static void dump_pdata_section(int sn, int size, char *contents, ULONG file_loc)
{
struct pd_s {
    ULONG BeginAddress;
    ULONG EndAddress;
    ULONG ExceptionHandler;
    ULONG HandlerData;
    ULONG PrologEndAddress;
    };
typedef struct pd_s PD;

PD *p;
int n_pds;
int i;
ULONG offset;
ULONG fa;

if (is_alpha_system == 0)
    {
    fprintf(outfile, "\nContents of Section %-8x                          Offset  FileAddr\n",
	sn);
    dump_section_data(size, contents, 0, file_loc);
    return;
    }

n_pds = size/sizeof(PD);
if (size % sizeof(PD))
    {
    fprintf(outfile, "??? section size (%d) not a multiple of PD size (%d)\n",
	size, sizeof(PD));
    n_errors++;
    fprintf(outfile, "\nContents of Section %-8x                          Offset  FileAddr\n",
	sn);
    dump_section_data(size, contents, 0, file_loc);
    return;
    }

fprintf(outfile, "\nContents of Section %-8x (Procedure Descriptors)\n\n", sn);
fprintf(outfile, "PrlgEnd E1  HdlrData  Handler E2  End     U1  Begin   U2  Offset    FileAddr  Exception Mode\n");
p = (PD *) contents;
offset = 0;
fa = file_loc;
for (i = 0; i < n_pds; i++)
    {
    int exc_mode;
    char *exc_mode_text;

    exc_mode = ((p->BeginAddress & 3) << 6) +
	       ((p->EndAddress & 3) << 4) + 
	       ((p->ExceptionHandler & 3) << 2) +
	       (p->PrologEndAddress & 3);
    switch (exc_mode)
	{
	case 0:
	    exc_mode_text = "(silent)";
	    break;

	case 1:
	    exc_mode_text = "(signal)";
	    break;

	case 2:
	    exc_mode_text = "(signal all)";
	    break;

	case 3:
	    exc_mode_text = "(full IEEE)";
	    break;

	case 4:
	    exc_mode_text = "(caller)";
	    break;

	default:
	    fprintf(outfile, "??? bad exception mode bits in procedure descriptor\n");
	    n_errors++;
	    exc_mode_text = "(??? not valid)";
	    break;
	}

    fprintf(outfile, "%8.8x %1.1x  %8.8x  %8.8x %1.1x  %8.8x %1.1x  %8.8x %1.1x  %8.8x  %8.8x  %2.2x %s\n",
	p->PrologEndAddress & (-4), p->PrologEndAddress & 3, p->HandlerData,
	p->ExceptionHandler & (-4), p->ExceptionHandler & 3,
	p->EndAddress & (-4), p->EndAddress & 3,
	p->BeginAddress & (-4), p->BeginAddress & 3,
	offset, fa, exc_mode, exc_mode_text);
    offset += sizeof(PD);
    fa += sizeof(PD);
    p++;
    } 
}


/* Process a NTCOFF Section Header
 *
 * FORMAL PARAMETERS:
 *
 *	i		(in) index into sec_header array of the section to be
 *			processed
 *
 * IMPLICIT INPUTS:
 *
 *	sec_header	pointer to the section header array
 *
 * IMPLICIT OUTPUTS:
 *
 *	infile		input file
 *	outfile		output file
 *
 * SIDE EFFECTS:
 *
 *	position of input file is indeterminate at routine exit
 */
static int process_section_header(int i)
{
char *funcname = "process_section_header";
char *bad_reltype = " ??? bad relocation type";
PIMAGE_SECTION_HEADER s = &sec_header[i];
int sc = s->Characteristics;
int type_sc = IMAGE_SCN_TYPE_NO_PAD;
int all_sc = type_sc | IMAGE_SCN_CNT_CODE | IMAGE_SCN_CNT_INITIALIZED_DATA |
		IMAGE_SCN_CNT_UNINITIALIZED_DATA | IMAGE_SCN_LNK_OTHER |
		IMAGE_SCN_LNK_INFO |
		IMAGE_SCN_LNK_REMOVE | IMAGE_SCN_LNK_COMDAT | 0x00f00000 |
		IMAGE_SCN_MEM_DISCARDABLE | IMAGE_SCN_MEM_NOT_CACHED |
		IMAGE_SCN_MEM_NOT_PAGED | IMAGE_SCN_MEM_SHARED |
		IMAGE_SCN_MEM_EXECUTE | IMAGE_SCN_MEM_READ |
		IMAGE_SCN_MEM_WRITE;
int alignment;
int last_type;
ULONG last_vaddr;
char sec_name[9];
char *sn;
char *contents;
ULONG reloc_size;
ULONG section_low_VA;
ULONG section_high_VA;
PIMAGE_RELOCATION r = NULL;
PIMAGE_RELOCATION ss;
ULONG ln_size;
PIMAGE_LINENUMBER ln = NULL;

if (s->Name[7])
    {
    strncpy(sec_name, (char *) &s->Name[0], 8);
    sec_name[8] = 0;
    sn = &sec_name[0];
    }
else
    sn = (char *) &s->Name[0];

if ((sn[0] == '/') && stringtab)
    {
    int namelen = strlen(sn);

    if ((namelen > 8) || (namelen < 2))
	{
	fprintf(outfile, "??? Bad indirect section name length: %d\n",
	    namelen);
	n_errors++;
	}
    else
	{
	int i;
	int name_loc = 0;

	for (i = 1; i < namelen; i++)
	    {
	    if ((sn[i] < '0') || (sn[i] > '9'))
		{
		fprintf(outfile, "??? Indirect section name offset conversion error\n");
		n_errors++;
		name_loc = -1;
		break;
		}

	    name_loc = name_loc * 10 + (sn[i] - '0');
	    }

	if (name_loc >= 0)
	    if ((name_loc < 4) || (name_loc >= stringtab_size))
		{
		fprintf(outfile, "??? Indirect section name offset out of range\n");
		n_errors++;
		}
	    else
		sn = stringtab + (name_loc - 4);
	}
    }

fprintf(outfile, "\n\nSection # %x (%s) Header\n", i, sn);
fprintf(outfile, "%-22.22sVirtual Size\n", flong(s->Misc.VirtualSize));
fprintf(outfile, "%-22.22sVirtual Address\n", flong(s->VirtualAddress));
section_low_VA = s->VirtualAddress;
fprintf(outfile, "%-22.22sRaw Data Size\n", flong(s->SizeOfRawData));
section_high_VA = section_low_VA + s->SizeOfRawData - 1;
fprintf(outfile, "%-22.22sFile Pointer to Raw Data\n",
    flong(s->PointerToRawData));
fprintf(outfile, "%-22.22sFile Pointer to Relocations\n",
    flong(s->PointerToRelocations));
fprintf(outfile, "%-22.22sFile Pointer to Line Numbers\n",
    flong(s->PointerToLinenumbers));
fprintf(outfile, "%-22.22sNumber of Relocations\n",
    fword(s->NumberOfRelocations));
fprintf(outfile, "%-22.22sNumber of Line Numbers\n",
    fword(s->NumberOfLinenumbers));
fprintf(outfile, "%-22.22sCharacteristics:\n", fword(sc));
if (sc & (~type_sc))
    {
    if (sc & IMAGE_SCN_TYPE_NO_PAD)
	fprintf(outfile, "%sno-pad section\n", indent);
    }
else
    fprintf(outfile, "%sregular section\n", indent);

if (sc & IMAGE_SCN_CNT_CODE)
    fprintf(outfile, "%scode\n", indent);

if (sc & IMAGE_SCN_CNT_INITIALIZED_DATA)
    fprintf(outfile, "%sinitialized data\n", indent);

if (sc & IMAGE_SCN_CNT_UNINITIALIZED_DATA)
    fprintf(outfile, "%suninitialized data\n", indent);

if (sc & IMAGE_SCN_LNK_OTHER)
    fprintf(outfile, "%sother information\n", indent);

if (sc & IMAGE_SCN_LNK_INFO)
    fprintf(outfile, "%scomments\n", indent);

if (sc & IMAGE_SCN_LNK_REMOVE)
    fprintf(outfile, "%swill not become part of image\n", indent);

if (sc & IMAGE_SCN_LNK_COMDAT)
    {
    fprintf(outfile, "%scommunal\n", indent);
    sec_info[i].is_comdat = 1;
    }

alignment = sc & 0x00f00000;
switch (alignment)
    {
    case 0:
	fprintf(outfile, "%sno alignment specified (16-byte defaulted)\n",
	    indent);
	break;

    case IMAGE_SCN_ALIGN_1BYTES:
	fprintf(outfile, "%sbyte-aligned\n", indent);
	break;

    case IMAGE_SCN_ALIGN_2BYTES:
	fprintf(outfile, "%s16-bit-aligned\n", indent);
	break;

    case IMAGE_SCN_ALIGN_4BYTES:
	fprintf(outfile, "%s32-bit-aligned\n", indent);
	break;

    case IMAGE_SCN_ALIGN_8BYTES:
	fprintf(outfile, "%s64-bit-aligned\n", indent);
	break;

    case IMAGE_SCN_ALIGN_16BYTES:
	fprintf(outfile, "%s128-bit-aligned\n", indent);
	break;

    case IMAGE_SCN_ALIGN_32BYTES:
	fprintf(outfile, "%s256-bit-aligned\n", indent);
	break;

    case IMAGE_SCN_ALIGN_64BYTES:
	fprintf(outfile, "%s512-bit-aligned\n", indent);
	break;

    default:
	{
	fprintf(outfile, "%s??? invalid alignment value %x\n", indent,
	    alignment);
	n_errors++;
	}
    }

if (sc & IMAGE_SCN_MEM_DISCARDABLE)
    fprintf(outfile, "%sdiscardable\n", indent);

if (sc & IMAGE_SCN_MEM_NOT_CACHED)
    fprintf(outfile, "%snot cachable\n", indent);

if (sc & IMAGE_SCN_MEM_NOT_PAGED)
    fprintf(outfile, "%snot pageable\n", indent);

if (sc & IMAGE_SCN_MEM_SHARED)
    fprintf(outfile, "%sshareable\n", indent);

if (sc & IMAGE_SCN_MEM_EXECUTE)
    fprintf(outfile, "%sexecutable\n", indent);

if (sc & IMAGE_SCN_MEM_READ)
    fprintf(outfile, "%sreadable\n", indent);

if (sc & IMAGE_SCN_MEM_WRITE)
    fprintf(outfile, "%swritable\n", indent);

if (sc & (~all_sc))
    {
    fprintf(outfile, "??? one or more unspecified characteristics bits set\n");
    n_errors++;
    }

/* make sure that sizes and pointers agree */

if (((s->SizeOfRawData != 0) && (s->PointerToRawData == 0) &&
	    ((sc & IMAGE_SCN_CNT_UNINITIALIZED_DATA) == 0)) ||
	((s->SizeOfRawData == 0) && (s->PointerToRawData != 0)))
    {
    fprintf(outfile, "??? size of raw data = %d but -> raw data = %x\n",
	s->SizeOfRawData, s->PointerToRawData);
    n_errors++;
    }

if (s->NumberOfLinenumbers)
    has_linenums = 1;

if (((s->NumberOfLinenumbers != 0) && (s->PointerToLinenumbers == 0)) ||
	((s->NumberOfLinenumbers == 0) && (s->PointerToLinenumbers != 0)))
    {
    fprintf(outfile, "??? number of line numbers = %d but -> line numbers = %x\n",
	s->NumberOfLinenumbers, s->PointerToLinenumbers);
    n_errors++;
    }

if (((s->NumberOfRelocations != 0) && (s->PointerToRelocations == 0)) ||
	((s->NumberOfRelocations == 0) && (s->PointerToRelocations != 0)))
    {
    fprintf(outfile, "??? number of relocations = %d but -> relocations = %x\n",
	s->NumberOfRelocations, s->PointerToRelocations);
    n_errors++;
    }

/* Read in and dump the section contents */

if ((s->SizeOfRawData > 0) && (s->PointerToRawData > 0))
    {
    if ((contents = malloc(s->SizeOfRawData)) == NULL)
	{
	fprintf(outfile, "??? Unable to allocate %u byte buffer to hold section data\n",
	    s->SizeOfRawData);
	n_errors++;
	}
    else if (!dfseek(infile, s->PointerToRawData))
	{
	fprintf(outfile, "??? error seeking to section raw data\n");
	n_errors++;
	}
    else if (!dfread(infile, s->SizeOfRawData, contents, funcname))
	{
	fprintf(outfile, "??? error reading section raw data\n");
	n_errors++;
	}
    else if (strcmp(sn, ".pdata") == 0)
	dump_pdata_section(i, s->SizeOfRawData, contents, s->PointerToRawData);
    else
	{
	fprintf(outfile, "\nContents of Section %-8x                          Offset  FileAddr\n",
	    i);
	dump_section_data(s->SizeOfRawData, contents, s->VirtualAddress,
	    s->PointerToRawData);
	}

    if (contents)
	free(contents);
    }

/* Read in and format the section relocations */

reloc_size = s->NumberOfRelocations * IMAGE_SIZEOF_RELOCATION;
if ((s->NumberOfRelocations == 0) || (s->PointerToRelocations == 0))
    ;
else if (s->PointerToRawData == 0)
    {
    fprintf(outfile, "??? section has relocations but no raw data\n");
    n_errors++;
    }
else if ((r = malloc(reloc_size)) == NULL)
    {
    fprintf(outfile, "??? Unable to allocate %u byte buffer to hold section relocations\n",
	reloc_size);
    n_errors++;
    }
else if (!dfseek(infile, s->PointerToRelocations))
    {
    fprintf(outfile, "??? error seeking to section relocations\n");
    n_errors++;
    }
else if (!dfread(infile, reloc_size, (char *) r, funcname))
    {
    fprintf(outfile, "??? error reading section relocations\n");
    n_errors++;
    }
else
    {
    int j;
    int check_symindex;
    ULONG faddr = s->PointerToRelocations;
    char *reltype;

    fprintf(outfile, "\nRelocations for Section %x\n", i);
    fprintf(outfile, "FileAddr  VirtAddr  Symtab Index           Relocation Type\n");
    last_type = 0;
    last_vaddr = 0;
    for (j = 0; j < s->NumberOfRelocations; j++)
	{
	check_symindex = 1;
	switch (file_header.Machine)
	    {
	    case IMAGE_FILE_MACHINE_I386:
		switch (r->Type)
		    {
		    case IMAGE_REL_I386_ABSOLUTE:
			reltype = " (ABSOLUTE)";
			break;

		    case IMAGE_REL_I386_DIR16:
			reltype = " (DIR16)";
			break;

		    case IMAGE_REL_I386_REL16:
			reltype = " (REL16)";
			break;

		    case IMAGE_REL_I386_DIR32:
			reltype = " (DIR32)";
			break;

		    case IMAGE_REL_I386_DIR32NB:
			reltype = " (DIR32NB)";
			break;

		    case IMAGE_REL_I386_SEG12:
			reltype = " (SEG12)";
			break;

		    case IMAGE_REL_I386_SECTION:
			reltype = " (SECTION)";
			break;

		    case IMAGE_REL_I386_SECREL:
			reltype = " (SECREL)";
			break;

		    case IMAGE_REL_I386_REL32:
			reltype = " (REL32)";
			break;

		    default:
			reltype = bad_reltype;
			n_errors++;
		    }

		break;

	    case IMAGE_FILE_MACHINE_R3000:
	    case IMAGE_FILE_MACHINE_R4000:
		switch (r->Type)
		    {
		    case IMAGE_REL_MIPS_ABSOLUTE:
			reltype = " (ABSOLUTE)";
			break;

		    case IMAGE_REL_MIPS_REFHALF:
			reltype = " (REFHALF)";
			break;

		    case IMAGE_REL_MIPS_REFWORD:
			reltype = " (REFWORD)";
			break;

		    case IMAGE_REL_MIPS_JMPADDR:
			reltype = " (JMPADDR)";
			break;

		    case IMAGE_REL_MIPS_REFHI:
			reltype = " (REFHI)";
			ss = r + 1;
			if ( (j == s->NumberOfRelocations - 1) ||
				(ss->Type != IMAGE_REL_MIPS_PAIR))
			    {
			    fprintf(outfile, "??? HIGH relocation not followed by PAIR\n");
			    n_errors++;
			    }

			break;

		    case IMAGE_REL_MIPS_REFLO:
			reltype = " (REFLO)";
			break;

		    case IMAGE_REL_MIPS_GPREL:
			reltype = " (GPREL)";
			break;

		    case IMAGE_REL_MIPS_LITERAL:
			reltype = " (LITERAL)";
			break;

		    case IMAGE_REL_MIPS_SECTION:
			reltype = " (SECTION)";
			break;

		    case IMAGE_REL_MIPS_SECREL:
			reltype = " (SECREL)";
			break;

		    case IMAGE_REL_MIPS_REFWORDNB:
			reltype = " (REFWORDNB)";
			break;

		    case IMAGE_REL_MIPS_PAIR:
			check_symindex = 0;
			reltype = " (PAIR)";
			if (last_type != IMAGE_REL_MIPS_REFHI)
			    {
			    fprintf(outfile, "??? PAIR relocation follows non-REFHI relocation\n");
			    n_errors++;
			    }
			else if (r->VirtualAddress != last_vaddr)
			    {
			    fprintf(outfile, "??? PAIR virtual address does not that of REFHI\n");
			    n_errors++;
			    }

			break;

		    default:
			reltype = bad_reltype;
			n_errors++;
		    }

		break;

	    case 0x183:
	    case 0x184:
		switch (r->Type)
		    {
		    case IMAGE_REL_ALPHA_ABSOLUTE:
			reltype = " (ABSOLUTE)";
			break;

		    case IMAGE_REL_ALPHA_REFLONG:
			reltype = " (REFLONG)";
			break;

		    case IMAGE_REL_ALPHA_REFQUAD:
			reltype = " (REFQUAD)";
			break;

		    case IMAGE_REL_ALPHA_GPREL32:
			reltype = " (GPREL32)";
			break;

		    case IMAGE_REL_ALPHA_LITERAL:
			reltype = " (LITERAL)";
			break;

		    case IMAGE_REL_ALPHA_LITUSE:
			reltype = " (LITUSE)";
			break;

		    case IMAGE_REL_ALPHA_GPDISP:
			reltype = " (GPDISP)";
			break;

		    case IMAGE_REL_ALPHA_BRADDR:
			reltype = " (BRADDR)";
			break;

		    case IMAGE_REL_ALPHA_HINT:
			reltype = " (HINT)";
			break;

		    case IMAGE_REL_ALPHA_INLINE_REFLONG:
			reltype = " (INLINE_REFLONG)";
			break;

		    case IMAGE_REL_ALPHA_REFHI:
			reltype = " (REFHI)";
			ss = r + 1;
			if ((j == s->NumberOfRelocations - 1) ||
				((ss->Type != IMAGE_REL_ALPHA_PAIR) &&
				 (ss->Type != IMAGE_REL_ALPHA_MATCH)))
			    {
			    fprintf(outfile, "??? HIGH relocation not followed by PAIR or MATCH\n");
			    n_errors++;
			    }

			break;

		    case IMAGE_REL_ALPHA_REFLO:
			reltype = " (REFLO)";
			break;

		    case IMAGE_REL_ALPHA_PAIR:
			check_symindex = 0;
			reltype = " (PAIR)";
			if (last_type != IMAGE_REL_ALPHA_REFHI)
			    {
			    fprintf(outfile, "??? PAIR relocation follows non-REFHI relocation\n");
			    n_errors++;
			    }
			else if (r->VirtualAddress != last_vaddr)
			    {
			    fprintf(outfile, "??? PAIR virtual address does not that of REFHI\n");
			    n_errors++;
			    }

			break;

		    case IMAGE_REL_ALPHA_MATCH:
			check_symindex = 0;
			reltype = " (MATCH)";
			if (last_type != IMAGE_REL_ALPHA_REFHI)
			    {
			    fprintf(outfile, "??? MATCH relocation follows non-REFHI relocation\n");
			    n_errors++;
			    }
			else if (r->VirtualAddress != last_vaddr)
			    {
			    fprintf(outfile, "??? MATCH virtual address does not that of REFHI\n");
			    n_errors++;
			    }
			break;

		    case IMAGE_REL_ALPHA_SECTION:
			reltype = " (SECTION)";
			break;

		    case IMAGE_REL_ALPHA_SECREL:
			reltype = " (SECREL)";
			break;

		    case IMAGE_REL_ALPHA_REFLONGNB:
			reltype = " (REFLONGNB)";
			break;

		    default:
			reltype = bad_reltype;
			n_errors++;
		    }

		break;

	    default:
		reltype = " ";
	    }

	fprintf(outfile, "%8.8x  %8.8x  %-22.22s  %-12.12s%s\n",
	    faddr, r->VirtualAddress, flong(r->SymbolTableIndex),
	    (fword(r->Type) + 4), reltype);
	if (check_symindex)
	    {
	    int n = r->SymbolTableIndex;

	    if (n > file_header.NumberOfSymbols)
		{
		fprintf(outfile, "??? symbol table index out of range\n");
		n_errors++;
		}
	    else if ((st[n].StorageClass == IMAGE_SYM_CLASS_STATIC) &&
		    (st[n].NumberOfAuxSymbols > 0) &&
		    (st[n].SectionNumber > 0) &&
		    (st[n].SectionNumber <= file_header.NumberOfSections))
		if (sec_header[st[n].SectionNumber].Characteristics &
			IMAGE_SCN_LNK_COMDAT)
		    {
		    fprintf(outfile, "??? relocation symbol is the comdat section symbol for section %x\n",
			st[n].SectionNumber);
		    n_errors++;
		    }
	    }

	if ((r->VirtualAddress < section_low_VA) ||
		(r->VirtualAddress > section_high_VA))
	    {
	    fprintf(outfile, "??? relocation outside section virtual address bounds (%8.8x:%8.8x)\n",
		section_low_VA, section_high_VA);
	    n_errors++;
	    }
/*	else if (r->VirtualAddress < last_vaddr)
	    {
	    fprintf(outfile, "??? relocation virtual address retrograde motion\n");
	    n_errors++;
	    }
*/
	last_type = r->Type;
	last_vaddr = r->VirtualAddress;
	faddr += IMAGE_SIZEOF_RELOCATION;
	r++;
	}
    }

if (r)
    free((char *) r);

/* Read in and format the line numbers */

ln_size = s->NumberOfLinenumbers * IMAGE_SIZEOF_LINENUMBER;
if ((s->NumberOfLinenumbers == 0) || (s->PointerToLinenumbers == 0))
    ;
else if (s->PointerToRawData == 0)
    {
    fprintf(outfile, "\n??? section has line numbers but no raw data\n");
    n_errors++;
    }
else if ((ln= malloc(ln_size)) == NULL)
    {
    fprintf(outfile, "\n??? Unable to allocate %u byte buffer to hold section line numbers\n",
	ln_size);
    n_errors++;
    }
else if (!dfseek(infile, s->PointerToLinenumbers))
    {
    fprintf(outfile, "??? error seeking to section line numbers\n");
    n_errors++;
    }
else if (!dfread(infile, ln_size, (char *) ln, funcname))
    {
    fprintf(outfile, "??? error reading section line numbers\n");
    n_errors++;
    }
else
    {
    ULONG faddr = s->PointerToLinenumbers;
    int j;
    int func_start = (-1);
    int func_end = 0;

    fprintf(outfile, "\nLine Numbers for Section %d\n", i);
    fprintf(outfile, "FileAddr VirtAddr Line #        ");
    fprintf(outfile, "FileAddr VirtAddr Line #");
    for (j = 0; j < s->NumberOfLinenumbers; j++)
	{
	if ((j % 2) == 0)
	    fprintf(outfile, "\n");

	fprintf(outfile, "%8.8x %8.8x %-12.12s  ", faddr,
	    ln->Type.VirtualAddress, (fsword(ln->Linenumber) + 4));
	if (ln->Linenumber == 0)
	    {
	    func_start = (-1);
	    func_end = 0;
	    if (ln->Type.SymbolTableIndex >= file_header.NumberOfSymbols)
		{
		fprintf(outfile, "\n??? symbol table index is out of range\n");
		n_errors++;
		}
	    else
		{
		PIMAGE_SYMBOL sym = &st[ln->Type.SymbolTableIndex];
		PIMAGE_AUX_SYMBOL aux = (PIMAGE_AUX_SYMBOL) (sym + 1);

		if (sym->SectionNumber != i)
		    {
		    fprintf(outfile, "\n??? function symbol is not in this section\n");
		    n_errors++;
		    }
		else if (sym->Type != 0x20)
		    {
		    fprintf(outfile, "\n??? symbol is not a function symbol\n");
		    n_errors++;
		    }
		else
		    {
		    func_start = sym->Value;
		    func_end = func_start + aux->Sym.Misc.TotalSize;
		    if (aux->Sym.FcnAry.Function.PointerToLinenumber != faddr)
			{
			fprintf(outfile, "\n??? aux of function symbol does not point back to linenum\n");
			n_errors++;
			}
		    }
		}
	    }
	else if ((func_start != (-1)) &&
		((ln->Type.VirtualAddress < func_start) ||
		 (ln->Type.VirtualAddress >= func_end)))
	    {
	    fprintf(outfile, "\n??? VirtualAddress not within function (%8.8x:%8.8x)\n",
		func_start, func_end);
	    n_errors++;
	    }

	ln++;
	faddr += IMAGE_SIZEOF_LINENUMBER;
	}

    fprintf(outfile, "\n");
    }

if (ln)
    free((char *) ln);

return 1;
}


/* Process the symbol table */

static void process_symbol_table(void)
{
char *funcname = "process_symbol_table";
ULONG faddr = file_header.PointerToSymbolTable;
int j;
int i;
PIMAGE_SYMBOL stab = st;

fprintf(outfile, "\n\nSymbol Table:\n");
fprintf(outfile, "FileAddr Indx Value    Sect Storage Class    Type               #Aux Name\n");
for (j = 0; j < file_header.NumberOfSymbols; j++)
    {
    char sect_num[5];
    char *storage_class;
    char sym_type[15];
    char *sym_name;
    static char short_name[9];
    static char error_name[80];
    ULONG base_value;
    int base_class;
    int base_type;
    int base_section;
    int n_aux;
    int k;
    int long_name;
    int long_nameloc;

    if (stab->N.Name.Short == 0)
	{
	long_name = 1;
	long_nameloc = stab->N.Name.Long;
	if (stab->N.Name.Long >= stringtab_size)
	    {
	    sprintf(error_name, "??? name offset %8x is outside string table",
		stab->N.Name.Long);
	    n_errors++;
	    sym_name = &error_name[0];
	    }
	else
	    sym_name = &stringtab[stab->N.Name.Long - 4];
	}
    else
	{
	long_name = 0;
	strncpy(short_name, (char *) &stab->N.ShortName[0], 8);
	sym_name = &short_name[0];
	}

    if (stab->SectionNumber == IMAGE_SYM_UNDEFINED)
	strcpy(sect_num, " UND");
    else if (stab->SectionNumber == IMAGE_SYM_ABSOLUTE)
	strcpy(sect_num, " ABS");
    else if (stab->SectionNumber == IMAGE_SYM_DEBUG)
	strcpy(sect_num, " DBG");
    else
	sprintf(sect_num, "%4.4x", stab->SectionNumber);

    base_class = stab->StorageClass;
    switch (stab->StorageClass)
	{
	case IMAGE_SYM_CLASS_END_OF_FUNCTION:
	    storage_class = "(end of func)";
	    break;

	case IMAGE_SYM_CLASS_NULL:
	    storage_class = "(null)";
	    break;

	case IMAGE_SYM_CLASS_AUTOMATIC:
	    storage_class = "(automatic)";
	    break;

	case IMAGE_SYM_CLASS_EXTERNAL:
	    storage_class = "(external)";
	    break;

	case IMAGE_SYM_CLASS_STATIC:
	    storage_class = "(static)";
	    break;

	case IMAGE_SYM_CLASS_REGISTER:
	    storage_class = "(register)";
	    break;

	case IMAGE_SYM_CLASS_EXTERNAL_DEF:
	    storage_class = "(external def)";
	    break;

	case IMAGE_SYM_CLASS_LABEL:
	    storage_class = "(label)";
	    break;

	case IMAGE_SYM_CLASS_UNDEFINED_LABEL:
	    storage_class = "(undef label)";
	    break;

	case IMAGE_SYM_CLASS_MEMBER_OF_STRUCT:
	    storage_class = "(struct memb)";
	    break;

	case IMAGE_SYM_CLASS_ARGUMENT:
	    storage_class = "(argument)";
	    break;

	case IMAGE_SYM_CLASS_STRUCT_TAG:
	    storage_class = "(struct tag)";
	    break;

	case IMAGE_SYM_CLASS_MEMBER_OF_UNION:
	    storage_class = "(union member)";
	    break;

	case IMAGE_SYM_CLASS_UNION_TAG:
	    storage_class = "(union tag)";
	    break;

	case IMAGE_SYM_CLASS_TYPE_DEFINITION:
	    storage_class = "(typedef)";
	    break;

	case IMAGE_SYM_CLASS_UNDEFINED_STATIC:
	    storage_class = "(undef static)";
	    break;

	case IMAGE_SYM_CLASS_ENUM_TAG:
	    storage_class = "(enum tag)";
	    break;

	case IMAGE_SYM_CLASS_MEMBER_OF_ENUM:
	    storage_class = "(enum member)";
	    break;

	case IMAGE_SYM_CLASS_REGISTER_PARAM:
	    storage_class = "(reg param)";
	    break;

	case IMAGE_SYM_CLASS_BIT_FIELD:
	    storage_class = "(bit field)";
	    break;

	case IMAGE_SYM_CLASS_BLOCK:
	    storage_class = "(block)";
	    break;

	case IMAGE_SYM_CLASS_FUNCTION:
	    storage_class = "(function)";
	    break;

	case IMAGE_SYM_CLASS_END_OF_STRUCT:
	    storage_class = "(end of struc)";
	    break;

	case IMAGE_SYM_CLASS_FILE:
	    storage_class = "(file)";
	    break;

	case IMAGE_SYM_CLASS_SECTION:
	    storage_class = "(section)";
	    break;

	case IMAGE_SYM_CLASS_WEAK_EXTERNAL:
	    storage_class = "(weak extern)";
	    break;

	default:
	    storage_class = "(??? bad)";
	    n_errors++;
	}

    sym_type[0] = '(';
    sym_type[1] = '\0';
    if (ISPTR(stab->Type))
	strcat(sym_type, "-> ");
    else if (ISFCN(stab->Type))
	strcat(sym_type, "Fn ");
    else if (ISARY(stab->Type))
	strcat(sym_type, "Ar ");

    base_type = stab->Type;
    switch (BTYPE(stab->Type))
	{
	case IMAGE_SYM_TYPE_NULL:
	    strcat(sym_type, "null)");
	    break;

	case IMAGE_SYM_TYPE_VOID:
	    strcat(sym_type, "void)");
	    break;

	case IMAGE_SYM_TYPE_CHAR:
	    strcat(sym_type, "char)");
	    break;

	case IMAGE_SYM_TYPE_SHORT:
	    strcat(sym_type, "short)");
	    break;

	case IMAGE_SYM_TYPE_INT:
	    strcat(sym_type, "int)");
	    break;

	case IMAGE_SYM_TYPE_LONG:
	    strcat(sym_type, "long)");
	    break;

	case IMAGE_SYM_TYPE_FLOAT:
	    strcat(sym_type, "float)");
	    break;

	case IMAGE_SYM_TYPE_DOUBLE:
	    strcat(sym_type, "double)");
	    break;

	case IMAGE_SYM_TYPE_STRUCT:
	    strcat(sym_type, "struct)");
	    break;

	case IMAGE_SYM_TYPE_UNION:
	    strcat(sym_type, "union)");
	    break;

	case IMAGE_SYM_TYPE_ENUM:
	    strcat(sym_type, "enum)");
	    break;

	case IMAGE_SYM_TYPE_MOE:
	    strcat(sym_type, "enum memb)");
	    break;

	case IMAGE_SYM_TYPE_BYTE:
	    strcat(sym_type, "byte)");
	    break;

	case IMAGE_SYM_TYPE_WORD:
	    strcat(sym_type, "word)");
	    break;

	case IMAGE_SYM_TYPE_UINT:
	    strcat(sym_type, "uint)");
	    break;

	case IMAGE_SYM_TYPE_DWORD:
	    strcat(sym_type, "dword)");
	    break;

	case 0x20:
	    strcat(sym_type, "null func)");
	    break;

	default:
	    strcat(sym_type, "??? bad)");
	    n_errors++;
	}

    base_value = stab->Value;
    fprintf(outfile, "%8.8x %4.4x %8.8x %s %2.2x%-14.14s %4.4x%-14.14s  %3d ",
	faddr, j, stab->Value, sect_num, stab->StorageClass, storage_class,
	stab->Type, sym_type, stab->NumberOfAuxSymbols);

    if (long_name)
	fprintf(outfile, "%x -> ", long_nameloc);

    fprintf(outfile, "%s\n", sym_name);
    if ((long_name) && (strlen(sym_name) <= 8))
	{
	fprintf(outfile, "??? name from string table is only %d characters long\n",
	    strlen(sym_name));
	n_errors++;
	}

    if (stab->StorageClass == IMAGE_SYM_CLASS_FILE)
        {
        if (j == 0)
	    first_sym_is_file = 1;

	/* File symbols must be named .file */

	if (strcmp(sym_name, ".file") != 0)
	    {
	    fprintf(outfile, "??? file symbol is not named .file\n");
	    n_errors++;
	    }

	/* File symbols must have a non-zero number of aux symbols */

	if (stab->NumberOfAuxSymbols == 0)
	    {
	    fprintf(outfile, "??? file symbol has no aux symbols\n");
	    n_errors++;
	    }
	}

    /* Check the names and number of aux symbols for function symbols */

    if (stab->StorageClass == IMAGE_SYM_CLASS_FUNCTION)
	if (strcmp(sym_name, ".bf") == 0)
	    {
	    if (stab->NumberOfAuxSymbols != 1)
		{
		fprintf(outfile, "??? .bf symbol must have exactly 1 aux\n");
		n_errors++;
		}
	    }
	else if (strcmp(sym_name, ".lf") == 0)
	    {
	    if (stab->NumberOfAuxSymbols != 0)
		{
		fprintf(outfile, "??? aux not allowed for .lf symbols\n");
		n_errors++;
		}
	    }
	else if (strcmp(sym_name, ".ef") == 0)
	    {
	    if (stab->NumberOfAuxSymbols != 1)
		{
		fprintf(outfile, "??? .ef symbol must have exactly 1 aux\n");
		n_errors++;
		}
	    }
	else
	    {
	    fprintf(outfile, "??? function symbol not named .bf, .lf, or .ef\n");
	    n_errors++;
	    }

    if ((stab->SectionNumber < IMAGE_SYM_DEBUG) ||
	    (stab->SectionNumber > file_header.NumberOfSections))
	{
	fprintf(outfile, "??? section number is out of range\n");
	n_errors++;
	base_section = 0;
	}
    else
	base_section = stab->SectionNumber;

    n_aux = stab->NumberOfAuxSymbols;
    if (base_class == IMAGE_SYM_CLASS_WEAK_EXTERNAL)
	{
	if (base_section != IMAGE_SYM_UNDEFINED)
	    {
	    fprintf(outfile, "??? weak external is not Undefined\n");
	    n_errors++;
	    }

	if (base_value != 0)
	    {
	    fprintf(outfile, "??? non-zero Value for weak external\n");
	    n_errors++;
	    }

	if (n_aux != 1)
	    {
	    fprintf(outfile, "??? wrong number of aux symbols for weak external\n");
	    n_errors++;
	    }
	}

    if ((n_aux + j) >= file_header.NumberOfSymbols)
	{
	fprintf(outfile, "??? aux symbol count overflows symbol table\n");
	n_errors++;
	}
    else if ((base_class == IMAGE_SYM_CLASS_FILE) && (n_aux > 0))
	{
	PIMAGE_AUX_SYMBOL ast;

	stab++;
	j++;
	ast = (PIMAGE_AUX_SYMBOL) stab;
	faddr += IMAGE_SIZEOF_SYMBOL;
	fprintf(outfile, "%8.8x %4.4x Aux: filename = \"%s\"\n",
	    faddr, j, &ast->File.Name[0]);
	stab += (n_aux - 1);
	j += (n_aux - 1);
	faddr += (IMAGE_SIZEOF_AUX_SYMBOL * (n_aux - 1));
	}
    else if ((base_class == IMAGE_SYM_CLASS_WEAK_EXTERNAL) && (n_aux > 0))
	{
	PIMAGE_AUX_SYMBOL ast;
	char *search_lib;

	stab++;
	j++;
	ast = (PIMAGE_AUX_SYMBOL) stab;
	faddr += IMAGE_SIZEOF_AUX_SYMBOL;
	switch (ast->Sym.Misc.TotalSize)
	    {
	    case 1:
		search_lib = "no ";
		break;

	    case 2:
		search_lib = "";
		break;

	    default:
		{
		fprintf(outfile, "??? weak sym aux record has bad search flag value of %8.8x\n",
		    ast->Sym.Misc.TotalSize);
		n_errors++;
		search_lib = "? ";
		}
	    }

	fprintf(outfile, "%8.8x %4.4x Aux %3.3d: wk default = %x (%slib search)\n",
	    faddr, j, 1, ast->Sym.TagIndex, search_lib);
	if (ast->Sym.TagIndex > file_header.NumberOfSymbols)
	    {
	    fprintf(outfile, "??? default symbol index is out of range\n");
	    n_errors++;
	    }
	}
    else
	for (k = 1; k <= n_aux; k++)
	    {
	    PIMAGE_AUX_SYMBOL ast;

	    stab++;
	    j++;
	    ast = (PIMAGE_AUX_SYMBOL) stab;
	    faddr += IMAGE_SIZEOF_AUX_SYMBOL;
	    fprintf(outfile, "%8.8x %4.4x Aux %3.3d: ", faddr, j, k);
	    if ((base_class == IMAGE_SYM_CLASS_STATIC) &&
		    (base_type == IMAGE_SYM_TYPE_NULL))
		{
		fprintf(outfile, "Length = %x(%d); # Relocs = %x(%d); # Line Nums = %x(%d)",
		    two(ast->Section.Length),
		    two(ast->Section.NumberOfRelocations),
		    two(ast->Section.NumberOfLinenumbers));
		if (base_section == 0)
		    fprintf(outfile, "\n");
		else
		    {
		    if (sec_header[base_section].Characteristics &
			    IMAGE_SCN_LNK_COMDAT)
			{
			char *sel_type;

			switch (ast->Section.Selection)
			    {
			    case IMAGE_COMDAT_SELECT_UNKNOWN:
				sel_type = "unknown";
				break;

			    case IMAGE_COMDAT_SELECT_NODUPLICATES:
				sel_type = "no duplicates";
				break;

			    case IMAGE_COMDAT_SELECT_ANY:
				sel_type = "any";
				break;

			    case IMAGE_COMDAT_SELECT_SAME_SIZE:
				sel_type = "same size";
				break;

			    case IMAGE_COMDAT_SELECT_EXACT_MATCH:
				sel_type = "exact match";
				break;

			    case IMAGE_COMDAT_SELECT_ASSOCIATIVE:
				sel_type = "associative";
				break;

			    default:
				sel_type = "??? bad type";
				n_errors++;
			    }

			fprintf(outfile, ";\n                       ");
			fprintf(outfile, "Checksum = %8.8x; Selection = %2x(%s)",
			    ast->Section.CheckSum,
			    ast->Section.Selection,
			    sel_type);
			if (ast->Section.Selection ==
				IMAGE_COMDAT_SELECT_ASSOCIATIVE)
			    {
			    fprintf(outfile, "; Assoc. sec = %d\n",
				ast->Section.Number);
			    if ((ast->Section.Number < 1) ||
				(ast->Section.Number >
				    file_header.NumberOfSections))
				{
				fprintf(outfile, "??? bad associated section number\n");
				n_errors++;
				}
			    }
			else
			    fprintf(outfile, "\n");
			}
		    else
			fprintf(outfile, "\n");

		    if (ast->Section.Length !=
			    sec_header[base_section].SizeOfRawData)
			{
			fprintf(outfile, "??? length does not match section header\n");
			n_errors++;
			}

		    if (ast->Section.NumberOfRelocations !=
			    sec_header[base_section].NumberOfRelocations)
			{
			fprintf(outfile, "??? number of relocations does not match section header\n");
			n_errors++;
			}

		    if (ast->Section.NumberOfLinenumbers !=
			    sec_header[base_section].NumberOfLinenumbers)
			{
			fprintf(outfile, "??? number of line numbers does not match section header\n");
			n_errors++;
			}
		    }
		}
	    else
		{
		fprintf(outfile, "Tag = %x", ast->Sym.TagIndex);
		if (ISTAG(BTYPE(base_type)))
		    fprintf(outfile, "; Line# = %d; Size = %d",
			ast->Sym.Misc.LnSz.Linenumber,
			ast->Sym.Misc.LnSz.Size);
		else if (base_class == IMAGE_SYM_CLASS_FUNCTION)
		    fprintf(outfile, "; Line# = %d",
			ast->Sym.Misc.TotalSize);
		else
		    fprintf(outfile, "; Size = %d", ast->Sym.Misc.TotalSize);

		if (ISARY(base_type))
		    fprintf(outfile, "; Dimensions = (%d, %d, %d, %d)\n",
			ast->Sym.FcnAry.Array.Dimension[0],
			ast->Sym.FcnAry.Array.Dimension[1],
			ast->Sym.FcnAry.Array.Dimension[2],
			ast->Sym.FcnAry.Array.Dimension[3]);
		else
		    {
		    fprintf(outfile, "; ->Line# = %x; ->Next Func = %x\n",
			ast->Sym.FcnAry.Function.PointerToLinenumber,
			ast->Sym.FcnAry.Function.PointerToNextFunction);
		    if (base_class != IMAGE_SYM_CLASS_FUNCTION)
			if ((ast->Sym.TagIndex >= 0) &&
				(ast->Sym.TagIndex <
				    file_header.NumberOfSymbols))
			    {
			    PIMAGE_SYMBOL s = &st[ast->Sym.TagIndex];

			    if ((strncmp((char *) &s->N.ShortName[0], ".bf", 8) !=
					0) ||
				    (s->StorageClass !=
					IMAGE_SYM_CLASS_FUNCTION))
				{
				fprintf(outfile,
				    "??? function aux tag does not point to .bf symbol\n");
				n_errors++;
				}
			    }
			else
			    {
			    fprintf(outfile, "??? function aux tag is out of range\n");
			    n_errors++;
			    }
		    }

		if (ast->Sym.TvIndex != 0)
		    {
		    fprintf(outfile, "??? Non-zero aux record TvIndex: %x\n",
			ast->Sym.TvIndex);
		    n_errors++;
		    }
		}
	    }

    /* logic checks for comdats */

    if ((base_section > 0) &&
	    (sec_info[base_section].is_comdat) &&
	    (base_class != IMAGE_SYM_CLASS_FUNCTION) &&
	    (base_class != IMAGE_SYM_CLASS_LABEL))
	if ((base_class == IMAGE_SYM_CLASS_STATIC) &&
		(base_type == IMAGE_SYM_TYPE_NULL) &&
		(n_aux > 0))
	    sec_info[base_section].sec_sym_seen = 1;
	else
	    {
	    if (!sec_info[base_section].sec_sym_seen)
		{
		fprintf(outfile, "??? comdat symbol precedes comdat's section symbol in symbol table\n");
		n_errors++;
		}

	    if (sec_info[base_section].comdat_sym_seen)
		{
		fprintf(outfile, "??? more than one comdat symbol for section %d\n",
		    base_section);
		n_errors++;
		}

	    sec_info[base_section].comdat_sym_seen = 1;
	    }

    stab++;
    faddr += IMAGE_SIZEOF_SYMBOL;
    }

/* If we have linenum records, the first symbol must be a .file symbol for
   ntsd and cdb to work properly */

if ((has_linenums == 1) && (first_sym_is_file == 0))
    {
    fprintf(outfile, "??? linenums present but first sym is not .file\n");
    n_errors++;
    }

/* Check that .file symbols are on a singly-linked list */

/* Find the first file symbol. */

stab = st;
for (j = 0; j < file_header.NumberOfSymbols; j++)
    {
    int k;
    int n_aux;

    if (stab->StorageClass == IMAGE_SYM_CLASS_FILE)
	break;

    n_aux = stab->NumberOfAuxSymbols;
    stab++;
    for (k = 0; (k < n_aux) && (j+k+1 < file_header.NumberOfSymbols); k++)
	{
	stab++;
	j++;
	}
    }

/* Now trace the list. */

i = 0;
if (j < file_header.NumberOfSymbols)
    do
	{
	if (stab->StorageClass != IMAGE_SYM_CLASS_FILE)
	    {
	    fprintf(outfile,
		"??? file symbol at index %x points to non-file symbol at %x\n",
		i, j);
	    n_errors++;
	    break;
	    }

	if (stab->N.ShortName[0] == 1)
	    {
	    fprintf(outfile,
		"??? circular link in file symbol list from %x to %x\n", i, j);
	    n_errors++;
	    break;
	    }

	stab->N.ShortName[0] = 1; /* leave a bread crumb */
	i = j;
	j = stab->Value;
	stab = 0;
	if ((j > 0) && (j < file_header.NumberOfSymbols))
	    stab = &st[j];
	else
	    if (j != 0)
		{
		fprintf(outfile,
		    "??? file symbol at index %x has out-of-range link value %x\n",
		    i, j);
		n_errors++;
		}
	}
    while (stab);

/* All file symbols now should have bread crumbs in their name. */

stab = st;
for (j = 0; j < file_header.NumberOfSymbols; j++)
    {
    int k;
    int n_aux;

    if ((stab->StorageClass == IMAGE_SYM_CLASS_FILE) &&
	    (stab->N.ShortName[0] != 1))
	{
	fprintf(outfile,
	    "??? file symbol at index %x is not in the file symbol list\n", j);
	n_errors++;
	}

    n_aux = stab->NumberOfAuxSymbols;
    stab++;
    for (k = 0; (k < n_aux) && (j+k+1 < file_header.NumberOfSymbols); k++)
	{
	stab++;
	j++;
	}
    }

/* Trace the list for function symbols */

stab = st;
for (j = 0; j < file_header.NumberOfSymbols; j++)
    {
    int k;
    int n_aux;

    if ((ISFCN(stab->Type)) && (stab->NumberOfAuxSymbols >= 1))
	break;

    n_aux = stab->NumberOfAuxSymbols;
    stab++;
    for (k = 0; (k < n_aux) && (j+k+1 < file_header.NumberOfSymbols); k++)
	{
	stab++;
	j++;
	}
    }

i = 0;
if (j < file_header.NumberOfSymbols)
    do
	{
	PIMAGE_AUX_SYMBOL ast;

	if (stab->Type == 0xffff)
	    {
	    fprintf(outfile,
		"??? circular link in file symbol list from %x to %x\n", i, j);
	    n_errors++;
	    break;
	    }

	if (!(ISFCN(stab->Type)) || (stab->NumberOfAuxSymbols < 1))
	    {
	    fprintf(outfile,
		"??? function symbol at index %x points to non-function symbol at %x\n",
		i, j);
	    n_errors++;
	    break;
	    }

	stab->Type = 0xffff; /* leave a bread crumb */
	i = j;
	ast = (PIMAGE_AUX_SYMBOL) (stab + 1);
	j = ast->Sym.FcnAry.Function.PointerToNextFunction;
	stab = 0;
	if ((j > 0) && (j < file_header.NumberOfSymbols))
	    stab = &st[j];
	else
	    if (j != 0)
		{
		fprintf(outfile,
		    "??? function symbol at index %x has out-of-range link value %x\n",
		    i, j);
		n_errors++;
		}
	}
    while (stab);

stab = st;
for (j = 0; j < file_header.NumberOfSymbols; j++)
    {
    int k;
    int n_aux;

    if ((ISFCN(stab->Type)) && (stab->NumberOfAuxSymbols >= 1) &&
	    (stab->SectionNumber != IMAGE_SYM_UNDEFINED) &&
	    (stab->SectionNumber != IMAGE_SYM_DEBUG))
	{
	fprintf(outfile,
	    "??? function symbol at index %x is not in the function symbol list\n", j);
	n_errors++;
	}

    n_aux = stab->NumberOfAuxSymbols;
    stab++;
    for (k = 0; (k < n_aux) && (j+k+1 < file_header.NumberOfSymbols); k++)
	{
	stab++;
	j++;
	}
    }

/* Trace the list for .bf symbols */

stab = st;
for (j = 0; j < file_header.NumberOfSymbols; j++)
    {
    int k;
    int n_aux;

    if ((stab->StorageClass == IMAGE_SYM_CLASS_FUNCTION) &&
	    (strncmp((char *) &stab->N.ShortName[0], ".bf", 8) == 0))
	break;

    n_aux = stab->NumberOfAuxSymbols;
    stab++;
    for (k = 0; (k < n_aux) && (j+k+1 < file_header.NumberOfSymbols); k++)
	{
	stab++;
	j++;
	}
    }

i = 0;
if (j < file_header.NumberOfSymbols)
    do
	{
	PIMAGE_AUX_SYMBOL ast;

	if (stab->N.ShortName[0] == 2)
	    {
	    fprintf(outfile,
		"??? circular link in .bf symbol list from %x to %x\n", i, j);
	    n_errors++;
	    break;
	    }

	if ((stab->StorageClass != IMAGE_SYM_CLASS_FUNCTION) ||
		(strncmp((char *) &stab->N.ShortName[0], ".bf", 8) != 0))
	    {
	    fprintf(outfile,
		"??? .bf symbol at index %x points to non-.bf symbol at %x\n",
		i, j);
	    n_errors++;
	    break;
	    }

	stab->N.ShortName[0] = 2; /* leave a bread crumb */
	i = j;
	ast = (PIMAGE_AUX_SYMBOL) (stab + 1);
	j = ast->Sym.FcnAry.Function.PointerToNextFunction;
	stab = 0;
	if ((j > 0) && (j < file_header.NumberOfSymbols))
	    stab = &st[j];
	else
	    if (j != 0)
		{
		fprintf(outfile,
		    "??? .bf symbol at index %x has out-of-range link value %x\n",
		    i, j);
		n_errors++;
		}
	}
    while (stab);

stab = st;
for (j = 0; j < file_header.NumberOfSymbols; j++)
    {
    int k;
    int n_aux;

    if ((stab->StorageClass == IMAGE_SYM_CLASS_FUNCTION) &&
	    (strncmp((char *) &stab->N.ShortName[0], ".bf", 8) == 0))
	{
	fprintf(outfile,
	    "??? .bf symbol at index %x is not in the .bf symbol list\n", j);
	n_errors++;
	}

    n_aux = stab->NumberOfAuxSymbols;
    stab++;
    for (k = 0; (k < n_aux) && (j+k+1 < file_header.NumberOfSymbols); k++)
	{
	stab++;
	j++;
	}
    }

fprintf(outfile, "\n");
}


/* Determine the output file name based on an input file name.  Strip off
   the current file type and replace it with the one given.  Return a
   malloc()'d buffer containing this string. */

static char *get_outfile_name(char *infile_name, char *outfile_type)
{
char *outbuff;
int i;

for (i = strlen(infile_name) - 1; i >= 0; i--)
    if (infile_name[i] == '.')
	break;

if (i < 0)
    {
    outbuff = malloc(strlen(infile_name) + strlen(outfile_type) + 1);
    strcpy(outbuff, infile_name);
    }
else
    {
    outbuff = malloc(i + strlen(outfile_type) + 1);
    outbuff[0] = 0;
    strncat(outbuff, infile_name, i);
    }

strcat(outbuff, outfile_type);
return outbuff;
}


/* Check for overlap between two ranges of numbers */

static int overlaps( ULONG b1, ULONG s1, ULONG b2, ULONG s2)
{
ULONG h1 = b1 + s1 - 1;
ULONG h2 = b2 + s2 - 1;

/* no overlap if any of the base or pointer values is zero */

if ((b1 ==0) || (s1 == 0) || (b2 == 0) || (s2 == 0))
    return 0;

return (((b1 >= b2) && (b1 <= h2)) ||
    ((h1 >= b2) && (h1 <= h2)) ||
    ((b1 <= b2) && (h1 >= h2)));
}


/* Check for overlapping file allocations between two sections */

static void check_section_overlap(int i, int j)
{
if (overlaps(sec_header[i].PointerToRawData, sec_header[i].SizeOfRawData,
	sec_header[j].PointerToRawData, sec_header[j].SizeOfRawData))
    {
    fprintf(outfile, "??? raw data overlap between sections %x and %x\n",
	i, j);
    n_errors++;
    }

if (overlaps(sec_header[i].PointerToRawData, sec_header[i].SizeOfRawData,
	sec_header[j].PointerToRelocations,
	sec_header[j].NumberOfRelocations * IMAGE_SIZEOF_RELOCATION))
    {
    fprintf(outfile, "??? section %x raw data overlaps section %x relocations\n",
	i, j);
    n_errors++;
    }

if (overlaps(sec_header[i].PointerToRawData, sec_header[i].SizeOfRawData,
	sec_header[j].PointerToLinenumbers,
	sec_header[j].NumberOfLinenumbers * IMAGE_SIZEOF_LINENUMBER))
    {
    fprintf(outfile, "??? section %x raw data overlaps section %x line numbers\n",
	i, j);
    n_errors++;
    }

if (overlaps(sec_header[i].PointerToRelocations,
	sec_header[i].NumberOfRelocations * IMAGE_SIZEOF_RELOCATION,
	sec_header[j].PointerToRawData, sec_header[j].SizeOfRawData))
    {
    fprintf(outfile, "??? section %x relocations overlap section %x raw data\n",
	i, j);
    n_errors++;
    }

if (overlaps(sec_header[i].PointerToRelocations,
	sec_header[i].NumberOfRelocations * IMAGE_SIZEOF_RELOCATION,
	sec_header[j].PointerToRelocations,
	sec_header[j].NumberOfRelocations * IMAGE_SIZEOF_RELOCATION))
    {
    fprintf(outfile, "??? relocations overlap for sections %x and %x\n",
	i, j);
    n_errors++;
    }

if (overlaps(sec_header[i].PointerToRelocations,
	sec_header[i].NumberOfRelocations * IMAGE_SIZEOF_RELOCATION,
	sec_header[j].PointerToLinenumbers,
	sec_header[j].NumberOfLinenumbers * IMAGE_SIZEOF_LINENUMBER))
    {
    fprintf(outfile, "??? section %x relocations overlap section %x line numbers\n",
	i, j);
    n_errors++;
    }

if (overlaps(sec_header[i].PointerToLinenumbers,
	sec_header[i].NumberOfLinenumbers * IMAGE_SIZEOF_LINENUMBER,
	sec_header[j].PointerToRawData, sec_header[j].SizeOfRawData))
    {
    fprintf(outfile, "??? section %x line numbers overlap section %x raw data\n",
	i, j);
    n_errors++;
    }

if (overlaps(sec_header[i].PointerToLinenumbers,
	sec_header[i].NumberOfLinenumbers * IMAGE_SIZEOF_LINENUMBER,
	sec_header[j].PointerToRelocations,
	sec_header[j].NumberOfRelocations * IMAGE_SIZEOF_RELOCATION))
    {
    fprintf(outfile, "??? section %x line numbers overlap section %x relocations\n",
	i, j);
    n_errors++;
    }

if (overlaps(sec_header[i].PointerToLinenumbers,
	sec_header[i].NumberOfLinenumbers * IMAGE_SIZEOF_LINENUMBER,
	sec_header[j].PointerToLinenumbers,
	sec_header[j].NumberOfLinenumbers * IMAGE_SIZEOF_LINENUMBER))
    {
    fprintf(outfile, "??? line numbers overlap for sections %x and %x\n",
	i, j);
    n_errors++;
    }
}


/* Insert a segment into the segment table, sorted in increasing order by size*/

static void insert_seg(ULONG ptr, ULONG size)
{
ULONG h = ptr + size - 1;
int i;
int j;

/* leave if this is a null segment */

if ((ptr == 0) || (size == 0))
    return;

/* find where this one goes */

for (i = 0; i < n_segs; i++)
    if (((ptr >= seg[i].base) && (ptr <= seg[i].high)) ||
	    ((h >= seg[i].base) && (h <= seg[i].high)) ||
	    ((ptr <= seg[i].base) && (h >= seg[i].high)))
	{
	/* overlap */

	seg[i].base = ((ptr < seg[i].base) ? ptr : seg[i].base);
	seg[i].high = ((h > seg[i].high) ? h : seg[i].high);
	return;
	}
    else if (ptr < seg[i].base)
	break;

/* i is the index into the seg array where the new segment should go.  Shift
   all others up one */

for (j = n_segs; j > i; j--)
    {
    seg[j].base = seg[j - 1].base;
    seg[j].high = seg[j - 1].high;
    }

seg[i].base = ptr;
seg[i].high = h;
n_segs++;
}

int main(int argc, char *argv[], char *envp)
{
char *outfile_name;
char *infile_name = argv[1];
char *funcname = "main";
ULONG signature;
int signature_size;
ULONG total_header_size;
ULONG total_symtab_size;
int i;
int j;
int seg_size;

#if VMS
if ((infile = fopen(infile_name, "rb", "ctx=stm")) == NULL)
#else
if ((infile = fopen(infile_name, "rb")) == NULL)
#endif
    {
    printf("??? Error opening input file \"%s\"\n", infile_name);
    exit(0);
    }


outfile_name = get_outfile_name(infile_name, ".anl");
if ((outfile = fopen(outfile_name, "w")) == NULL)
    {
    printf("??? Error opening output file \"%s\"\n", outfile_name);
    exit(0);
    }

free(outfile_name);
fprintf(outfile, "Analysis of NT Object file %s\n\n", infile_name);

signature_size = sizeof(ULONG);
if (!dfread(infile, sizeof(ULONG), &signature, funcname))
    {
    fprintf(outfile, "??? error reading file signature\n");
    printf("??? error reading file signature\n");
    exit(0);
    }

if (signature == IMAGE_DOS_SIGNATURE)
    {
    fprintf(outfile, "This is a DOS executable image\n");
    report_errors();
    exit(0);
    }
else if (signature == IMAGE_OS2_SIGNATURE)
    {
    fprintf(outfile, "This is an OS/2 NE-format image\n");
    report_errors();
    exit(0);
    }
else if (signature == IMAGE_OS2_SIGNATURE_LE)
    {
    fprintf(outfile, "This is an OS/2 LE-format image\n");
    report_errors();
    exit(0);
    }
else if (signature == IMAGE_NT_SIGNATURE)
    fprintf(outfile, "This is a NT image\n");
else
    /* no signature (an object file); back up */

    signature_size = 0;
    if (!dfseek(infile, 0))
	{
	fprintf(outfile, "??? error seeking to start of file header\n");
	printf("??? error seeking to start of file header\n");
	exit(0);
	}

if (!process_file_header())
    {
    report_errors();
    exit(0);
    }

if (file_header.SizeOfOptionalHeader)
    {
    fprintf(outfile, "\n");
    if (!process_opt_header())
	{
	report_errors();
	exit(0);
	}
    }

/* read in the section table */

if (file_header.NumberOfSections)
    {
    int i;

    if ((sec_header = (PIMAGE_SECTION_HEADER)
			calloc(file_header.NumberOfSections + 1,
			    IMAGE_SIZEOF_SECTION_HEADER)) == NULL)
	{
	fprintf(outfile, "??? calloc failed for %d section headers\n",
	    file_header.NumberOfSections);
	printf("??? calloc failed for %d section headers\n",
	    file_header.NumberOfSections);
	report_errors();
	exit(0);
	}

    if ((sec_info = (struct sec_info_s *) calloc(file_header.NumberOfSections + 1,
					    sizeof(struct sec_info_s))) == NULL)
	{
	ULONG n = (file_header.NumberOfSections + 1) *
		    sizeof(struct sec_info_s);

	fprintf(outfile, "??? calloc failed for %d bytes of memory for section info array\n",
	    n);
	printf("??? calloc failed for %d bytes of memory for section info array\n",
	    n);
	report_errors();
	exit(0);
	}

    if (!dfread(infile,
	    IMAGE_SIZEOF_SECTION_HEADER * file_header.NumberOfSections,
	    &sec_header[1], funcname))
	{
	fprintf(outfile, "??? error reading section headers\n");
	printf("??? error reading section headers\n");
	exit(0);
	}
    }

/* Read in the string table.  Seek to the spot right past the symbol table,
   read the next 4 bytes (which contain the string table length as a binary
   value), then allocate and read the table itself. */

if (file_header.PointerToSymbolTable == 0)
    {
    fprintf(outfile, "??? file header symbol table pointer is zero\n");
    n_errors++;
    stringtab_size = 0;
    }
else
    {
    int seek_loc =  file_header.PointerToSymbolTable +
	    file_header.NumberOfSymbols * IMAGE_SIZEOF_SYMBOL;

    if (!dfseek(infile, seek_loc))
	{
	fprintf(outfile, "??? error seeking to start of string table at file address %x\n",
	    seek_loc);
	n_errors++;
	stringtab_size = 0;
	}
    else
	if (!dfread(infile, 4, &stringtab_size, funcname))
	    {
	    fprintf(outfile, "??? error reading string table size\n");
	    n_errors++;
	    stringtab_size = 0;
	    }

    fprintf(outfile, "\nString table size: %s\n", flong(stringtab_size));
    if (stringtab_size != 0)
	if (stringtab_size < 4)
	    {
	    fprintf(outfile, "??? bad string table size\n");
	    n_errors++;
	    stringtab_size = 0;
	    }
	else
	    stringtab_size -= 4;

    if (stringtab_size != 0)
	{
	if ((stringtab = malloc(stringtab_size)) == NULL)
	    {
	    fprintf(outfile, "??? malloc of buffer for string table failed\n");
	    n_errors++;
	    printf("??? malloc of buffer for string table failed\n");
	    stringtab = 0;
	    stringtab_size = 0;
	    }
	else if (!dfread(infile, stringtab_size, stringtab, funcname))
	    {
	    fprintf(outfile, "??? read of string table failed\n");
	    n_errors++;
	    free(stringtab);
	    stringtab = 0;
	    stringtab_size = 0;
	    }
	}
    }

/* Check for overlaps between section data in the file, and for gaps */

total_header_size = signature_size + IMAGE_SIZEOF_FILE_HEADER +
    file_header.SizeOfOptionalHeader +
    (IMAGE_SIZEOF_SECTION_HEADER * file_header.NumberOfSections);
total_symtab_size = file_header.NumberOfSymbols * IMAGE_SIZEOF_SYMBOL +
    stringtab_size + 4;
seg_size = sizeof(struct mem_seg) * (file_header.NumberOfSections * 3 + 2);
if ((seg = malloc(seg_size)) == NULL)
    {
    fprintf(outfile, "??? unable to allocate %d bytes for memory segment table\n",
	seg_size);
    n_errors++;
    }
else
    {
    ULONG last_faddr;

    /* set up segments describing headers and symbol table */

    seg[0].base = 0;
    seg[0].high = total_header_size - 1;
    n_segs = 1;
    if (file_header.PointerToSymbolTable)
	{
	seg[1].base = file_header.PointerToSymbolTable;
	seg[1].high = seg[1].base + total_symtab_size - 1;
	n_segs++;
	}

    /* Insert segments for each section */

    for (i = 1; i <= file_header.NumberOfSections; i++)
	{
	insert_seg(sec_header[i].PointerToRawData, sec_header[i].SizeOfRawData);
	insert_seg(sec_header[i].PointerToRelocations,
	    sec_header[i].NumberOfRelocations * IMAGE_SIZEOF_RELOCATION);
	insert_seg(sec_header[i].PointerToLinenumbers,
	    sec_header[i].NumberOfLinenumbers * IMAGE_SIZEOF_LINENUMBER);
	}

    /* report any gaps, taking into account section rounding to quadword
       boundaries */

    last_faddr = seg[0].high;
    for (i = 1; i < n_segs; i++)
	{
	if ((seg[i].base > last_faddr) &&
		((seg[i].base - (last_faddr + 1)) > 15))
	    {
	    fprintf(outfile, "??? area in file not described by headers: %8.8x:%8.8x\n",
		last_faddr + 1, seg[i].base - 1);
	    n_errors++;
	    }

	last_faddr = seg[i].high;
	}
    }

if (seg_size)
    free(seg);

for (i = 1; i < file_header.NumberOfSections; i++)
    for (j = i + 1; j <= file_header.NumberOfSections; j++)
	check_section_overlap(i, j);

for (i = 1; i < file_header.NumberOfSections; i++)
    {
    if (overlaps(sec_header[i].PointerToRawData, sec_header[i].SizeOfRawData,
	    0, total_header_size))
	{
	fprintf(outfile, "??? section %d raw data overlaps headers\n", i);
	n_errors++;
	}

    if (overlaps(sec_header[i].PointerToRawData, sec_header[i].SizeOfRawData,
	    file_header.PointerToSymbolTable, total_symtab_size))
	{
	fprintf(outfile, "??? section %d raw data overlaps symbol table\n", i);
	n_errors++;
	}

    if (overlaps(sec_header[i].PointerToRelocations,
	    sec_header[i].NumberOfRelocations * IMAGE_SIZEOF_RELOCATION,
	    0, total_header_size))
	{
	fprintf(outfile, "??? section %d relocations overlap headers\n", i);
	n_errors++;
	}

    if (overlaps(sec_header[i].PointerToRelocations,
	    sec_header[i].NumberOfRelocations * IMAGE_SIZEOF_RELOCATION,
	    file_header.PointerToSymbolTable, total_symtab_size))
	{
	fprintf(outfile, "??? section %d relocations overlap symbol table\n",
	    i);
	n_errors++;
	}

    if (overlaps(sec_header[i].PointerToLinenumbers,
	    sec_header[i].NumberOfLinenumbers * IMAGE_SIZEOF_LINENUMBER,
	    0, total_header_size))
	{
	fprintf(outfile, "??? section %d line numbers overlap headers\n", i);
	n_errors++;
	}

    if (overlaps(sec_header[i].PointerToLinenumbers,
	    sec_header[i].NumberOfLinenumbers * IMAGE_SIZEOF_LINENUMBER,
	    file_header.PointerToSymbolTable, total_symtab_size))
	{
	fprintf(outfile, "??? section %d line numbers overlap symbol table\n",
	    i);
	n_errors++;
	}
    }

/* Read in the symbol table */

if (((st_size = file_header.NumberOfSymbols * IMAGE_SIZEOF_SYMBOL) != 0) &&
	(file_header.PointerToSymbolTable != 0))
    if ((st= malloc(st_size)) == NULL)
	{
	fprintf(outfile, "\n??? Unable to allocate %u byte buffer to hold symbol table\n",
	    st_size);
	n_errors++;
	}
    else if (!dfseek(infile, file_header.PointerToSymbolTable))
	{
	fprintf(outfile, "??? error seeking to symbol table\n");
	n_errors++;
	}
    else if (!dfread(infile, st_size, (char *) st, funcname))
	{
	fprintf(outfile, "??? error reading symbol table\n");
	n_errors++;
	}

/* Process each section */

for (i = 1;  i <= file_header.NumberOfSections;  i++)
    process_section_header(i);

/* Process the symbol table */

if (st)
    process_symbol_table();

report_errors();
exit(0);
}
