#ifndef _2D_H
#define _2D_H

#if  defined(__DLL__)
#  define _EXPORT _export
#else
#  define _EXPORT _CLASSTYPE
#endif


extern "C" {
#include "windows.h"
}


#include <stddef.h>
#include <string.h>
#include <math.h>

#include "cl_list.cpp"

#define  MEM_ERROR           "Unable to Allocate Memory"

#define  NAME_SIZE           20
#define  WIN_TITLE_SIZE      100
#define  WIN_PTR1            "Clumsy_Ptr1"
#define  WIN_PTR2            "Clumsy_Ptr2"

#define  BLACK               RGB( 0, 0, 0 )              // color black
#define  WHITE               RGB( 0xFF, 0xFF, 0xFF )     // color white
#define  RED                 RGB( 0xFF, 0, 0 )           // color red
#define  GREEN               RGB( 0, 0xFF, 0 )           // color green
#define  BLUE                RGB( 0, 0, 0xFF )           // color blue


extern const  HBRUSH NO_BRUSH;                           // null brush

extern HANDLE hInst;
extern HANDLE hPrevInst;
extern int    cmdShow;
extern LPSTR  cmdLine;

extern void * operator new(size_t size);
extern void operator delete(void * obj);


LRESULT CALLBACK _export WinProc( HWND hWnd, UINT iMessage,
                                  WPARAM wParam, LPARAM lParam );


typedef char boolean;


// window object class
class _EXPORT WinObj
{
public:
  // display a message with a stop icon
  virtual
  void        Error( LPSTR str );

  // Display a message with an exclamation icon
  virtual
  void        Warning( LPSTR str );

  // Display a message box
  virtual
  int         Message( UINT type, LPSTR str );
};


// class with a cleanup flag
// this class is inherited by any class that needs a flag to check for
// a cleanup before an object fo that class is destroyed.
class _EXPORT CleanUpCheck
{
protected:
  boolean     cleanUpFlag;

public:
              CleanUpCheck()                         { cleanUpFlag = FALSE; }
  void        CleanUp( boolean cleanFlag=TRUE )  { cleanUpFlag = cleanFlag; }
};


// class for defining coordinates of a point in space
// space is 2-dimentional in this version 
class _EXPORT Coord
{
public:
  float       x, y;

              Coord();           // default constructor initializes x = y = 0
              Coord( float xCoor, float yCoor );
  Coord       operator+( Coord& p );                  // return(this + p)
  Coord       operator-();                            // return(-this)
  Coord       operator-( Coord& p );                  // return(this - p)
  Coord       operator*( Coord& p );                  // return(this * p)
  Coord       Rot( float angle, Coord& p );           // return(this rotated
                                                      // by angle around p)         
  Coord&      operator+=( Coord& p );                 // this += p
  Coord&      operator-=( Coord& p );                 // this -= p
  Coord&      operator*=( Coord& p );                 // this *= p
  Coord&      RotBy( float angle, Coord& p );         // rotate this by angle
                                                      // around p
};


// Center is a global coordinate point representing (0,0)
extern Coord Center;


// abstract class for all graphic objects
class _EXPORT VisualObject : public WinObj
{
public:
  virtual     ~VisualObject()  {};                      // virtual destructor
  // definition of virtual methods that apply to all graphic objects
  virtual
  void        Translate( Coord& transValue ) = 0;
  virtual
  void        Rotate( float angle, Coord& center = Center ) = 0;
  virtual
  void        Scale( Coord& scale, Coord& center = Center) = 0;
  virtual
  void        Draw( HDC dc, float angle = 0 ) = 0;
};


// define GraphicList as a linked list of visual object references
typedef LinkedList<VisualObject&> GraphicList;


// class for graphic objects consisting of a collection of graphic objects
class _EXPORT ComplexGraphic : public VisualObject, public CleanUpCheck
{
private:
  GraphicList container;

public:
  virtual     ~ComplexGraphic();
  void        AddObject( VisualObject& vObj );                 // adds object
  void        Translate( Coord& transValue );
  void        Rotate( float angle, Coord& center = Center );
  void        Scale( Coord& scale, Coord& center = Center );
  void        Draw( HDC dc, float angle = 0 );
};


// class for a space system for graphics 
class _EXPORT Space : public ComplexGraphic
{
private:
  long        infinity;                           // not used in this version
};


// class for storing Windows pens which are used to define the format
// for drawing lines and arcs
class _EXPORT PenObject
{
protected:                                   
  LOGPEN      lp;                           // pen information

public:
              PenObject( HPEN penValue );   // create the pen from an already
                                            // existing pen
              PenObject( COLORREF color );  // create a pen from a color
  void        SetPen( HPEN penValue );      // changes the pen to a new one
  void        SetPen( COLORREF color );     // changes the pen to a new color
  COLORREF    GetColor();                   // returns the color of a pen
  int         GetWidth();                   // returns the width of a pen
  UINT        GetStyle();                   // returns the style of a pen
  HPEN        ActivatePen( HDC dc );        // activates the pen on a device
                                            // context
  void        DeactivatePen( HDC dc, HPEN originalPen );
                                            // removes the pen from a device
                                            // context
};


// class for storing Windows brushes which are used to define the format
// for filling areas
class _EXPORT BrushObject
{
protected:
  LOGBRUSH    lb;
public:
              BrushObject( HBRUSH brushValue );
              BrushObject( COLORREF color );
  void        SetBrush( HBRUSH brushValue );
  void        SetBrush( COLORREF color );
  COLORREF    GetColor();
  int         GetHatch();
  UINT        GetStyle();
  HBRUSH      ActivateBrush( HDC dc );
  void        DeactivateBrush( HDC dc, HPEN originalBrush );
};


// graphic class for points
class _EXPORT Point : public PenObject, public VisualObject, protected Coord
{
public:
              Point( float x, float y, HPEN pen ); 
              Point( float x, float y, COLORREF color=BLACK );
  void        Translate( Coord& transValue );
  void        Rotate( float angle, Coord& center = Center );
  void        Scale( Coord& scale, Coord& center = Center );
  void        Draw( HDC dc, float angle = 0 );
  Coord&      GetCoord();
};


// graphic class for lines
class _EXPORT Line : public PenObject, public VisualObject
{
protected:
  Coord       p1, p2;

public:

              Line( Coord& p1, Coord& p2, COLORREF color=BLACK );
              Line( Coord& p1, Coord& p2, HPEN pen );
  void        Translate( Coord& transValue );
  void        Rotate( float angle, Coord& center = Center );
  void        Scale( Coord& scale, Coord& center = Center );
  void        Draw( HDC dc, float angle = 0 );
  Coord&      GetCoord1();
  Coord&      GetCoord2();
};


// graphic class for rectangles
class _EXPORT CRectangle : public BrushObject, public Line
{
private:
  void        Init( float angle );

protected:
  Coord       p3, p4;

public:
              CRectangle( Coord& topLeft, Coord& bottomRight, float angle,
                          HPEN pen, HBRUSH brush=NO_BRUSH);
              CRectangle( Coord& topLeft, Coord& bottomRight, float angle,
                          HPEN pen, COLORREF brushColor );
              CRectangle( Coord& topLeft, Coord& bottomRight, float angle,
                          COLORREF penColor, HBRUSH brush=NO_BRUSH );
              CRectangle( Coord& topLeft, Coord& bottomRight, float angle,
                          COLORREF penColor, COLORREF brushColor );
  void        Translate( Coord& transValue );
  void        Rotate( float angle, Coord& center = Center );
  void        Scale( Coord& scale, Coord& center = Center );
  void        Draw( HDC dc, float angle = 0 );
  Coord&      GetCoord3();
  Coord&      GetCoord4();
};


// graphic class for ellipses
class _EXPORT CEllipse : public CRectangle
{
public:
              CEllipse( Coord& topLeft, Coord& bottomRight, HPEN pen,
                        HBRUSH brush=NO_BRUSH );
              CEllipse( Coord& topLeft, Coord& bottomRight, HPEN pen,
                        COLORREF brushColor );
              CEllipse( Coord& topLeft, Coord& bottomRight, COLORREF penCol,
                        HBRUSH brush = NO_BRUSH);
              CEllipse( Coord& topLeft, Coord& bottomRight, COLORREF pCol,
                        COLORREF brushColor );
  void        Draw( HDC dc, float angle = 0 );
};


// graphic class for circles
class _EXPORT Circle : public CEllipse
{
private:
  Coord       center;
  float       radius;
  void        RecalcEllipse( Coord& center, float radius );

public:
              Circle( Coord& center, float radius, HPEN pen,
                      HBRUSH brush=NO_BRUSH );
              Circle( Coord& center, float radius, HPEN pen,
                      COLORREF brushColor );
              Circle( Coord& center, float radius, COLORREF penColor,
                      HBRUSH brush=NO_BRUSH );
              Circle( Coord& center, float radius, COLORREF penColor,
                      COLORREF brushColor );
};


// graphic class for chords
class _EXPORT CChord : public CEllipse
{
protected:
  Coord       line1, line2;

public:
              CChord( Coord& topLeft, Coord& bottomRight,
                      Coord& line1, Coord& line2,
                      HPEN pen, HBRUSH brush=NO_BRUSH );
              CChord( Coord& topLeft, Coord& bottomRight,
                      Coord& line1, Coord& line2,
                      HPEN pen, COLORREF brushColor );
              CChord( Coord& topLeft, Coord& bottomRight,
                      Coord& line1, Coord& line2,
                      COLORREF penColor, HBRUSH brush=NO_BRUSH );
              CChord( Coord& topLeft, Coord& bottomRight,
                      Coord& line1, Coord& line2,
                      COLORREF penCol, COLORREF brushColor );
  void        Translate( Coord& transValue );
  void        Rotate( float angle, Coord& center = Center );
  void        Scale( Coord& scale, Coord& center = Center );
  void        Draw( HDC dc, float angle = 0 );
};


// graphic class for arcs
class _EXPORT CArc : public CChord
{
public:
              CArc( Coord& topLeft, Coord& bottomRight,
                    Coord& line1, Coord& line2, HPEN pen );
              CArc( Coord& topLeft, Coord& bottomRight,
                    Coord& line1, Coord& line2, COLORREF penColor=BLACK );
  void        Draw( HDC dc, float angle = 0 );
};


// graphic class for pies
class _EXPORT CPie : public CChord
{
public:
              CPie( Coord& topLeft, Coord& bottomRight,
                    Coord& line1, Coord& line2,
                    HPEN pen, HBRUSH brush=NO_BRUSH );
              CPie( Coord& topLeft, Coord& bottomRight,
                    Coord& line1, Coord& line2,
                    HPEN pen, COLORREF brushColor );
              CPie( Coord& topLeft, Coord& bottomRight,
                    Coord& line1, Coord& line2,
                    COLORREF penColor, HBRUSH brush=NO_BRUSH );
              CPie( Coord& topLeft, Coord& bottomRight,
                    Coord& line1, Coord& line2,
                    COLORREF penColor, COLORREF brushColor );
  void        Draw( HDC dc, float angle = 0 );
};


// graphic class for Windows window classes
class _EXPORT WindowClass : public WinObj
{
private:
  char        name[NAME_SIZE];     // name of the class
  UINT        style;               // style of the class
  HICON       icon;                // icon for minimized window of this class
  HCURSOR     cursor;              // cursor for a window of this class
  HBRUSH      background;          // background for a window of this class
  LPCSTR      menuName;            // menu name for the menu of windows
  HINSTANCE   instance;            // instance of the application

public:
              WindowClass( char *className );
              ~WindowClass();
  boolean     Register();        // register the class
  boolean     UnRegister();      // unregister the class.  Done by destructor
  LPSTR       GetName();         // get the class name
  // the following setting have to be done before the class is registered
  // for them to take effect
  void        SetIcon( HICON newIcon );             // sets the minimize icon
  HICON       GetIcon();
  void        SetCursor( HCURSOR newCursor );       // sets the window cursor
  HCURSOR     GetCursor();
  void        SetBackground( HBRUSH newBack );      // sets the bkgnd color
  HBRUSH      GetBackground();
  void        SetMenu( LPCSTR newMenu );            // set the window menu
  LPCSTR      GetMenu();
  void        SetInstance( HINSTANCE newInstance ); // sets application
  HINSTANCE   GetInstance();  
};


// keyboard structure
typedef struct {
  int         repeatCount:16,
              scanCode:8,
              fExtended:1,
              reserved1:4,
              fAlt:1,
              fPrev:1,
              fTransition:1;
} KeyCodes;


// class for defining windows
class _EXPORT Window: public WinObj, public CleanUpCheck
{
private:
  HWND        handle;                        // Windows handle for the window
  char        title[WIN_TITLE_SIZE];         // title of the window
  LONG        style;                         // window style
  int         x, y;                          // position of the window
  int         width, height;                
  HWND        parent;                        // parent of the window
  HMENU       menu;                      

protected:
  static
  HANDLE      hAccel;                        // accelerator keys

public:
              Window();
              Window( HWND handle );         // create window object for
                                             // existing window
              ~Window();
  HWND        GetHandle();
  void        Create( WindowClass& wc );     // create the window

  virtual
  LONG        MessageProc( UINT msg, WPARAM wParam, LPARAM lParam );
  void        Move( RECT *rc, BOOL repaint=TRUE);
  BOOL        Show( int nCmdShow = SW_SHOW );
  LPSTR       GetTitle();
  int         SetTitle( LPSTR pBuffer );
  void        Close();
  void        Delete();

#ifdef __BORLANDC__
  #pragma warn -par
#endif
  // Virtual event methods that handle some of Windows events
  virtual BOOL  InitMenu( HMENU menu )                   { return( FALSE ); };
  virtual BOOL  QueryClose()                             { return( TRUE ); };
  virtual BOOL  Size( UINT width, UINT height, UINT type );
  virtual BOOL  Char( UINT value, UINT repeat )          { return( FALSE ); };
  virtual BOOL  KeyDown(UINT virtKey, KeyCodes keyCodes) { return( FALSE ); };
  virtual BOOL  LButtonDown( POINT cursor, UINT keys )   { return( FALSE ); };
  virtual BOOL  LButtonDblClk( POINT cursor, UINT keys ) { return( FALSE ); };
  virtual BOOL  MouseMove( POINT cursor, UINT keys )     { return( FALSE ); };
  virtual BOOL  MouseActivate( HWND hTop, UINT hitTest ) { return( FALSE ); };
  virtual BOOL  LButtonUp( POINT cursor, UINT keys )     { return( FALSE ); };
  virtual BOOL  RButtonDown( POINT cursor, UINT keys )   { return( FALSE ); };
  virtual BOOL  RButtonUp( POINT cursor, UINT keys )     { return( FALSE ); };
  virtual BOOL  Paint();
  virtual BOOL  SetFocus ( HWND hPrev )                  { return( FALSE ); };
  virtual BOOL  KillFocus ( HWND hNext )                 { return( FALSE ); };
  virtual BOOL  Command(UINT Id,UINT Code,HWND hControl);

#ifdef __BORLANDC__
  #pragma warn +par
#endif

};


extern boolean SetWindowPtr( HWND hWnd, void *ptr );
extern Window& GetWindowPtr( HWND hWnd );     // given window handle, returns
                                              // a pointer to the window
                                              // object


// class for defining graphics windows
class _EXPORT GraphicWindow : public Window
{
protected:
  Space&      space;                   // the space containing the graphics
                                       // to be displayed in this window
  float       viewAngle;               // angle of rotation for the view
  Coord       viewPoint;               // the point in space corresponding
                                       // to the center of the window
  int         bkMode;                  // Windows background mode
  boolean     centerAlways;            // if true, the space will be centered

public:
              GraphicWindow( Space& s );
              GraphicWindow( Space& s, HWND handle );
  void        Create( WindowClass& wc );
  void        SetViewPoint( Coord& vp );
  void        CenterView();
  void        CenterAlways(boolean centerFlag=TRUE) {centerAlways=centerFlag;}
  void        SetViewAngle( float angle );
  float       GetViewAngle();
  BOOL        Paint();
  int         GetBkMode();
  void        SetBkMode( int background );
};


// application class
class _EXPORT ClumsyApp : public WinObj
{
protected:
  virtual
  void        MessageLoop();
  int         status;                            // status of the application

public:
              ClumsyApp( HINSTANCE hInstance, HINSTANCE hPrevInstance,
                         LPSTR lpszCmdLine, int nCmdShow );
  virtual
  int         Run();                             // run the application
  int         GetStatus();                       // return the status
};


#endif