/* This file is DEBUG.C
**
** contains :
**
**	- debugging stuff for rsx
**
** Copyright (c) Rainer Schnitker '92 '93
*/

#include <stdio.h>
#include <stdlib.h>
#include <bios.h>
#include <string.h>
#include <ctype.h>
#include "DPMI.h"
#include "DPMIDOS.h"
#include "PROCESS.H"
#include "RSX.h"
#include "COPY32.h"
#include "START32.h"
#include "CDOSX32.h"
#include "ADOSX32.h"
#include "EXCEP32.h"
#include "DEBUG.H"
#include "FPU.H"
#include "syms.h"

typedef enum {
    Unknown, CONT, STEP, NEXT, LIST, DUMP, DOS,
    BREAK_SET, BREAK_LIST, BREAK_CLEAR, BREAK_ENABLE, BREAK_DISABLE,
    BREAK_DATA_READ, BREAK_DATA_WRITE,
    REGS, SET, WHERE, FIND, NPX, PROCESS, USER, SEL, HELP, QUIT
};

typedef struct {
    char *cp;
    int t;
} item;

static item cmds[] =
{
    "go", CONT, "g", CONT,
    "cont", CONT, "c", CONT,
    "step", STEP, "s", STEP,
    "next", NEXT, "n", NEXT,
    "list", LIST, "l", LIST,
    "dump", DUMP, "d", DUMP,
    "reg", REGS, "r", REGS,
    "where", WHERE, "w", WHERE,
    "find", FIND, "f", FIND,
    "bp", BREAK_SET,
    "bc", BREAK_CLEAR,
    "bd", BREAK_DISABLE,
    "be", BREAK_ENABLE,
    "bl", BREAK_LIST,
    "bdr", BREAK_DATA_READ,
    "bdw", BREAK_DATA_WRITE,
    "set", SET,
    "process", PROCESS,
    "user", USER,
    "dos", DOS,
    "sel", SEL,
    "npx", NPX,
    "help", HELP, "h", HELP, "?", HELP,
    "quit", QUIT, "q", QUIT,
    0, 0
};

extern struct {
    char *name;
    int size;
    int ofs;
} regs[];

void memget(DWORD where, char *buf, WORD size)
{
    cpy32_16(DS, where, buf, (DWORD) size);
}

static void print_387_control_word(unsigned short control)
{
    printf("control %04X: ", control);
    printf("compute to ");
    switch ((control >> 8) & 3) {
    case 0:
	printf("24 bits; ");
	break;
    case 1:
	printf("(bad); ");
	break;
    case 2:
	printf("53 bits; ");
	break;
    case 3:
	printf("64 bits; ");
	break;
    }
    printf("round ");
    switch ((control >> 10) & 3) {
    case 0:
	printf("NEAREST; ");
	break;
    case 1:
	printf("DOWN; ");
	break;
    case 2:
	printf("UP; ");
	break;
    case 3:
	printf("CHOP; ");
	break;
    }
    if (control & 0x3f) {
	printf("\nmask:");
	if (control & 0x0001)
	    printf(" INVALID");
	if (control & 0x0002)
	    printf(" DENORM");
	if (control & 0x0004)
	    printf(" DIVZ");
	if (control & 0x0008)
	    printf(" OVERF");
	if (control & 0x0010)
	    printf(" UNDERF");
	if (control & 0x0020)
	    printf(" LOS");
	printf(";");
    }
    printf("\n");
    if (control & 0xe080)
	printf("warning: reserved bits on: %04X\n", control & 0xe080);
}

static void print_387_status_word(unsigned short status)
{
    printf("status %04X: ", status);
    if (status & 0xff) {
	printf("exceptions:");
	if (status & 0x0001)
	    printf(" INVALID");
	if (status & 0x0002)
	    printf(" DENORM");
	if (status & 0x0004)
	    printf(" DIVZ");
	if (status & 0x0008)
	    printf(" OVERF");
	if (status & 0x0010)
	    printf(" UNDERF");
	if (status & 0x0020)
	    printf(" LOS");
	if (status & 0x0040)
	    printf(" FPSTACK");
	printf("\n");
    }
    printf("flags: %d%d%d%d; ",
	   (status & 0x4000) != 0,
	   (status & 0x0400) != 0,
	   (status & 0x0200) != 0,
	   (status & 0x0100) != 0);

    printf("top %d\n", (status >> 11) & 7);
}


static void printregs()
{
    printf("cs:eip=%04X:%08lX eflags=%08lX  orgesp=%08lX\n"
	 "eax=%08lX ebx=%08lX ecx=%08lX edx=%08lX  ebp=%08lX esp=%08lX\n"
	   "esi=%08lX edi=%08lX  cs=%04X ds=%04X es=%04X ss=%04X fs=%04X gs=%04X\n",
	   CS, EIP, EFLAGS, ESPORG, EAX, EBX, ECX, EDX, EBP, ESP,
	   ESI, EDI, CS, DS, ES, SS, FS, GS);
}

static char prompt[] = "cmd:";

int get_input(char *str, int size)
{
    static char oldstr[40];
    char ins_flag;
    int strpos;
    int strend;

    unsigned int key, scan;
    unsigned char ascii = 0;
    int i;

    strpos = 0;
    strend = 0;
    ins_flag = 1;
    memset(str, 0, size);


    for (;;) {
	while (_bios_keybrd(kready))
	    _bios_keybrd(kread);
	key = _bios_keybrd(kread);
	scan = key >> 8;
	ascii = (unsigned char) (key & 0xff);

	if (ascii == 0xE0)
	    ascii = 0;

	if (ascii >= 32) {	/* show char */
	    if (strpos == strend) {
		putchar(ascii);
		*(str + strpos) = ascii;
		strend++;
	    } else if (ins_flag) {	/* shift right chars */
		for (i = strend; i >= strpos; --i)
		    *(str + i + 1) = *(str + i);
		*(str + strpos) = ascii;
		printf("%s",str + strpos);
		for (i = strend; i > strpos; i--)
		    putchar('\b');
		strend++;
	    } else {
		putchar(ascii);
		*(str + strpos) = ascii;
	    }
	    strpos++;
	    continue;
	}
	/* backspace or delete */
	else if ((ascii == '\b' && strpos != 0)
		 || (scan == 0x53 && strend != strpos)) {
	    if (ascii == '\b') {
		putchar('\b');
		strpos--;
	    }
	    for (i = strpos; i < strend; ++i)
		*(str + i) = *(str + i + 1);
	    strend--;
	    *(str + strend) = '\0';
	    printf("%s",str + strpos);
	    putchar(' ');
	    putchar('\b');
	    for (i = strend; i > strpos; i--)
		putchar('\b');
	    continue;
	} else if (ascii == 13) {
	    strcpy(oldstr, str);
	    break;
	} else if (ascii == 27) {
	    for (; strpos != 0; strpos--) {
		putchar('\b');
		putchar(' ');
		putchar('\b');
	    }
	    memset(str, 0, size);
	    strcpy(str, oldstr);
	    strpos = strend = 0;
	    continue;
	} else if (scan == 0x4b && strpos != 0) {
	    putchar('\b');
	    strpos--;
	    continue;
	} else if (scan == 0x48) {	/* up */
	    for (; strpos != 0; strpos--) {
		putchar('\b');
		putchar(' ');
		putchar('\b');
	    }
	    memset(str, 0, size);
	    strcpy(str, oldstr);
	    printf("%s",str);
	    strpos = strend = strlen(str);
	    continue;
	} else if (scan == 0x4d && strpos < strend) {	/* -> */
	    putchar(*(str + strpos));
	    strpos++;
	    continue;
	} else if (scan == 0x47) {	/* home */
	    for (; strpos != 0; strpos--)
		putchar('\b');
	    strpos = 0;
	    continue;
	} else if (scan == 0x4F) {	/* end */
	    printf("%s",str + strpos);
	    strpos = strend;
	    continue;
	} else if (scan == 0x52)/* ins */
	    ins_flag ^= 1;
    }
    return strend;
}

#define MAX_HARD        4
#define BP_FLAG_MASK    0x0FFF
#define BP_ENABLE       0x1000
#define BP_DISABLE      0x2000
#define BP_SYS          0x4000

typedef struct {
    DWORD address;
    WORD handle;
    WORD flags;
    WORD state;
} HardBreakPoint;
static HardBreakPoint hbp[MAX_HARD];

static int set_hard_break(DWORD at_address, WORD flags)
{
    int i;

    if (flags & BP_SYS)
	i = 0;
    else {
	for (i = 1; i < MAX_HARD; i++)
	    if (hbp[i].flags == 0)
		break;
	if (i == MAX_HARD) {
	    printf("no break free\n");
	    return -1;
	}
    }
    if (SetDebugWatchpoint(npz->memaddress + at_address,
			   flags & BP_FLAG_MASK, &hbp[i].handle)) {
	printf("error set\n");
	return -1;
    } else {
	hbp[i].address = at_address;
	hbp[i].flags = flags | BP_ENABLE;
	return i;
    }
}

static int del_hard_break(WORD no)
{
    if (no >= MAX_HARD || hbp[no].flags == 0) {
	printf("invalid break-no\n");
	return -1;
    }
    if (no == 0 && !(hbp[no].flags & BP_SYS)) {
	printf("break0 not for user\n");
	return -1;
    }
    if (ClearDebugWatchpoint(hbp[no].handle)) {
	printf("error clear\n");
	return -1;
    } else {
	hbp[no].address = 0;
	hbp[no].flags = 0;
	hbp[no].handle = 0;
	return 0;
    }
}

static int disable_hard_break(WORD no)
{
    if (no <= 1 && no >= MAX_HARD) {
	printf("ivalid break-no\n");
	return -1;
    }
    if (hbp[no].flags & BP_DISABLE) {
	printf("already disabled\n");
	return -1;
    }
    if (ClearDebugWatchpoint(hbp[no].handle)) {
	printf("error clear\n");
	return -1;
    } else {
	hbp[no].flags &= BP_FLAG_MASK;
	hbp[no].flags |= BP_DISABLE;
	return 0;
    }
}

static int enable_hard_break(WORD no)
{
    if (no <= 1 && no >= MAX_HARD) {
	printf("ivalid break-no\n");
	return -1;
    }
    if (hbp[no].flags & BP_ENABLE) {
	printf("already enabled\n");
	return -1;
    }
    if (SetDebugWatchpoint(npz->memaddress + hbp[no].address,
			hbp[no].flags & BP_FLAG_MASK, &hbp[no].handle)) {
	printf("error set\n");
	return -1;
    } else {
	hbp[no].flags &= BP_FLAG_MASK;
	hbp[no].flags |= BP_ENABLE;
	return 0;
    }
}

/*
** if the memory block base address has been changed (after DPMI:get memory),
** we must update all breakpoints to the new address
*/
void debug_update_breakpoints(void)
{
    WORD no;
    printf("update breakpoints\n");
    for (no = 0; no < MAX_HARD; no++) {
	if (hbp[no].flags & BP_ENABLE) {
	    printf("update break%d\n", no);
	    if (!ClearDebugWatchpoint(hbp[no].handle))
		SetDebugWatchpoint(npz->memaddress + hbp[no].address,
			  hbp[no].flags & BP_FLAG_MASK, &hbp[no].handle);
	}
    }
}


static DWORD next_inst;
static int cmd;
static union i387_union *npx;

/* entry after breakpoint,single step */

void debugger(char *filename, int deb_cmd)
{
    DWORD temp, list_inst;
    int lastjmp, lastj, i, j, resume_flag;
    char input[40];
    char *argv[4];
    char *name;

    /* if filename set, first call */
    if (deb_cmd == DEB_INIT) {
	for (i = 0; i < MAX_HARD; i++)
	    hbp[i].flags = 0;
	printf("debugging file %s\n", filename);
	syms_init(filename);
	if (copro==3)	    /* turn lookahead */
	    emu_switch(MATH_NEW, PTRACED);
    }
    /* cleanup */
    else if (deb_cmd == DEB_EXIT) {
	for (i = 0; i < MAX_HARD; i++)
	    if (hbp[i].flags & BP_ENABLE)
		del_hard_break(i);
	return;
    }
    /* trap exception */
    else if (deb_cmd == 1) {
	resume_flag = 0;
	for (i = 1; i < MAX_HARD; i++)
	    if (hbp[i].flags) {
		GetStateDebugWatchpoint(hbp[i].handle, &j);
		if (j) {
		    printf("breakpoint%d\n", i);
		    ResetDebugWatchpoint(hbp[i].handle);
		    resume_flag = 1;
		}
	    }
    }
    if (hbp[0].flags & BP_ENABLE)
	del_hard_break(0);

    if (deb_cmd == 3)
	EIP -= 1;		/* remove int3 count */

    if (ESP == ESPORG)		/* if no iret_frame, make one */
	ESP -= 12;

    if (DS == emu_sel)
	printf("Caution! fpu-emulator is now debugged! (do cont)\n");

    next_inst = list_inst = unassemble(EIP, 1);

    for (;;) {
	printf("%s", prompt);

	get_input(input, 40);

	/* make strings */
	j = 0;
	for (i = 0; input[i] != 0; i++)
	    if (input[i] == ' ') {
		input[i] = 0;
		if (input[i+1] == ' ')
		    continue;
		if (input[i+1] == 0)
		    break;
		argv[j++] = input + i + 1;
	    }
	argv[j] = 0;

	if (*input == 0) {	/* only return */
	    for (i = sizeof(prompt); i != 0; i--) {
		putchar('\b'), putchar(' '), putchar('\b');
	    argv[0] = NULL;
	    }
	} else {		/* new input */
	    putchar('\n');
	    list_inst = next_inst;

	    for (i = 0; cmds[i].cp; i++)
		if (strcmp(cmds[i].cp, input) == 0) {
		    cmd = cmds[i].t;
		    i = -1;
		    break;
		}
	    if (i != -1 && *input)
		cmd = Unknown;
	}

	switch (cmd) {
	case HELP:
	    printf("go <v>   g - go or continue execution (until <v>)\n");
	    printf("cont <v> c - go or continue execution (until <v>)\n");
	    printf("step     s - step through instruction\n");
	    printf("next     n - step to next instruction\n");
	    printf("list <v> l - list instructions at <v>\n");
	    printf("dump v   d - dump memory at <v>\n");
	    printf("reg      r - show registers\n");
	    printf("where    w - display list of active functions\n");
	    printf("find     f - find a symbol/location (wildcard)\n");
	    printf("bp v       - set breakpoint at v\n");
	    printf("bx no      - x=List/Clear/Dis-/Enable breakpoint no\n");
	    printf("bdx v      - x=Read/Write watchpoint in data <v>\n");
	    printf("set m v    - register or memory to value\n");
	    printf("process    - list processes\n");
	    printf("sel s <n>  - show n selectors start with s\n");
	    printf("user       - list process-user-tab\n");
	    printf("npx        - print copro regs\n");
	    printf("dos argv   - execute DOS program\n");
	    printf("quit     q - terminate debugger\n");
	    printf("help     ? - this text\n");
	    break;

	case CONT:
	    if (argv[0]) {	/* weitere eingabe */
		temp = syms_name2val(argv[0]);
		if (!undefined_symbol && !verify_illegal(npz, temp, 4)) {
		    set_hard_break(temp, BREAK_CODE | BP_SYS);
		}
	    }
	    if (resume_flag)
		EFLAGS |= RESUME_FLAG;
	    back_from_syscall();

	case STEP:
	    j = (WORD) read32(DS, EIP);
	    if (j == 0x21CD)	/* don't step int0x21 */
		goto next;
	    if (copro==3)	/* check floating point instr */
		if ((j & 0xf8) == 0xd8 || (j & 0xf8ff) == 0xd866 || j == 0x9b)
		    goto next;
	    EFLAGS |= SINGLE_STEP;
	    if (resume_flag)
		EFLAGS |= RESUME_FLAG;
	    back_from_syscall();

	case NEXT:
	  next:
	    if (last_unassemble_unconditional || last_unassemble_jump)
		EFLAGS |= SINGLE_STEP;
	    else
		set_hard_break(next_inst, BREAK_CODE | BP_SYS);
	    if (resume_flag)
		EFLAGS |= RESUME_FLAG;
	    back_from_syscall();

	case LIST:
	    if (argv[0]) {	/* weitere eingabe */
		temp = syms_name2val(argv[0]);
		if (!undefined_symbol)
		    list_inst = temp;
	    }
	    lastjmp = last_unassemble_unconditional;
	    lastj = last_unassemble_jump;
	    for (i = 0; i < 10; i++)
		if (!verify_illegal(npz, list_inst, 16))
		    list_inst = unassemble(list_inst, 0);
	    last_unassemble_unconditional = lastjmp;
	    last_unassemble_jump = lastj;
	    break;

	case DUMP:
	    if (argv[0]) {	/* weitere eingabe? */
		temp = syms_name2val(argv[0]);
		if (!undefined_symbol && !verify_illegal(npz, temp, 4)) {
		    printf("%s: adr 0x%08lx val 0x%08lx\n",
			   argv[0], temp, read32(DS, temp));
		} else
		    break;
	    }
	    for (i = 0; i < 8; i++) {
		if (verify_illegal(npz, temp, 4 * sizeof(DWORD)))
		    break;
		printf("%04X:%08lX: ", DS, temp);
		for (j = 0; j < 4; ++j) {
		    DWORD r;
		    r = read32(DS, temp);
		    printf("%02X %02X %02X %02X ",
			   (BYTE) r,
			   (BYTE) (r >> 8),
			   (BYTE) (r >> 16),
			   (BYTE) (r >> 24));
		    temp += 4;
		}
		putchar('\n');
	    }
	    break;

	case BREAK_SET:
	    if (argv[0]) {	/* weitere eingabe */
		temp = syms_name2val(argv[0]);
		if (!undefined_symbol && !verify_illegal(npz, temp, 4))
		    if ((i = set_hard_break(temp, BREAK_CODE)) != -1)
			printf("bp no %d in code at address %08lX\n",
			       i, temp);
	    }
	    break;

	case BREAK_DATA_READ:
	    if (argv[0]) {	/* weitere eingabe */
		temp = syms_name2val(argv[0]);
		if (!undefined_symbol && !verify_illegal(npz, temp, 4))
		    if ((i = set_hard_break(temp, BREAK_DATA_W4)) != -1)
			printf("bp no %d in data at address %08lX\n",
			       i, temp);
	    }
	    break;

	case BREAK_DATA_WRITE:
	    if (argv[0]) {	/* weitere eingabe */
		temp = syms_name2val(argv[0]);
		if (!undefined_symbol && !verify_illegal(npz, temp, 4))
		    if ((i = set_hard_break(temp, BREAK_DATA_RW4)) != -1)
			printf("bp no %d in data at address %08lX\n"
			       ,i, temp);
	    }
	    break;

	case BREAK_CLEAR:
	    if (argv[0]) {
		sscanf(argv[0], "%d", &i);
		del_hard_break(i);
	    }
	    break;

	case BREAK_ENABLE:
	    if (argv[0]) {
		sscanf(argv[0], "%d", &i);
		enable_hard_break(i);
	    }
	    break;

	case BREAK_DISABLE:
	    if (argv[0]) {
		sscanf(argv[0], "%d", &i);
		disable_hard_break(i);
	    }
	    break;

	case BREAK_LIST:
	    printf("Breakpoint list:\n");
	    for (i = 0; i < MAX_HARD; i++)
		if (hbp[i].flags) {
		    name = syms_val2name(hbp[i].address, &temp);
		    printf("bp%X at %08lX ", i, hbp[i].address);
		    if (name)
			printf("%s", name);
		    if (temp)
			printf("+%lX ", temp);
		    printf(" type: %s ",
			   (hbp[i].flags & 0x0f00) ? "data" : "code");
		    if (hbp[i].flags & BP_DISABLE)
			printf("- disabled");
		    printf("\n");
		}
	    break;

	case REGS:
	    printregs();
	    unassemble(EIP, 1);
	    break;

	case WHERE:
	    {
		DWORD v, vaddr, delta;

		printf("0x%08lX %s", EIP, syms_val2name(EIP, &temp));
		name = syms_val2line(EIP, &i, 0);
		if (name)
		    printf(", line %d in file %s", i, name);
		else if (temp)
		    printf("%+ld", temp);
		putchar('\n');

		v = EBP;
		do {
		    if (v == 0)
			break;
		    temp = read32(DS, v);
		    if (temp == 0)
			break;
		    vaddr = read32(DS, v + 4);
		    printf("0x%08lx %s", vaddr, syms_val2name(vaddr, &delta));
		    name = syms_val2line(vaddr, &i, 0);
		    if (name)
			printf(", line %d in file %s", i, name);
		    else if (delta)
			printf("%+ld", delta);
		    putchar('\n');
		    v = temp;
		}
		while ((v >= ESP) && (v < npz->stackp32));
	    }
	    break;

	case FIND:
	    if (argv[0] == NULL)/* weitere eingabe? */
		break;
	    if (strpbrk(argv[0], "*?") != NULL) {
		syms_listwild(argv[0]);
		break;
	    }
	    temp = syms_name2val(argv[0]);
	    if (!undefined_symbol && !verify_illegal(npz, temp, 4)) {
		name = syms_val2name(EIP, &temp);
		printf("0x%08lx %s", EIP, name);
		if (temp)
		    printf("+%lx", temp);
		name = syms_val2line(EIP, &i, 0);
		if (name)
		    printf(", line %d in file %s", i, name);
		putchar('\n');
	    }
	    break;

	case PROCESS:
	    for (i = 1; i <= N_PRZ; i++)
		if (process[i].pptr)
		    printf("prg_id %d parent_id %d\n",
			   process[i].pid, process[i].pptr->pid);
	    break;

	case USER:
	    {
		DESCRIPTOR desc;

		printf("prg_id %d parent_id %d\n"
		       "textsize: %lX - %lX\n"
		       "data+bss: %lX - %lX\n"
		       "stack: %lX - %lX\n"
		       "heap: %lX -  %lX\n",
		       npz->pid, npz->pptr->pid,
		       npz->text_start, npz->text_end,
		       npz->data_start, npz->data_end,
		       npz->stack_down, npz->stack_top,
		       npz->init_brk, npz->brk_value
		       );
		printf("CS sel %X: ", npz->code32sel);
		GetDescriptor(npz->code32sel, &desc);
		PrintDescriptor(&desc);
		printf("DS sel %X: ", npz->data32sel);
		GetDescriptor(npz->data32sel, &desc);
		PrintDescriptor(&desc);
		printf("SS sel %X: ", npz->data32sel+8);
		GetDescriptor(npz->data32sel+8, &desc);
		PrintDescriptor(&desc);
		break;
	    }

	case SET:
	    if (argv[0] == NULL && argv[1] == NULL)
		break;
	    temp = syms_name2val(argv[1]);
	    if (undefined_symbol)
		break;
	    j = 0;
	    for (i = 0; regs[i].name; i++)
		if (strcmp(regs[i].name, argv[0]) == 0) {
		    j = 1;
		    switch (regs[i].size) {
		    case 1:
			*((BYTE *) npz + regs[i].ofs) = (BYTE) temp;
			break;
		    case 2:
			*(WORD *) ((BYTE *) npz + regs[i].ofs) = (WORD) temp;
			break;
		    case 4:
			*(DWORD *) ((BYTE *) npz + regs[i].ofs) = temp;
			break;
		    }
		}
	    if (j)
		break;
	    else {
		DWORD pokeat = syms_name2val(argv[0]);
		if (undefined_symbol)
		    break;
		store32(DS, pokeat, temp);
	    }
	    break;

	case SEL:
	    {
		DESCRIPTOR desc;

		if (argv[0] == NULL)
		    break;
		if (argv[1] == NULL)
		    i = 1;
		else
		    i = atoi(argv[1]);

		temp = syms_name2val(argv[0]);
		if (undefined_symbol)
		    break;
		for (j = (int) temp; j < (int) temp + i * 8; j += 8) {
		    printf("sel %X: ", j);
		    if (GetDescriptor(j, &desc)) {
			printf("- invalid -\n");
			break;
		    }
		    PrintDescriptor(&desc);
		}
	    }
	    break;

	case NPX:
	    if (copro == 0) {
		printf("No 387 option used\n");
		break;
	    } else if (copro == 3) {
		DESCRIPTOR desc;

		printf("emu387:\nstacksize: %lX\n",process[0].stacksize);
		printf("CS sel %X: ", process[0].code32sel);
		GetDescriptor(process[0].code32sel, &desc);
		PrintDescriptor(&desc);
		printf("DS sel %X: ", process[0].data32sel);
		GetDescriptor(process[0].data32sel, &desc);
		PrintDescriptor(&desc);

		npx = &(npz->npx);
		cpy32_16(emu_sel, copro_struct, npx, sizeof(union i387_union));

		if (npx->soft.cwd == 0) {
		    printf("no floating foint instruction is done\n");
		    break;
		}
		print_387_control_word((WORD) npx->soft.cwd);
		print_387_status_word((WORD) npx->soft.swd);
		printf("tag: %04lX\n", npx->soft.twd);

		for (i = 0; i < 8; i++) {
		    printf("st(%d)  ", i);
		    if (npx->soft.regs[i].sign)
			putchar('-');
		    else
			putchar('+');
		    printf("%08X %08X e %04x\n",
			   npx->soft.regs[i].sigh,
			   npx->soft.regs[i].sigl,
			   npx->soft.regs[i].exp);
		}
		printf("lookahead %s\n", (npx->soft.res1) ? "on" : "off");
	    }
	    break;

	case DOS:
	    {
		char buffer[128];
		DWORD org_eax;

		if (argv[0] == NULL)
		    break;
		strcpy(buffer, argv[0]);

		org_eax = EAX;
		if (realmode_prg(buffer, argv, org_env))
		    printf("error spawn %s\n", buffer);
		EAX = org_eax;
	    }
	    break;

	case QUIT:
	    for (i = 0; i < MAX_HARD; i++)
		if (hbp[i].flags & BP_ENABLE)
		    del_hard_break(i);
	    shut_down(0);
	    /* never */

	case Unknown:
	    printf("unknown command, try (H)elp or ?\n");
	    break;

	}			/* switch */

    }				/* for (;;) */
}
