#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <process.h>
#include <errno.h>
#include <sys/ptrace.h>
#include <sys/user.h>
#include <sys/reg.h>
#include "syms.h"

#ifdef __GO32__
#define SEEK_SET 0
#endif

typedef unsigned long CORE_ADDR;
typedef unsigned char byte;

extern CORE_ADDR print_insn(CORE_ADDR,char *);
extern int get_input(char *, int);

struct breakpoint
{
  unsigned addr;
  unsigned save;
} bp ;

struct reg
{
  char *name;
  int regno;
  unsigned value;
};

#define N_EIP 8

static struct reg regs[] =
{
  {"eax", EAX},
  {"ebx", EBX},
  {"ecx", ECX},
  {"edx", EDX},
  {"esi", ESI},
  {"edi", EDI},
  {"esp", UESP},
  {"ebp", EBP},
  {"eip", EIP},
  {NULL, 0}
};

typedef enum {
    Unknown, CONT, STEP, NEXT, LIST, DUMP, DOS,
    BREAK_SET, BREAK_LIST, BREAK_CLEAR, BREAK_ENABLE, BREAK_DISABLE,
    BREAK_DATA_READ, BREAK_DATA_WRITE,
    REGS, SET, WHERE, FIND, NPX, PROCESS, USER, SEL, HELP, QUIT
};

typedef struct {
    char *cp;
    int t;
} item;

static item cmds[] =
{
    "go", CONT, "g", CONT,
    "cont", CONT, "c", CONT,
    "step", STEP, "s", STEP,
    "next", NEXT, "n", NEXT,
    "list", LIST, "l", LIST,
    "dump", DUMP, "d", DUMP,
    "reg", REGS, "r", REGS,
    "where", WHERE, "w", WHERE,
    "find", FIND, "f", FIND,
    "bp", BREAK_SET,
    "bc", BREAK_CLEAR,
    "bd", BREAK_DISABLE,
    "be", BREAK_ENABLE,
    "bl", BREAK_LIST,
    "bdr", BREAK_DATA_READ,
    "bdw", BREAK_DATA_WRITE,
    "set", SET,
    "process", PROCESS,
    "user", USER,
    "dos", DOS,
    "sel", SEL,
    "npx", NPX,
    "help", HELP, "h", HELP, "?", HELP,
    "quit", QUIT, "q", QUIT,
    0, 0
};


static int pid;
static int uaddr;
static long dump_addr;
static char prompt[] = "cmd:";
static char **org_env;
static char jumping;
static unsigned long next_inst;
static unsigned long this_inst;
static int cmd;

/* function missing in i386-pin */
void print_address(CORE_ADDR addr, char *print_buf)
{
    unsigned long delta;
    char *s;

    s = syms_val2name(addr,&delta);
    strcat (print_buf, s);
    if (delta) {
	char ascii[40];
	sprintf(ascii, " + %d", delta);
	strcat(print_buf,ascii);
    }
}

/* function missing in i386-pin */
void read_memory(CORE_ADDR addr, char *buf, int len)
{
    unsigned long pc,r,i;

    for (pc = addr, i=0 ; pc < addr+len ; pc++,i++) {
	errno = 0;
	r = ptrace (PTRACE_PEEKTEXT, pid, pc, 0);
	if (errno != 0)
	    break;
	else
	    buf[i] = (char) r;
    }
}

struct exe_hdr {
    unsigned short signatur;
    unsigned short low;
    unsigned short high;
    unsigned short reloc;
    unsigned short hdr_para;
    } ;

struct emx_hdr {
    char sig[18];
    char next_off[4];
} ;

int skip_exe_hdr(int filehandle, unsigned long *headoff)
{
    struct exe_hdr exehdr;

    read(filehandle, &exehdr, sizeof(struct exe_hdr));

#ifdef __EMX__
    if (exehdr.signatur == 0x5a4d) {	/* falls exe-kopf */
	struct emx_hdr emxhdr;
	*headoff = ((unsigned long) exehdr.hdr_para) * 16;
	if (lseek(filehandle, *headoff, SEEK_SET) == -1) {
	    goto fail;
	}
	read(filehandle, &emxhdr, sizeof(struct emx_hdr));
	if (memcmp(emxhdr.sig, "emx", 3) != 0) {
	    goto fail;
	}
	*headoff = * (int *) emxhdr.next_off;
    }
    if (lseek(filehandle, *headoff, SEEK_SET) == -1)
	goto fail;
    return 0;

  fail:
    *headoff = 0;
    lseek(filehandle, 0, SEEK_SET);
    return -1;

#else
    if (exehdr.signatur == 0x5a4d) {	/* falls exe-kopf */
	*headoff += (unsigned long) exehdr.high * 512L;
	if (exehdr.low)
	    *headoff += (unsigned long) exehdr.low - 512L;
    }
    if (lseek(filehandle, *headoff, SEEK_SET) != -1)
	return 0;
    else {
	*headoff = 0;
	lseek(filehandle, 0, SEEK_SET);
	return -1;
    }
    return 0;
#endif
}

int ptrace_get_register(int off)
{
    return ptrace (PTRACE_PEEKUSER, pid, uaddr + off, 0);
}


static void show_regs (void)
{
  int i, j, r;

  j = 4;
  for (i = 0; regs[i].name != NULL; ++i)
    {
      errno = 0;
      r = ptrace (PTRACE_PEEKUSER, pid, uaddr + regs[i].regno * 4, 0);
      if (errno != 0) perror ("ptrace");
      regs[i].value = r;
      if (j == 4)
        {
          j = 0; putchar ('\n');
        }
      ++j;
      printf ("%s=%.8x ", regs[i].name, r);
    }
  errno = 0;
  r = ptrace (PTRACE_PEEKTEXT, pid, regs[N_EIP].value, 0);
  if (errno == 0)
    for (i = 0; i < 4; ++i)
      {
        printf ("%.2x ", r & 0xff);
        r >>= 8;
      }
  putchar ('\n');
}


static int run (int cmd)
{
    int s, t, p, r;

    s = ptrace (cmd, pid, 0, 0);
    if (s < 0)
	perror ("ptrace");
    else {
	p = wait (&t);
	if (p == -1)
	    perror ("wait");
	if ((t & 0377) != 0177) {
	    printf ("Program terminated (%d)\n", (t >> 8) & 0xff);
	    exit (0);
        }
	errno = 0;
	r = ptrace (PTRACE_PEEKUSER, pid, uaddr + EIP * 4, 0);
	if (errno != 0)
	    perror ("ptrace");
	if (bp.addr == r - 1)
	    ptrace (PTRACE_POKEUSER, pid, uaddr + EIP * 4, --r);
    }
    return r;
}


static void dump (long n)
{
  int i, j, r;
  char *p;

  dump_addr = n;
  for (i = 0; i < 8; ++i)
    {
      printf ("%.8lx: ", dump_addr);
      for (j = 0; j < 8; ++j)
        {
          errno = 0;
          r = ptrace (PTRACE_PEEKTEXT, pid, dump_addr, 0);
          if (errno != 0)
            fputs ("?? ?? ", stdout);
          else
            printf ("%.2x %.2x ", r & 0xff, (r >> 8) & 0xff);
          dump_addr += 2;
        }
      putchar ('\n');
    }
}

void insert_bp(int eip)
{
    errno = 0;
    bp.addr = 0;
    bp.save = ptrace(PTRACE_PEEKTEXT, pid, eip, 0);
    if (errno != 0) {
	printf("PEEKTEXT fail\n");
	return;
    }
    ptrace(PTRACE_POKETEXT, pid, eip, (bp.save & ~0xff) | 0xcc);
    if (errno != 0) {
	printf("POKETEXT fail\n");
	return;
    }
    bp.addr = eip;
}

void remove_bp(void)
{
    if (bp.addr == 0)
	return;
    errno = 0;
    ptrace(PTRACE_POKETEXT, pid, bp.addr, bp.save);
    if (errno != 0) {
	printf("PEEKTEXT fail\n");
	return;
    }
    bp.addr = 0;
}


CORE_ADDR unassemble(CORE_ADDR adr, char print)
{
    char *name,*lname;
    unsigned long delta, r;
    int linenum;
    static char print_buffer[80];

    if (print) {
	name = syms_val2name(adr, &delta);
	if (!delta && (name[0] != '0')) {
	    printf("%s()", name);
	    lname = syms_val2line(adr, &linenum, 1);
	    if (lname)
		printf(" (%s#%d):\n", lname, linenum);
	    else
		printf(":\n");
	} else {
	    lname = syms_val2line(adr, &linenum, 1);
	    if (lname)
		printf("#%d:\n", linenum);
	}
    }

    if (print)
	printf("%08X : ", adr);

    print_buffer[0] = 0;
    r = print_insn(adr,print_buffer);
    if (print_buffer[0] == 'j')
	jumping = 1;
    else
	jumping = 0;

    if (print) {
	printf("%s\n", print_buffer);
	fflush(stdout);
    }

    return r;
}


static void command_loop (void)
{
    unsigned long temp, list_inst;
    int lastjmp, lastj, i, j, len, resume_flag;
    char input[40];
    char *argv[10];
    char *name;
    char show_inst;

    show_inst = 1;

    while (1) {
	this_inst = ptrace (PTRACE_PEEKUSER, pid, uaddr + EIP * 4, 0);
	next_inst = this_inst + unassemble(this_inst, show_inst);

	printf("%s", prompt);
	fflush(stdout);

	len = get_input(input, 40);

	/* make strings */
	j = 0;
	for (i = 0; input[i] != 0; i++)
	    if (input[i] == ' ') {
		input[i] = 0;
		if (input[i+1] == ' ')
		    continue;
		if (input[i+1] == 0)
		    break;
		argv[j++] = input + i + 1;
	    }
	argv[j] = 0;

	if (*input == 0) {	/* only return */
	    for (i = sizeof(prompt); i != 0; i--) {
		putchar('\b'), putchar(' '), putchar('\b');
	    fflush(stdout);
	    argv[0] = NULL;
	    }
	} else {		/* new input */
	    putchar('\n');
	    fflush(stdout);

	    list_inst = this_inst;

	    for (i = 0; cmds[i].cp; i++)
		if (strcmp(cmds[i].cp, input) == 0) {
		    cmd = cmds[i].t;
		    i = -1;
		    break;
		}
	    if (i != -1 && *input)
		cmd = Unknown;
	}

	switch (cmd) {

	case HELP:
	    printf("go       g - go or continue execution\n");
	    printf("cont     c - continue execution\n");
	    printf("step     s - step through instruction\n");
	    printf("next     n - step to next instruction\n");
	    printf("list <v> l - list instructions at <v>\n");
	    printf("dump <v> d - dump memory at <v>\n");
	    printf("reg      r - show registers\n");
	    printf("find     f - find a symbol/location (wildcard)\n");
	    printf("dos argv   - execute DOS program\n");
	    printf("quit     q - terminate debugger\n");
	    printf("help     ? - this text\n");
	    show_inst = 0;
	    break;

	case CONT:
	    if (argv[0]) {
		temp = syms_name2val(argv[0]);
		if (undefined_symbol)
		    break;
		insert_bp(temp);
		run(PTRACE_RESUME);
		remove_bp();
	    }
	    else
		run (PTRACE_RESUME);
	    show_inst = 1;
	    break;

	case STEP:
            run (PTRACE_STEP);
	    show_inst = 1;
	    break;

	case NEXT:
	    if (jumping)
		run (PTRACE_STEP);
	    else {
		insert_bp(next_inst);
		run(PTRACE_RESUME);
		remove_bp();
	    }
	    show_inst = 1;
	    break;

	case LIST:
	    if (argv[0]) {	/* weitere eingabe */
		temp = syms_name2val(argv[0]);
		if (!undefined_symbol)
		    list_inst = temp;
		else
		    list_inst = this_inst;
	    }
	    for (i=1;i<=10;i++)
		list_inst += unassemble(list_inst, 1);
	    show_inst = 0;
	    break;

	case DUMP:
	    if (argv[0] == NULL)
		break;
	    else
		temp = syms_name2val(argv[0]);
	    if (!undefined_symbol)
		dump(temp);
	    show_inst = 0;
	    break;

        case REGS:
            show_regs();
	    show_inst = 1;
	    break;

	case DOS:
	    if (argv[0] == NULL)
		break;
	    if (spawnve(P_WAIT, argv[0], (const char * const *)&(argv[0]),
			(const char * const *)org_env) < 0)
		printf("error spawn %s\n", argv[0]);
	    show_inst = 0;
	    break;

	case FIND:
	    if (argv[0] == NULL)
		break;
	    if (strpbrk(argv[0], "*?") != NULL) {
		syms_listwild(argv[0]);
		break;
	    }
	    temp = syms_name2val(argv[0]);
	    if (!undefined_symbol) {
		name = syms_val2name(this_inst, &temp);
		printf("0x%08lx %s", this_inst, name);
		if (temp)
		    printf("+%lx", temp);
		name = syms_val2line(this_inst, &i, 0);
		if (name)
		    printf(", line %d in file %s", i, name);
		putchar('\n');
	    }
	    break;

	case QUIT:
	    return ;

	case Unknown:
	    printf("unknown command, try (H)elp or ?\n");
	    show_inst = 0;
	    break;

	}   /* switch */

    } /* while */
}

int main (int argc, char **argv, char **envp)
{
    struct user u;

    if (argc<2) {
	fputs ("Usage: debug prog [args]\n", stderr);
	exit (1);
    }
    org_env = envp;

    pid = spawnve (P_DEBUG , argv[1], (char **) &(argv[1]), envp);
    if (pid < 0) {
	perror ("spawnve");
	exit (2);
    }
    errno = 0;
    uaddr = ptrace (PTRACE_PEEKUSER, pid, (char *)&u.u_ar0 - (char *)&u, 0);
    if (errno != 0)
	perror ("ptrace");
    uaddr -= 0xe0000000;

    printf("debugging file %s\n",argv[1]);
    syms_init(argv[1]);

    command_loop ();

    return (0);
}

