/*
~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
IWF.C (Image WorkFrame)

AUTHOR: Craig Muller P. Eng. 1991,1992,1993
        cmuller@ccu.umanitoba.ca
        Computer Vision Laboratory
        Mech. Engn.,Univ. of Manitoba
        Winnipeg, Manitoba. R3T 2N2

DESCRIPTION:
This is the main program file for the Image WorkFrame
System or IWF for short. The purpose of this software is to provide a
general purpose program interface for image processing and image manipulation
in a windows environment. The code can be cut an pasted to produce other
programs of used as is. The program also allows the user to control the
MVP-AT image capture board to perform capture and store of images.

All the code is designed in a modular fashion so that projects in image
processing or conputer vision do not need to re-write all the basic code
for image manipulation. Each additional software project is designed as an
addition to the and has its own header file which is included here.
Programmers should limit their code modifications to the program module
which contains their project.

Coding conventions:

   ~~~ boxing for module title and description.
   --- boxing for procedures which can be called by the user.
   === boxing for exports which cannot be called by the user.

~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
*/
#include <windows.h>

#pragma hdrstop

#include <stdio.h>
#include <string.h>
#include <io.h>
#include <math.h>

#include "iwf.h"
#include "module.h"

#define  PGMNAME "IWF"
#define  MODULE  "IWF"

#define   VGA      640                 // VGA resolution width
#define  SVGA      800                 // Super VGA resolution width
#define   XGA     1024                 // XGA resolution width
#define   CAD     1280                 // CAD/CAM resolution width

//========================================
// Exported procedure prototypes.
//========================================
long far PASCAL _export WP_Main (HWND hWnd,WORD wMsg,WORD wParam,LONG lParam);
BOOL FAR PASCAL _export DP_About(HWND hDlg,WORD wMsg,WORD wParam,LONG lParam);
BOOL FAR PASCAL _export DP_Info(HWND hDlg,WORD wMsg,WORD wParam,LONG lParam);
BOOL FAR PASCAL _export DP_Scale(HWND hDlg,WORD wMsg,WORD wParam,LONG lParam);
BOOL FAR PASCAL _export DP_Transform(HWND hDlg,WORD wMsg,WORD wParam,LONG lParam);
BOOL FAR PASCAL _export DP_Frame(HWND hDlg,WORD wMsg,WORD wParam,LONG lParam);
BOOL FAR PASCAL _export DP_UserFile(HWND hDlg,WORD wMsg,WORD wParam,LONG lParam);


//---------------------------------------
// Private procedures.
//---------------------------------------
static void Load_Settings(void);

int UserDlgMessage(MSG *msg);
int ToolDlgMessage(MSG *msg);

void Labeller(void);
void FillColors(void);
void iwf_FrameOp(int action);
int Load_Item(FILE *fp,char *keyword,int *buf,int cItems);

IMAGE *MedianImage(IMAGE *image);
IMAGE *SobelImage(IMAGE *image,WORD Threshold,BOOL Overlay);

//---------------------------------------
// Public variables.
//---------------------------------------

HANDLE    hInst;                          /* current instance handle   */

HWND  _hWndImage [MAXIMAGES];          // Window handles for images.
HWND  _hWndModule[MAXMODULES];         // Window handles for modules
HWND  _hWndTool  [MAXTOOLS];           // Window handles for buttons
HWND  hWndMod=NULL;                    // Window handle for module.
HWND  hWndSrc=NULL;                    // Window handle for source image.
HWND  hWndDst=NULL;                    // Window handle for destination image.

char  _name[8][20];                    // Button names
char  _cwdImage[80];                   // Image working directory

IMAGE  *_undo=NULL;                    // Undo image buffer pointer.

PALETTEENTRY peGray  [64];             // Gray scale palette for gray images
PALETTEENTRY pePseudo[64];             // Pseudo color palette for alternate
PALETTEENTRY peUser [128];             // User programmable section
PALETTEENTRY peMain [256];             // Primary system palette

HPALETTE hPalMain;                      // Handle to the current palette.

extern WORD hist[256];

// Default user file profile
char _ext[5] = ".usr";
int _hsize  =   0;
int _psize  =   8;
int _pshift =   8;
int _width  = 512;
int _height = 480;

//---------------------------------------
// Module prototypes.
//---------------------------------------
#if defined(MODULE0)
int MODULE0(HWND hWnd);
#endif
#if defined(MODULE1)
int MODULE1(HWND hWnd);
#endif
#if defined(MODULE2)
int MODULE2(HWND hWnd);
#endif
#if defined(MODULE3)
int MODULE3(HWND hWnd);
#endif
#if defined(MODULE4)
int MODULE4(HWND hWnd);
#endif
#if defined(MODULE5)
int MODULE5(HWND hWnd);
#endif
#if defined(MODULE6)
int MODULE6(HWND hWnd);
#endif
#if defined(MODULE7)
int MODULE7(HWND hWnd);
#endif

//--------------------------------------
// External dialog windows.
//--------------------------------------
#if defined (DIALOG0)
extern HWND DIALOG0;
#endif
#if defined (DIALOG1)
extern HWND DIALOG1;
#endif
#if defined (DIALOG2)
extern HWND DIALOG2;
#endif
#if defined (DIALOG3)
extern HWND DIALOG3;
#endif
#if defined (DIALOG4)
extern HWND DIALOG4;
#endif
#if defined (DIALOG5)
extern HWND DIALOG5;
#endif
#if defined (DIALOG6)
extern HWND DIALOG6;
#endif
#if defined (DIALOG7)
extern HWND DIALOG7;
#endif





/*
==========================================================================
int PASCAL WinMain(HANDLE hInstance,HANDLE hPrevInstance,
                   LPSTR  lpCmdLine,int    nCmdShow)

Description:

WINDOWS ENTRY POINT CODE.

Windows initial entry point for the program. This is the startup code for
the program. It initializes the top level window, sets up code sharing
(if another instance of the program is running), and begins a message loop.
This code should not be modified by module programmers.

If this function must abort before entering the msg loop, it returns the
conventional value NULL.
==========================================================================
*/
int PASCAL WinMain(HANDLE hInstance,        /* current instance       */
						 HANDLE hPrevInstance,    /* previous instance      */
						 LPSTR  lpCmdLine,        /* cmd line               */
						 int    nCmdShow)         /* show type (open/icon)  */
   {
   int  x0,y0,w,h,cx,cy;
   MSG  msg;
   HWND hWnd;                              
   WNDCLASS wc;

   hInst = hInstance;                            // Save the instance handle.
   if (!hPrevInstance)                           // Other instances running?
      {
      wc.lpszClassName = MODULE"Main";           // Name for CreateWindow.
      wc.style         = CS_DBLCLKS;             // Qualifying double clicks 
      wc.lpfnWndProc   = (WNDPROC)WP_Main;       // Func to get msgs
      wc.cbClsExtra    = 0;                      // No per-class extra data. 
      wc.cbWndExtra    = 0;                      // No per-window extra data 
      wc.hInstance     = hInstance;              // App that owns the class  
      wc.hIcon         = LoadIcon(hInst,MODULE); // Loads icon 
      wc.hCursor       = LoadCursor(NULL,IDC_ARROW);
      wc.hbrBackground = COLOR_APPWORKSPACE+1;
      wc.lpszMenuName  = MODULE;                 // Name of menu resource

      if (!RegisterClass(&wc))                   // Initialize new window class.
         {
         char sz[80];

         sprintf("Unable to register <%s> class",(char *)wc.lpszClassName);
         MessageBeep(MB_ICONEXCLAMATION);
         MessageBox(NULL,sz,"Error",MB_OK | MB_ICONEXCLAMATION);
         return (FALSE);                         // Exits if unable.
         }
      }

   /*
   Perform initializations that apply to a specific instance, those that 
   cannot be shared by multiple instances. Save the instance handle in
   static variable, which will be used in many subsequence calls from
   this application to Windows.

   WRIPS is currently compatible with 3 display resolutions:

   1. 640x480  VGA             (VGA)
   2. 800x600  Super VGA      (SVGA)
   3. 1024x768 IBM 8514A/XGA   (XGA)
   4. 1280x1024 std CAD/CAM    (CAD)
   */

   x0=y0=0;
   w=h=CW_USEDEFAULT;
   cx = GetSystemMetrics(SM_CXSCREEN);      // Get the screen width in pixels
   cy = GetSystemMetrics(SM_CYSCREEN);      // Get the screen height in pixels

   switch (cx)
      {
      case  VGA: x0=  0; y0=  0; w= cx-  0; h=cy-  0; break;
      case SVGA: x0=  0; y0=  0; w= cx-  0; h=cy- 60; break;
      case  XGA: x0=  0; y0=  0; w= cx-  0; h=cy- 80; break;
      case  CAD: x0=100; y0=100; w= cx-200; h=cy-200; break;
      }


   hWnd = CreateWindow(
      PGMNAME"Main",                        // See RegisterClass() call.
      "Image Workframe",                    // Text for window title bar.
      WS_OVERLAPPEDWINDOW | WS_CLIPCHILDREN,// Use standard window style
      x0,y0,w,h,                            // Set window startup coords 
      NULL,                                 // No parent for overlapping
      NULL,                                 // Use the window class menu. 
      hInstance,                            // This instance owns this wnd 
      NULL);                                // Pointer not needed.

   if (!hWnd)                               // If no window created
      {
      char sz[80];

      sprintf("Unable to create <%s> desktop window",PGMNAME);
      MessageBeep(MB_ICONEXCLAMATION);
      MessageBox(NULL,sz,"Error",MB_OK | MB_ICONEXCLAMATION);
      return (FALSE);                       // Exits if unable.
      }

   ShowWindow(hWnd,nCmdShow);               // Show the window 
   UpdateWindow(hWnd);                      // Sends WM_PAINT msg 
   lpCmdLine=lpCmdLine;                     // Suppress warning.

   // Acquire and dispatch messages until a WM_QUIT msg is received.
   // This function exits the application instance by returning the 
   // value passed by PostQuitMessage().

   while (GetMessage(&msg,NULL,NULL,NULL))
      {
		if (!ToolDlgMessage(&msg) && !UserDlgMessage(&msg))
         {                                  // Dispatch to a window
         TranslateMessage(&msg);            // Xlates virtual key codes
         DispatchMessage(&msg);             // Dispatches msg to window
         }
      }
   return(msg.wParam);                      // Returns PostQuitMessage
   }


/*
---------------------------------------------------------------------
int ToolDlgMessage(MSG *msg)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Dispatches messages designated for one of the tools includes in IWF.
---------------------------------------------------------------------
*/
int ToolDlgMessage(MSG *msg)
   {
   int processed=0;
   extern HWND hDlgPalMgr;
   extern HWND hDlgProfiler;
   extern HWND hDlgConvolver;
   //extern HWND hDlgMatrox;

   if (!processed && IsWindow(hDlgPalMgr   ) && IsDialogMessage(hDlgPalMgr   ,msg)) processed=1;
   if (!processed && IsWindow(hDlgProfiler ) && IsDialogMessage(hDlgProfiler ,msg)) processed=1;
   if (!processed && IsWindow(hDlgConvolver) && IsDialogMessage(hDlgConvolver,msg)) processed=1;
   //if (!processed && IsWindow(hDlgMatrox   ) && IsDialogMessage(hDlgMatrox   ,msg)) processed=1;

   return(processed);
   }

/*
---------------------------------------------------------------------
int UserDlgMessage(MSG *msg)
~~~~~~~~~~~~~~~~~~~~~~~~~~~~
Dispatches messages designated for one of the user modules.
---------------------------------------------------------------------
*/
int UserDlgMessage(MSG *msg)
   {
   int processed=0;

   #if defined (DIALOG0)
   if (!processed && IsWindow(DIALOG0) && IsDialogMessage(DIALOG0,msg)) processed=1;
   #endif
   #if defined (DIALOG1)
   if (!processed && IsWindow(DIALOG1) && IsDialogMessage(DIALOG1,msg)) processed=1;
   #endif
   #if defined (DIALOG2)
   if (!processed && IsWindow(DIALOG2) && IsDialogMessage(DIALOG2,msg)) processed=1;
   #endif
   #if defined (DIALOG3)
   if (!processed && IsWindow(DIALOG3) && IsDialogMessage(DIALOG3,msg)) processed=1;
   #endif
   #if defined (DIALOG4)
   if (!processed && IsWindow(DIALOG4) && IsDialogMessage(DIALOG4,msg)) processed=1;
   #endif
   #if defined (DIALOG5)
   if (!processed && IsWindow(DIALOG5) && IsDialogMessage(DIALOG5,msg)) processed=1;
   #endif
   #if defined (DIALOG6)
   if (!processed && IsWindow(DIALOG6) && IsDialogMessage(DIALOG6,msg)) processed=1;
   #endif
   #if defined (DIALOG7)
   if (!processed && IsWindow(DIALOG7) && IsDialogMessage(DIALOG7,msg)) processed=1;
   #endif

   msg = msg;
   return(processed);
   }


/*
==========================================================================
long far PASCAL _export WP_Main(HWND hWnd,WORD wMsg,WORD wParam,LONG lParam)

Description:

MAIN MESSAGE DISPATCHER CODE.

This is the windows main messaging loop which is specifically constructed to
provide a message dispatcher for multiple modules of code attached to WRIPS.

This procedure handles all the primary message handling from the window.
This includes not only the main menu selections but specific messages
generates by the Windows environment. This code should only be modified to
allow the module programmers to had menu access for their code.
==========================================================================
*/
long far PASCAL _export WP_Main(HWND hWnd,WORD wMsg,WORD wParam,LONG lParam)
   {
   switch (wMsg)
      {
      case WM_CREATE:
      // Create the push buttons for launching user modules.
      {
      int  i;
      DWORD dwStyle;

      dwStyle = WS_CHILD | WS_VISIBLE | BS_PUSHBUTTON;

      for (i=0; i<8; ++i)
         strcpy(_name[i],"");

      #if defined(NAME0)
      strcpy(_name[0],NAME0);
      #endif
      #if defined(NAME1)
      strcpy(_name[1],NAME1);
      #endif
      #if defined(NAME2)
      strcpy(_name[2],NAME2);
      #endif
      #if defined(NAME3)
      strcpy(_name[3],NAME3);
      #endif
      #if defined(NAME4)
      strcpy(_name[4],NAME4);
      #endif
      #if defined(NAME5)
      strcpy(_name[5],NAME5);
      #endif
      #if defined(NAME6)
      strcpy(_name[6],NAME6);
      #endif
      #if defined(NAME7)
      strcpy(_name[7],NAME7);
      #endif

      for (i=0; i<7 && _name[i][0]; ++i)
         _hWndTool[i] = CreateWindow("button",_name[i],dwStyle,i*74,0,75,25,hWnd,1000+i,hInst,NULL);
      }

      _undo = NULL;                              // Set undo buffer to empty.
      Status(hWnd);                              // Launch the status window.

      hPalMain = CreateLogPalette(256);          // Create a logical system palette
      FillColors();                              // Fill the color tables

      PostMessage(hWnd,WM_USER  ,0,0);           // Post Message to Menu
      PostMessage(hWnd,WM_COMMAND,999,0);       // Post Message to About Box
      break;

      // THE IMAGE DESKTOP NEEDS REPAINTING
      case WM_PAINT:
      {
      PAINTSTRUCT ps;

      BeginPaint(hWnd,&ps);                      // Set up a DC for paint
      EndPaint(hWnd,&ps);                        // Return the display DC
      }
      break;

      // THE IMAGE DESKTOP HAS BEEN SIZED OR MOVED.
      case WM_SIZE:
      case WM_MOVE:
      PostMessage(hWndStatus,wMsg,hWnd,lParam);  // Post msg to status window.
      break;

      // SYSTEM PALETTED CHANGED.
      case WM_PALETTECHANGED:
      {
      int n;

      PostMessage(hWndStatus,wMsg,hWnd,lParam);  // Post msg to status window.

      // Re-paint the image windows which do not have input focus.
      for (n=0; n<MAXIMAGES; ++n)
         if (IsWindow(_hWndImage[n]) && (_hWndImage[n]!=(HWND)wParam))
             InvalidateRect(_hWndImage[n],NULL,FALSE);
      }
      break;

      // IMAGE ACTIVE/INACTIVE
      case WM_USER:
      {
      int i;
      HMENU hMenu;

      Load_Settings();
      hMenu = GetMenu(hWnd);
      if (IsWindow(wParam))                      // Active image handle.
         {
         EnableMenuItem(GetSubMenu(hMenu,0),2,MF_ENABLED | MF_BYPOSITION);
         EnableMenuItem(GetSubMenu(hMenu,0),3,MF_ENABLED | MF_BYPOSITION);
         for (i=1; i<=4; ++i)
            EnableMenuItem(hMenu,i,MF_ENABLED | MF_BYPOSITION);
         }
      else                                       // No active image.
         {
         EnableMenuItem(GetSubMenu(hMenu,0),2,MF_GRAYED | MF_BYPOSITION);
         EnableMenuItem(GetSubMenu(hMenu,0),3,MF_GRAYED | MF_BYPOSITION);
         for (i=1; i<=4; ++i)
            EnableMenuItem(hMenu,i,MF_GRAYED | MF_BYPOSITION);
         }
      DrawMenuBar(hWnd);
      }
      break;

      // A MAIN MENU COMMAND HAS BEEN ISSUED
      case WM_COMMAND:
      if (wParam>100 && wParam<200)
         {
         switch (wParam)
	         {
            //------------------------- FILE MENU -----------------------------
            // NEW FILE
	         case 101:
            FileNew (hWnd);
            break;

            // OPEN AN EXISTING FILE
	         case 102:
            FileOpen(hWnd);
            break;

            // SAVE THE SOURCE IMAGE
	         case 103:
            FileSave(hWnd);
            break;

            // SAVE THE SOURCE IMAGE AS A NEW NAME
	         case 104:
            FileSaveAs(hWnd);
            break;

            // CLOSE IWF
	         case 108:
            PostMessage(hWnd,WM_CLOSE,0,0);
            break;

            // SET USER FILE OPTIONS
            case 109:
            CallDialogBox(hWnd,DP_UserFile,"USER_FILE");
            break;
            }
         }

      if (wParam>200 && wParam<900 && IsWindow(hWndSrc))
         {
         IMAGE *image;

	      if (IsWindow(hWndSrc))
		      image = GetImage(hWndSrc);           // Get attached data set.
         else
            {
            MessageBox(NULL,"No source image!","WP_Main",MB_OK);
            break;
            }

         SetCursor(LoadCursor(NULL,IDC_WAIT));   // Set Hourglass cursor.
         switch (wParam)
	         {
            //------------------------- EDIT MENU -----------------------------
            // UNDO/REDO THE LAST CHANGES TO THE IMAGE
            case 210:
            if (!GetUndoImage())
               {
               MessageBox(NULL,"Nothing to undo","Image Workframe",MB_OK);
               break;
               }
            SetImage(hWndSrc,_undo);                   // Attach undo image
            if (_undo->hres!=image->hres || _undo->vres!=image->vres)
               PostMessage(hWndSrc,WM_USER,0,0);          // Re-do window
            _undo = image;                             // Set undo image to previous
            break;

            case 220:                            break;
            case 230:                            break;
            case 240:                            break;

            // MAKE A DUPLICATE OF THE SOURCE IMAGE
            case 250:
            DuplicateImageWindow(hWndSrc);
            break;

            // SELECT THE ENTIRE IMAGE AREA
            case 260:
	         SetRect(&image->rc,0,0,image->hres,image->vres);
            break;

            //------------------------- VIEW MENU -----------------------------
            // SET THE IMAGE DISPLAY TO 100%
            case 310:
            if (GetScale(hWndSrc)!=100)
               {
               SetScale(hWndSrc,100);
	            PostMessage(hWndSrc,WM_USER,0,0);    // Force image window rebuild
               }
            break;

            // ZOOM IN THE IMAGE
            case 320:
            {
            int scale;

            scale = GetScale(hWndSrc);
            if (scale<400)
               {
               SetScale(hWndSrc,scale*2);
	            PostMessage(hWndSrc,WM_USER,0,0);    // Force image window rebuild
               }
            }
            break;

            // ZOOM OUT THE IMAGE
            case 330:
            {
            int scale;

            scale = GetScale(hWndSrc);
            if (scale>25)
               {
               SetScale(hWndSrc,scale/2);
	            PostMessage(hWndSrc,WM_USER,0,0);    // Force image window rebuild
               }
            }
            break;

            // DISPLAY THE IMAGE INFO
            case 340:
            CallDialogBox(hWnd,DP_Info,"INFO");
            break;

            //------------------------- IMAGE TOOLS ------------------------
            // PERFORM IMAGE SCALING
            case 410:
            CallDialogBox(hWnd,DP_Scale,"SCALE");
            break;

            // PERFORM IMAGE TRANSFORMATION
            case 420:
            CallDialogBox(hWnd,DP_Transform,"TRANSFORM");
            break;

            // PERFORM FRAME TO FRAME OPERATIONS
            case 430:
            CallDialogBox(hWnd,DP_Frame,"FRAME");
            break;

            // MATROX MVP-AT FRAME GRABBER TOOL
            case 450:
            //Matrox(hWnd);
            break;

            //------------------------- REGION TOOLS ----------------------
            // START HISTOGRAM TOOL
            case 610:
            Histogram(hWnd);
            break;

            // START PROFILER TOOL
            case 620:
            Profiler(hWnd);
            break;

            // START IMAGE CONVOLUTION TOOL
            case 630:
            Convolver(hWnd);
            break;

            // START RENDER TOOL
            case 640:
            Render(hWnd);
            break;
            }

         SetCursor(LoadCursor(NULL,IDC_ARROW));     // Restore arrow cursor.
         InvalidateRect(hWndSrc,NULL,FALSE);        // Invalidate image
         }

      if (wParam>900 && wParam<1000)
         switch (wParam)
            {
            // DISPLAY NOT AVAILABLE BOX
				case 998:
            MessageBox(hWnd,"Item not yet available","IWF",MB_OK);
            break;

            // DISPLAY ABOUT BOX
				case 999:
            CallDialogBox(hWnd,DP_About,"ABOUT");
            break;
            }

      // USER MODULESS
      if (wParam >= 1000)
         switch (wParam)
            {
            // Creation procedures for user modules
            #if defined(MODULE0)
            case 1000: MODULE0(hWnd); break;
            #endif
            #if defined(MODULE1)
            case 1001: MODULE1(hWnd); break;
            #endif
            #if defined(MODULE2)
            case 1002: MODULE2(hWnd); break;
            #endif
            #if defined(MODULE3)
            case 1003: MODULE3(hWnd); break;
            #endif
            #if defined(MODULE4)
            case 1004: MODULE4(hWnd); break;
            #endif
            #if defined(MODULE5)
            case 1005: MODULE5(hWnd); break;
            #endif
            #if defined(MODULE6)
            case 1006: MODULE6(hWnd); break;
            #endif
            #if defined(MODULE7)
            case 1007: MODULE7(hWnd); break;
            #endif
            }
      break;

      case WM_CLOSE:                             // REQUEST TO EXIT PGM.
      {
      char sz[80];
    
      sprintf(sz,"Exit Image Workframe?");
      if (MessageBox(hWnd,sz,"EXIT",MB_YESNO | MB_ICONQUESTION) == IDYES)
         {
         DestroyWindow(hWnd);
         }
      }
      break;

      case WM_DESTROY:                           // WINDOW IS TO BE DESTROYED.
      if (_undo)
			_undo = DestroyImage(_undo);
      DeleteObject(hPalMain);                    // Delete palette.
      PostQuitMessage(0);                        // Tell program to quit.
      break;

      default:                                   // Pass on if unprocessed.
      return(DefWindowProc(hWnd, wMsg, wParam, lParam));
      }

   return(NULL);
   }

/*
------------------------------------------------------------------------
Load_Settings(void)

Description:
Loads the module settings from disk file.
------------------------------------------------------------------------
*/
static void Load_Settings(void)
   {
   FILE *fp;

   fp = fopen("iwf.ini","r");
   if (!fp)
      {
      MessageBeep(-1);
      MessageBox(NULL,"Cannot open initialization file",MODULE,MB_OK);
      return;
      }

   Load_Item(fp,"header"    ,&_hsize,1);
   Load_Item(fp,"pixelsize" ,&_psize, 1);
   Load_Item(fp,"pixelshift",&_pshift,1);
   Load_Item(fp,"imagewidth",&_width ,1);
   Load_Item(fp,"imageheight",&_height,1);

   fclose(fp);
   }



/*
---------------------------------------------------------------------
HPALETTE CreateLogPalette(int size)

Creates a logical palette for mapping into the system palette.
---------------------------------------------------------------------
*/
HPALETTE CreateLogPalette(int size)
   {
   LOGPALETTE *npPal;
   HPALETTE hPal;

   npPal = (LOGPALETTE *)LocalAlloc(LPTR,sizeof(LOGPALETTE)+size*sizeof(PALETTEENTRY));
   npPal->palVersion    = 0x300;              // Windows version
   npPal->palNumEntries = size;               // Palette size

   hPal = CreatePalette((LPLOGPALETTE)npPal); // Create a logical palette

   LocalFree((LOCALHANDLE)npPal);             // Free the palette info structure

   return(hPal);             
   }


/*
---------------------------------------------------------------------

---------------------------------------------------------------------
*/
void FillColors(void)
      {
      int i,intensity;

      // Create a 64 level grey section for normal image display.
      for (i=0; i<64; i++)
         {
         peGray[i].peRed   = i*4;
         peGray[i].peGreen = i*4;
         peGray[i].peBlue  = i*4;
         peGray[i].peFlags = NULL;
         }

      for (i=0; i<125; i++)
         {
         peUser[i].peRed   = (i/ 1)%5 * 63;
         peUser[i].peGreen = (i/ 5)%5 * 63;
         peUser[i].peBlue  = (i/25)%5 * 63;
         peUser[i].peFlags = NULL;
         }

      for (i=0; i<64; ++i)
         {
         pePseudo[i].peRed   = 0;
         pePseudo[i].peGreen = 0;
         pePseudo[i].peBlue  = 0;
         pePseudo[i].peFlags = NULL;
         }

      for (i=0; i<8; ++i)
         {
         intensity = (i*16+i);

         pePseudo[i+ 0].peRed   = 0;
         pePseudo[i+ 8].peRed   = 0;
         pePseudo[i+16].peRed   = 0;
         pePseudo[i+24].peRed   = 0;
         pePseudo[i+32].peRed   = 0   + intensity;
         pePseudo[i+40].peRed   = 128 + intensity;
         pePseudo[i+48].peRed   = 255;
         pePseudo[i+56].peRed   = 255;

         pePseudo[i+ 0].peBlue  = 128 + intensity;
         pePseudo[i+ 8].peBlue  = 255;
         pePseudo[i+16].peBlue  = 255 - intensity;
         pePseudo[i+24].peBlue  = 127 - intensity;
         pePseudo[i+32].peBlue  = 0;
         pePseudo[i+40].peBlue  = 0;
         pePseudo[i+48].peBlue  = 0;
         pePseudo[i+56].peBlue  = 0;

         pePseudo[i+ 0].peGreen = 0;
         pePseudo[i+ 8].peGreen =   0 + intensity;
         pePseudo[i+16].peGreen = 128 + intensity;
         pePseudo[i+24].peGreen = 255 - intensity;
         pePseudo[i+32].peGreen = 128;
         pePseudo[i+40].peGreen = 128 + intensity;
         pePseudo[i+48].peGreen = 255 - intensity;
         pePseudo[i+56].peGreen = 127 - intensity;
         }

      for (i=0; i<64; ++i)
         {
         peMain[i+  0] = peGray[i];
         peMain[i+ 64] = pePseudo[i];
         peMain[i+128] = peUser[i];
         peMain[i+192] = peUser[i+64];
         }
      }

/*
--------------------------------------------------------------------------

--------------------------------------------------------------------------
*/
void SetUndoImage(IMAGE *image)
	{
	if (_undo)                                    // Check for undo image
		_undo = DestroyImage(_undo);                 // Destroy the undo image
	if (_undo)                                    // Check for undo image
		{                                          // Error if it exists
		MessageBox(NULL,"Unable to free undo buffer","Image Workframe",MB_OK);
		return;
		}
	_undo = image;
	}

/*
--------------------------------------------------------------------------

--------------------------------------------------------------------------
*/
IMAGE *GetUndoImage(void)
	{
	return(_undo);
	}

//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~ End of IWF.C ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
