/* NTFS authentication */

#include <windows.h>
#include <stdio.h>
#include <io.h>
#include <string.h>
#include <malloc.h>

#include "ntfsauth.h"

#define PERR(a) printf(a);

#define GROUP   "Group"
#define OWNER   "Owner"
#define WORLD   "World"
#define USER    "User"

struct NTFSMAP
{
	u_long      unixId;
	char*       Name;
	char*       Type;
	ACCESS_MASK Mask;
	PSID        psid;
	LPSTR       Domain;
	struct NTFSMAP*    next;
};

typedef struct NTFSMAP NtfsMap;

struct NTFSHOST
{
	LPSTR       Host;
	NtfsMap*    User;
	NtfsMap*    Group;
	NtfsMap*    World;
	struct NTFSHOST*   next;
};
typedef struct NTFSHOST NtfsHost;

struct NTFSDRIVE
{
	LPSTR   Type;
	LPSTR   Drive;
	NtfsHost*   Host;
	struct NTFSDRIVE*  next;
};
typedef struct NTFSDRIVE     NtfsDrive;


static NtfsHost* NtfsInitHost( LPSTR Host);
static NtfsMap* NtfsInitMap( LPSTR Host, LPSTR Type);
static FILE* OpenMapFile( LPSTR FileName);
static BOOL GetNextMapEntry( FILE* fp, u_long* punixId, LPSTR NTName);
static void CloseMapFile( FILE* fp);
static NtfsMap* NtfsGetAttrib( LPSTR FileName, LPSTR Type);
static NtfsMap* NtfsGetOwner( LPSTR FileName);
static NtfsMap* NtfsGetWorld( LPSTR FileName);
static NtfsMap* NtfsGetGroup( LPSTR FileName);
static NtfsDrive* NtfsGetDrive( LPSTR FileName);
static BOOL NtfsSetOwner(  PSECURITY_DESCRIPTOR psd, NtfsDrive* pDrive, 
			u_long unixId);
static BOOL NtfsSetRights(  PACL pAcl, NtfsDrive* pDrive, 
			u_long unixId, u_long na_mode, LPSTR type);
static BOOL NtfsSetGroup(  PACL pAcl, NtfsDrive* pDrive, 
			u_long oldUnixId, u_long newUnixId);
static BOOL NtfsChangeDacl( PACL pAcl, ACCESS_MASK Mask, 
				PSID pOldSid, PSID pNewSid);

static NtfsDrive*   FirstDrive= NULL;
static NtfsHost*    FirstHost = NULL;
		   
BOOL NtfsSetUnixAttrib( char* FileName, struct nfsfattr* attr)
{
#define                                 SZ_ABS_SD_BUF  500
#define                                 SZ_DACL_BUF    500
#define                                 SZ_SACL_BUF    500
#define                                 SZ_SID_OWN_BUF 500
#define                                 SZ_SID_PG_BUF  500
	UCHAR                ucBufAbs          [SZ_ABS_SD_BUF];
	UCHAR                ucBufDacl         [SZ_DACL_BUF];
	UCHAR                ucBufDacl2        [SZ_DACL_BUF];
	UCHAR                ucBufSacl         [SZ_SACL_BUF];
	UCHAR                ucBufCtrl         [sizeof(PSECURITY_DESCRIPTOR_CONTROL)];
	UCHAR                ucBufSidOwn       [SZ_SID_OWN_BUF];
	UCHAR                ucBufSidPG        [SZ_SID_PG_BUF];
	DWORD                dwSDLength       = 0;
	DWORD                dwDACLLength     = SZ_DACL_BUF;
	DWORD                dwSACLLength     = SZ_SACL_BUF;
	DWORD                dwSidOwnLength   = SZ_SID_OWN_BUF;
	DWORD                dwSidPGLength    = SZ_SID_PG_BUF;
	DWORD                dwSDLengthNeeded;
	PSECURITY_DESCRIPTOR psdSrelFileSD    = NULL;
	PSECURITY_DESCRIPTOR psdAbsFileSD     = (PSECURITY_DESCRIPTOR)&ucBufAbs;
	PSECURITY_DESCRIPTOR_CONTROL psdcCtrl = (PSECURITY_DESCRIPTOR_CONTROL)&ucBufCtrl;
	PACL                 paclDacl         = (PACL)&ucBufDacl;
	PACL                 paclDacl2        = (PACL)&ucBufDacl2;
	PACL                 paclSacl         = (PACL)&ucBufSacl;
	PSID                 psidSidOwn       = (PSID)&ucBufSidOwn;
	PSID                 psidSidPG        = (PSID)&ucBufSidPG;
	BOOL                 bDaclPresent;
	BOOL                 bDaclDefaulted;
	BOOL                 bSaclPresent;
	BOOL                 bSaclDefaulted;
	BOOL                 bOwnerDefaulted;
	BOOL                 bGroupDefaulted;
	BOOL                 bSDSelfRelative;
	DWORD                dwRevision;
	ACCESS_ALLOWED_ACE*  pAce;
	DWORD       AceIndex;

	struct nfsfattr oldattr;
	struct nfsfattr* poldattr = &oldattr;
	NtfsDrive*  pDrive;
	u_long      na_mode;
	u_long      ModeMask;
	u_long      OldId;
	u_long      NewId;
	SECURITY_INFORMATION    si;

	//oldattr = 0;

#define PERR(a) printf("NTFSAUTH.C NtfsSetUnixAttr () failedat %s\n",a)

	printf( "    NtfsSetAttr( %s, Uid %ld, Gid %ld, Mode %lx)\n", 
		   FileName, attr->na_uid, attr->na_gid, attr->na_mode);

	pDrive = NtfsGetDrive( FileName);

	if( ( pDrive == NULL) || (stricmp( pDrive->Type, "NTFS") ) )
	{ PERR("pDrive == NULL || ...");
	  return(FALSE);
	}

	si = (SECURITY_INFORMATION)( OWNER_SECURITY_INFORMATION
							   | GROUP_SECURITY_INFORMATION
							   | DACL_SECURITY_INFORMATION
							  /* | SACL_SECURITY_INFORMATION */ );

	dwSDLength = dwSDLengthNeeded = 0;
	GetFileSecurity( FileName, si, psdSrelFileSD, dwSDLength,
			(LPDWORD)&dwSDLengthNeeded);
	if( !dwSDLengthNeeded)
	{ PERR("GetFileSecurity1");
	  return(FALSE);
	}

	dwSDLength = dwSDLengthNeeded;
	if (NULL == ( psdSrelFileSD  = (PSECURITY_DESCRIPTOR) 
					_alloca(dwSDLengthNeeded) ) )
	{ PERR("NULL == psdrel..");
	  return(FALSE);
	}

	if (!GetFileSecurity(FileName, si, psdSrelFileSD, dwSDLength,
			(LPDWORD)&dwSDLengthNeeded))
	{ PERR("GetFileSecurity2");
	  return(FALSE);
	}


	if (!InitializeSecurityDescriptor(psdAbsFileSD,
				   SECURITY_DESCRIPTOR_REVISION))
	{ PERR("InitializeSecurityDescriptor");
	  return FALSE;
	}
	if (!GetSecurityDescriptorControl(psdSrelFileSD,
			psdcCtrl,
			&dwRevision))
	{ PERR("GetSecurityDescriptorControl");
	  return FALSE;
	}

	bSDSelfRelative = (SE_SELF_RELATIVE & *psdcCtrl);

	if (bDaclPresent = (SE_DACL_PRESENT   & *psdcCtrl))
	{                // SE_DACL_DEFAULTED ignored if SE_DACL_PRESENT not set
	  bDaclDefaulted = (SE_DACL_DEFAULTED & *psdcCtrl);
	}
	else
	{ // No DACL at all
	}

	if (!GetSecurityDescriptorDacl(psdSrelFileSD,
			&bDaclPresent,      // fDaclPresent flag
			&paclDacl,
			&bDaclDefaulted))   // is/is not a default DACL
	{ PERR("GetSecurityDescriptorDacl");
	  return FALSE;
	}
	if (!SetSecurityDescriptorDacl(psdAbsFileSD,
			bDaclPresent,       // fDaclPresent flag
			paclDacl,
			bDaclDefaulted))    // is/is not a default DACL
	{ PERR("SetSecurityDescriptorDacl");
	  return FALSE;
	}


	if (bSaclPresent = (SE_SACL_PRESENT   & *psdcCtrl))
	{                // SE_SACL_DEFAULTED ignored if SE_SACL_PRESENT not set
	  bSaclDefaulted = (SE_SACL_DEFAULTED & *psdcCtrl);
	}
	else
	{ // No SACL at all
	}

	if (!GetSecurityDescriptorSacl(psdSrelFileSD,
			&bSaclPresent,      // fSaclPresent flag
			&paclSacl,
			&bSaclDefaulted))   // is/is not a default SACL
	{ PERR("GetSecurityDescriptorSacl");
	  return FALSE;
	}
	if (!SetSecurityDescriptorSacl(psdAbsFileSD,
			bSaclPresent,       // fSaclPresent flag
			paclSacl,
			bSaclDefaulted))    // is/is not a default SACL
	{ PERR("SetSecurityDescriptorSacl");
	  return FALSE;
	}

	bOwnerDefaulted = (SE_OWNER_DEFAULTED & *psdcCtrl);

	if (!GetSecurityDescriptorOwner(psdSrelFileSD,
			&psidSidOwn,
			&bOwnerDefaulted))   // is/is not a default Owner
	{ PERR("GetSecurityDescriptorOwner");
	  return FALSE;
	}
	if (!SetSecurityDescriptorOwner(psdAbsFileSD,
			psidSidOwn,
			bOwnerDefaulted))    // is/is not a default Owner
	{ PERR("SetSecurityDescriptorOwner");
	  return FALSE;
	}

	bGroupDefaulted = (SE_GROUP_DEFAULTED & *psdcCtrl);

	if (!GetSecurityDescriptorGroup(psdSrelFileSD,
			&psidSidOwn,
			&bGroupDefaulted))   // is/is not a default Group
	{ PERR("GetSecurityDescriptorGroup");
	  return FALSE;
	}
	if (!SetSecurityDescriptorGroup(psdAbsFileSD,
			psidSidOwn,
			bGroupDefaulted))    // is/is not a default Group
	{ PERR("SetSecurityDescriptorGroup");
	  return FALSE;
	}

	if( !InitializeAcl( paclDacl2, SZ_DACL_BUF, paclDacl->AclRevision))
	{ PERR("InitializeAcl");
	  return FALSE;
	}

	AceIndex = 0;

	for( AceIndex = 0; AceIndex < paclDacl->AceCount; AceIndex++)
	{
		if( FALSE == GetAce( paclDacl, AceIndex, (LPVOID)&pAce) )
			continue;
		if( pAce->Header.AceType == ACCESS_ALLOWED_ACE_TYPE)
		{
			if( FALSE == AddAccessAllowedAce( paclDacl2, 
						paclDacl->AclRevision, 
						pAce->Mask, &(pAce->SidStart)))
			{ PERR("AddAccessAllowedAce");
				return FALSE;
			}
		}
		else if( pAce->Header.AceType == ACCESS_DENIED_ACE_TYPE)
		{
			if( FALSE == AddAccessDeniedAce( paclDacl2, 
						paclDacl2->AclRevision, 
						pAce->Mask, &(pAce->SidStart)))
			{ PERR("AddAccessDeniedAce");
				return FALSE;
			}
		}
	}

	oldattr.na_uid = 0xffffffff;
	oldattr.na_gid = 0xffffffff;
	oldattr.na_mode = 0xffffffff;

	NtfsGetUnixAttrib( FileName, &oldattr);
	if( attr->na_mode == -1)
		attr->na_mode == oldattr.na_mode;
	if( attr->na_uid == -1)
		attr->na_uid = oldattr.na_uid;
	if( attr->na_gid == -1)
		attr->na_gid = oldattr.na_gid;

	ModeMask = UPERM_OWNER;
	na_mode = (attr->na_mode & ModeMask);
	OldId = oldattr.na_uid;
	NewId = attr->na_uid;
	
	if( OldId != NewId )
	{
		NtfsSetOwner( psdAbsFileSD, pDrive, NewId);
		if( FALSE == NtfsSetRights( paclDacl2, pDrive, NewId, na_mode, OWNER))
		{ PERR("NtfsSetowner(owner)");
			return FALSE;
		}
	}    
	if (na_mode != (oldattr.na_mode & ModeMask) )
	{
		if( FALSE == NtfsSetRights( paclDacl2, pDrive, NewId, na_mode, OWNER))
		{ PERR("NtfsSetRights(ownerrights)");
			return FALSE;
		}
	}

	ModeMask = UPERM_GROUP;
	na_mode = (attr->na_mode & ModeMask);
	OldId = oldattr.na_gid;
	NewId = attr->na_gid;

	if( OldId != NewId )
	{
		if( FALSE == NtfsSetGroup( paclDacl2, pDrive, OldId, NewId))
		{ PERR("NtfsSetGroup(group)");
			return FALSE;
		}
	}

	if (na_mode != (oldattr.na_mode & ModeMask) )
	{
		if( FALSE == NtfsSetRights( paclDacl2, pDrive, NewId, na_mode, GROUP))
		{ PERR("NtfsSetRights(GroupRights)");
			return FALSE;
		}
	}

	ModeMask = UPERM_WORLD;
	na_mode = (attr->na_mode & ModeMask);
	if( NULL == pDrive->Host->World)
		return TRUE;

	OldId = pDrive->Host->World->unixId;
	NewId = OldId;

	if ( na_mode != (oldattr.na_mode & ModeMask) )
	{
		if( FALSE == NtfsSetRights( paclDacl2, pDrive, NewId, na_mode, WORLD))
		{ PERR("NtfsSetRights(WorldRights)");
			return FALSE;
		}
	}

	if (!SetSecurityDescriptorDacl(psdAbsFileSD,
			bDaclPresent,       // fDaclPresent flag
			paclDacl2,
			bDaclDefaulted))    // is/is not a default DACL
	{ PERR("SetSecurityDescriptorDacl");
	  return FALSE;
	}
	   
	if( !SetFileSecurity( FileName, si, psdAbsFileSD))
	{ PERR("SetFileSecurity");
	  printf("%ld = GetLastError()\n", GetLastError());
	  return FALSE;
	}
	return TRUE;
}

static BOOL NtfsSetGroup( PACL pAcl, NtfsDrive* pDrive, 
		u_long oldUnixId, u_long newUnixId )
{

	NtfsMap*    pOldMap;
	NtfsMap*    pNewMap;
	PSID        pOldSid;

	if( NULL == ( pOldMap = pNewMap = pDrive->Host->Group) )
		return FALSE;

	while( pOldMap != NULL && pOldMap->unixId != oldUnixId)
		pOldMap = pOldMap->next;

	if( pOldMap == NULL)
		pOldSid = NULL;
	else
		pOldSid = pOldMap->psid;
					   

	while( pNewMap != NULL && pNewMap->unixId != newUnixId)
		pNewMap = pNewMap->next;

	if( pNewMap == NULL)
		return FALSE;

	return NtfsChangeDacl( pAcl, 0xffffffff, pOldSid, pNewMap->psid);
}

static BOOL NtfsChangeDacl( PACL pAcl, ACCESS_MASK Mask, 
				PSID pOldSid, PSID pNewSid)
{
	DWORD       AceIndex;
	ACCESS_ALLOWED_ACE* pAce = NULL;

	if( pOldSid == NULL && pNewSid == NULL)
		return FALSE;

	AceIndex = 0;


	if( pOldSid)
	{
		if( !IsValidSid( pOldSid) )
			return FALSE;

		for( AceIndex = pAcl->AceCount ; AceIndex--; )
		{
			if( FALSE == GetAce( pAcl, AceIndex, (LPVOID)&pAce) )
				continue;
			if( (pAce->Header.AceType != ACCESS_ALLOWED_ACE_TYPE))
			   continue;

			if( EqualSid( (PSID)&(pAce->SidStart), pOldSid) )
			{
				if( FALSE == DeleteAce( pAcl, AceIndex))
					return FALSE;
				if( Mask == 0xffffffff)
					Mask = pAce->Mask;
			}
		}
	}
	if( Mask && pNewSid)
	{
		if( FALSE == AddAccessAllowedAce( 
				pAcl, pAcl->AclRevision, Mask, pNewSid))
			return FALSE;
	}

	return TRUE;
}

static BOOL NtfsSetRights(  PACL pAcl, NtfsDrive* pDrive, 
			u_long unixId, u_long na_mode, LPSTR type)
{
	NtfsMap*    pMap;
	BOOL        result = FALSE;
	u_long      Mask;

	if( !stricmp( type, WORLD))
		pMap = pDrive->Host->World;
	else
	{
		if( !stricmp( type, OWNER))
			pMap = pDrive->Host->User;

		if( !stricmp( type, GROUP))
			pMap = pDrive->Host->Group;

		while( (pMap != NULL) && (pMap->unixId != unixId))
			pMap = pMap->next;
	}
	if( pMap == NULL || pMap->psid == NULL)
		return FALSE;

	Mask = 0;
	if( na_mode & UPERM_READ)
		Mask |= GENERIC_READ;
	if( na_mode & UPERM_WRITE)
		Mask |= GENERIC_WRITE;
	if( na_mode & UPERM_EXEC)
		Mask |= GENERIC_EXECUTE;
	if( Mask == ( GENERIC_EXECUTE| GENERIC_READ | GENERIC_WRITE) )
		Mask = GENERIC_ALL;

	pMap = pMap;

	return NtfsChangeDacl( pAcl, Mask, pMap->psid, pMap->psid);
}

static BOOL NtfsSetOwner(  PSECURITY_DESCRIPTOR psd, NtfsDrive* pDrive, 
			u_long unixId)
{

	NtfsMap*    pMap;

	pMap = pDrive->Host->User;
	while( pMap != NULL)
	{
		if( pMap->unixId == unixId)
			break;
		pMap = pMap->next;
	}
	if( pMap == NULL)
		return FALSE;


	return SetSecurityDescriptorOwner( psd, pMap->psid, FALSE);
}


BOOL NtfsGetUnixAttrib( char* FileName, struct nfsfattr* attr)
{
	NtfsMap* pMap;
	NtfsDrive* pDrive;
	struct _finddata_t  findbuf;
	long fhand;

//printf( "    NtfsGetAttr( %s, Uid %ld, Gid %ld, Mode %lx)\n", 
//            FileName, attr->na_uid, attr->na_gid, attr->na_mode);

	attr->na_mode = 0;

	pDrive = NtfsGetDrive( FileName);

	if( ( pDrive == NULL) || (stricmp( pDrive->Type, "NTFS") ) )
		return FALSE;     // not NTFS, leave alone


	fhand = _findfirst( FileName, &findbuf);
	if ( fhand < 0 )
	{ PERR("_findfirst");
		return FALSE;
	}

	_findclose( fhand);

	if( findbuf.attrib & _A_SUBDIR)
		attr->na_mode = UPERM_DIR;

	attr->na_mode &= ((~UPERM_OWNER) & (~UPERM_GROUP) & (~UPERM_WORLD));
	pMap = NtfsGetOwner( FileName);
	if( pMap != NULL)
	{
		attr->na_uid = pMap->unixId;

		if( pMap->Mask & GENERIC_ALL)
			attr->na_mode |= UPERM_OWNER;
		if( pMap->Mask & (FILE_READ_ATTRIBUTES | GENERIC_READ))
			attr->na_mode |= (UPERM_READ & UPERM_OWNER);

		if( pMap->Mask & (FILE_WRITE_ATTRIBUTES | GENERIC_WRITE))
			attr->na_mode |= (UPERM_WRITE & UPERM_OWNER);

		if( pMap->Mask & (FILE_EXECUTE | GENERIC_EXECUTE))
			attr->na_mode |= (UPERM_EXEC & UPERM_OWNER);
	}
	else
	{
		attr->na_uid = 0xfffffffe;
	}

	pMap = NtfsGetGroup( FileName);
	if( pMap != NULL)
	{
		attr->na_gid = pMap->unixId;
		if( pMap->Mask & GENERIC_ALL)
			attr->na_mode |= UPERM_GROUP;
		if( pMap->Mask & (FILE_READ_ATTRIBUTES | GENERIC_READ))
			attr->na_mode |= (UPERM_READ & UPERM_GROUP);

		if( pMap->Mask & (FILE_WRITE_ATTRIBUTES | GENERIC_WRITE))
			attr->na_mode |= (UPERM_WRITE & UPERM_GROUP);

		if( pMap->Mask & (FILE_EXECUTE | GENERIC_EXECUTE))
			attr->na_mode |= (UPERM_EXEC & UPERM_GROUP);
	}
	else
	{
		attr->na_gid = 0xfffffffe;
	}

	pMap = NtfsGetWorld( FileName);
	if( pMap != NULL)
	{
		if( pMap->Mask & GENERIC_ALL)
			attr->na_mode |= UPERM_WORLD;

		if( pMap->Mask & (FILE_READ_ATTRIBUTES | GENERIC_READ))
			attr->na_mode |= (UPERM_READ & UPERM_WORLD);

		if( pMap->Mask & (FILE_WRITE_ATTRIBUTES | GENERIC_WRITE))
			attr->na_mode |= (UPERM_WRITE & UPERM_WORLD);

		if( pMap->Mask & (FILE_EXECUTE | GENERIC_EXECUTE))
			attr->na_mode |= (UPERM_EXEC & UPERM_WORLD);
	}
	else
	{
	}
	return TRUE;
}


BOOL NtfsCompDriveType( LPSTR FileName, LPSTR DriveType)
{

	NtfsDrive* pDrive;

	pDrive = NtfsGetDrive( FileName);
	if( pDrive == NULL)
		return FALSE;

	if( stricmp( pDrive->Type, DriveType))
		return FALSE;

	return TRUE;
}

NtfsDrive* NtfsGetDrive( LPSTR FileName)
{

	char        DriveLetter[3];
	NtfsDrive* pDrive = FirstDrive;

	if( FileName[1] != ':')
		return NULL;

	strncpy( DriveLetter, FileName, 2);

	DriveLetter[2] = 0;   
   

	while( pDrive != NULL)    
	{
		if( !stricmp( pDrive->Drive, DriveLetter) )
			return pDrive;
		pDrive = pDrive->next;
	}

	return NULL;
}

BOOL NtfsInitDrive( LPSTR FileName)
{

	NtfsDrive*  pDrive = FirstDrive;
	NtfsDrive** ppDrive = NULL;
	LPSTR       lpszRemoteName = NULL;
	DWORD       cchRemoteName = 0;
	LPSTR       p;
	char        Drive[4];


	// Is drive already in list

	if( FileName[1] != ':')
		return FALSE;

	strncpy( Drive, FileName, 2);

	Drive[2] = 0;   
	Drive[3] = 0;
   

	while( pDrive != NULL)
	{
		if( !stricmp (pDrive->Drive, Drive) )
			return TRUE;
		pDrive = pDrive->next;
	}

	if( ERROR_MORE_DATA != WNetGetConnection( Drive, lpszRemoteName, 
					&cchRemoteName) )
		return FALSE;

	if( NULL == (lpszRemoteName = (LPSTR) malloc( cchRemoteName + 1)))
		return FALSE;

	// get hostname

	
	if( NO_ERROR != WNetGetConnection( Drive, lpszRemoteName, 
					&cchRemoteName) )
	{
		if( ERROR_NOT_CONNECTED != GetLastError() )
		{
			free( lpszRemoteName);
			return FALSE;
		}
	}
	else
	{
		p = strchr( lpszRemoteName + 2, '\\') ;
		*p = 0;
	}

	// Insert drive into List
	ppDrive = &FirstDrive;
	while( *ppDrive != NULL)
	{
		ppDrive = &((*ppDrive)->next);
	}

	pDrive = (NtfsDrive*) malloc( sizeof(NtfsDrive) );
	pDrive->Drive = (LPSTR) malloc( strlen( Drive) +1);
	strcpy( pDrive->Drive, Drive);
	pDrive->next = NULL;
	pDrive->Host = NtfsInitHost( lpszRemoteName);
	free( lpszRemoteName);

	{
		char    SystemName[40];
		DWORD   nSystemNameSize = sizeof(SystemName);

		Drive[2] = '\\';   
		if (FALSE == GetVolumeInformation(Drive,
						   NULL,
						   0,
						   NULL,
						   NULL,
						   NULL,
						   SystemName,
						   nSystemNameSize))
		{
			pDrive->Type = (LPSTR) malloc(sizeof("NONE"));
			strcpy(pDrive->Type, "NONE");
		}
		else
		{
			pDrive->Type = (LPSTR) malloc(strlen(SystemName) + 1);
			strcpy( pDrive->Type, SystemName);
		}
	}   
	*ppDrive = pDrive;

	return TRUE;
}

static NtfsHost* NtfsInitHost( LPSTR Host)
{
	NtfsHost*   pHost = FirstHost;
	NtfsHost**  ppHost = NULL;

	while( pHost != NULL)
	{
		if( !stricmp (pHost->Host, Host) )
			return pHost;
		pHost = pHost->next;
	}
	// Insert Host into List
	ppHost = &FirstHost;
	while( *ppHost != NULL)
	{
		ppHost = &((*ppHost)->next);
	}

	pHost = (NtfsHost*) malloc( sizeof(NtfsHost) );
	pHost->Host = (LPSTR) malloc( strlen( Host) +1);
	strcpy( pHost->Host, Host);
	pHost->next = NULL;
	*ppHost = pHost;
	pHost->User = NtfsInitMap( Host, USER);
	pHost->Group = NtfsInitMap( Host, GROUP);
	pHost->World = NtfsInitMap( Host, WORLD);
	return pHost;
}

NtfsMap* NtfsGetOwner( LPSTR FileName)
{
	return NtfsGetAttrib( FileName, OWNER);
}

NtfsMap* NtfsGetWorld( LPSTR FileName)
{
	return NtfsGetAttrib( FileName, WORLD);
}


NtfsMap* NtfsGetGroup( LPSTR FileName)
{
	return NtfsGetAttrib( FileName, GROUP);
}

static NtfsMap* NtfsGetAttrib( LPSTR FileName, LPSTR Type)
{

	char        DriveLetter[3];
	NtfsMap*    pMap = NULL;
	NtfsDrive*  pDrive = NULL;
	PSID        psid;
	PSECURITY_DESCRIPTOR    psd = NULL;
	DWORD       lsd = 0;
	DWORD       rlsd = 0;
	BOOL        Defaulted;
	BOOL        DaclPresent;            
	PACL        pAcl;
	DWORD       AceIndex;
	ACCESS_ALLOWED_ACE* pAce;
	SECURITY_INFORMATION    si;

	
	if( FileName[1] != ':')
		return NULL;

	strncpy( DriveLetter, FileName, 2);
	DriveLetter[2] = 0;   
   
	if( NULL == ( pDrive = NtfsGetDrive( DriveLetter) ) )
		return NULL;

	if( !stricmp( Type, OWNER) )
	{
		si = OWNER_SECURITY_INFORMATION;
		GetFileSecurity( FileName, si, psd, lsd, &rlsd);

		if( 0 == rlsd )
			return NULL;

		psd = (PSECURITY_DESCRIPTOR) malloc( rlsd);
		lsd = rlsd;
		while(1)
		{
			if( FALSE == GetFileSecurity( FileName, si, psd, lsd, &rlsd) )
				break;
			if( FALSE == GetSecurityDescriptorOwner( psd, &psid
					, &Defaulted))
				break;
			pMap = pDrive->Host->User;
			while( pMap != NULL)
			{
				if( (pMap->psid != NULL) && ( EqualSid( psid, pMap->psid) ) )
					break;
				pMap = pMap->next;
			}
			break;
		}
		free (psd);
		psd = NULL;

		if( pMap == NULL)
			return NULL;
	}

	lsd = rlsd = 0;
	si = DACL_SECURITY_INFORMATION;
	GetFileSecurity( FileName, si, psd, lsd, &rlsd);


	if( 0 == rlsd )
		return NULL;

	psd = (PSECURITY_DESCRIPTOR) malloc( rlsd);
	lsd = rlsd;

	while(1)
	{
		if( FALSE == GetFileSecurity( FileName, si, psd, lsd, &rlsd) )
			break;

		if( FALSE == GetSecurityDescriptorDacl( psd, &DaclPresent, &pAcl
			, &Defaulted))
				break;
		if( FALSE == DaclPresent)
			break;

		AceIndex = 0;

		for( AceIndex = 0; AceIndex < pAcl->AceCount; AceIndex++)
		{
			if( FALSE == GetAce( pAcl, AceIndex, (LPVOID)&pAce) )
				continue;
			if( pAce->Header.AceType != ACCESS_ALLOWED_ACE_TYPE)
				continue;

			psid = (PSID) & (pAce->SidStart);

			
			if( !stricmp( Type, OWNER) )
			{
				if( EqualSid( psid, pMap->psid) )
				{
					pMap->Mask = pAce->Mask;
					break;
				}
				continue;
			}
			else
			{
				if( !stricmp( Type, GROUP) )
					pMap = pDrive->Host->Group;
				else if( !stricmp( Type, WORLD) )
					pMap = pDrive->Host->World;
				while( pMap != NULL)
				{
					if( (pMap->psid != NULL) && ( EqualSid( psid, pMap->psid) ) )
					   break;
					pMap = pMap->next;
				}
				if( pMap != NULL)
				{
					pMap->Mask = pAce->Mask;
					break;
				}
			}
		}
		break;
	}


	free( psd);

	return pMap;
}

static NtfsMap* NtfsInitMap( LPSTR Host, LPSTR Type)
{

	FILE*       pFile= NULL;
	NtfsMap*    pLastMap = NULL;
	NtfsMap*    pMap = NULL;
	u_long      unixId;
	char        NTName[80];
	DWORD       lSid;
	DWORD       lDomain;
	SID_NAME_USE   smu;

	if( NULL == (pFile = OpenMapFile( Type)))
		return NULL;

	while( TRUE == GetNextMapEntry( pFile, &unixId, NTName))
	{
		pMap = (NtfsMap*) malloc( sizeof(NtfsMap));
		pMap->Name = (LPSTR) malloc( strlen( NTName) + 1);
		strcpy( pMap->Name, NTName);
		pMap->psid = NULL;
		pMap->Mask =0xffffffff;
		pMap->Domain = NULL;
		lSid = 0;
		lDomain = 0;
		LookupAccountName( Host, NTName, pMap->psid, &lSid, pMap->Domain,
				&lDomain, &smu);

		if( lSid != 0)      // Found Sid
		{
			pMap->psid = (PSID) malloc( lSid);
			pMap->Domain = (LPSTR) malloc( lDomain); 

			LookupAccountName( Host, NTName, pMap->psid, &lSid, 
					pMap->Domain,&lDomain, &smu);

//            if( ( stricmp( Type, USER) && (smu == SidTypeUser))
//                || ( stricmp( Type, GROUP) && (smu == SidTypeGroup)) )


		}

		pMap->unixId = unixId;
		pMap->next = pLastMap;
		pLastMap = pMap;
	}

	fclose( pFile);
	return pMap;
}


static FILE* OpenMapFile( LPSTR FileName)
{
	FILE*   fp;

	if ( NULL == ( fp = fopen( FileName, "r") ) )
		return NULL;

}

static BOOL GetNextMapEntry( FILE* fp, u_long* punixId, LPSTR NTName)
{
	char    buffer[128];


	while( 1)
	{
		if( NULL == fgets( buffer, sizeof(buffer), fp ) )
			return FALSE;
	
		if( 2 == sscanf( buffer, "%d %s", punixId, NTName))
			return TRUE;
	}

	return TRUE;
}

static void CloseMapFile( FILE* fp)
{
	fclose( fp);
}

