Option Explicit

Global Const Version = "1.22"

'--------------------
' DLL Routines
'--------------------
Declare Sub DragAcceptFiles Lib "Shell" (ByVal hWnd As Integer, ByVal fAccept As Integer)
Declare Function GetModuleUsage Lib "Kernel" (ByVal hinst As Integer) As Integer
Declare Function GetWindowsDirectory Lib "Kernel" (ByVal lpBuffer As String, ByVal nSize As Integer) As Integer
Declare Function SendMessage Lib "User" (ByVal hWnd As Integer, ByVal wMsg As Integer, ByVal wParam As Integer, ByVal lParam As Any) As Long
Declare Function SystemParametersInfo Lib "User" (ByVal uAction As Integer, ByVal uParam As Integer, ByVal lpvParam As String, ByVal fuWinIni As Integer) As Integer
Declare Function GetPrivateProfileInt Lib "Kernel" (ByVal SectionName$, ByVal KeyName$, ByVal Default%, ByVal IniFileName$) As Integer
Declare Function GetPrivateProfileString Lib "Kernel" (ByVal SectionName$, ByVal KeyName$, ByVal Default$, ByVal Ret$, ByVal Size%, ByVal IniFileName$) As Integer
Declare Function WritePrivateProfileString Lib "Kernel" (ByVal SectionName$, ByVal KeyName$, ByVal Value$, ByVal IniFileName$) As Integer
Declare Function ErasePrivateProfileString Lib "Kernel" Alias "WritePrivateProfileString" (ByVal SectionName As String, ByVal KeyName As String, ByVal Value As Any, ByVal IniFileName$) As Integer
Declare Function ErasePrivateProfileSection Lib "Kernel" Alias "WritePrivateProfileString" (ByVal SectionName As String, ByVal KeyName As Any, ByVal Value As Any, ByVal IniFileName$) As Integer
Declare Function WriteProfileString Lib "Kernel" (ByVal SectionName$, ByVal KeyName$, ByVal Value$) As Integer

Declare Function FindWindow Lib "User" (ByVal lpClassName As Any, ByVal lpCaption As Any) As Integer
Declare Function ShowWindow Lib "User" (ByVal Handle As Integer, ByVal Cmd As Integer) As Integer
Declare Function SFocus Lib "User" Alias "SetFocus" (ByVal Handle As Integer) As Integer
Declare Function IsWindow Lib "User" (ByVal hWnd) As Integer

'--------------------
' Windows Message constants
'--------------------
Global Const WM_USER = &H400
Global Const LB_ERR = (-1)
Global Const LB_NOTFOUND = LB_ERR
Global Const LB_FINDSTRINGEXACT = (WM_USER + 35)
Global Const LB_GETHORIZONTALEXTENT = (WM_USER + 20)
Global Const LB_SETHORIZONTALEXTENT = (WM_USER + 21)
Global Const LB_SETTABSTOPS = (WM_USER + 19)
Global Const LB_SearchFromTop = -1

'--------------------
' Common dialog box File Open/Save flags
'--------------------
Global Const OFN_HIDEREADONLY = &H4&

'--------------------
' MsgBox constants
'--------------------
Global Const MB_OK = 0
Global Const MB_OKCANCEL = 1
Global Const MB_RETRYCANCEL = 5
Global Const MB_ICONSTOP = 16
Global Const MB_ICONEXCLAMATION = 48
Global Const IDOK = 1
Global Const IDCANCEL = 2
Global Const IDRETRY = 4
Global Const IDOVERWRITE = IDOK
Global Const IDSKIP = IDCANCEL

'--------------------
' Dir() attributes
'--------------------
Global Const ATTR_DIRECTORY = 16

'--------------------
' Mouse pointer values
'--------------------
Global Const IDC_DEFAULT = 0
Global Const IDC_NODROP = 12
Global Const IDC_WAIT = 11

'--------------------
' Keyboard constants
'--------------------
Global Const KEY_ESCAPE = &H1B
Global Const KEY_RETURN = &HD

'--------------------
' Global Constants
'--------------------
Global Const SPI_SETDESKWALLPAPER = 20
Global Const SPIF_UPDATEINIFILE = 1
Global Const SPIF_SENDWININICHANGE = 2

Global Const Blank = 0&

Global Const FormIsNormal = 0
Global Const FormIsMinimized = 1
Global Const FormIsMaximized = 2

'--------------------
' Program Constants & Defaults
'--------------------
' Title of main program window
Global Const WindowTitle$ = "Wallpaper Changer"

' Max minutes (1 week)
Global Const MaxMinutes$ = "10080"

' Values for IniSelectionMethod
Global Const RandomSelection = 0
Global Const SequentialSelection = 1

' Values for IniWallpaperFromDirectory
Global Const UseDirectory = 0
Global Const UseCustomList = 1

' Whether to tile/center a file (skip used to process dropped files)
Global Const CenterFile = 0
Global Const TileFile = 1
Global Const SkipFile = 2

' Whether to use Chr(9) or "\t" for the tab char in list boxes
Global aTab$

' Wallpaper sources (directory or custom list)
Global Const WallpaperFromDirectory = 0
Global Const WallpaperFromCustomList = 1

' Error values
Global Const ERR_NONE = 0
Global Const ERR_BADDEVICE = 68
Global Const ERR_DRIVENOTREADY = 71
Global Const ERR_CONVERT = -32767

' VarType() values
Global Const V_INTEGER = 2
Global Const V_LONG = 3
Global Const V_STRING = 8

' Current mode (used to prevent recursion and stop some
' events i.e., do not change  wallpaper during startup
' unless user has set the option
Global Const ModeIdle = 1
Global Const ModeInitializing = 2
Global Const ModeUpdateComboBox = 4
Global Const ModeChanging = 8
Global Const ModeUpdateCheckMark = 16

'--------------------
' Ini file settings
'--------------------
Global WinIniFileName$                'Will set to correct value during initialization
Global Const WallpaperSection$ = "Desktop"
Global Const WallpaperKeyName$ = "Wallpaper"
Global Const WallpaperTileKeyName$ = "TileWallpaper"

Global Const IniFileName$ = "PaperChg.Ini"
Global Const CustomListBitmapPrefix$ = "Bitmap"
Global Const CustomListTilePrefix$ = "Tile"

' Program settings
Global Const IniMin = 1
Global Const IniVersion = 1
Global Const IniMinutesSinceChange = 2

' Selection Options settings
Global Const IniMinSelectionSettings = 3
Global Const IniMaxSelectionSettings = 7
Global Const IniWallpaperSource = 3
Global Const IniDirectory = 4
Global Const IniTile = 5
Global Const IniSelectionMethod = 6
Global Const IniChangeFrequency = 7

' Misc Options settings
Global Const IniMinMiscSettings = 8
Global Const IniMaxMiscSettings = 12
Global Const IniEnabled = 8
Global Const IniHideIcon = 9
Global Const IniCumulativeTime = 10
Global Const IniChangeOnStartup = 11
Global Const IniQuitAfterFirstChange = 12

' Last Wallpaper settings
Global Const IniMinWallpaperSettings = 13
Global Const IniMaxWallpaperSettings = 14
Global Const IniLastWallpaperFromDirectory = 13
Global Const IniLastWallpaperFromCustomList = 14

' Custom List settings
Global Const IniNumberOfBitmaps = 15
Global Const IniMax = 15

' IniSetting layout
Global Const IniSection = 0
Global Const IniKeyName = 1
Global Const IniType = 2
Global Const IniDefault = 3
Global Const IniValue = 4
Global IniSetting(IniMin To IniMax, 4) As Variant

'--------------------
' Global Variables
'--------------------

' Number of minutes since either the last wallpaper
' change or since startup
Global Minutes As Long

' Index (within BitmapFiles) of last wallpaper displayed
Global LastWallpaperIndex As Integer

' Name of last wallpaper displayed
Global LastWallpaperFileName As String

' Aliases for fMain.BitmapFiles and fMain.ComboBitmapFiles
' This just makes referring to the actuals objects easier.
' However, these are not set until initialization is completed
Global BitmapFiles As FileListBox
Global ComboBitmapFiles As ComboBox

' Current mode (used to prevent recursion and stop some
' events i.e., do not change  wallpaper during startup
' unless user has set the option
Global Mode As Integer

'-----------------------------------------------------------
' DESCRIPTION
'   Converts a Boolean type ("Y", "YES", "N", "NO" or Integer)
'   to an Integer
'
' NOTES
'   True:  "Y", "YES" or aBoolean <> 0
'   False: "N", "NO" or aBoolean = 0
'-----------------------------------------------------------
Function BoolToInt (aBoolean As Variant) As Integer
    Select Case Trim(UCase(CStr(aBoolean)))
    Case "Y", "YES", "-1", "1"
        BoolToInt = True
    Case "N", "NO", "0"
        BoolToInt = False
    Case Else
        If VarType(aBoolean) = V_INTEGER Or VarType(aBoolean) = V_LONG Then
            If aBoolean <> 0 Then
                BoolToInt = True
            Else
                BoolToInt = False
            End If
        Else
            BoolToInt = ERR_CONVERT
        End If
    End Select
End Function

'-----------------------------------------------------------
' DESCRIPTION
'   Centers aForm on the screen.
'-----------------------------------------------------------
Sub CenterForm (aForm As Form)
    aForm.Left = (Screen.Width - aForm.Width) / 2
    aForm.Top = (Screen.Height - aForm.Height) / 2
End Sub

Sub ChangeWallpaper ()
    Dim PrevMode As Integer
    Dim Tile As Integer
    Dim FileName As String

    Screen.MousePointer = IDC_WAIT
    PrevMode = Mode
    Mode = Mode Or ModeChanging

    If ComboBitmapFiles.ListCount >= 1 Then
        GetNewWallpaper FileName, Tile
        If fMain.CheckTile <> (Tile = TileFile) Then
            fMain.CheckTile = Tile = TileFile
        Else
            SetWallpaper FileName, Tile
        End If
        Minutes = 0

        If IniSetting(IniQuitAfterFirstChange, IniValue) Then
            End
        End If
    End If

    Mode = PrevMode
    Screen.MousePointer = IDC_DEFAULT
End Sub

Function CheckIfNewVersion ()
    Dim VersionFromIni As String

    VersionFromIni = RetrieveIniSetting(IniFileName$, CStr(IniSetting(IniVersion, IniSection)), CStr(IniSetting(IniVersion, IniKeyName)), IniSetting(IniVersion, IniDefault), CStr(IniSetting(IniVersion, IniType)))
    If VersionFromIni <> Version Then
        IniSetting(IniVersion, IniValue) = Version
        SaveIniSettings IniMin, IniMax
        CheckIfNewVersion = True
    Else
        CheckIfNewVersion = False
    End If
End Function

Sub FillBitmapFilesListBox ()
    Dim iLoop As Integer
    Dim Tile As Integer
    Dim aString As String
    Dim FileName As String

    Screen.MousePointer = IDC_WAIT

    fMain.ComboBitmapFiles.Clear
    If IniSetting(IniWallpaperSource, IniValue) = UseDirectory Then
        fMain.BitmapFiles.Path = IniSetting(IniDirectory, IniValue)

        For iLoop = 0 To fMain.BitmapFiles.ListCount - 1
           fMain.ComboBitmapFiles.AddItem fMain.BitmapFiles.List(iLoop)
        Next

        aString = IniSetting(IniLastWallpaperFromDirectory, IniValue)
        LastWallpaperIndex = GetWallpaperIndex(aString)
        Tile = IniSetting(IniTile, IniValue)
    Else
        For iLoop = 0 To IniSetting(IniNumberOfBitmaps, IniValue) - 1
            FileName = RetrieveIniSetting(IniFileName, CVar(IniSetting(IniNumberOfBitmaps, IniSection)), CustomListBitmapPrefix$ + Format$(iLoop), "", "String")
            fMain.ComboBitmapFiles.AddItem FileName
        Next

        LastWallpaperIndex = IniSetting(IniLastWallpaperFromCustomList, IniValue)
        If LastWallpaperIndex >= IniSetting(IniNumberOfBitmaps, IniValue) Then
            LastWallpaperIndex = IniSetting(IniNumberOfBitmaps, IniValue) - 1
        End If

        If LastWallpaperIndex >= 0 Then
            Tile = RetrieveIniSetting(IniFileName, CVar(IniSetting(IniNumberOfBitmaps, IniSection)), CustomListTilePrefix$ + Format$(LastWallpaperIndex), CenterFile, "Tile")
        Else
            Tile = CenterFile
        End If
    End If

    fMain.CheckTile = (Tile = TileFile)
    'FileName = SetNewWallpaper(LastWallpaperIndex)
    fMain.ComboBitmapFiles.ListIndex = LastWallpaperIndex
    Screen.MousePointer = IDC_DEFAULT
End Sub

'-----------------------------------------------------------
' DESCRIPTION
'   Determines the drive portion of FileSpec.
'
' RETURNS
'   The drive specification
'-----------------------------------------------------------
Function GetDriveFromPath (FileSpec As String) As String
    Dim DirPos As Integer

    ' Check of UNC drive
    If Left(FileSpec, 2) = "\\" Then
        DirPos = InStr(3, FileSpec, "\")
        If DirPos <> 0 Then                ' Invalid UNC but chug along
            DirPos = InStr(DirPos, FileSpec, "\")
        End If
        If DirPos = 0 Then
            DirPos = Len(FileSpec) + 1
        End If
    Else
        DirPos = InStr(1, FileSpec, ":")   ' If DirPos = 0 then will get current drive
        If DirPos <> 0 Then                ' Include the ":"
            DirPos = DirPos + 1
        End If
    End If

    ' If DirPos <> 0 then got drive, other no drive
    If DirPos <> 0 Then
        GetDriveFromPath = Left(FileSpec, DirPos - 1)
    Else
        GetDriveFromPath = ""
    End If
End Function

Sub GetNewWallpaper (FileName As String, Tile As Integer)
    Dim FileIndex As Integer
    Dim TheDir As String

    If IniSetting(IniSelectionMethod, IniValue) = RandomSelection Then
        FileIndex = Int(fMain.ComboBitmapFiles.ListCount * Rnd)
    Else
        If LastWallpaperIndex < ComboBitmapFiles.ListCount - 1 Then
            FileIndex = LastWallpaperIndex + 1
        ElseIf ComboBitmapFiles.ListCount < 1 Then
            FileIndex = -1
        Else
            FileIndex = 0
        End If
    End If

    ' FileName = fMain.ComboBitmapFiles.List(FileIndex)
    FileName = SetNewWallpaper(FileIndex)
    
    If IniSetting(IniWallpaperSource, IniValue) = UseDirectory Then
        Tile = IniSetting(IniTile, IniValue)
    Else
        If FileIndex >= 0 Then
            Tile = RetrieveIniSetting(IniFileName, CVar(IniSetting(IniNumberOfBitmaps, IniSection)), CustomListTilePrefix$ + Format$(FileIndex), CenterFile, "Tile")
        End If
    End If
End Sub

'-----------------------------------------------------------
' Determine what the index of the current wallpapaer is
' (i.e., what index within the fMain.BitmapFiles.List)
'-----------------------------------------------------------
Function GetWallpaperIndex (WallpaperName As String)
    Dim iLoop As Integer
    Dim WallpaperIndex As Integer

    WallpaperIndex = -1
    iLoop = 0
    Do While WallpaperIndex = -1 And iLoop < fMain.BitmapFiles.ListCount
       If WallpaperName = fMain.BitmapFiles.List(iLoop) Then
          WallpaperIndex = iLoop
       Else
          iLoop = iLoop + 1
       End If
    Loop

    If WallpaperIndex = -1 And fMain.BitmapFiles.ListCount > 0 Then
        WallpaperIndex = 0
    End If

    GetWallpaperIndex = WallpaperIndex
End Function

'-----------------------------------------------------------
' DESCRIPTION
'   Calls the windows API to get the Windows directory.
'
' RETURNS
'   The Windows directory path (not the system directory)
'-----------------------------------------------------------
Function GetWindowsDir () As String
    Dim numchars As Integer
    Dim WinDir As String

    WinDir = String(256, 0)                      ' Size Buffer
    numchars = GetWindowsDirectory(WinDir, 256)  ' Make API Call
    GetWindowsDir = Left(WinDir, numchars)              ' Trim Buffer
    'GetWindowsDir = StrToDir(WinDir)
End Function

Function IndexInListBox (aListBoxHandle As Integer, StartPos As Integer, StringToSearchFor As String)
    IndexInListBox = SendMessage(aListBoxHandle, LB_FINDSTRINGEXACT, StartPos, StringToSearchFor)
End Function

Sub InitVars ()
    Dim IsNewVersion As Integer
    Dim WinWallpaper As String
    Dim WinWallpaperDir As String
    Dim WinWallpaperDrive As String

    aTab$ = Chr$(9)
    'aTab$ = "\t"

    WinIniFileName$ = GetWindowsDir() + "\win.ini"
    WinWallpaper = RetrieveIniSetting(WinIniFileName$, WallpaperSection$, WallpaperKeyName$, "", "String")
    ParseFileSpec WinWallpaper, WinWallpaperDrive, WinWallpaperDir, WinWallpaper
    WinWallpaperDir = WinWallpaperDrive + WinWallpaperDir
    If WinWallpaperDir = "." Then
        WinWallpaperDir = GetWindowsDir()
    End If

    ' The current version of this software
    IniSetting(IniVersion, IniSection) = "Wallpaper Changer"
    IniSetting(IniVersion, IniKeyName) = "Version"
    IniSetting(IniVersion, IniType) = "String"
    IniSetting(IniVersion, IniDefault) = Version

    ' The number of minutes sinces wallpaper was last changed (used with cumulative time option)
    IniSetting(IniMinutesSinceChange, IniSection) = "Wallpaper Changer"
    IniSetting(IniMinutesSinceChange, IniKeyName) = "Minutes Since Last Change"
    IniSetting(IniMinutesSinceChange, IniType) = "Integer"
    IniSetting(IniMinutesSinceChange, IniDefault) = 0

    ' Whether to select the wallpaper from a directory or list file
    IniSetting(IniWallpaperSource, IniSection) = "Options"
    IniSetting(IniWallpaperSource, IniKeyName) = "Wallpaper Source"
    IniSetting(IniWallpaperSource, IniType) = "Wallpaper Source"
    IniSetting(IniWallpaperSource, IniDefault) = "Directory"

    ' The directory to select wallpaper from if selecting from a directory
    IniSetting(IniDirectory, IniSection) = "Options"
    IniSetting(IniDirectory, IniKeyName) = "Wallpaper Directory"
    IniSetting(IniDirectory, IniType) = "String"
    IniSetting(IniDirectory, IniDefault) = WinWallpaperDir

    ' Whether to tile the wallpaper if selecting wallpaper from a directory
    IniSetting(IniTile, IniSection) = "Options"
    IniSetting(IniTile, IniKeyName) = "Tile Or Center"
    IniSetting(IniTile, IniType) = "Tile"
    IniSetting(IniTile, IniDefault) = TileToTileStr(RetrieveIniSetting(WinIniFileName$, WallpaperSection$, WallpaperTileKeyName$, "Center", "Tile"))

    ' Whether to select the next wallpaper sequentially or randomly
    IniSetting(IniSelectionMethod, IniSection) = "Options"
    IniSetting(IniSelectionMethod, IniKeyName) = "Selection Method"
    IniSetting(IniSelectionMethod, IniType) = "Selection Method"
    IniSetting(IniSelectionMethod, IniDefault) = "Random"

    ' How often to change the wallpaper (in minutes)
    IniSetting(IniChangeFrequency, IniSection) = "Options"
    IniSetting(IniChangeFrequency, IniKeyName) = "Change Frequency"
    IniSetting(IniChangeFrequency, IniType) = "Integer"
    IniSetting(IniChangeFrequency, IniDefault) = 120

    ' Whether the wallpaper changer is enabled
    IniSetting(IniEnabled, IniSection) = "Options"
    IniSetting(IniEnabled, IniKeyName) = "Enabled"
    IniSetting(IniEnabled, IniType) = "Boolean"
    IniSetting(IniEnabled, IniDefault) = "Yes"

    ' Whether to hide the icon (i.e., hide instead of minimize)
    IniSetting(IniHideIcon, IniSection) = "Options"
    IniSetting(IniHideIcon, IniKeyName) = "Hide Icon"
    IniSetting(IniHideIcon, IniType) = "Boolean"
    IniSetting(IniHideIcon, IniDefault) = "No"

    ' Whether to keep cumulative time (pick up where left off when exited program)
    IniSetting(IniCumulativeTime, IniSection) = "Options"
    IniSetting(IniCumulativeTime, IniKeyName) = "Cumulative Time"
    IniSetting(IniCumulativeTime, IniType) = "Boolean"
    IniSetting(IniCumulativeTime, IniDefault) = "No"

    ' Whether to change the wallpaper on startup
    IniSetting(IniChangeOnStartup, IniSection) = "Options"
    IniSetting(IniChangeOnStartup, IniKeyName) = "Change On Startup"
    IniSetting(IniChangeOnStartup, IniType) = "Boolean"
    IniSetting(IniChangeOnStartup, IniDefault) = "No"

    ' Whether to quit after startup
    IniSetting(IniQuitAfterFirstChange, IniSection) = "Options"
    IniSetting(IniQuitAfterFirstChange, IniKeyName) = "Quit After First Change"
    IniSetting(IniQuitAfterFirstChange, IniType) = "Boolean"
    IniSetting(IniQuitAfterFirstChange, IniDefault) = "No"

    ' The last wallpaper selected (and displayed) from the wallpaper directory
    IniSetting(IniLastWallpaperFromDirectory, IniSection) = "Options"
    IniSetting(IniLastWallpaperFromDirectory, IniKeyName) = "Last Wallpaper From Directory"
    IniSetting(IniLastWallpaperFromDirectory, IniType) = "String"
    IniSetting(IniLastWallpaperFromDirectory, IniDefault) = WinWallpaper

    ' The last wallpaper selected (and displayed) from the custom list
    IniSetting(IniLastWallpaperFromCustomList, IniSection) = "Custom List"
    IniSetting(IniLastWallpaperFromCustomList, IniKeyName) = "Last Wallpaper"
    IniSetting(IniLastWallpaperFromCustomList, IniType) = "Integer"
    IniSetting(IniLastWallpaperFromCustomList, IniDefault) = 1

    ' The number of items in the custom list
    IniSetting(IniNumberOfBitmaps, IniSection) = "Custom List"
    IniSetting(IniNumberOfBitmaps, IniKeyName) = "Number Of Bitmaps"
    IniSetting(IniNumberOfBitmaps, IniType) = "Integer"
    IniSetting(IniNumberOfBitmaps, IniDefault) = 0

    RetrieveIniSettings IniMin, IniMax
    IsNewVersion = CheckIfNewVersion()
End Sub

'-----------------------------------------------------------
' DESCRIPTION
'   Converts an Integer to a Boolean type
'
' NOTES
'   True:  aInteger != 0
'   False: aInteger == 0
'-----------------------------------------------------------
Function IntToBool (aInteger As Variant) As String
    If VarType(aInteger) = V_STRING Then
        If aInteger = "0" Then
            IntToBool = "No"
        ElseIf aInteger = "1" Or aInteger = "-1" Then
            IntToBool = "Yes"
        Else
            IntToBool = ""
        End If
    ElseIf VarType(aInteger) = V_INTEGER Or VarType(aInteger) = V_LONG Then
        If aInteger <> 0 Then
            IntToBool = "Yes"
        Else
            IntToBool = "No"
        End If
    Else
        IntToBool = ""
    End If
End Function

Function IsValidDirectory (ByVal aDirectory As String)
    Dim Found As Integer
    Dim ErrorMsg As String
    Dim TheDir As String

    'Error 68 = Device Unavailable (bad drive)
    'Error 76 = Path Not Found (filename instead of directory)
    Do
        On Error Resume Next
        TheDir = Dir(StrToDir(aDirectory), ATTR_DIRECTORY)
        Found = (TheDir = ".")
        Select Case Err
            Case ERR_BADDEVICE, ERR_DRIVENOTREADY
                ErrorMsg = GetDriveFromPath(aDirectory) + " is not ready"
                If MsgBox(ErrorMsg, MB_RETRYCANCEL Or MB_ICONSTOP, WindowTitle$) <> IDRETRY Then
                    Exit Do
                End If
            Case Else
                Exit Do
        End Select
    Loop While Err <> ERR_NONE

    If Found Then
        IsValidDirectory = ERR_NONE
    Else
        IsValidDirectory = Err
    End If
End Function

Sub Main ()
    Screen.MousePointer = IDC_WAIT
    Mode = ModeInitializing
    Randomize

    ' Get defaults using current settings or
    ' using Windows API calls
    InitVars

    Set BitmapFiles = fMain.BitmapFiles
    Set ComboBitmapFiles = fMain.ComboBitmapFiles
    fMain.Show

    FillBitmapFilesListBox

    If IniSetting(IniChangeOnStartup, IniValue) Then
        ChangeWallpaper
    End If
    'fMain.CheckTile = IniSetting(IniTile, IniValue)

    Mode = ModeIdle
    Screen.MousePointer = IDC_DEFAULT
End Sub

'-----------------------------------------------------------
' DESCRIPTION
'   Take FileSpec and determine the Drive, Path (w/ trailing "\")
'   and FileName.
'
'   If FileSpec does not contain a Path, then Path is set to ".\".
'   If FileSpec does not contain a Drive, then Drive is set to the current drive.
'   If FileSpec does not contain a file spec, then FileName is set to "".
'
' NOTES
'   Everything after the last \ is assumed to be a filespec not
'   a directory.  No checking is done to see if the spec after
'   the \ is a file or directory; IT IS a file.
'-----------------------------------------------------------
Sub ParseFileSpec (FileSpec As String, Drive As String, Path As String, FileName As String)
    Dim DirPos As Integer
    Dim LastPos As Integer
    Dim Pos As Integer

    Drive = GetDriveFromPath(FileSpec)
    DirPos = Len(Drive) + 1
    
    If Drive = "" Then
        Drive = GetDriveFromPath(CurDir + ":")
    End If
    If Drive = "" Then
        MsgBox "Program Error (I guess).  Cannot determine current drive.", MB_ICONSTOP, "Error"
        End
    End If
    
    ' Check for a path specification
    Pos = DirPos - 1
    Do
        LastPos = Pos
        Pos = InStr(Pos + 1, FileSpec, "\")
    Loop Until Pos = 0

    ' If no path specification, use current dir (i.e. ".")
    If LastPos = DirPos - 1 Then
        Path = ".\"
        LastPos = DirPos
    Else
        Path = Mid(FileSpec, DirPos, LastPos - DirPos + 1)
        LastPos = LastPos + 1
    End If

    If LastPos = 0 Or LastPos > Len(FileSpec) Then
        FileName = ""
    Else
        FileName = Mid(FileSpec, LastPos)
    End If
End Sub

'-----------------------------------------------------------
' DESCRIPTION
'   Retrieve a value from an .ini file.
'
'   All string values are assumed to be not null, if
'   it is null then the default value is assigned to
'   it and written to the .ini file.
'
'   All integer values are assumed to be non-negative
'   values.  All invalid or negative values are
'   assigned the default value and then written to
'   the .ini file.
'
' SPECIAL PROCESSING FOR IniType
'   When a value is retrieved from an .ini file, special
'   processing for IniType is done as described below:
'       "Boolean": See function BoolToInt
'       "Tile":    See TileMsgToTile
'       "Wallpaper Source": Not case sensitive
'                  "0", "D", "DIR" and "DIRECTORY" are converted to
'                          to UseDirectory
'                  "1", "C", "CUSTOM" and "CUSTOM LIST" are converted
'                          to UseCustomList
'       "Selection Method": Not case sensitive
'                  "0", "R" and "RANDOM" are converted to RandomSelection
'                  "1", "S" and "SEQUENTIAL" are converted to SequentialSelection
'-----------------------------------------------------------
Function RetrieveIniSetting (IniFileName As String, IniSectionName As String, IniKeyName As String, Default As Variant, IniType As String) As Variant
    Const CantBe$ = ""
    Const MaxLen% = 1024

    Dim ErrorUseDefault As Integer
    Dim numchars As Integer
    Dim Result As Integer
    Dim ErrorMsg As String
    Dim TempValue As String
    Dim Value As String

    ErrorUseDefault = False

    Value = Space(MaxLen%)
    numchars = GetPrivateProfileString(IniSectionName, IniKeyName, CantBe$, Value, MaxLen%, IniFileName)
    Value = Trim(Left(Value, numchars))
    If Value = CantBe$ Then
        Value = Default
        Result = WriteAPrivateProfileString(IniFileName, IniSectionName, IniKeyName, Value)
    End If

    Select Case IniType
    Case "Boolean"
        Result = BoolToInt(Value)
        If Result = ERR_CONVERT Then
            ErrorUseDefault = True
        Else
            RetrieveIniSetting = Result
        End If
    
    Case "Integer", "String"
        RetrieveIniSetting = Value
    
    Case "Tile"
        Result = TileStrToTile(Value)
        If Result = ERR_CONVERT Then
            ErrorUseDefault = True
        Else
            RetrieveIniSetting = Result
        End If
    
    Case "Wallpaper Source"
        TempValue = UCase(Value)
        If TempValue = "0" Or TempValue = "D" Or TempValue = "DIR" Or TempValue = "DIRECTORY" Then
           RetrieveIniSetting = UseDirectory
        ElseIf TempValue = "1" Or TempValue = "C" Or TempValue = "CUSTOM" Or TempValue = "CUSTOM LIST" Then
            RetrieveIniSetting = UseCustomList
        Else
            ErrorUseDefault = True
        End If
    
    Case "Selection Method"
        TempValue = UCase(Value)
        If TempValue = "0" Or TempValue = "R" Or TempValue = "RANDOM" Then
           RetrieveIniSetting = RandomSelection
        ElseIf TempValue = "1" Or TempValue = "S" Or TempValue = "SEQUENTIAL" Then
            RetrieveIniSetting = SequentialSelection
        Else
            ErrorUseDefault = True
        End If
    
    Case Else
        MsgBox "Program Error (Bad IniType) in RetriveIniSetting", MB_OK Or MB_ICONSTOP, WindowTitle$
        End
    End Select

    If ErrorUseDefault Then
        ErrorMsg = "Invalid Value in " + IniFileName + Chr(13) + "Section = " + IniSectionName + Chr(13) + IniKeyName + "=" + TempValue
        ErrorMsg = ErrorMsg + String(2, 13) + "Using default value: " + CStr(Default)
        MsgBox ErrorMsg, MB_OK Or MB_ICONSTOP, WindowTitle$
        Result = SaveIniSetting(IniFileName, IniSectionName, IniKeyName, Default, IniType)
        RetrieveIniSetting = Default
    End If
End Function

'-----------------------------------------------------------
' DESCRIPTION
'   Retrives the specified .ini settings.  To retrieve all of the
'   .ini file settings in IniSetting(), use:
'       RetriveIniSettings(IniMin, IniMax)
'-----------------------------------------------------------
Sub RetrieveIniSettings (MinIndex As Integer, MaxIndex As Integer)
    Dim aLoop As Integer

    For aLoop = MinIndex To MaxIndex
        IniSetting(aLoop, IniValue) = RetrieveIniSetting(IniFileName, CVar(IniSetting(aLoop, IniSection)), CVar(IniSetting(aLoop, IniKeyName)), IniSetting(aLoop, IniDefault), CVar(IniSetting(aLoop, IniType)))
    Next
End Sub

'-----------------------------------------------------------
' DESCRIPTION
'   Save a value to an .ini file.
'
' SPECIAL PROCESSING FOR IniType
'   When a value saved to an .ini file, special processing
'   for IniType is done as described below:
'       "Boolean": See function IntToBool
'       "Tile":    See function TileToTileMsg
'       "Wallpaper Source":
'                  UseDirectory converted to "Directory"
'                  UseCustomList converted to "Custom List"
'       "Selection Method":
'                  RandomSelection converted to "Random"
'                  SequentialSelection converted to "Sequential"
'-----------------------------------------------------------
Function SaveIniSetting (IniFileName As String, IniSectionName As String, IniKeyName As String, KeyValue As Variant, IniType As String) As Integer
    Const CantBe$ = ""
    Const MaxLen% = 1024

    Dim numchars As Integer
    Dim Result As Integer
    Dim ErrorMsg As String
    Dim TempValue As String
    Dim Value As String

    Select Case IniType
    Case "Boolean"
        Value = IntToBool(KeyValue)
        If Value = "" Then
            MsgBox "Program Error (Bad KeyValue) in SaveIniSetting", MB_OK Or MB_ICONSTOP, WindowTitle$
            End
        End If
    
    Case "Integer", "String"
        Value = KeyValue
    
    Case "Tile"
        Value = TileToTileStr(KeyValue)
        If Value = "" Then
            MsgBox "Program Error (Bad KeyValue) in SaveIniSetting", MB_OK Or MB_ICONSTOP, WindowTitle$
            End
        End If
    
    Case "Wallpaper Source"
        If KeyValue = UseDirectory Then
            Value = "Directory"
        ElseIf KeyValue = UseCustomList Then
            Value = "Custom List"
        Else
            MsgBox "Program Error (Bad KeyValue) in SaveIniSetting", MB_OK Or MB_ICONSTOP, WindowTitle$
            End
        End If
    
    Case "Selection Method"
        If KeyValue = RandomSelection Then
            Value = "Random"
        ElseIf KeyValue = SequentialSelection Then
            Value = "Sequential"
        Else
            MsgBox "Program Error (Bad KeyValue) in SaveIniSetting", MB_OK Or MB_ICONSTOP, WindowTitle$
            End
        End If
    
    Case Else
        MsgBox "Program Error (Bad IniType) in SaveIniSetting", MB_OK Or MB_ICONSTOP, WindowTitle$
        End
    End Select
    
    Result = WriteAPrivateProfileString(IniFileName, IniSectionName, IniKeyName, Trim(Value))
    SaveIniSetting = Result
End Function

'-----------------------------------------------------------
' DESCRIPTION
'   Save the specified .ini settings.  To save all of the
'   .ini file settings in IniSetting(), use:
'       SaveIniSettings(IniMin, IniMax)
'-----------------------------------------------------------
Sub SaveIniSettings (MinIndex As Integer, MaxIndex As Integer)
    Dim aLoop As Integer
    Dim Result As Integer
    Dim aSection As String
    Dim aType As String
    Dim aValue As Variant
    Dim aKeyName As String

    For aLoop = MinIndex To MaxIndex
        aSection = IniSetting(aLoop, IniSection)
        aType = IniSetting(aLoop, IniType)
        aKeyName = IniSetting(aLoop, IniKeyName)
        aValue = IniSetting(aLoop, IniValue)
        Result = SaveIniSetting(IniFileName, aSection, aKeyName, aValue, aType)
    Next
End Sub

Sub SetLastWallpaper (LastFileName As String, LastIndex As Integer)
    LastWallpaperIndex = LastIndex
    LastWallpaperFileName = LastFileName
    fMain.ComboBitmapFiles.ListIndex = LastWallpaperIndex
End Sub

Function SetNewWallpaper (FileIndex As Integer)
    Dim FileName As String
    Dim TheDir As String
    Dim aTile As Integer

    FileName = fMain.ComboBitmapFiles.List(FileIndex)
    
    If IniSetting(IniWallpaperSource, IniValue) = UseDirectory Then
        TheDir = StrToDir(fMain.BitmapFiles.Path)
        IniSetting(IniLastWallpaperFromDirectory, IniValue) = FileName
        FileName = TheDir + FileName
    Else
        IniSetting(IniLastWallpaperFromCustomList, IniValue) = FileIndex
    End If

    SetLastWallpaper FileName, FileIndex
    SetNewWallpaper = FileName
End Function

Sub SetWallpaper (WallpaperFileName As String, Tile As Integer)
    Dim Result As Integer
    Dim Drive As String
    Dim FileName As String
    Dim Path As String
    Dim WinTile As Integer
    
    ParseFileSpec WallpaperFileName, Drive, Path, FileName
    WinTile = RetrieveIniSetting(WinIniFileName$, WallpaperSection$, WallpaperTileKeyName$, True, "Tile")
    If Tile <> WinTile Then
        SetWallpaperTiling Tile
        If FileName = LastWallpaperFileName Then
            Result = SystemParametersInfo(SPI_SETDESKWALLPAPER, 0, "(none)", 0)
        End If
    End If

    Result = SystemParametersInfo(SPI_SETDESKWALLPAPER, 0, WallpaperFileName, 3)
    SaveIniSettings IniMinWallpaperSettings, IniMaxWallpaperSettings
End Sub

Sub SetWallpaperTiling (Tile As Integer)
    Dim lResult As Long
    Dim Result As Integer
    Dim aType As String

'    If IniSetting(IniWallpaperSource, IniValue) = UseDirectory Then
'        IniSetting(IniTile, IniValue) = Tile
'    End If
    Result = WriteProfileString(WallpaperSection$, WallpaperTileKeyName$, Format(Abs(Tile = TileFile)))
    lResult = SendMessage(-1, 26, 0, WallpaperSection)
End Sub

'-----------------------------------------------------------
' DESCRIPTION
'   Activates a previous instance of FormName if one is found.
'
' RETURNS
'   Handle to the previous instance found.
'   If no previous instance found, then 0 is returned.
'-----------------------------------------------------------
Function ShowPrevInstance (FormName As String) As Integer
    Dim Handle As Integer
    Dim Result As Integer
    Dim hPrevInstance As Integer

    hPrevInstance = FindWindow(0&, FormName)

    If hPrevInstance <> 0 Then   ' Show copy
        Result = ShowWindow(hPrevInstance, 1)
        Result = SFocus(hPrevInstance)
    End If

    ShowPrevInstance = hPrevInstance
End Function

Function StrToDir (ByVal aPathSpec As String) As String
    If Right(aPathSpec, 1) <> "\" Then
        StrToDir = aPathSpec + "\"
    Else
        StrToDir = aPathSpec
    End If
End Function

'-----------------------------------------------------------
' DESCRIPTION
'   Converts a TileStr to a Tile type (integer)
'
' NOTES
'   Not case sensitive
'       "0", "N", "NO", "C" and "CENTER" are converted to CenterFile
'       "1", "-1", "Y", "YES", "T" and "TILE" are converted to TileFile
'-----------------------------------------------------------
Function TileStrToTile (ByVal Tile As Variant) As Integer
    Tile = Trim(UCase(CStr(Tile)))
    If Tile = "0" Or Tile = "N" Or Tile = "NO" Or Tile = "C" Or Tile = "CENTER" Then
        TileStrToTile = CenterFile
    ElseIf Tile = "1" Or Tile = "-1" Or Tile = "Y" Or Tile = "YES" Or Tile = "T" Or Tile = "TILE" Then
        TileStrToTile = TileFile
    Else
        TileStrToTile = ERR_CONVERT
    End If
End Function

'-----------------------------------------------------------
' DESCRIPTION
'   Converts a Tile type (integer) to a TileStr
'
' NOTES
'   Not case sensitive
'       CenterFile and False are converted to "Center"
'       TileFile and True are converted to "Tile"
'-----------------------------------------------------------
Function TileToTileStr (ByVal Tile As Integer) As String
    If Tile = CenterFile Or Tile = False Then
        TileToTileStr = "Center"
    ElseIf Tile = TileFile Or Tile = True Then
        TileToTileStr = "  Tile"
    Else
        TileToTileStr = ""
    End If
End Function

Function WriteAPrivateProfileString (ByVal IniFileName$, ByVal SectionName$, ByVal KeyName$, ByVal Value$) As Integer
    Dim Success
    Dim Msg$

    Success = WritePrivateProfileString(SectionName$, KeyName$, Value$, IniFileName$)
    If Success Then     ' Evaluate results.
    Else
        Msg$ = "Error writing to ini file." + Chr$(13)
        Msg$ = Msg$ + "Ini File: " + IniFileName$ + Chr$(13)
        Msg$ = Msg$ + "Section: " + SectionName$ + Chr$(13)
        Msg$ = Msg$ + "Key: " + KeyName$ + Chr$(13)
        Msg$ = Msg$ + "Value: " + Value$
        MsgBox Msg$, MB_OK Or MB_ICONSTOP, WindowTitle$
    End If

    WriteAPrivateProfileString = Success
End Function

