/*  <- declarations */
/*
 * These routines can help in importing/exporting graphics files.
 *
 * Written by Hippocrates Sendoukas, Los Angeles, September 1993
 */
#include <windows.h>
#include <stdio.h>
#include <string.h>
#include "graphio.h"
#include "filter.h"
#define strlwr(s)	AnsiLower(s)
#define strcmp(a,b)	lstrcmp((a),(b))
#define strcpy(a,b)	lstrcpy((a),(b))
#define	strlen(s)	lstrlen(s)
#define malloc(n)	mymalloc(n)
#define	free(p)		myfree(p)
#define	MAXSIZE		4096		/* Size of buffer for the contents */
					/* of the WIN.INI filter section */

/*
 * Placeable metafile header
 */
typedef struct
  {
    BYTE	key[4];
    HANDLE	h;
    RECT	bbox;
    WORD	inch;
    DWORD	reserved;
    WORD	checksum;
  } PMF_header;

static HANDLE lochinst;
static LPSTR pwmf_sig = "\xD7\xCD\xC6\x9A";

/*  -> declarations */
/*  <- get_ext */

/*
 * Return a pointer to the extension of a filename, or NULL if there
 * is no extension
 */
static LPSTR PASCAL NEAR
get_ext(LPSTR s)
{
  LPSTR p,q;

  if ( (p=strrchr(s,'\\')) == NULL )	p = s;
  if ( (q=strrchr(p, '/')) == NULL )	q = p;
  if ( (p=strrchr(q, '.')) != NULL )	p++;
  return p;
}

/*  -> get_ext */
/*  <- mymalloc */

/*
 * Replacement for malloc(); uses no static variables
 */
static LPSTR PASCAL NEAR
mymalloc(unsigned size)
{
  HANDLE	h;
  LPSTR		p;

  if ( (h=GlobalAlloc(GMEM_MOVEABLE,size)) == 0 ) return NULL;
  if ( (p=GlobalLock(h)) == NULL ) GlobalFree(h);
  return p;
}

/*  -> mymalloc */
/*  <- myfree */

/*
 * Replacement for free(); uses no static variables
 */
static void PASCAL NEAR
myfree(LPSTR p)
{
  HANDLE h;

  h = LOWORD(GlobalHandle(HIWORD(p)));
  GlobalUnlock(h);
  GlobalFree(h);
}

/*  -> myfree */
/*  <- exists */

/*
 * Test for file existence
 */
static int PASCAL NEAR
exists(LPSTR s)
{
  OFSTRUCT of;

  return( OpenFile(s,&of,OF_EXIST) != -1 );
}

/*  -> exists */
/*  <- impaux */

/*
 * Call a graphics filter to import the file
 */
static int PASCAL NEAR
impaux(HDC hdc,HWND hwnd,LPSTR fname,LPSTR libname,LPPICTINFO pi,
       int dopref,LPSTR filtopt)
{
  HANDLE	hlib,hpref;
  WORD		result;
  PFN_INFO	pfninfo;
  PFN_IMPORT	pfnimport;
  PFN_PREF	pfnpref;
  PFN_PREF2	pfnpref2;
  PFN_OUTPUT	pfnoutput;
  FILESPEC	fs;

  memset(pi,0,sizeof(PICTINFO));
  if (!exists(libname)) return GRIMP_NOFILTER;
  if ( (hlib=LoadLibrary(libname)) < 32 )  return GRIMP_NOFILTER;
  hpref = 0;
  result = (WORD) -1;
  memset(&fs,0,sizeof(fs));
  strcpy(fs.fullname,fname);
  if (!GetProcAddress(hlib,"GetFilterVersion"))		/* Version 1.0 */
    {
      pfninfo   = (PFN_INFO)	GetProcAddress(hlib,"GetFilterInfo");
      pfnpref   = (PFN_PREF)	GetProcAddress(hlib,"GetFilterPref");
      pfnimport = (PFN_IMPORT)	GetProcAddress(hlib,"ImportGR");
      if (pfnimport)
        {
          if ( pfninfo  &&  (*pfninfo)(0,filtopt,&hpref,NULL)!=2 )
            {
              FreeLibrary(hlib);
              return GRIMP_NOINIFILT;
            }
          if (dopref && pfnpref)  (*pfnpref)(lochinst,hwnd,hpref,1);
          result = (*pfnimport)(hdc,&fs,pi,hpref);
        }
    }
  else							/* Version 2.0 */
    {
      pfnpref2  = (PFN_PREF2)	GetProcAddress(hlib,"GetFilterPref");
      pfnoutput = (PFN_OUTPUT)	GetProcAddress(hlib,"OutputGR");
      if (pfnoutput)
        {
          if (pfnpref2) (*pfnpref2)(lochinst,hwnd,&hpref,fs.ftype,NULL,&fs);
          result = (*pfnoutput)(0,hdc,&fs,NULL,pi,hpref,NULL,FALSE);
        }
    }
  FreeLibrary(hlib);
  if (hpref) GlobalFree(hpref);
  return (result==IE_OK) ? GRIMP_OK :
         (result==IE_MEM_FULL) ? GRIMP_NOMEM : GRIMP_NOCONV;
}

/*  -> impaux */
/*  <- MyReadMetafile */

/*
 * Read a standard or placeable metafile from disc
 */
static int PASCAL NEAR
MyReadMetafile(LPSTR s,LPPICTINFO pi)
{
  int		fh;
  unsigned	delta;
  HANDLE	hmem;
  char huge	*memptr;
  DWORD		n;
  PMF_header	pmh;
  METAHEADER	mh;
  OFSTRUCT	of;

  memset(pi,0,sizeof(PICTINFO));
  fh = OpenFile(s,&of,OF_READ | OF_SHARE_COMPAT);
  if (fh<0) return GRIMP_NOFILE;
  if ( _lread(fh,(LPSTR)&pmh,sizeof(pmh)) != sizeof(pmh) ||
       memcmp(pmh.key,pwmf_sig,4) )
    {
      _lclose(fh);
      return ( (pi->hmf=GetMetaFile(s)) == 0 ) ? GRIMP_NOCONV : GRIMP_OK;
    }
  if (_lread(fh,(LPSTR)&mh,sizeof(METAHEADER)) != sizeof(METAHEADER) ||
       mh.mtSize==0 )
    {
      _lclose(fh);
      return GRIMP_NOCONV;
    }
  if ( (hmem=GlobalAlloc(GMEM_MOVEABLE,n = 2L*mh.mtSize)) == 0 ||
       (memptr=GlobalLock(hmem)) == NULL )
    {
      if (hmem) GlobalFree(hmem);
      _lclose(fh);
      return GRIMP_NOMEM;
    }
  _llseek(fh,sizeof(PMF_header),0);
  while (n)
    {
      delta = (n > 16384) ? 16384 : (unsigned)n;
      _lread(fh,(LPSTR)memptr,delta);
      n -= delta;
      memptr += delta;
    }
  _lclose(fh);
  GlobalUnlock(hmem);
  if ( (pi->hmf=SetMetaFileBits(hmem)) == 0 )
    {
      GlobalFree(hmem);
      return GRIMP_NOCONV;
    }
  else
    {
      memcpy(&(pi->bbox),&(pmh.bbox),sizeof(pmh.bbox));
      pi->inch = pmh.inch;
      return GRIMP_OK;
    }
}

/*  -> MyReadMetafile */
/*  <- import_graphic              public */

/*
 * Import a graphics file; this DLL can read both kinds of metafiles
 * without external support; other types of files are imported through
 * filters.
 */
int FAR PASCAL
import_graphic(HDC hdc,HWND hwnd,LPSTR fname,LPPICTINFO pi,int dopref,
               LPSTR inisection)
{
  int		wildcard,n;
  HCURSOR	oldcur;
  char		*p,*namext,*filtext,*buf,*buf2,*filtopt;

  if (!fname || !pi || !inisection) return GRIMP_BADPARMS;
  strlwr(fname);
  if ( (namext=get_ext(fname)) == NULL )
    {
      namext = fname + strlen(fname);
      *namext++ = '.';
      *namext = '\0';
      wildcard = 1;
    }
  else
    {
      if (!exists(fname)) return GRIMP_NOFILE;
      wildcard = 0;
    }
  oldcur = SetCursor(LoadCursor(0,IDC_WAIT));
  ShowCursor(TRUE);
  if (wildcard || !strcmp(namext,"wmf"))
    {
      strcpy(namext,"wmf");
      if ( (n=MyReadMetafile(fname,pi)) != GRIMP_NOFILE )
        {
          ShowCursor(FALSE);
          SetCursor(oldcur);
          return n;
        }
    }
  if ( (buf=malloc(MAXSIZE)) == NULL )
    {
      ShowCursor(FALSE);
      SetCursor(oldcur);
      return GRIMP_NOMEM;
    }
  buf2 = buf + MAXSIZE/2;
  GetProfileString(inisection,NULL,"",buf,MAXSIZE/2);
  n = GRIMP_NOTYPE;
  for (p=buf; *p;  p += strlen(p)+1)
    {
      GetProfileString(inisection,p,"",buf2,MAXSIZE/2);
      if ( (filtext=strchr(buf2,',')) == NULL ) continue;
      *filtext++ = '\0';
      while (*filtext==' ' || *filtext=='\t')  filtext++;
      if ( (filtopt=strchr(filtext,',')) != NULL )  *filtopt++ = '\0';
      strlwr(filtext);
      if (wildcard)
        {
          if (strlen(filtext)>3) continue;
          strcpy(namext,filtext);
          if (!exists(fname)) continue;
          n = impaux(hdc,hwnd,fname,buf2,pi,dopref,filtopt);
          if (n==GRIMP_OK) break;
        }
      else if (strcmp(namext,filtext)==0)
        {
          n = impaux(hdc,hwnd,fname,buf2,pi,dopref,filtopt);
          if (n==GRIMP_OK) break;
        }
    }
  free(buf);
  if (n==GRIMP_NOTYPE && wildcard)  *--namext = '\0';
  ShowCursor(FALSE);
  SetCursor(oldcur);
  return n;
}

/*  -> import_graphic              public */
/*  <- getcheck */

/*
 * Compute the checksum for a placeable metafile header
 */
static WORD PASCAL NEAR
getcheck(PMF_header *buf)
{
  WORD	*p = (WORD *) buf;
  WORD	c,n;

  c = *p++;
  for (n=9;  n>0;  n--)
    c ^= *p++;
  return c;
}

/*  -> getcheck */
/*  <- make_pmf_header */

/*
 * Construct a valid placeable metafile header
 */
static void PASCAL NEAR
make_pmf_header(PMF_header *mh,LPRECT r,int inch)
{
  strcpy((LPSTR)(mh->key),pwmf_sig);
  mh->h			= 0;
  memcpy(&(mh->bbox),r,sizeof(RECT));
  mh->inch		= inch;
  mh->reserved		= 0;
  mh->checksum		= getcheck(mh);
}

/*  -> make_pmf_header */
/*  <- write_pwmf */

/*
 * Write a placeable metafile
 */
static int PASCAL NEAR
write_pwmf(LPSTR fname,PMF_header *mh,char huge *p)
{
  int		fh,status;
  unsigned	n;
  DWORD		sz;
  OFSTRUCT	of;

//  if ( (fh=_lcreat(fname,0))<0 ) return GREXP_BADFNAME;
  fh = OpenFile(fname,&of,OF_CREATE);
  if (fh<0) return GREXP_BADFNAME;
  status = GREXP_NODISC;
  if (_lwrite(fh,(LPSTR)mh,sizeof(PMF_header)) == sizeof(PMF_header) )
    {
      sz = ((METAHEADER *)p) -> mtSize;
      sz *= 2;
      for (;;)
        {
          n = (sz>16384) ? 16384 : (unsigned)sz;
          if (n==0  ||  _lwrite(fh,(LPSTR)p,n)!=n)  break;
          p += n;
          sz -= n;
        }
      if (sz==0)  status = GREXP_OK;
    }
  _lclose(fh);
  if (status != GREXP_OK) remove(fname);
  return status;
}

/*  -> write_pwmf */
/*  <- save_meta                   public */

/*
 * Save a metafile to disc
 */
int FAR PASCAL
save_meta(LPPICTINFO pi,int mm,LPSTR fname,int placeable)
{
  LPSTR		p;
  int		n,inch;
  HDC		hdc;
  HANDLE	newmf;
  PMF_header	mh;

  if (pi==NULL || pi->hmf==0 || fname==NULL) return GREXP_BADPARMS;
  if (placeable)
    {
      if (IsRectEmpty(&(pi->bbox))) return GREXP_BADPARMS;
      if ( (newmf=CopyMetaFile(pi->hmf,NULL)) == 0 ) return GREXP_NOMEM;
      if ( (newmf=GetMetaFileBits(newmf)) == 0 ) return GREXP_NOMEM;
      if ( (p=GlobalLock(newmf)) == NULL )
        {
          DeleteMetaFile(SetMetaFileBits(newmf));
          return GREXP_NOMEM;
        }
      if ( (inch=pi->inch) == 0 )
        switch (mm)
          {
            case MM_LOENGLISH:	inch = 100;	break;
            case MM_HIENGLISH:	inch = 1000;	break;
            case MM_TWIPS:	inch = 1440;	break;
            case MM_LOMETRIC:	inch = 254;	break;
            case MM_HIMETRIC:
            case MM_ISOTROPIC:
            case MM_ANISOTROPIC: inch = 2540;	break;
            default:
              inch = GetDeviceCaps(hdc=GetDC(0),LOGPIXELSX);
              ReleaseDC(0,hdc);
          }
      make_pmf_header(&mh,&(pi->bbox),inch);
      n = write_pwmf(fname,&mh,p);
      GlobalUnlock(newmf);
      DeleteMetaFile(SetMetaFileBits(newmf));
      return n;
    }
  else
    {
      if ( (newmf=CopyMetaFile(pi->hmf,fname)) == 0 ) return GREXP_NODISC;
      DeleteMetaFile(newmf);
      return GREXP_OK;
    }
}

/*  -> save_meta                   public */
/*  <- get_openmask                public */

/*
 * Construct a mask suitable for the "List Files of Type" listbox of
 * the "Open" common dialog.
 */
int FAR PASCAL
get_openmask(LPSTR outbuf,int outbufsz,LPSTR inisection)
{
  int	n1,n2,retcode;
  LPSTR	ip,names,entry,q,q2;

  if (outbufsz<=44) return GRIMP_NOSPACE;
  if (!inisection || !outbuf) return GRIMP_BADPARMS;
  memcpy(outbuf,"All files (*.*)\0*.*\0Metafiles (*.wmf)\0*.wmf\0",44);
  outbuf += 44;
  *outbuf = '\0';
  if ( (names=malloc(MAXSIZE)) == NULL ) return GRIMP_NOMEM;
  entry = names + MAXSIZE/2;
  GetProfileString(inisection,NULL,"",names,MAXSIZE/2);
  retcode = GRIMP_OK;
  for (ip=names; *ip;  ip += strlen(ip)+1)
    {
      GetProfileString(inisection,ip,"",entry,MAXSIZE/2);
      if ( (q=strchr(entry,',')) == NULL ) continue;
      q++;
      while (*q==' ' || *q=='\t') q++;
      if ( (q2=strchr(q,',')) != NULL ) *q2 = '\0';
      if ( (q2=strchr(ip,'(')) != NULL  && q2[1]=='.') *q2 = '\0';
      strlwr(q);
      n1 = strlen(ip);
      n2 = strlen(q)+2;
      if ( (outbufsz -= n1+3+n2+1+n2+1) <= 0 )
        {
          retcode = GRIMP_NOSPACE;
          break;
        }
      wsprintf(outbuf,"%s (*.%s)",ip,q);
      outbuf += n1 + 3 + n2 + 1;
      wsprintf(outbuf,"*.%s",q);
      outbuf += n2 + 1;
      if (q2) *q2 = '(';
    }
  *outbuf = '\0';
  free(names);
  return retcode;
}

/*  -> get_openmask                public */
/*  <- LibMain */

#pragma argsused
int FAR PASCAL
LibMain(HANDLE hinst,WORD dataseg,WORD heapsize,LPSTR cmdline)
{
  lochinst = hinst;
  if (heapsize>0) UnlockData(0);
  return 1;
}
/*  -> LibMain */
