/*
 *  MS-dependent painter code
 */


#include <InterViews\btable.h>
#include <InterViews\bitmap.h>
#include <InterViews\paint.h>
#include <InterViews\canvas.h>
#include <InterViews\painter.h>
#include <InterViews\raster.h>
#include <InterViews\transformer.h>
#include <InterViews\X11\palette.h>
#include <InterViews\X11\dash.h>
#include <InterViews\X11\painterrep.h>
#include <InterViews\X11\worldrep.h>

static const PointListSize = 50;
static POINT Points[PointListSize];

inline POINT* AllocPts (int n) {
    return ((n <= PointListSize) ? Points : new POINT[n]);
}

inline void FreePts (POINT* v) {
    if (v != Points) {
	delete v;
    }
}

inline Coord* AllocCoords (int n) {
    return (new Coord[n]);
}

inline void FreeCoords (Coord* c) {
    delete c;
}

Painter::Painter() {
    rep = new PainterRep;
    Init();
}

Painter::Painter(Painter* copy) {
    rep = new PainterRep;
    Copy(copy);
    rep->Copy(copy->rep);
    if (copy->rep->xor) {
	Begin_xor();
    }
}

Painter::~Painter() {
    Unref(matrix);
    Unref(font);
    Unref(br);
    Unref(foreground);
    Unref(background);
    Unref(pattern);
    delete rep;
    if (dash != nil) {
	delete dash;
    }
}

void Painter::FillBg(boolean b) {
    if (rep->fillbg != b) {
	if (rep->xor) {
	    End_xor();
	}
	rep->fillbg = b;
    }
}

boolean Painter::BgFilled() {
    return rep->fillbg;
}

void Painter::SetColors(Color* f, Color* b) {
    if (rep->xor)
	End_xor();
    if (f != nil && foreground != f && (f->PixelValue() != -1)) {
	Unref(foreground);
	foreground = f;
	foreground->Reference();
	rep->foreground = f;
    }
    if (b != nil && background != b && (b->PixelValue() != -1)) {
	Unref(background);
	background = b;
	background->Reference();
	rep->background = b;
    }
}

void Painter::SetPattern(Pattern* pat) {
    if (rep->xor)
	End_xor();
    if (pattern != pat) {
	Unref(pattern);
	pattern = pat;
	if (pattern != nil) {
	    pattern->Reference();
	    rep->pattern = pattern;
	}
    }
}

void Painter::SetBrush(Brush* b) {
    if (rep->xor)
	End_xor();
    if (br != b) {
	Unref(br);
	br = b;
	if (dash != nil) {
	    delete dash;
	    dash = nil;
	}
	if (br != nil) {
	    br->Reference();
	    if (br != single && br->rep->count != 0) {
		if (br->Width() == 1) {
		    dash = new PixelDash;
		} else {
		    dash = new BitmapDash(br->Width());
		}
		dash->SetPattern(br);
	    }
	}
    }
}

void Painter::SetFont(Font* f) {
    if (font != f) {
	Unref(font);
	font = f;
	if (font != nil) {
	    font->Reference();
	    rep->font = f;
	}
    }
}

void Painter::Clip(
    Canvas* c, Coord left, Coord bottom, Coord right, Coord top
) {
    Coord tmp;
    int width, height;

    bottom = c->height - bottom -1;
    top = c->height - top -1;
    if (left > right) {
	tmp = right; right = left; left = tmp;
    };
    if (bottom > top) {
	tmp = bottom; bottom = top; top = tmp;
    };
    width = right - left +1;
    height = top - bottom +1;
    if ((left == 0) && (bottom == 0) &&
	(width == c->width) && (height == c->height)) {
	rep->SetClipAll ();
    } else {
	rep->SetClipRectangle(left, bottom, right, top);
    }
}

void Painter::NoClip() {
    rep->SetClipAll();
}

void Painter::SetOverwrite(boolean children) {
    if (rep->overwrite != children) {
	rep->overwrite = children;
    }
}

void Painter::SetPlaneMask(int m) {
}

void Painter::Map(Canvas* c, Coord x, Coord y, Coord& mx, Coord& my) {
    if (matrix == nil) {
	mx = x; my = y;
    } else {
	matrix->Transform(x, y, mx, my);
    }
    mx += xoff;
    my = c->height - 1 - (my + yoff);
}

void Painter::MapList(
    Canvas* c, Coord x[], Coord y[], int n, Coord mx[], Coord my[]
) {
    register Coord* xp, * yp, * mxp, * myp;
    Coord* lim;

    xp = x; yp = y;
    mxp = mx; myp = my;
    lim = &x[n];
    if (matrix == nil) {
	for (; xp < lim; xp++, yp++, mxp++, myp++) {
	    *mxp = *xp + xoff;
	    *myp = c->height - 1 - (*yp + yoff);
	}
    } else {
	for (; xp < lim; xp++, yp++, mxp++, myp++) {
	    matrix->Transform(*xp, *yp, *mxp, *myp);
	    *mxp += xoff;
	    *myp = c->height - 1 - (*myp + yoff);
	}
    }
}

void Painter::MapList(
    Canvas* c, float x[], float y[], int n, Coord mx[], Coord my[]
) {
    register float* xp, * yp;
    register Coord* mxp, * myp;
    float tmpx, tmpy, * lim;

    xp = x; yp = y;
    mxp = mx; myp = my;
    lim = &x[n];
    if (matrix == nil) {
	for (; xp < lim; xp++, yp++, mxp++, myp++) {
	    *mxp = round(*xp + xoff);
	    *myp = round(c->height - 1 - (*yp + yoff));
	}
    } else {
	for (; xp < lim; xp++, yp++, mxp++, myp++) {
	    matrix->Transform(*xp, *yp, tmpx, tmpy);
	    *mxp = round(tmpx + xoff);
	    *myp = round(c->height - 1 - (tmpy + yoff));
	}
    }
}

void Painter::Begin_xor() {
    if (!rep->xor) {
	rep->xor = true;
	rep->foreground = black;
	rep->pattern = nil;
    }
}

void Painter::End_xor() {
    if (rep->xor) {
	rep->xor = false;
	rep->foreground = foreground;
	rep->pattern = pattern;
    }
}

inline char _txkey (int i) {
    if (i >= 0) {
	return (
	    i < 32 ? i
	    : i < 160 ? (24 + (i>>2))
	    : i < 672 ? (54 + (i>>4))
	    : 127
	);
    } else {
	return (
	    i > -32 ? i
	    : i > -160 ? (-24 - (i>>2))
	    : i > -672 ? (-54 - (i>>4))
	    : -127
	);
    }
}

static int TxKey(Transformer* t, int x, int y) {
    if (t == nil) {
	return 0;
    } else {
	Coord x1, y1, x2, y2, x3, y3;
	t->Transform(0, 0, x1, y1);
	t->Transform(0, y, x2, y2);
	t->Transform(x, 0, x3, y3);
	return (
	      (_txkey(x2 - x1) << 24)
	    + (_txkey(y2 - y1 - y) << 16)
	    + (_txkey(x3 - x1 - x) << 8)
	    + (_txkey(y3 - y1))
	);
    }
}

void Painter::Stencil(Canvas* c, Coord x, Coord y, Bitmap* d, Bitmap* m) {
    if (rep->xor) {
	End_xor();
    }
    rep->Get_DC(c);
    int tx = TxKey(matrix, d->Width(), d->Height());
    if (tx == 0) {
	HBITMAP hBitmap, hBitmap1, holdBitmapSourceDC, holdBitmapMaskDC, holdBitmapDC;
	HDC hsourceDC, hmaskDC, hDC;
	HBRUSH hBrush, holdBrush;
	COLORREF fg, bg;
	Coord dx, dy;
	int w, h;

	w = d->rep->width;
	h = d->rep->height;
	fg = PALETTEINDEX(foreground->PixelValue());
	bg = PALETTEINDEX(background->PixelValue());
	hBitmap  = CreateBitmap(w, h, 1, 1, NULL);
	hBitmap1 = CreateBitmap(w, h, 1, 1, NULL);
	hDC       = CreateCompatibleDC(rep->hDC);
	hsourceDC = CreateCompatibleDC(rep->hDC);
	hmaskDC   = CreateCompatibleDC(rep->hDC);

	holdBitmapDC = SelectObject(hDC, hBitmap);
	holdBitmapSourceDC = SelectObject(hsourceDC, (HBITMAP)d->Map());
	if (m == nil) {
	    holdBitmapMaskDC = SelectObject(hmaskDC, hBitmap1);
	    PatBlt(hmaskDC, 0, 0, w, h, WHITENESS);
	} else {
	    holdBitmapMaskDC = SelectObject(hmaskDC, (HBITMAP)m->Map());
	}

	Map(c, x + d->Left(), y + d->Top(), dx, dy);

        BitBlt(rep->hDC, dx, dy, w, h, hDC, 0, 0, WHITENESS);
	BitBlt(hDC, 0, 0, w, h, hmaskDC, 0, 0, SRCCOPY);
	BitBlt(hDC, 0, 0, w, h, hsourceDC, 0, 0, MERGECOPY);
	hBrush = CreateSolidBrush(fg);
	holdBrush = SelectObject(rep->hDC, hBrush);
	BitBlt(rep->hDC, dx, dy, w, h, hDC, 0, 0, 0xE20746L);
	SelectObject(rep->hDC, holdBrush);
	DeleteObject(hBrush);

	BitBlt(hDC, 0, 0, w, h, hsourceDC, 0, 0, SRCCOPY);
	BitBlt(hDC, 0, 0, w, h, hmaskDC, 0, 0, SRCERASE);
	hBrush = CreateSolidBrush(bg);
	holdBrush = SelectObject(rep->hDC, hBrush);
	BitBlt(rep->hDC, dx, dy, w, h, hDC, 0, 0, 0xE20746L);
	SelectObject(rep->hDC, holdBrush);
	DeleteObject(hBrush);

	SelectObject(hmaskDC, holdBitmapMaskDC);
	SelectObject(hsourceDC, holdBitmapSourceDC);
	SelectObject(hDC, holdBitmapDC);
	DeleteObject(hBitmap1);
	DeleteObject(hBitmap);
	DeleteDC(hmaskDC);
	DeleteDC(hsourceDC);
	DeleteDC(hDC);

    } else {
	if (m != nil) {
	    DrawTransformedImage(
		(HBITMAP)d->rep->GetData(), x + d->Left(), y + d->Bottom(),
		(HBITMAP)m->rep->GetData(), x + m->Left(), y + m->Bottom(),
		(HWND)c->Id(), c->height, -xoff, -yoff,
		true,
		PALETTEINDEX(foreground->PixelValue()),
		PALETTEINDEX(background->PixelValue()),
		rep->hDC, matrix,
		d->rep->width, d->rep->height,
		m->rep->width, m->rep->height
	    );
	} else {
	    DrawTransformedImage(
		(HBITMAP)d->rep->GetData(), x + d->Left(), y + d->Bottom(),
		nil, 0, 0,
		(HWND)c->Id(), c->height, -xoff, -yoff,
		true,
		PALETTEINDEX(foreground->PixelValue()),
		PALETTEINDEX(background->PixelValue()),
		rep->hDC, matrix,
		d->rep->width, d->rep->height,
		0, 0
	    );
	}
    }
    rep->Release_DC(c);
}

void Painter::RasterRect(Canvas* c, Coord x, Coord y, Raster* r) {
    int tx = TxKey(matrix, r->Width(), r->Height());
    if (tx == 0) {
	Coord mx, my;
	Map(c, x, y + r->Height() - 1, mx, my);
	HDC hDC = GetDC((HWND)c->Id());
	HDC hMemDC = CreateCompatibleDC(hDC);
	HBITMAP holdBitmap = SelectObject(hMemDC, (HBITMAP)r->rep->GetData());
	BitBlt(hDC, mx, my, r->Width(), r->Height(), hMemDC, 0, 0, SRCCOPY);
	SelectObject(hMemDC, holdBitmap);
	DeleteDC(hMemDC);
	ReleaseDC((HWND)c->Id(), hDC);
    }
    else {
	rep->Get_DC(c);
	DrawTransformedImage(
	    (HBITMAP)r->rep->GetData(), x, y,
	    nil, 0, 0,
	    (HWND)c->Id(), c->Height(), -xoff, -yoff,
	    false,
	    PALETTEINDEX(foreground->PixelValue()),
	    PALETTEINDEX(background->PixelValue()),
	    rep->hDC, matrix,
	    r->rep->width, r->rep->height,
	    0, 0
	);
	rep->Release_DC(c);
    }
}

void Painter::Text(Canvas* c, const char* s, int len, Coord x, Coord y) {
    Coord x0, y0;
    Coord ybase = y + font->Height() - 1;
    Coord ytop = y + font->Height() - 1;
    int txstring = TxKey(matrix, font->Width(s, len), font->Height());

    if (style & Reversed) {
	SetColors(GetBgColor(), GetFgColor());
    }
    if (txstring == 0) {
	Map(c, x, ybase, x0, y0);
	rep->Get_DC(c);
	TextOut(rep->hDC, x0, y0, (LPSTR)s, len);
	if (style & Boldface) {
	    TextOut(rep->hDC, x0-1, y0, (LPSTR)s, len);
	}
	rep->Release_DC(c);
    } else {
	Coord curx = x;
	float fx0, fy0;
	Transformer notrans(matrix);
	notrans.Transform(0.0, 0.0, fx0, fy0);
	notrans.Translate(-fx0, -fy0);
	int txchar = TxKey(matrix, font->Width("M"), font->Height());

	for (int i = 0; i < len; ++i) {
	    Coord nextx = curx + font->Width(s+i, 1);
	    if (rep->fillbg) {
		ClearRect(c, curx, y, nextx, ytop);
	    }
	    Map(c, curx, ybase, x0, y0);
	    rep->Get_DC(c);
	    TextOut(rep->hDC, x0, y0, (LPSTR)s+i, 1);
	    if (style & Boldface) {
		TextOut(rep->hDC, x0-1, y0, (LPSTR)s+i, 1);
	    }
	    rep->Release_DC(c);
	    curx = nextx;
	}
    }
    if (style & Outlined) {
    }
    if (style & Underlined) {
	    Line(c, x, ybase, x + font->Width(s, len) - 1, ybase);
    }
    if (style & Reversed) {
	    SetColors(GetBgColor(), GetFgColor());
    }
}

COLORREF GetFg (Color* fg) {
    return (PALETTEINDEX(fg->PixelValue()));
}

void Painter::Point(Canvas* c, Coord x, Coord y) {
    Coord mx, my;
    Map(c, x, y, mx, my);

    if ((dash == nil) || rep->xor) {
	rep->Get_DC(c);
	SetPixel(rep->hDC, mx, my, GetFg(rep->foreground));
	rep->Release_DC(c);
    } else {
	dash->Init(c, rep);
	dash->Point(mx, my);
    }
}

void Painter::MultiPoint(Canvas* c, Coord x[], Coord y[], int n) {
    register int i;
    Coord mx, my;

    if ((dash == nil) || rep->xor) {
	rep->Get_DC(c);
	for (i = 0; i < n; i++) {
	    Map(c, x[i], y[i], mx, my);
	    SetPixel(rep->hDC, mx, my, GetFg(rep->foreground));
	}
	rep->Release_DC(c);
    } else {
	dash->Init(c, rep);
	for (i = 0; i < n; i++) {
	    Map(c, x[i], y[i], mx, my);
	    dash->Point(mx, my);
	}
    }
}

void Painter::Line(Canvas* c, Coord x1, Coord y1, Coord x2, Coord y2) {
    Coord mx1, my1, mx2, my2;

    Map(c, x1, y1, mx1, my1);
    Map(c, x2, y2, mx2, my2);

    if ((dash == nil) || rep->xor) {
	rep->Get_DC(c);
	SetPixel(rep->hDC, mx1, my1, GetFg(rep->foreground));
	::MoveTo(rep->hDC, mx1, my1);
	::LineTo(rep->hDC, mx2, my2);
	SetPixel(rep->hDC, mx2, my2, GetFg(rep->foreground));
	rep->Release_DC(c);
    } else {
	dash->Init(c, rep);
	dash->Line(mx1, my1, mx2, my2);
    }
}

void Painter::Rect(Canvas* c, Coord x1, Coord y1, Coord x2, Coord y2) {
    if (matrix != nil && matrix->Rotated() && !matrix->Rotated90()) {
	Coord x[4], y[4];

	x[0] = x[3] = x1;
	x[1] = x[2] = x2;
	y[0] = y[1] = y1;
	y[2] = y[3] = y2;
	if ((dash == nil) || rep->xor) {
	    Polygon(c, x, y, 4);
	} else {
	    dash->Init(c, rep);
	    dash->Polygon(x, y, 4);
	}
    } else {
	Coord left, bottom, right, top, tmp;

	Map(c, x1, y1, left, bottom);
	Map(c, x2, y2, right, top);
	if (left > right) {
	    tmp = left; left = right; right = tmp;
	}
	if (top > bottom) {
	    tmp = bottom; bottom = top; top = tmp;
	}
	if ((dash == nil) || rep->xor) {
	    rep->Get_DC(c);
	    if (rep->fill_next_primitive) {
		right++; bottom++;
	    }
	    Rectangle(rep->hDC, left, top, right+1, bottom+1);
	    rep->Release_DC(c);
	} else {
	    dash->Init(c, rep);
	    dash->Rect(left, top, right, bottom);
	}
    }
}

/*
 * XDrawRectangle() und XFillRectangle() ergeben 2 unterschiedlich groe
 * Rechtecke (siehe Referenz-160). Das von Rectangle() in Windows erzeugte
 * Rechteck entspricht dem von XFillRectangle() (bezglich der Gre).
 */

void Painter::FillRect(Canvas* c, Coord x1, Coord y1, Coord x2, Coord y2) {
    rep->fill_next_primitive = true;
    Rect(c, x1, y1, x2, y2);
}

void Painter::ClearRect(Canvas* c, Coord x1, Coord y1, Coord x2, Coord y2) {
    rep->foreground = background;
    Pattern* curpat = pattern;
    SetPattern(solid);
    FillRect(c, x1, y1, x2, y2);
    rep->foreground = foreground;
    SetPattern(curpat);
}

void Painter::Circle(Canvas* c, Coord x, Coord y, int r) {
    if (matrix != nil && (matrix->Stretched() || matrix->Rotated())) {
	Ellipse(c, x, y, r, r);
    } else {
	if ((dash == nil) || rep->xor) {
	    Coord left, top, right, bottom;
	    Map(c, x-r, y+r, left, top);
	    Map(c, x+r, y-r, right, bottom);
	    rep->Get_DC(c);
	    if (rep->fill_next_primitive) {
		right++; top--;
	    }
	    ::Ellipse(rep->hDC, left, top, right, bottom);
	    rep->Release_DC(c);
	} else {
	    Coord mx, my;
	    Map(c, x, y, mx, my);
	    dash->Init(c, rep);
	    dash->Circle(mx, my, r);
	}
    }
}

void Painter::FillCircle(Canvas* c, Coord x, Coord y, int r) {
    rep->fill_next_primitive = true;
    Circle(c, x, y, r);
}

void Painter::MultiLine(Canvas* c, Coord x[], Coord y[], int n) {
    register int i;

    if ((dash == nil) || rep->xor) {
	register POINT* v;
	v = AllocPts(n);
	for (i = 0; i < n; i++) {
	    Map(c, x[i], y[i], v[i].x, v[i].y);
	}
	rep->Get_DC(c);
	Polyline(rep->hDC, v, n);
	rep->Release_DC(c);
	FreePts(v);
    } else {
	Coord* mx = AllocCoords(n);
	Coord* my = AllocCoords(n);
	for (i = 0; i < n; i++) {
	    Map(c, x[i], y[i], mx[i], my[i]);
	}
	dash->Init(c, rep);
	dash->PolyLine(mx, my, n);
	FreeCoords(mx);
	FreeCoords(my);
    }
}

void Painter::MultiLineNoMap(Canvas* c, Coord x[], Coord y[], int n) {
    register int i;

    if ((dash == nil) || rep->xor) {
	register POINT* v;
	v = AllocPts(n);
	for (i = 0; i < n; i++) {
	    v[i].x = x[i];
	    v[i].y = y[i];
	}
	rep->Get_DC(c);
	Polyline(rep->hDC, (LPPOINT)v, n);
	rep->Release_DC(c);
	FreePts(v);
    } else {
	dash->Init(c, rep);
	dash->PolyLine(x, y, n);
    }
}

void Painter::Polygon(Canvas* c, Coord x[], Coord y[], int n) {
    register int i;

    if ((dash == nil) || rep->xor) {
	register POINT* v;
	v = AllocPts(n+1);
	for (i = 0; i < n; i++) {
	    Map(c, x[i], y[i], v[i].x, v[i].y);
	}
	if (x[i-1] != x[0] || y[i-1] != y[0]) {
	    v[i] = v[0];
	    ++i;
	}
	rep->Get_DC(c);
	::Polygon(rep->hDC, (LPPOINT)v, i);
	rep->Release_DC(c);
	FreePts(v);
    } else {
	Coord* mx = AllocCoords(n);
	Coord* my = AllocCoords(n);
	for (i = 0; i < n; i++) {
	    Map(c, x[i], y[i], mx[i], my[i]);
	}
	dash->Init(c, rep);
	dash->Polygon(mx, my, n);
	FreeCoords(mx);
	FreeCoords(my);
    }
}

void Painter::FillPolygonNoMap(Canvas* c, Coord x[], Coord y[], int n) {
    register POINT* v;
    register int i;

    v = AllocPts(n);
    for (i = 0; i < n; i++) {
	v[i].x = x[i];
	v[i].y = y[i];
    }

    rep->fill_next_primitive = true;
    rep->Get_DC(c);
    ::Polygon(rep->hDC, (LPPOINT)v, n);
    rep->Release_DC(c);

    FreePts(v);
}

void Painter::FillPolygon(Canvas* c, Coord x[], Coord y[], int n) {
    rep->fill_next_primitive = true;
    Polygon(c, x, y, n);
}

void Painter::Copy(
    Canvas* src, Coord x1, Coord y1, Coord x2, Coord y2,
    Canvas* dst, Coord x0, Coord y0
) {
    Coord sx1, sy1, sx2, sy2, sx3, sy3, sx4, sy4, dx1, dy1;
    Transformer t(matrix);

    t.Transform(x1, y1, sx1, sy1);
    t.Transform(x1, y2, sx2, sy2);
    t.Transform(x2, y2, sx3, sy3);
    t.Transform(x2, y1, sx4, sy4);
    t.Transform(x0, y0, dx1, dy1);

    int minx = min(sx1, min(sx2, min(sx3, sx4)));
    int maxx = max(sx1, max(sx2, max(sx3, sx4)));
    int miny = min(sy1, min(sy2, min(sy3, sy4)));
    int maxy = max(sy1, max(sy2, max(sy3, sy4)));

    int w = maxx - minx + 1;
    int h = maxy - miny + 1;
    int sx = minx + xoff;
    int sy = src->height - 1 - (maxy + yoff);
    int dx = dx1 - (sx1 - minx) + xoff;
    int dy = dst->height - 1 - (dy1 - (sy1 - maxy) + yoff);

    w = min(w, src->width - sx);
    h = min(h, src->height - sy);


    if ((sx1 == sx2 || sy1 == sy2) && (sx1 == sx4 || sy1 == sy4)) {
	if (src->status == CanvasOffscreen) {
/*
 *
 */
	} else {
	    HDC hDC = GetDC((HWND)src->id);
	    HDC hMemDC = CreateCompatibleDC(hDC);
	    HBITMAP hBitmap = CreateCompatibleBitmap(hDC, w, h);

	    HBITMAP holdBitmap = SelectObject(hMemDC, hBitmap);
	    BitBlt(hMemDC, 0, 0, w, h, hDC, sx, sy, SRCCOPY);
	    ReleaseDC((HWND)src->id, hDC);

	    hDC = GetDC((HWND)dst->id);
	    BitBlt(hDC, dx, dy, w, h, hMemDC, 0, 0, SRCCOPY);
	    ReleaseDC((HWND)dst->id, hDC);

	    SelectObject(hMemDC, holdBitmap);
	    DeleteDC(hMemDC);
	    DeleteObject(hBitmap);

	    dst->WaitForCopy();
	}
    } else {
	if (src->status == CanvasOffscreen) {
/*
 *
 */
	} else {
	    POINT v[4];
	    v[0].x = sx1; v[0].y = sy1;
	    v[1].x = sx2; v[1].y = sy2;
	    v[2].x = sx3; v[2].y = sy3;
	    v[3].x = sx4; v[3].y = sy4;
	    HRGN hRgn = CreatePolygonRgn((LPPOINT)v, 4, WINDING);
	    HDC hDC = GetDC((HWND)src->id);
	    HDC hMemDC = CreateCompatibleDC(hDC);
	    HBITMAP hBitmap = CreateCompatibleBitmap(hDC, w, h);

	    HBITMAP holdBitmap = SelectObject(hMemDC, hBitmap);
	    BitBlt(hMemDC, 0, 0, w, h, hDC, sx, sy, SRCCOPY);
	    ReleaseDC((HWND)src->id, hDC);

	    hDC = GetDC((HWND)dst->id);
	    HRGN holdRgn = SelectObject(hDC, hRgn);
	    BitBlt(hDC, dx, dy, w, h, hMemDC, 0, 0, SRCCOPY);
	    SelectObject(hDC, holdRgn);
	    ReleaseDC((HWND)dst->id, hDC);

	    SelectObject(hMemDC, holdBitmap);
	    DeleteDC(hMemDC);
	    DeleteObject(hBitmap);
	    DeleteObject(hRgn);

	    dst->WaitForCopy();
	}
    }
}

void Painter::Read(
    Canvas* c, void* dst, Coord x1, Coord y1, Coord x2, Coord y2
) {
    /* unimplemented -- use Raster::Raster(Canvas*, ...) */
}

void Painter::Write(
    Canvas* c, const void* src, Coord x1, Coord y1, Coord x2, Coord y2
) {
    /* unimplemented -- use Painter::RasterRect(Canvas*, ...) */
}

