/******************************************************************************
*                                    ctm3d                                    *
*                                                                             *
*                          Homogeneous Coordinates                            *
*                          -----------------------                            *
*                                                                             *
*      Homogeneous coordinates allow transformations to be represented by     *
*      matrices. A 3x3 matrix is used for 2D transformations, and a 4x4 matrix*
*      for 3D transformations.                                                *
*                                                                             *
*      THIS MODULE IMPLEMENTS ONLY 3D TRANSFORMATIONS.                        *
*                                                                             *
*      in homogeneous coordination the point P(x,y,z) is represented as       *
*      P(w*x, w*y, w*z, w) for any scale factor w!=0.                         *
*      in this module w == 1.                                                 *
*                                                                             *
* Transformations:                                                            *
*          1. translation                                                     *
*                  [x, y, z] --> [x + Dx, y + Dy, z + Dz]                     *
*                                                                             *
*                                                                           *
*                              1  0  0  0                                   *
*              T(Dx, Dy, Dz) = 0  1  0  0                                   *
*                              0  0  1  0                                   *
*                              Dx Dy Dz 1                                   *
*                                                                           *
*          2. scaling                                                         *
*                  [x, y, z] --> [Sx * x, Sy * y, Sz * z]                     *
*                                                                             *
*                                                                           *
*                          Sx 0  0  0                                       *
*              S(Sx, Sy) = 0  Sy 0  0                                       *
*                          0  0  Sz 0                                       *
*                          0  0  0  1                                       *
*                                                                           *
*                                                                             *
*          3. rotation                                                        *
*                                                                             *
*              a) Around the Z axis:                                          *
*                                                                             *
*                  [x, y, z] --> [x*cost - t*sint, x*sint + y*cost, z]        *
*                                                                           *
*                      cost  sint   0   0                                   *
*              Rz(t) = -sint cost   0   0                                   *
*                      0     0      1   0                                   *
*                      0     0      0   1                                   *
*                                                                           *
*                                                                             *
*              b) Around the X axis:                                          *
*                                                                             *
*                  [x, y, z] --> [x, y*cost - z*sint, y*sint + z*cost]        *
*                                                                           *
*                      1     0     0    0                                   *
*              Rx(t) = 0     cost  sint 0                                   *
*                      0    -sint  cost 0                                   *
*                      0     0     0    1                                   *
*                                                                           *
*                                                                             *
*              c) Around the Y axis:                                          *
*                                                                             *
*                  [x, y, z] --> [xcost + z*sint, y, z*cost - x*sint]         *
*                                                                           *
*                      cost  0   -sint  0                                   *
*              Ry(t) = 0     1    0     0                                   *
*                      sint  0    cost  0                                   *
*                      0     0    0     1                                   *
*                                                                           *
*                                                                             *
*   transformation of the vector [x,y,z,1] by transformation matrix T is given *
*   by the formula:                                                           *
*                                                                           *
*              [x', y', z', 1] = [x,y,z,1] T                                *
*                                                                           *
* Optimizations:                                                              *
*   The most general composition of R, S and T operations will produce a matrix*
*   of the form:                                                              *
*                                                                           *
*              r11    r12     r13    0                                      *
*              r21    r22     r23    0                                      *
*              r31    r32     r33    0                                      *
*              tx     ty      tz     1                                      *
*                                                                           *
*   The task of matrix multiplication can be simplified by                    *
*      x' = x*r11 + y*r21 + z*r31 + tx                                        *
*      y' = x*r12 + y*r22 + z*r32 + ty                                        *
*      z' = x*r13 + y*r23 + z*r33 + tz                                        *
*                                                                             *
*                                                                             *
* See also:                                                                   *
*      "Fundamentals of Interactive Computer Graphics" J.D FOLEY A.VAN DAM    *
*      Adison-Weslely ISBN 0-201-14468-9 pp 245-265                           *
*                                                                             *
******************************************************************************/

#ifndef _ctm3d_h
#define _ctm3d_h

#if !defined(_hdr3d_h)
#include "hdr3d.h"
#endif

class ctm {

public: 

   double r11, r12, r13;
   double r21, r22, r23;
   double r31, r32, r33;
   double tx,  ty,  tz ;

   ctm(); // same as constructor setunit in pascal 
   ctm(ctm& src); // same as copy constructor in pascal

   void copy(ctm& src);
   
   void setUnit();
   void save(ctm& dest);

   void translate(double Dx, double Dy, double Dz);
   void translateX(double Dx);
   void translateY(double Dy);
   void translateZ(double Dz);

   void rotateX(double t);
   void rotateY(double t);
   void rotateZ(double t);

   void scale(double Sx, double Sy, double Sz);
   void scaleX(double Sx);
   void scaleY(double Sy);
   void scaleZ(double Sz);

   void Left_translate(double Dx, double Dy, double Dz);

   void Left_translateX(double Dx);
   void Left_translateY(double Dy);
   void Left_translateZ(double Dz);

   void Left_rotateX(double t);
   void Left_rotateY(double t);
   void Left_rotateZ(double t);

   void Left_scale(double Sx, double Sy, double Sz);

   void Left_scaleX(double Sx);
   void Left_scaleY(double Sy);
   void Left_scaleZ(double Sz);

   void transform(point3d& t, point3d p);

   void inv_transform(point3d& p); // transform the input

   void inverse(); // M ^-1 : Bring the matrix to -1 power
   void multiply(const ctm c); // multiply from right self * c
   void multiply_2(ctm& a, ctm& b);
   
}; // ctm object definition

typedef ctm *ctmPtr;
#endif
