//***************************************************************
// From the book "Win32 System Services: The Heart of Windows NT"
// by Marshall Brain
// Published by Prentice Hall
//
// Copyright 1994, by Prentice Hall.
//
// This code demonstrates how to multi-thread the mandelbrot
// program. One thread handles the user interface while 
// another handles redrawing.
//***************************************************************

// mandel1.cpp

#include <afxwin.h>
#include "menus.h"

#define NUM_ITERATIONS 64

const double left = -1.0;
const double right = 1.0;
const double top = -1.0;
const double bottom = 1.0;

DWORD colors[64];

typedef struct
{
	double real;
	double imag;
} complex;

typedef struct
{
	WORD height;
	WORD width;
} mandelParams;

// Define the application object class
class CManApp : public CWinApp
{
public:
	virtual BOOL InitInstance();
};

// Define the edit window class
class CManWindow : public CFrameWnd
{
private:
	HANDLE threadHandle;
	mandelParams params;
public:
	CManWindow();
	void RunMandel();
	void SetPix(int x, int y, WORD iter);
	afx_msg void OnPaint();
	afx_msg void OnDoDraw();
	afx_msg void OnExit();
	DECLARE_MESSAGE_MAP()
};

// Create an instance of the application object
CManApp manApp;

// member function used to set pixel colors 
// in the window
void CManWindow::SetPix(int x, int y, WORD iter)
{
	CClientDC dc(this);
	dc.SetPixel(x, y, colors[iter]);
}

// the thread function which does the drawing
DWORD MandelThread(mandelParams *params)
{
	double xstep, ystep;
	double x, y;
	int i,j;
	WORD iter;
	complex k;
	complex z;
	double real, imag, spread;

	ystep = (double) (bottom - top) / 
		params->height;
	xstep = (double) (right - left) / params->width;
	
	for (y=top, j=0; y <= bottom; y += ystep, j++)
	{
		for (x=left, i=0; x<=right; x += xstep, i++)
		{
			k.real = x;
			k.imag = y;
			z.real=z.imag=0.0;

			for (iter=0; iter<NUM_ITERATIONS-1; 
				iter++)
			{
				real = z.real + k.real;
				imag = z.imag + k.imag;
				z.real = real * real - 
					imag * imag;
				z.imag = 2 * real * imag;
				spread = z.real * z.real + 
					z.imag * z.imag;
				if (spread > 4.0)
					break;
			}
			((CManWindow *)manApp.m_pMainWnd)->
				SetPix(i, j, iter);
		}
	}
	return(0);
}

// member function used to instigate 
// the drawing thread
void CManWindow::RunMandel()
{
	DWORD threadID;
	CRect r;

	GetClientRect(&r);
	params.height=r.Height();
	params.width=r.Width();

	threadHandle=CreateThread(NULL, 0,
		(LPTHREAD_START_ROUTINE) MandelThread,
		&params, 0, &threadID);
}

// The message map
BEGIN_MESSAGE_MAP(CManWindow, CFrameWnd)
	ON_WM_PAINT()
	ON_COMMAND(IDM_DODRAW, OnDoDraw)
	ON_COMMAND(IDM_EXIT, OnExit)
END_MESSAGE_MAP()

// Handler for the Start/Stop menu option
void CManWindow::OnDoDraw()
{
	DWORD threadStatus;
	BOOL status;

	status = GetExitCodeThread(threadHandle,
		&threadStatus);
	if (threadStatus == STILL_ACTIVE)
	{
		// stop the existing thread
		TerminateThread(threadHandle, 0);
		CloseHandle(threadHandle);
	}
	// clear the window
	CClientDC dc(this);
	CRect r;
	GetClientRect(&r);
	dc.PatBlt(0, 0, r.Width(), r.Height(), 
		WHITENESS);
	// redraw
	RunMandel();
}

// Handler for WM_PAINT messages
void CManWindow::OnPaint()
{
	ValidateRect(NULL);
}

// Handler for the Exit menu option
void CManWindow::OnExit()
{
	CloseHandle(threadHandle);
	DestroyWindow();
}

// CManWindow constructor
CManWindow::CManWindow()
{
	WORD x;
	BYTE red=0, green=0, blue=0;

	Create( NULL, "Threaded Mandel Example",
		WS_OVERLAPPEDWINDOW,
		CRect(0,0,150,150), NULL, "MainMenu" );
	for (x=0; x<64; x++)
	{
		colors[x] = RGB(red, green, blue);
		if (!(red += 64)) 
			if (!(green += 64)) 
				blue += 64;
	}
	colors[63] = RGB(255,255,255);
}

// Initialize the CManApp m_pMainWnd data member
BOOL CManApp::InitInstance()
{
	m_pMainWnd = new CManWindow();
	m_pMainWnd -> ShowWindow( m_nCmdShow );
	m_pMainWnd -> UpdateWindow();

	return TRUE;
}
