// ObjectWindows - (C) Copyright 1992 by Borland International
//
// spheres.cpp
//
// Modified by Philip VanBaren (71214,2302 or phillipv@engin.umich.edu)
// to enable the screen saver to work with NDW's Sleeper program.
//
// The main modifications are:
//   1) Window class is "WindowsScreenSaverClass"  (this is required for the
//        saver to work properly with Sleeper.)
//   2) IdleAction does screen saver things on only 1 out of 100 calls,
//        thereby allowing background events (backups, printing, etc.) to
//        retain most of the processing power when they need it.
//   3) "-c", "/c", "c", "-s", "/s", "s", and "" are accepted as valid command
//        line parameters.  (Sleeper passes "c" and "-s" to start the configure
//        and saver routines, respectively.)
//   4) Allow some slop in mouse movement before aborting the screen saver,
//        so that the screen saver doesn't abort whenever the desk is bumped.

#include <owl.h>
#include <dialog.h>
#include <bwcc.h>
#include <math.h>
#include <stdio.h>
#include "tscrnsav.h"
#include "spheres.h"

char szAppName[] = "ScreenSaver.Spheres";
int max_spheres;

HDC hDC;
HPALETTE hPalette,hpaletteold;
int grayscale;

#define PALETTE_SIZE 200

LOGPALETTE *plgpl;	/* address of LOGPALETTE structure	*/

_CLASSDEF( TMyScrnSavWindow )
class TMyScrnSavWindow : public TScrnSavWindow
{
   int xlen,ylen;
   int i;
   int radius;
   int cx,cy;
   int red,green,blue;
   int spheres;
   double x_offset,y_offset;
   HPEN hpen;
public:
   TMyScrnSavWindow( PTWindowsObject AParent, LPSTR ATitle,
                     PTModule AModule = NULL );
   ~TMyScrnSavWindow();
   virtual void  GetWindowClass( WNDCLASS & AWndClass );
   virtual void  AnimateScreen();
};

TMyScrnSavWindow::TMyScrnSavWindow( PTWindowsObject AParent,
                                    LPSTR ATitle,
                                    PTModule AModule ) :
TScrnSavWindow( AParent, ATitle, AModule )
{
    randomize();
    xlen = GetSystemMetrics( SM_CXSCREEN );
    ylen = GetSystemMetrics( SM_CYSCREEN );
    i=0;
    radius=0;
    spheres=max_spheres;
}

TMyScrnSavWindow::~TMyScrnSavWindow()
{
   if(grayscale && hPalette)
   {
      SelectPalette(hDC,hpaletteold,0);
      DeleteObject(hPalette);
   }
   ReleaseDC( HWindow, hDC );
}

void TMyScrnSavWindow::GetWindowClass( WNDCLASS & AWndClass )
{
    TScrnSavWindow::GetWindowClass( AWndClass );
    AWndClass.hbrBackground = ( HBRUSH )GetStockObject( BLACK_BRUSH );
    AWndClass.style |= CS_OWNDC;
}

void TMyScrnSavWindow::AnimateScreen()
{
   HBRUSH hbrush,hbrushold;
   int x1,y1,x2,y2;
   double ci;
   int c1,c2;
   static int setup=1;

   if(setup)
   {
      hDC = GetDC( HWindow );
      hpen = (HPEN)GetStockObject(NULL_PEN);

      /*
       *  Initialize the palette if in grayscale mode.
       */
      if(grayscale)
      {
         plgpl = (LOGPALETTE*) LocalAlloc(LPTR,sizeof(LOGPALETTE)
                                    + PALETTE_SIZE * sizeof(PALETTEENTRY));

         plgpl->palNumEntries = PALETTE_SIZE;
         plgpl->palVersion = 0x300;

         for(int k=0;k<PALETTE_SIZE;k++)
         {
            plgpl->palPalEntry[k].peRed =   256-PALETTE_SIZE+k;
            plgpl->palPalEntry[k].peGreen = 256-PALETTE_SIZE+k;
            plgpl->palPalEntry[k].peBlue =  256-PALETTE_SIZE+k;
            plgpl->palPalEntry[k].peFlags = NULL;
         }

         hPalette = CreatePalette(plgpl);
         LocalFree((HLOCAL) plgpl);

         hpaletteold=SelectPalette(hDC,hPalette,0);

         RealizePalette(hDC);
      }
      setup=0;
   }

   if ( hDC )
   {
      if(i<(radius-1))
      {
         i+=2;

         x1=cx-radius+(i*x_offset);
         x2=x1+2*(radius-i);
         y1=cy-radius+(i*y_offset);
         y2=y1+2*(radius-i);

         ci=sin((double)i/radius);

         if(grayscale)
         {
            hbrush=CreateSolidBrush(PALETTEINDEX(floor(ci*(PALETTE_SIZE-0.5))));
         }
         else
         {         
            ci=ci*1.1+0.2;
            if(ci<1)
            {
               hbrush = CreateSolidBrush(RGB(ci*red,ci*green,ci*blue));
            }
            else
            {
               ci=(ci-1)/0.3;
               hbrush = CreateSolidBrush(RGB(red+ci*(255-red),green+ci*(255-green),blue+ci*(255-blue)));
            }
         }
         hbrushold = (HBRUSH)SelectObject(hDC,hbrush);
         SelectObject(hDC,hpen);

         Ellipse(hDC,x1,y1,x2,y2);

         SelectObject(hDC,hbrushold);
         DeleteObject(hbrush);
      }
      else
      {
         i=0;
         spheres++;
         if(spheres>=max_spheres)
         {
            InvalidateRect(HWindow,NULL,TRUE);
            spheres=0;
            /*
             *  Compute a random "light source direction"
             */
            x_offset=(double)random(1000)/1000+0.5;
            y_offset=(double)random(1000)/1000+0.5;
         }
         /*
          *  Compute a random radius and color
          */
         radius=random(xlen/10)+xlen/30;
         cx=random(xlen-2*radius)+radius;
         cy=random(ylen-2*radius)+radius;

         c1=random(6);
         c2=random(255);
         if(c1==0)
         {
            red=c2;
            green=0;
            blue=255;
         }
         else if(c1==1)
         {
            red=c2;
            green=255;
            blue=0;
         }
         else if(c1==2)
         {
            red=255;
            green=c2;
            blue=0;
         }
         else if(c1==3)
         {
            red=0;
            green=c2;
            blue=255;
         }
         else if(c1==4)
         {
            red=0;
            green=255;
            blue=c2;
         }
         else
         {
            red=255;
            green=0;
            blue=c2;
         }
      }
   }
}


_CLASSDEF( TMyScrnSavDlg )
class TScrnSavDlg : public TDialog
{
public:
   TScrnSavDlg( PTWindowsObject AParent,
                LPSTR AName,
                PTModule AModule = NULL ) : TDialog( AParent, AName, AModule )
                {}
   void WMInitDialog(TMessage&);
   void Ok(TMessage&) = [ID_FIRST + IDOK];
   void Cancel(TMessage&) = [ID_FIRST + IDCANCEL];
};

void TScrnSavDlg::WMInitDialog(TMessage&)
{
   SetDlgItemInt(HWindow,IDD_MAXSPHERES,max_spheres,FALSE);
   CheckDlgButton(HWindow,IDD_GRAYSCALE,grayscale);
}

void TScrnSavDlg::Ok(TMessage&)
{
   char ach[20];
   GetDlgItemText(HWindow,IDD_MAXSPHERES,ach,20);
   max_spheres=atoi(ach);
   grayscale=IsDlgButtonChecked(HWindow,IDD_GRAYSCALE);

   if(max_spheres<1) max_spheres=1;
   wsprintf(ach,"%d",max_spheres);
   WritePrivateProfileString("Screen Saver.Spheres","Spheres",ach,"CONTROL.INI");
   wsprintf(ach,"%d",grayscale);
   WritePrivateProfileString("Screen Saver.Spheres","Grayscale",ach,"CONTROL.INI");

   SendMessage(HWindow,WM_SYSCOMMAND,SC_CLOSE,0L);
}

void TScrnSavDlg::Cancel(TMessage&)
{
   SendMessage(HWindow,WM_SYSCOMMAND,SC_CLOSE,0L);
}


_CLASSDEF( TMyScrnSavApp )
class TMyScrnSavApp : public TScrnSavApp
{
    public:
        TMyScrnSavApp( LPSTR AName, HINSTANCE AnInstance,
                       HINSTANCE APrevInstance,
                       LPSTR ACmdLine, int ACmdShow ) :
        TScrnSavApp( AName, AnInstance, APrevInstance, ACmdLine, ACmdShow )
        {
           max_spheres=GetPrivateProfileInt("Screen Saver.Spheres","Spheres",50,"CONTROL.INI");
           if(max_spheres<1) max_spheres=1;
           grayscale=GetPrivateProfileInt("Screen Saver.Spheres","Grayscale",0,"CONTROL.INI");
        }
        virtual void  InitScrnSavWindow();
        virtual void  InitConfigDialog();
};


void TMyScrnSavApp::InitScrnSavWindow()
{
    pScrnSavWnd = new TMyScrnSavWindow( NULL, szAppName );
}


void TMyScrnSavApp::InitConfigDialog()
{
   /*
    *  Force BWCC.DLL to be linked.
    */
   BWCCGetVersion();
   pConfigureDialog = new TScrnSavDlg( NULL, "CONFIGUREDIALOG" );
}


int PASCAL WinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance,
                    LPSTR lpCmdLine, int nCmdShow )
{
    TMyScrnSavApp App( szAppName, hInstance,
                       hPrevInstance, lpCmdLine, nCmdShow );
    App.Run();
    return( App.Status );
}
