// Contents ---------------------------------------------------------------
//
//   ntime.c -- NTIME a Freeware Time Protocol Client
//
//   Version 1.0.1, a Windows Socket Time Client
//
//   Copyright (C) Frederick W. Bent 1994
//   All rights reserved.
//
//
// Redistribution and use in source and binary forms are permitted provided
// that the above copyright notice and this paragraph are duplicated in all
// such forms and that any documentation, advertising materials, and other
// materials related to such distribution and use acknowledge that the
// software was developed by Frederick W. Bent.  In addition, if you wish
// to distribute this program in source and/or binary forms with other
// samples of WinSock programs, you must first contact the author so that
// I can keep accurate records of its usage.  The name of the author may
// not be used to endorse or promote products derived from this software
// without specific prior written permission. Specifically, do not modify
// this source in any way and re-distribute it without the author's prior
// consent.  THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OF
// IMPLIED WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES
// OF MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
//
//
// Description
//
//	See RFC 868.
//
// Ends -------------------------------------------------------------------

// History ----------------------------------------------------------------
//
// 6/28/94  1.0.0  Fred Bent	Wrote this thing
// 8/18/94  1.0.1  Fred Bent	Fixed some UDP issues
//
// Ends -------------------------------------------------------------------

// Legal Stuff ------------------------------------------------------------
//
// Finger Version 3.1, a Windows Sockets Time Client
//
// Copyright 1992, 1993 Network Research Corporation
//
// Permission to use, modify, and distribute this software and its
// documentation for any purpose and without fee is hereby granted, provided
// that the above copyright notice appears in all copies and that both
// that copyright notice and this permission notice appear in supporting
// documentation.  NRC makes no claims as to the suitability of this software
// for any purpose.
//
// Ends ---------------------------------------------------------------------

#pragma	warn -par

// Interface Dependencies -------------------------------------------------

#define USE_CTL3D
#define STRICT

#include <windows.h>
#include <windowsx.h>
#include <stdlib.h>
#include <stdio.h>	// sscanf
#include <string.h>
#include <winsock.h>
#ifdef	USE_CTL3D
   #include <ctl3d.h>
#endif
#include <stdarg.h>
#include <time.h>
#include "dsplist.h"
#include "ntime.h"

#define MAXTEXT   132
#define THUMBPOS  LOWORD(lParam)    // Win 16
#define WM_RUN	(WM_USER + 100)

#define INI_FILE	"WSNTIME.INI"
#define HELP_FILE	"WSNTIME.HLP"

#define DEFAULT_UDPTIME		192
#define DEFAULT_UDPRETRY	3
#define DEFAULT_TCPTIME		30

#define SECTION_MAIN	  "NetTime"
#define ENTRY_ONLYTCP     "UseTCP"
#define ENTRY_DOUPDATE	  "UpdateSysTime"
#define ENTRY_DEFAULTHOST "TimeServer"
#define ENTRY_TIMECORR	  "AddSeconds"
#define ENTRY_TCPTIMEOUT  "TCPTimeout"
#define ENTRY_UDPTIMEOUT  "UDPTimeout"
#define ENTRY_UDPRETRY	  "UDPRetry"

#define MAXHOSTTABLE	5
#define ENTRY_HOST1	  "Host1"
#define ENTRY_HOST2	  "Host2"
#define ENTRY_HOST3	  "Host3"
#define ENTRY_HOST4	  "Host4"
#define ENTRY_HOST5	  "Host5"


// Function prototypes ----------------------------------------------------

int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine, int nCmdShow);
LRESULT CALLBACK FrameWndProc(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);

BOOL CALLBACK HostDlgProc(HWND hDlg, UINT Msg, WPARAM wParam, LPARAM lParam);
BOOL CALLBACK AboutDlgProc(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);

BOOL InitInstance(HINSTANCE hInstance, int nCmdShow);
BOOL InitApp(HINSTANCE hInstance);

LRESULT DoPaint(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT DoRun(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT DoSize(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT DoCreate(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT DoCommand(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT DoDestroy(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT DoMouseMove(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT DoMenuHost(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT DoOpenHost(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT DoMenuOpen(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT DoVScroll(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT DoActivate(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT DoClose(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT DoMenuExit(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT DoMenuAbout(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT DoMenuHelp(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);
LRESULT DoInitMenuPopup(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam);

void AddHostToPopup(PSTR szHostName);

VOID Repaint(VOID);
VOID PosView(int nlines);
VOID ReportWSError(int Err);
VOID RelScroll(HWND hWnd, int nlines);
VOID SetWinCaption(PSTR str);
VOID SetScroll(VOID);
VOID WinPrintf( HDC   hdc, int row, UINT   col, LPSTR pszFormat, ... );
VOID MakeHelpPathName(HINSTANCE hInst, LPSTR szFileName);
PSTR WSErrorString(UINT ResourceID);

// Global variables -------------------------------------------------------

char     szHostName[MAXHOST+1] = ""; // name of host to connect to
char     szUser[MAXUSER+1] = "";     // query for this user id (can be null)
char     szAppName[] = APP_NAME;     // application's title
char	 szAppClass[] = "WSNTime.Class";

char	 szHostTable[MAXHOSTTABLE][MAXHOST+1];
char	 szEntryHost[MAXHOSTTABLE][6] = { ENTRY_HOST1, ENTRY_HOST2, ENTRY_HOST3, ENTRY_HOST4, ENTRY_HOST5 };

HLOCAL	pLineItems = NULL;           // ptr to display list LINEITEMS
int      nLineItems = 0;            // number of items in display list
HLOCAL	pTopLine;                 // pts to topmost displayable LINEITEM
int      nTopLine = 0;              // line number of topmost displayed line
int      nClientLines;              // # of text lines in view
int      CharY;                     // pixel character height
int	 CharX;
HWND     hFrame;                    // ntime main window handle
HMENU    hMenu;                     // main window menu handle
HINSTANCE    hInst;                     // this instance of ntime
HCURSOR  hCursor;                   // current cursor (either wait or normal)
WSADATA  WSAData;             		// windows sockets info return

BOOL	bCommandLineArgs = FALSE;
BOOL	bShouldUseTCP = TRUE;
BOOL	bUpdateSystemTime = FALSE;
BOOL	bRunning = FALSE;

char	szHelpFileName[MAX_PATH];


DECODEWORD frameMsgs[] =            // windows messages & handlers
{
     {WM_ACTIVATE,   DoActivate}
   , {WM_CLOSE,      DoClose}
   , {WM_COMMAND,    DoCommand}
   , {WM_CREATE,     DoCreate}
   , {WM_DESTROY,    DoDestroy}
   , {WM_MOUSEMOVE,  DoMouseMove}
   , {WM_PAINT,      DoPaint}
   , {WM_RUN,	     DoRun}
   , {WM_INITMENUPOPUP, DoInitMenuPopup}
   , {WM_SIZE,       DoSize}
   , {WM_VSCROLL,    DoVScroll}
   , {WM_KEYDOWN,    DoVScroll}
};

DECODEWORD menuItems[] =            // menu items & associated handlers
{
     {IDM_OPEN,	     DoMenuOpen}
   , {IDM_SETUP,     DoMenuHost}
   , {IDM_EXIT,      DoMenuExit}
   , {IDM_ABOUT,     DoMenuAbout}
   , {IDM_HELP_INDEX, DoMenuHelp}
   , {IDM_HOST1,     DoOpenHost}
   , {IDM_HOST2,     DoOpenHost}
   , {IDM_HOST3,     DoOpenHost}
   , {IDM_HOST4,     DoOpenHost}
   , {IDM_HOST5,     DoOpenHost}
};


// Function

	BOOL	HandleCommandLine( LPSTR lpszCmdLine )

// Summary ----------------------------------------------------------------
//
//	This function attempts to read any command-line arguments that
//	this program has been passed.  The arguments are copied into
//	the szHostname and szUser global variables.
//
// Parameters
//
//	lpszCmdLine
//
// Returns
//
//	BOOL
//
//	Returns TRUE if there was an argument, FALSE otherwise
//
// Ends -------------------------------------------------------------------
{
	int	status;
        char	szBuffer[128];

	if (lstrlen(lpszCmdLine) == 0 ) return(FALSE);

	lstrcpyn((LPSTR)szBuffer, lpszCmdLine, sizeof(szBuffer));
	status = sscanf(szBuffer, "%s %s", szHostName, szUser);

	if ( status < 1 ) szHostName[0] = '\0';
	if ( status < 2 ) szUser[0] = '\0';

	if ( status > 0 ) return(TRUE);
	else return FALSE;
}


//
// WinMain -- windows calls this to start the application.
//
int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
    LPSTR lpCmdLine, int nCmdShow)

// Summary ----------------------------------------------------------------
//
//
// Parameters
//
//	hInstance	The instance handle of this application
//
//	hPrevInstance	The handle of the previous instance of this
//			application
//
//	lpCmdLine	A far pointer to a null-terminated command-line
//
//	nCmdShow	An integer that specifies the application's window
//			display.
//
// Ends -------------------------------------------------------------------


	{
		MSG msg;                                  // holds current message
		int err;
#ifdef	USE_CTL3D
		WORD	ctl3d_ver;
#endif


	   	hInst = hInstance;                        // save our instance handle

#ifdef USE_CTL3D
		ctl3d_ver = Ctl3dGetVer();

		if ( ( HIBYTE(ctl3d_ver) < CTL3DVERSION_MAJOR ) ||
		     ( HIBYTE(ctl3d_ver) == CTL3DVERSION_MAJOR &&
		       LOBYTE(ctl3d_ver) < CTL3DVERSION_MINOR) )
		{
			MessageBox(NULL, WSErrorString(IDS_CTL3D_VER), szAppName, MB_ICONEXCLAMATION | MB_OK);
			return(FALSE);
		}
		Ctl3dRegister(hInstance);
	   	Ctl3dAutoSubclass(hInstance);
#endif

		if (!hPrevInstance)                       // if first instance,
	      		if (!InitApp(hInstance))               // register window classes
		      	{
				MessageBox(hFrame, WSErrorString(FE_INIT), szAppName, MB_ICONSTOP | MB_OK);
#ifdef USE_CTL3D
				Ctl3dUnregister(hInstance);
#endif
			 	return(FALSE);
		      	}

		if ( HandleCommandLine(lpCmdLine) )
		{
			nCmdShow = SW_SHOWMINNOACTIVE;
			bCommandLineArgs = TRUE;
		} else {
			bCommandLineArgs = FALSE;
		}

		if (!InitInstance(hInstance, nCmdShow))   // per instance initialization &
		{                                         // window creation
			MessageBox(hFrame, WSErrorString(FE_INIT), szAppName, MB_ICONSTOP | MB_OK);
#ifdef USE_CTL3D
			Ctl3dUnregister(hInstance);
#endif
			return(FALSE);
		}

		if ((err = WSAStartup(WSVERSION_REQ, &WSAData)) != 0 ) // register task with
		{                                         // winsock tcp/ip API
			ReportWSError(err);
#ifdef USE_CTL3D
			Ctl3dUnregister(hInstance);
#endif
			DestroyWindow(hFrame);            // kill application window & signal app exit
			return(FALSE);
		}

		if ( ( LOBYTE(WSAData.wVersion) < WSVERSION_MAJOR ) ||
		     ( LOBYTE(WSAData.wVersion) == WSVERSION_MAJOR &&
		       HIBYTE(WSAData.wVersion) < WSVERSION_MINOR) )
		{
			MessageBeep(MB_ICONSTOP);
			MessageBox(hFrame, WSErrorString(IDS_BAD_VERSION), szAppName, MB_ICONSTOP | MB_OK);
			DestroyWindow(hFrame);
#ifdef USE_CTL3D
			Ctl3dUnregister(hInstance);
#endif
			WSACleanup();
			return(FALSE);
		}

		/* If auto-run mode then start it now... */
		if (bCommandLineArgs) SendMessage(hFrame, WM_RUN, 0, 0L);

		while (GetMessage(&msg, NULL, 0, 0))      // loop til WM_QUIT
		{
		      TranslateMessage(&msg);
		      DispatchMessage(&msg);
		}

		WSACleanup();                             // disconnect from winsock
#ifdef USE_CTL3D
		Ctl3dUnregister(hInstance);
#endif
		return msg.wParam;                        // return to windows
	}


// Function

	BOOL InitApp(HINSTANCE hInstance)

// Summary ----------------------------------------------------------------
//
// InitApp -- initialization for all instances of application.
// Registers main window class.
//
// Paramters
//
//	hInstance	This instance of the NTime app
//
//	nCmdShow	Now to start the application
//
// Returns
//
//	BOOL
//
//	Returns TRUE is successful, FALSE otherwise
//
// Ends -------------------------------------------------------------------

	{
		WNDCLASS    wndclass;

		InitNetApp();  // initializes (per application) network module

		wndclass.style         = CS_HREDRAW | CS_VREDRAW;
	   	wndclass.lpfnWndProc   = FrameWndProc;
	   	wndclass.cbClsExtra    = 0;
	   	wndclass.cbWndExtra    = 0;
	   	wndclass.hInstance     = hInstance;
	   	wndclass.hIcon         = LoadIcon(hInst, "NTimeIcon");
		wndclass.hCursor       = NULL;
#ifdef USE_CTL3D
		wndclass.hbrBackground = CreateSolidBrush(GetSysColor(COLOR_BTNSHADOW));
#else
		wndclass.hbrBackground = CreateSolidBrush(GetSysColoe(COLOR_WINDOW));
#endif
		wndclass.lpszMenuName  = "NTimeMenu";
	   	wndclass.lpszClassName = szAppClass;

		return(RegisterClass(&wndclass));
	}


// Function

	BOOL InitInstance(HINSTANCE hInstance, int nCmdShow)

// Summary ----------------------------------------------------------------
//
//	This function is used to initialize this instance of a WSNTime app
//	and create its main window.  It reads the .INI file to setup
//	some of the global variables.
//
//	It also initializes the various global variables, and calls to
//	start the network window.
//
// Paramters
//
//	hInstance	This instance of the NTime app
//
//	nCmdShow	Now to start the application
//
// Returns
//
//	BOOL
//
//	Returns TRUE is successful, FALSE otherwise
//
// Ends -------------------------------------------------------------------

	{
	   HDC hdc;
	   TEXTMETRIC tm;                      // contains font dimensions
	   RECT rect;                          // outer dimensions of window
	   int  nTCPTimeOutValue;
	   int i;

	   hFrame = CreateWindow( szAppClass, szAppName
			, WS_POPUP | WS_SYSMENU | WS_CAPTION | WS_VSCROLL | WS_MINIMIZEBOX
			, CW_USEDEFAULT, CW_USEDEFAULT
			, CW_USEDEFAULT, CW_USEDEFAULT
			, NULL, NULL, hInstance, NULL);

	   if (hFrame == NULL)
	      return(FALSE);

	   hCursor = LoadCursor(NULL, IDC_ARROW);
	   hMenu = GetMenu(hFrame);
	   bRunning = FALSE;

	   /* Read the settings in the .INI file */
	   nTCPTimeOutValue = GetPrivateProfileInt( SECTION_MAIN
						, ENTRY_TCPTIMEOUT
						, DEFAULT_TCPTIME
						, (LPCSTR) INI_FILE );

	   nUDPTimeOutValue = GetPrivateProfileInt( SECTION_MAIN
						, ENTRY_UDPTIMEOUT
						, DEFAULT_UDPTIME
						, (LPCSTR) INI_FILE );

	   nUDPNumberRetry = GetPrivateProfileInt( SECTION_MAIN
						, ENTRY_UDPRETRY
						, DEFAULT_UDPRETRY
						, (LPCSTR) INI_FILE );

	   bShouldUseTCP = GetPrivateProfileInt( SECTION_MAIN
				, ENTRY_ONLYTCP
				, TRUE
				, (LPCSTR) INI_FILE);


	   MakeHelpPathName(hInstance, (LPSTR)szHelpFileName);

	   if (!InitNetInst(hFrame, nTCPTimeOutValue, nUDPTimeOutValue, nUDPNumberRetry))
		return(FALSE);  // initialize (per instance) the network module

	   /* Setup for the output */
	   hdc = GetDC(hFrame);
	   SelectObject(hdc, GetStockObject(ANSI_VAR_FONT));
	   GetTextMetrics(hdc, &tm);
	   CharY = tm.tmHeight + tm.tmExternalLeading;
	   CharX = tm.tmAveCharWidth;
#ifdef USE_CTL3D
	   SetTextColor(hdc, GetSysColor(COLOR_BTNTEXT));
	   SetBkColor(hdc, GetSysColor(COLOR_BTNSHADOW));
#else
	   SetTextColor(hdc, GetSysColor(COLOR_WINDOWTEXT));
	   SetBkColor(hdc, GetSysColor(COLOR_WINDOW));
#endif
	   ReleaseDC(hFrame, hdc);

	   // set initial window width & height to 80x10 chars
	   GetWindowRect(hFrame, &rect);

	   MoveWindow( hFrame, rect.left, rect.top,
		       80 * CharX + GetSystemMetrics(SM_CXVSCROLL),
	               10 * CharY + GetSystemMetrics(SM_CYCAPTION) +
	               GetSystemMetrics(SM_CYMENU), FALSE);

	   /* Show the window */
	   ShowWindow(hFrame, nCmdShow);
	   UpdateWindow(hFrame);

	   if (!bCommandLineArgs)
	   {
		   bUpdateSystemTime = GetPrivateProfileInt( SECTION_MAIN
					, ENTRY_DOUPDATE
					, FALSE
					, (LPCSTR) INI_FILE );

		    GetPrivateProfileString( SECTION_MAIN
					   , ENTRY_DEFAULTHOST
					   , "\0"
					   , (LPSTR) szHostName
					   , sizeof(szHostName)
					   , (LPCSTR) INI_FILE);

		    GetPrivateProfileString( SECTION_MAIN
					   , ENTRY_TIMECORR
					   , "\0"
					   , (LPSTR) szUser
					   , sizeof(szUser)
					   , (LPCSTR) INI_FILE);

                    /* Setup the host tables */
		    for( i = 0; i < MAXHOSTTABLE; i++ )
                    {
			    GetPrivateProfileString( SECTION_MAIN
						   , (LPSTR) szEntryHost[i]
						   , "\0"
						   , (LPSTR) szHostTable[i]
						   , MAXHOST
						   , (LPCSTR) INI_FILE);
		    }

		    /* Add the hosts, if any, to the menu */
		    AddHostToPopup(NULL);

	   } else {
		for ( i = 0; i < MAXHOSTTABLE; i++ )
		{
			szEntryHost[i][0] = '\0';
		}
		bUpdateSystemTime = TRUE;
	   }


	   return(TRUE);
	}


// Function

	void SaveHosts(void)

// Summary ----------------------------------------------------------------
//
//	This function saves the time servers that are currently
//	present in the Host Menu to the .INI file.
//
// Paramters
//
//	none
//
// Returns
//
//	Nothing
//
// Ends -------------------------------------------------------------------

{
    int i;

    for( i = 0; i < MAXHOSTTABLE; i++ )
    {
	    WritePrivateProfileString( SECTION_MAIN
				   , (LPSTR) szEntryHost[i]
				   , (LPSTR) szHostTable[i]
				   , (LPCSTR) INI_FILE);
    }
}


// Function

	BOOL	IsHostInTable(PSTR szHostName)

// Summary ----------------------------------------------------------------
//
//	This function searches for the specifies time host in the
//	Host menu.
//
// Paramters
//
//	szHostName	The ASCIIZ string giving the host to search for.
//
// Returns
//
//	BOOL
//
//	Returns TRUE is the string is in the menu, FALSE otherwise
//
// Ends -------------------------------------------------------------------

{
	int	i;


	if (szHostName == NULL) return FALSE;

        i = 0;
	while (strcmp(szHostTable[i], szHostName))
	{
		i++;
                if ( i == MAXHOSTTABLE ) break;
	}

	return (i != MAXHOSTTABLE);
}


// Function

	void AddHostToPopup(PSTR szHostName)

// Summary ----------------------------------------------------------------
//
//	This function adds the given host to the Host Menu popup.
//	If the host is already in the menu, nothing is done.  If
//	the menu is full (i.e. 5 hosts) then the last one is
//	removed from the menu.
//
// Paramters
//
//	szHostName	The ASCIIZ string that is to be added to the
//			menu.
//
// Returns
//
//	Nothing.
//
// Ends -------------------------------------------------------------------

{
	HMENU 	hMenu;
	HMENU 	hMenuPopup;
	int	nCount;
	int	i;
	char	szTempBuffer[MAXHOST + 4];


        if (IsHostInTable(szHostName)) return;

	hMenu = GetMenu(hFrame);
	hMenuPopup = GetSubMenu(hMenu, 0);	// Host menu...
	nCount = GetMenuItemCount(hMenuPopup);

	if ( nCount < 5 )	// Have added separator already?
	{
		if ((szHostName != NULL) || (szHostTable[0][0] != '\0'))
			AppendMenu(hMenuPopup, MF_SEPARATOR, 0, NULL);
		nCount = GetMenuItemCount(hMenuPopup);
	}

	if (szHostName != NULL )
	{
		for ( i = 0; ((i < MAXHOSTTABLE) && (nCount >= 5)); i++ )
		{
			RemoveMenu( hMenuPopup, (IDM_HOST1 + i), MF_BYCOMMAND );
			nCount = GetMenuItemCount(hMenuPopup);
		}

		for ( i = (MAXHOSTTABLE-1); i > 0; i-- )
		{
			strcpy(szHostTable[i], szHostTable[i-1]);
		}
		strcpy(szHostTable[0], szHostName);
	}

	for ( i = 0; i < MAXHOSTTABLE; i++ )
	{
		if ( szHostTable[i][0] != '\0' )
		{
			sprintf(szTempBuffer, "&%1d ", (i+1));
                        strcat(szTempBuffer, szHostTable[i]);
			AppendMenu(hMenuPopup, MF_STRING, (IDM_HOST1 + i), szTempBuffer);
                }
	}
	DrawMenuBar(hFrame);
}

 
// Callback function

	LRESULT CALLBACK FrameWndProc(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// FrameWndProc -- callback function for application frame (main) window.
// Decodes message and routes to appropriate message handler. If no handler
// found, calls DefWindowProc.
// 
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TRUE.
//
// Ends -------------------------------------------------------------------

	{
	   int i;
	
	   for (i = 0; i < dim(frameMsgs); i++)
	   {
	      if (Msg == frameMsgs[i].Code)
		 return(*frameMsgs[i].Fxn)(hWnd, Msg, wParam, lParam);
	   }

	   return(DefWindowProc(hWnd, Msg, wParam, lParam));
	}


// Window function

	LRESULT DoCommand(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// DoCommand -- demultiplexes WM_COMMAND messages resulting from menu
// selections, and routes to corresponding menu item handler.  Sends back
// any unrecognized messages to windows.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TRUE.
//
// Ends -------------------------------------------------------------------

	{
	   int i;

	   for (i = 0; i < dim(menuItems); i++)
	   {
	      if (wParam == menuItems[i].Code)
		 return(*menuItems[i].Fxn)(hWnd, Msg, wParam, lParam);
	   }

	   return(DefWindowProc(hWnd, Msg, wParam, lParam));
	}


// Window function

	LONG DoMenuHost(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// DoMenuHost -- handles the "Setup..." menu item.  We prompt for a
// Domain Name System (DNS) host name, or a "dotted IP address" (e.g.,
// 129.216.202.5). 
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TRUE.
//
// Ends -------------------------------------------------------------------

	{
		DLGPROC lpfnProc;
                int ret;

	   	// prompt for the host's domain name or ip address

	   	lpfnProc = (DLGPROC) MakeProcInstance((FARPROC)HostDlgProc, hInst);
		ret = DialogBox(hInst, "HostBox", hWnd, lpfnProc);
		FreeProcInstance((FARPROC)lpfnProc);

		if (ret == IDB_CONNECT)
		{
			PostMessage(hWnd, WM_COMMAND, IDM_OPEN, 0L);
		}

		return(FALSE);
	}


// Menu Function

	LONG DoMenuOpen(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
//	This function is called when the user has selected the Host Open
//	menu command.  This causes the client to attempt to connect to
//	the server and retrieve the data from the remote server.
//
//	The Open and Setup menu options are disabled while the request
//	is being serviced.  Only ONE request at a time is allowed.
//
//	The function invokes NETWRK_ module FingerStart()
// 	routine to initiate a conversation with the time server on the
// 	remote host.  NETWRK_, in turn, calls TimeFinish() to signal
// 	completion.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TRUE.
//
// Ends -------------------------------------------------------------------

	{
		SetWinCaption(NULL);
		AddHostToPopup(szHostName);
		bRunning = TRUE;
		SetCursor(hCursor = LoadCursor(NULL, IDC_WAIT));
		FreeLineList(pLineItems);	// dispose old display
		pLineItems = NULL;
		nLineItems = 0;
		PosView(0);                               // position view to top
		FingerStart(szHostName);
                Repaint();
		return(FALSE);
	}



// Menu Function

	LONG DoOpenHost(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
//	This function is called when the user has selected the Host Open
//	menu command.  This causes the client to attempt to connect to
//	the server and retrieve the data from the remote server.
//
//	The Open and Setup menu options are disabled while the request
//	is being serviced.  Only ONE request at a time is allowed.
//
//	The function invokes NETWRK_ module FingerStart()
// 	routine to initiate a conversation with the time server on the
// 	remote host.  NETWRK_, in turn, calls TimeFinish() to signal
// 	completion.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TRUE.
//
// Ends -------------------------------------------------------------------

	{
		HMENU	hMenu;
		HMENU	hMenuPopup;
                char	szString[MAXHOST + 4];

		hMenu = GetMenu(hWnd);
		hMenuPopup = GetSubMenu(hMenu, 0);

		GetMenuString(hMenuPopup, wParam, (LPSTR) szString, sizeof(szString), MF_BYCOMMAND);
		CheckMenuItem(hMenuPopup, wParam, MF_CHECKED | MF_BYCOMMAND );

		strncpy(szHostName, &szString[3], MAXHOST);

		PostMessage(hWnd, WM_COMMAND, IDM_OPEN, 0L);

		return(FALSE);
	}


// Windows function

	LONG DoRun(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
//	This function is called when command line arguments have been
//	supplied by the user, indicating that the program should be
//	run in automatic mode using the command-line arguments.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TRUE.
//
// Ends -------------------------------------------------------------------

	{
		PostMessage(hWnd, WM_COMMAND, IDM_OPEN, 0L);
		return(FALSE);
	}


// Function

	VOID TimeFinish(UINT Err)

// Summary ----------------------------------------------------------------
//
// TimeFinish -- invoked when the time operation is complete,
// this function updates the display list & repaints the frame window
// client area.  The menu is re-enabled.
//
// Paramters
//
//	Err	The error if any
//
// Returns
//
//	Nothing
//
// Ends -------------------------------------------------------------------

	{
		HMENU	hMenuPopup;
		int	i;

		FingerStop();				// tidy up network stuff

		GetDisplayList(&pLineItems, &nLineItems);

	      	SetWinCaption(szHostName);                // set win title to host name
	      	PosView(0);                               // position view to top
	      	SetScroll();                              // rescale (or delete)
	      	Repaint();                                // scrollbar & force a repaint

		bRunning = FALSE;			// No request pending
		SetCursor(hCursor = LoadCursor(NULL, IDC_ARROW));

		if (bCommandLineArgs)
			SendMessage(hFrame, WM_CLOSE, 0, 0);

		for ( i = 0; i < MAXHOSTTABLE; i++ )
		{
			if ( szHostTable[i][0] != '\0' )
			{
				hMenu = GetMenu(hFrame);
				hMenuPopup = GetSubMenu(hMenu, 0);

				CheckMenuItem(hMenuPopup, (IDM_HOST1 + i), MF_UNCHECKED | MF_BYCOMMAND);
				EnableMenuItem(hMenu, (IDM_HOST1 + i), MF_ENABLED | MF_BYCOMMAND);
	                }
		}
	}


// Windows function

	LONG DoMenuExit(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// DoMenuExit -- allows close via menu item.  Same as sys menu close.
//
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TRUE.
//
// Ends -------------------------------------------------------------------

	{
		SendMessage(hWnd, WM_CLOSE, 0, 0);
		return(FALSE);
	}


// Windows function

	LRESULT DoMenuAbout(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// DoMenuAbout -- respond to "About..." menu selection by invoking the
// "About" dialog box.
//
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TRUE.
//
// Ends -------------------------------------------------------------------

	{
		DLGPROC lpProcAbout;

		lpProcAbout = (DLGPROC) MakeProcInstance((FARPROC)AboutDlgProc, hInst);
	   	DialogBox(hInst, "AboutBox", hWnd, lpProcAbout);
	   	FreeProcInstance((FARPROC)lpProcAbout);

		return(FALSE);
	}


// Windows Function

	LRESULT DoMenuHelp(HWND hWnd, UINT wMsg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// 	Respond to "Help" menu selection by invoking the WinHelp system.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
// Ends -------------------------------------------------------------------

	{

		WinHelp(hWnd,szHelpFileName,HELP_INDEX,0L);
		return(FALSE);
	}


// Windows function

	LONG DoDestroy(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// DoDestroy -- posts a WM_QUIT message to the task's win queue, which
// causes the main translate & dispatch loop to exit, and the app to
// terminate.
//
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TRUE.
//
// Ends -------------------------------------------------------------------

	{
	   PostQuitMessage(0);
	   return(FALSE);
	}


// Window Function

	LRESULT DoClose(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// DoClose -- cleans up display list & tells windows to deallocate
// our window.  If there is a request pending, then the user is told
// about this fact.
//
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TRUE.
//
// Ends -------------------------------------------------------------------

{
	char	szBuffer[128];
	int	result;


	if (!bCommandLineArgs)
        {
		if (( sSocket != INVALID_SOCKET ) || bRunning )
		{
			LoadString(hInst, IDS_EXIT_ACTIVE, (LPSTR)szBuffer, sizeof(szBuffer));
	
			MessageBeep(MB_ICONEXCLAMATION);
			result = MessageBox(hWnd, (LPSTR) szBuffer, szAppName,
					MB_ICONEXCLAMATION | MB_OKCANCEL | MB_DEFBUTTON2 );
		}
		else result = IDYES;
//		{
//			LoadString(hInst, IDS_EXITING, (LPSTR)szBuffer, sizeof(szBuffer));
//			 result = MessageBox(hWnd, szBuffer, szAppName,
//					MB_ICONQUESTION | MB_YESNO);
//		}

        }

	if ((result == IDYES) || (result == IDOK) || bCommandLineArgs)
	{
	   SaveHosts();
	   NetClose();
	   FreeLineList(pLineItems);
	   pLineItems = NULL;
           nLineItems = 0;
	   DestroyWindow(hWnd);		// WM_DESTROY
	}

	return(FALSE);
}


// Windows Function

	LRESULT DoCreate(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TRUE.
//
// Ends -------------------------------------------------------------------

	{
		return(FALSE);
	}



// Window Function

	LRESULT DoInitMenuPopup(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// Called prior to the displaying of the menu popup.  LOWORD(lParam) is the
// index of the popup in the menu.  This let's us disable the Open menu
// item of the Host popup menu if there is no valid host name present,
// or if there is a request pending.
//
// Note that the "Setup..." menu item is disabled while the
// time operation is in progress.  This prevents the user from
// altering the settings before the first request is finished.  The "Open"
// menu item is also disabled; NTime is not designed to process simultaneous
// requests.
//
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TRUE.
//
// Ends -------------------------------------------------------------------

	{
	   UINT	wEnable;
	   int	i;


	   if(HIWORD(lParam))
	   {
	     HMENU hSystemMenu;

	     hSystemMenu = GetSystemMenu(hWnd, FALSE);
	     EnableMenuItem(hSystemMenu, SC_SIZE, MF_GRAYED);
	     EnableMenuItem(hSystemMenu, SC_MAXIMIZE, MF_GRAYED);
	   }

	   if (LOWORD(lParam) == 0)
	   {
		if (( szHostName[0] == '\0' ) || bRunning )
			wEnable = MF_GRAYED;
		else
			wEnable = MF_ENABLED;
			EnableMenuItem(hMenu, IDM_OPEN, wEnable | MF_BYCOMMAND);

		wEnable = bRunning ? MF_GRAYED : MF_ENABLED;
		EnableMenuItem(hMenu, IDM_SETUP, wEnable | MF_BYCOMMAND);

		/*
		 * Disable the various host entries
		 */
		for ( i = 0; i < MAXHOSTTABLE; i++ )
		{
			if ( szHostTable[i][0] != '\0' )
			{
				EnableMenuItem(hMenu, (IDM_HOST1 + i), wEnable | MF_BYCOMMAND);
	                }
		}
	   }

	   return(FALSE);
	}


// Window Function

	LONG DoActivate(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// DoActivate -- grabs the keyboard focus whenever our deminimized window
// is activated.  This is so we can respond to VK_HOME, VK_END, etc.
// virtual keys for scrolling. HIWORD(lParam) is TRUE for minimized, while
// LOWORD(wParam) is FALSE for activation message.
// 
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TRUE.
//
// Ends -------------------------------------------------------------------

	{
		if (!HIWORD(lParam) && LOWORD(wParam))
	      		SetFocus(hFrame);

		return FALSE;
	}


// Function

	LONG DoMouseMove(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// DoMouseMove -- resets the cursor back to the current cursor (either
// a wait, or normal cursor) because Windows will otherwise redraw it
// using the window's class.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TRUE.
//
// Ends -------------------------------------------------------------------

	{
		SetCursor(hCursor);
		return(FALSE);
	}


// Function

	VOID PosView(int nlines)

// Summary ----------------------------------------------------------------
//
// PosView -- repositions the view relative to the top of the display list.
// The view is a logical "window" onto the display list.  The frame window's
// client area is painted with the view's contents.
//
//
// Paramters
//
//	nlines	The position in the display
//
// Returns
//
//	Nothing.
//
// Ends -------------------------------------------------------------------

	{
	   HLOCAL	pline;
	   HLOCAL	ptemp;
	   LINEITEM	*p;
	   int i;

	   pline = pLineItems;              // root of LINEITEM list

	   for (i = 0; i < nlines; i++)
	   {
	      if (pline == NULL)
		 break;
	      else
	      {
		 p = LocalLock(pline);
		 if ( p != NULL )
	         {
			 ptemp = p->next;
			 LocalUnlock(pline);
			 pline = ptemp;
		 } else pline = NULL;
	      }
	   }

	   pTopLine = pline;                // ptr to LINEITEM in topmost view line
	   nTopLine =+ nlines;              // offset of topmost view line
	}


// Window Function

	LRESULT DoPaint(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// DoPaint -- Paint the client area with the contents of the view.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TRUE.
//
// Ends -------------------------------------------------------------------

	{
	   HDC hdc;                   // scratch device context
	   TEXTMETRIC	tm;
	   PAINTSTRUCT ps;            // scratch paint structure
	   HLOCAL	pline;        // pts to topmost displayable LINEITEM
	   HLOCAL	hLocal;
	   int i;
	   int CharY;

	   pline = pTopLine;

	   hdc = BeginPaint(hWnd, &ps);
	   SelectObject(hdc, GetStockObject(ANSI_VAR_FONT));
	   GetTextMetrics(hdc, &tm);
	   CharY = tm.tmHeight + tm.tmExternalLeading;
	//   CharX = tm.tmAveCharWidth;
#ifdef USE_CTL3D
	   SetTextColor(hdc, GetSysColor(COLOR_BTNHIGHLIGHT));
	   SetBkColor(hdc, GetSysColor(COLOR_BTNSHADOW));
#else
	   SetTextColor(hdc, GetSysColor(COLOR_WINDOWTEXT));
	   SetBkColor(hdc, GetSysColor(COLOR_WINDOW));
#endif

	   for (i = 0; i <= nClientLines; i++)
	   {
	      if (pline != NULL)
	      {
		 LINEITEM	*p;
		 char	*sztext;

		 p = LocalLock(pline);
		 if ( p != NULL )
		 {
			hLocal = p->next;
			if (p->hText != NULL)
	                {
				sztext = LocalLock(p->hText);
				if (sztext != NULL)
			        {
					 TextOut(hdc, 0, i * CharY, (LPSTR)sztext, p->len);
					 LocalUnlock(p->hText);
				}
	                }
			LocalUnlock(pline);
			pline = hLocal;
		 }
	      } else // end of display list...
	   break;
	   } // for

	   EndPaint(hWnd, &ps);
	   return(FALSE);
	}


// Function

	VOID Repaint(VOID)

// Summary ----------------------------------------------------------------
//
// Repaint -- force refresh of client window.
// 
// Returns
//
//	Nothing.
//
// Ends -------------------------------------------------------------------

	{
		InvalidateRect(hFrame, NULL, TRUE);
	}


// Function

	VOID SetScroll(VOID)

// Summary ----------------------------------------------------------------
//
// SetScroll -- sets the vertical scroll range to the length of the display
// list.  The Scrollbar disappears when the list fits within the view.
//
// Returns
//
//	Nothing.
//
// Ends -------------------------------------------------------------------

{
   if (hFrame != NULL)
   {
	   if (nLineItems > nClientLines)
	      SetScrollRange(hFrame, SB_VERT, 0, nLineItems - nClientLines, FALSE);
	   else
	      SetScrollRange(hFrame, SB_VERT, 0, 0, FALSE);

	   SetScrollPos(hFrame, SB_VERT, nTopLine, TRUE);
   }
}


// number of lines below the bottom of the view.
#define NLINESBELOW (nLineItems - nTopLine - nClientLines)

// Windows Function

	LRESULT DoVScroll(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// DoVScroll -- process WM_VSCROLL & WM_KEYDOWN for main window.
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TRUE.
//
// Ends -------------------------------------------------------------------

{
   switch (LOWORD(wParam))
   {
      case VK_HOME:
      case SB_TOP:
         if (nTopLine > 0)
            RelScroll(hWnd, -nTopLine);
         break;

      case VK_END:
      case SB_BOTTOM:
         if (NLINESBELOW)
            RelScroll(hWnd, NLINESBELOW);
	 break;

      case VK_PRIOR:
      case SB_PAGEUP:
         if (nTopLine > 0)
            RelScroll(hWnd, max(-nClientLines, -nTopLine));
         break;

      case VK_NEXT:
      case SB_PAGEDOWN:
         if (NLINESBELOW)
            RelScroll(hWnd, min(nClientLines, NLINESBELOW));
         break;

      case VK_UP:
      case SB_LINEUP:
         if (nTopLine > 0)
            RelScroll(hWnd, -1);
         break;

      case VK_DOWN:
      case SB_LINEDOWN:
         if (NLINESBELOW)
            RelScroll(hWnd, 1);
         break;

      case SB_THUMBTRACK:
         RelScroll(hWnd, THUMBPOS - nTopLine);
	 break;
   }

   SetScrollPos(hFrame, SB_VERT, nTopLine, TRUE);
   return(FALSE);
}


// Function

	VOID RelScroll(HWND hWnd, int nlines)

// Summary ----------------------------------------------------------------
//
// RelScroll -- scroll up/down nlines from present position
//
//
// Parameters
//
//	hWnd	Handle of the window
//
//	nlines	The number of lines to scroll
//
// Returns
//
//	Nothing.
//
// Ends -------------------------------------------------------------------

{
	PosView(nTopLine + nlines);
	ScrollWindow(hWnd, 0, -nlines * CharY, NULL, NULL);
	UpdateWindow(hWnd);
}


// Window Function

	LRESULT DoSize(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// DoSize -- respond to WM_SIZE by recalculating the number of text lines
// in the main window's client area.
//
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	LRESULT
//
//	Returns FALSE of the dialog message was handled, otherwise TURE.
//
// Ends -------------------------------------------------------------------

{
	HDC	hdc;
	TEXTMETRIC	tm;


	hdc = GetDC(hFrame);
	SelectObject(hdc, GetStockObject(ANSI_VAR_FONT));
	GetTextMetrics(hdc, &tm);
	CharY = tm.tmHeight + tm.tmExternalLeading;
	CharX = tm.tmAveCharWidth;
	ReleaseDC(hFrame, hdc);

	nClientLines = HIWORD(lParam) / CharY;
	PosView(0);
	SetScroll();

	return(FALSE);
}


// Function

	VOID SetWinCaption(PSTR szString)

// Summary ----------------------------------------------------------------
//
// SetWinCaption -- set the frame window caption according to last
// host fingered.
//
// Parameters
//
//	szString	The caption string
//
// Returns
//
//	Nothing.
//
// Ends -------------------------------------------------------------------

{
   char szcaption[128];

   strcpy(szcaption, szAppName);
   if ( szString != NULL )
   {
	   strcat(szcaption, " - ");
           if ( strlen(szString) + strlen(szcaption) < sizeof(szcaption))
		   strcat(szcaption, szString);
   }

   SetWindowText(hFrame, (LPSTR)szcaption);
}


#define MAXLINE	128
// Function

	PSTR WSErrorString(UINT ResourceID)

// Summary ----------------------------------------------------------------
//
//	Translates WinSock error number into an appropriate string.
//
//
// Parameters
//
//      ResourceID
//
//	The resource number/WS error number of the string to be printed.
//
// Return
//
//	PSTR	A pointer to the string that is to be printed.
//
// Ends -------------------------------------------------------------------

	{
		int i;
		static char szBuffer[MAXLINE];

		if ( !LoadString(hInst, ResourceID, (LPSTR)szBuffer, MAXLINE) )
			wsprintf( (LPSTR) szBuffer, "Windows Sockets error %d", ResourceID );

		return(szBuffer);
	}


// Function

	VOID ReportWSError(int Err)

// Summary ----------------------------------------------------------------
//
// ReportWSError -- prompt user with a windows sockets error message.
//
//
// Parameters
//
//	Err	The WinSock error.
//
// Returns
//
//	Nothing.
//
// Ends -------------------------------------------------------------------

	{
	   int i;
	   char szerr[128];


	   strcpy(szerr, WSErrorString(Err));
	   MessageBeep( MB_ICONSTOP );
	   MessageBox(hFrame, (LPSTR)szerr, (LPSTR)szAppName, MB_ICONSTOP | MB_OK);
	}


// Function

	VOID ReportFingerErr(UINT Err, int nWSErr)

// Summary ----------------------------------------------------------------
//
// ReportFingerErr -- prompt user with a ntime specific error
//
//
// Parameters
//
//	Err	The ntime error message
//
//	nWSErr	The WinSock error message
//
// Returns
//
//	Nothing.
//
// Ends -------------------------------------------------------------------

	{
	   int i;
	   char szerr[128];
	   HDC	hdc;


	   hdc = GetDC(hFrame);
	   strcpy(szerr, WSErrorString(Err));
	   WinPrintf(hdc, -1, 0, "%s", (LPSTR) szerr);

	   if (nWSErr > 0)
	   {
		strcat(szerr, "\n");
		strcat(szerr, WSErrorString(nWSErr));
	   }
	   WinPrintf(hdc, -1, 0, "%s", (LPSTR) WSErrorString(nWSErr));
	   ReleaseDC(hFrame, hdc);

	   MessageBeep( MB_ICONSTOP );
	   MessageBox(hFrame, (LPSTR)szerr, (LPSTR)szAppName, MB_ICONSTOP | MB_OK);

	}


// Dialog callback function

	BOOL CALLBACK HostDlgProc(HWND hDlg, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
// 	HostDlgProc -- dialog box proc for "setup dialog".
// 	This box queries user for a host name in response to the user's
// 	selection of the "Setup..." main menu item.
//
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	BOOL
//
//	Returns TRUE of the dialog message was handled, otherwise FALSE.
//
// Ends -------------------------------------------------------------------

{
   BOOL	fTranslated;
   char	string[25];


   switch(Msg)
   {
      case WM_INITDIALOG:
         SetDlgItemText(hDlg, IDC_HOSTNAME, szHostName);
	 SetDlgItemText(hDlg, IDC_USER, szUser);

	 SetDlgItemInt(hDlg, IDC_UDPRETRY, nUDPNumberRetry, FALSE);
	 SetDlgItemInt(hDlg, IDC_UDPTIME, nUDPTimeOutValue, FALSE);

	 bShouldUseTCP = GetPrivateProfileInt( SECTION_MAIN
					, ENTRY_ONLYTCP
					, TRUE
					, (LPCSTR) INI_FILE);
	 CheckDlgButton(hDlg, IDC_USETCP, bShouldUseTCP);

	 bUpdateSystemTime = GetPrivateProfileInt( SECTION_MAIN
					, ENTRY_DOUPDATE
					, FALSE
					, (LPCSTR) INI_FILE);
	 CheckDlgButton(hDlg, IDC_UPDATE, bUpdateSystemTime);

	 EnableWindow(GetDlgItem(hDlg, IDB_CONNECT), strlen(szHostName));
	 EnableWindow(GetDlgItem(hDlg, IDC_UDPRETRY), !bShouldUseTCP);
	 EnableWindow(GetDlgItem(hDlg, IDC_UDPTIME), !bShouldUseTCP);
	 EnableWindow(GetDlgItem(hDlg, IDT_UDP1), !bShouldUseTCP);
	 EnableWindow(GetDlgItem(hDlg, IDT_UDP2), !bShouldUseTCP);

	 GetPrivateProfileString( SECTION_MAIN
				, ENTRY_DEFAULTHOST
				, (LPSTR) szHostName
                                , (LPSTR) szHostName
				, sizeof(szHostName)
				, INI_FILE );
	 return TRUE;

      case WM_SYSCOLORCHANGE:
#ifdef USE_CTL3D
	 Ctl3dColorChange();
#endif
	 return TRUE;

 
      case WM_COMMAND:
         switch(wParam)
	 {
            case IDB_CONNECT:
            case IDOK:
                GetDlgItemText(hDlg, IDC_HOSTNAME, szHostName, MAXHOST);
		GetDlgItemText(hDlg, IDC_USER, szUser, MAXUSER);
		nUDPTimeOutValue = GetDlgItemInt(hDlg, IDC_UDPTIME, (BOOL FAR *) &fTranslated, FALSE);
		if ( !fTranslated )
		{
			nUDPTimeOutValue = DEFAULT_UDPTIME;
		 	SetDlgItemInt(hDlg, IDC_UDPTIME, nUDPTimeOutValue, FALSE);
                }

		nUDPNumberRetry = GetDlgItemInt(hDlg, IDC_UDPRETRY, (BOOL FAR *) &fTranslated, FALSE);
		if ( !fTranslated )
		{	nUDPNumberRetry = DEFAULT_UDPRETRY;
			SetDlgItemInt(hDlg, IDC_UDPRETRY, nUDPNumberRetry, FALSE);
		}

		bShouldUseTCP = IsDlgButtonChecked(hDlg, IDC_USETCP);
		if ( bShouldUseTCP )
		{
			WritePrivateProfileString(SECTION_MAIN
				, ENTRY_ONLYTCP
				, (LPSTR) "1"
				, (LPSTR) INI_FILE);
		} else {
			WritePrivateProfileString(SECTION_MAIN
				, ENTRY_ONLYTCP
				, (LPSTR) "0"
				, (LPSTR) INI_FILE);
		}

		bUpdateSystemTime = IsDlgButtonChecked(hDlg, IDC_UPDATE);
		if ( bUpdateSystemTime )
		{
			WritePrivateProfileString(SECTION_MAIN
				, ENTRY_DOUPDATE
				, (LPSTR) "1"
				, (LPSTR) INI_FILE);
		} else {
			WritePrivateProfileString(SECTION_MAIN
				, ENTRY_DOUPDATE
				, (LPSTR) "0"
				, (LPSTR) INI_FILE);
		}
		WritePrivateProfileString(SECTION_MAIN
				, ENTRY_DEFAULTHOST
				, (LPSTR) szHostName
				, (LPSTR) INI_FILE);

		WritePrivateProfileString(SECTION_MAIN
				, ENTRY_TIMECORR
				, (LPSTR) szUser
				, (LPSTR) INI_FILE);

		GetDlgItemText(hDlg, IDC_UDPRETRY, (LPSTR)string, sizeof(string));
		WritePrivateProfileString(SECTION_MAIN
				, ENTRY_UDPRETRY
				, (LPSTR) string
				, (LPSTR) INI_FILE);  

		GetDlgItemText(hDlg, IDC_UDPTIME, (LPSTR)string, sizeof(string));
		WritePrivateProfileString(SECTION_MAIN
				, ENTRY_UDPTIMEOUT
				, (LPSTR) string
				, (LPSTR) INI_FILE);  

	       EndDialog(hDlg, wParam);
	       return TRUE;

            case IDCANCEL:
	       EndDialog(hDlg, IDCANCEL);
	       return TRUE;

	    case IDC_USETCP:
		if (SendMessage(GetDlgItem(hDlg,IDC_USETCP), BM_GETCHECK, 0, 0L))
		{
			 EnableWindow(GetDlgItem(hDlg, IDC_UDPRETRY), FALSE);
			 EnableWindow(GetDlgItem(hDlg, IDC_UDPTIME), FALSE);
			 EnableWindow(GetDlgItem(hDlg, IDT_UDP1), FALSE);
			 EnableWindow(GetDlgItem(hDlg, IDT_UDP2), FALSE);
		} else {
			 EnableWindow(GetDlgItem(hDlg, IDC_UDPRETRY), TRUE);
			 EnableWindow(GetDlgItem(hDlg, IDC_UDPTIME), TRUE);
			 EnableWindow(GetDlgItem(hDlg, IDT_UDP1), TRUE);
			 EnableWindow(GetDlgItem(hDlg, IDT_UDP2), TRUE);
		}
                return TRUE;

	    case IDC_HOSTNAME:
	       if (HIWORD(lParam) == EN_CHANGE)
			EnableWindow(GetDlgItem(hDlg, IDB_CONNECT),
				(BOOL) SendMessage((HWND) LOWORD(lParam), WM_GETTEXTLENGTH, 0, 0L));
               return TRUE;
         }
         break;
   }

   return FALSE;
}


//
// AboutDlgProc -- callback for the "About" dialog box
//
BOOL CALLBACK AboutDlgProc(HWND hDlg, UINT Msg, WPARAM wParam, LPARAM lParam)

// Summary ----------------------------------------------------------------
//
//
// Parameters
//
//	hWnd	Handle of the window.
//
//	Msg	Message.
//
//	wParam	First message paramter.
//
//	lParam	Second message parameter.
//
//
// Returns
//
//	BOOL
//
//	Returns TRUE of the dialog message was handled, otherwise FALSE.
//
// Ends -------------------------------------------------------------------

	{

		switch(Msg)
		{
			default: return FALSE;

			case WM_INITDIALOG:
                        	return TRUE;

			case WM_SYSCOLORCHANGE:
#ifdef USE_CTL3D
				Ctl3dColorChange();
#endif
				break;

			case WM_COMMAND:
			{
				switch(wParam)
				{
				case IDOK:
				case IDCANCEL:
					EndDialog(hDlg, TRUE);
					break;

				default : return FALSE;
				}
			} // WM_COMMAND
		}
		return TRUE;
	}


#define MAX_PRINTF_OUTPUT	128
/*******************************************************************

    NAME:       WinPrintf

    SYNOPSIS:   A printf-like interface to TextOut

    ENTRY:      hdc - Display context for the text.

                row - Starting row (* tmHeight).

                col - Starting column (* tmAveCharWidth).

                pszFormat - A printf-like format string.

                ... - Other printf-like arguments as needed.

********************************************************************/
VOID WinPrintf( HDC   hdc, int row, UINT   col, LPSTR pszFormat, ... )
{
    TEXTMETRIC	tm;
    int	CharX, CharY;
    char szOutput[MAX_PRINTF_OUTPUT];
    int  cbOutput;
    va_list ArgList;
    static int prevrow;

    va_start( ArgList, pszFormat );
    cbOutput = wvsprintf( szOutput, pszFormat, ArgList );
    va_end( ArgList );

    SelectObject(hdc, GetStockObject(ANSI_VAR_FONT));
    GetTextMetrics(hdc, &tm);
    CharY = tm.tmHeight + tm.tmExternalLeading;
    CharX = tm.tmAveCharWidth;
#ifdef USE_CTL3D
    SetTextColor(hdc, GetSysColor(COLOR_BTNHIGHLIGHT));
    SetBkColor(hdc, GetSysColor(COLOR_BTNSHADOW));
#else
    SetTextColor(hdc, GetSysColor(COLOR_WINDOWTEXT));
    SetBkColor(hdc, GetSysColor(COLOR_WINDOW));
#endif

    if ( row < 0 )
    {
    	prevrow++;
	row = prevrow;
    } else { prevrow = row; }

    TextOut( hdc,
	     col * CharX,
	     row * CharY,
             szOutput,
	     cbOutput );


    if (bCommandLineArgs)
    	SetWinCaption(szOutput);

    strcat(szOutput, "\r\n");
    PushChars(szOutput, strlen(szOutput));

    GetDisplayList(&pLineItems, &nLineItems); // list and get new one

    PosView(0);                               // position view to top
    SetScroll();                              // rescale (or delete)
//	Repaint();                                // scrollbar & force a repaint


}   // WinPrintf





// Function

	void MakeHelpPathName(HINSTANCE hInst, LPSTR szFileName)

// Summary ----------------------------------------------------------------
//
//	This function will attempt to generate the full path name of
//	the help file from the path of the executable.  It assumes that
//	the .HLP file is in the same directory as the executable.
//
// Parameters
//
//
//	hInst
//
//	Identifies the module or the instance of the program.
//
//	szFileName
//
//	The buffer to accept the name of the help file.
//
//
// Return
//
//	nothing
//
// Ends -------------------------------------------------------------------

	{
		LPSTR   lpcFileName;
		int     nFileNameLen;
		const char	szHelpFile[] = HELP_FILE;

		nFileNameLen = GetModuleFileName(hInst, szFileName, MAX_PATH);
		lpcFileName = szFileName + nFileNameLen;

		/* Start looking in reverse for a backslash or full colon */
		while (lpcFileName > szFileName)
		{
			if ((*lpcFileName == '\\') || (*lpcFileName == ':'))
			{
				*(++lpcFileName) = '\0';
		break;
			}
			nFileNameLen--;
			lpcFileName--;
		}

		/* Is there enough room to append the help file name? */
		if ( ( nFileNameLen + lstrlen((LPSTR)szHelpFile) ) < MAX_PATH)
		{
			lstrcat( szFileName, (LPSTR) szHelpFile );
		}
		else
		{
			lstrcat( szFileName, "?" );
		}
	
		return;
	}
