/*

FLIST -- Generates file lists based on privilege levels and keys

Version 2.1  (8/26/92)

Written by Bob Quinlan of Austin, Texas, USA
Sysop of Red October at 512-834-2593 (1:382/111)

Copyright 1992 by Bob Quinlan

Compatible with Maximus 2.00 and 2.01


Special thanks to David Powers for encouragement, ideas, and code!


This program generates a list of files that a user with a particular
privilege level and set of keys could access.  It is intended to make
the procedure independent of changes to the structure of your file
areas.  Instead of specifying which areas should be included you just
specify what sort of user you want the list to apply to.

You can also create lists of new files.  Just use /D to specify how many
days of files you want to list.

    /Afile      Area file.  You can use this to specify a path to
                AREA.DAT or to refer to another file (although it must
                follow the AREA.DAT format).  The filename must be
                included along with the path.  Defaults to AREA.DAT in
				the current directory.

	/B          Include barricaded areas.  If this switch is not
				included no barricaded areas will be included.

    /C          Include comments from the FILES.BBS file lists.  By
                default only files and their descriptions are included.

                Using this switch has one side-effect!  Normally area
                headers are only generated for areas that have files in
                them.  /C will cause them to be generated for any area
                that has comments as well.  Do not use /C if you only
                want areas that have files in them to appear!  (I
				recommend not using /C with /D so that you don't show
                lots of apparently empty file areas.)

    /Ddays      Days.  Excludes all files more than this many days old.
                Use this switch to create lists of new files.

	/Ffile      Output file.  The pathname of the file you want the file
				list written to.  Defaults to FLIST.TXT in the current
				area.

    /HAfile     Area header file.  The contents of this file will be
                added to the report for each area header.  There are
                several sequences you can embed in the file that have
				special meanings:

                    %N will be replaced with the area name.
                    %I will be replaced with the area information.
                    %D will be replaced with the report date.
                    %T will be replaced with the report time.
                    %A will be replaced with the age of the files
                       included in the report.  Do not use this except
                       with the /D command line parameter!

                    Any of these may be followed by an L, R, or C
                    immediately followed by a number.  These will left,
                    right, or center justify the value in a field of the
					specified width.

                    You can also follow any of the above with an NF or
                    with an Fx.  NF stands for "no fill" and has no
                    effect (see examples).  Fx stands for "fill" and
                    will cause the entire field to be replaced by an
                    equivalent number of the character 'x'.  This is
                    useful when drawing boxes.

                This switch is not required.  If no area header file is
                specified a default header will be used.

    /HBfile     Beginning header file.  The contents of this file will
				be added to the beginning of the report.  There are
                several sequences you can embed in the file that have
                special meanings.  All of the codes listed under /HA are
                allowed except for %N and %I.

    /HEfile     End header file.  The contents of this file will be
                added to the end of the report.  There are several
                sequences you can embed in the file that have special
                meanings.  All of the codes listed under /HA are allowed
                except for %N and %I.

    /Kkeys      Keys to use for access to locked areas.  Keys are
				represented by 1-8 and A-X.  An asterisk (*) turns on
				all of the keys.  Multiple keys can be specified in one
				/K argument (such as /K12B).

	/N          No-wrap.  Leave long description lines intact.  The
				default is to wrap them.

	/O          Include offline files.  By default files that are listed
				but not present are not included in the list.

	/Ppriv      Privilege to use for access to areas.  Only the first
				letter of the level name is required.  Legal values are:
					Twit
					Disgrace
					Limited
					Normal
					Worthy
					Privil
					Favored
					Extra
					Clerk
					AsstSysop
					Sysop
					Hidden
				Defaults to Normal.

    /Q          Quiet.  Eliminates the progress display.

	/T          Truncate long description lines.  The default is to wrap
				them.

Let me show you an example of how to use the %N and %I parameters in an
area header file (defined with the /HA switch).  The purpose of those
parameters is to let you build boxes that will conform to the size of
the text produced by %N and %I.  Here is an example of a simple header
box using these parameters:

%NF%IFĿ
  %NNF    %INF  
%NF%IF

Note that the characters following the %NF and %IF look like a part of
the regular text, but are actually the characters that will be used to
fill out the parameters to the desired length.

This will expand to something that looks like this:

Ŀ
  MAX    Maximus BBS files and related utilities  


Now you can see why the apparently useless NF modifier can be useful. It
pads out the shorter embedded codes so that they visually match those in
the surrounding lines.  The following would produce exactly the same
result as the previous example, but it is a lot harder to figure out how
everything will match up:

%NF%IFĿ
  %N    %I  
%NF%IF

Here is an example using fixed width fields.  This also right justifies
the area name and left justifies the area information:

%NR9F%IL50FĿ
 %NR9NF  %IL50NF 
%NR9F%IL50F

Which will expand to something that looks like this:

Ŀ
       MAX  Maximus BBS files and related utilities            



FLIST returns the following ERRORLEVELS:
    0 = success
    1 = bad command line parameter
    2 = unable to open AREA.DAT
    3 = unable to open header file
    4 = unable to open output file
    5 = not enough memory to run
    6 = string overflow processing header

NOTICE:  You may use, copy, and distribute this program freely as long
as you insure that both the executable and the documentation (.DOC)
files are included in the distribution package.  The source code does
not need to be included.  You may modify this program and document, so
long as reasonable credit is given to the original author if a
substantial portion of the original remains intact.  The author is not
responsible for any losses which may occur either directly or indirectly
as a result of using this program.

This program uses the Maximus structures written by Scott J. Dudley.

HISTORY:
Version 2.1   (8/26/92) -- Added the /Q switch to eliminate the progress
                           display.  Now returns different errorlevels
                           for each possible error.  Fixed problem with
                           headers before comments.
Version 2.0   (8/23/92) -- Added more error checking.
Version 1.9   (8/22/92) -- Added a progress display.  Fixed a memory
                           corruption bug.
Version 1.8   (8/20/92) -- Added fixed width fields with left, right,
						   and centered justification for embedded
						   variables.  Changed the previous codes for
						   embedded variables!  Removed built-in help to
                           reduce memory requirements.
Version 1.7   (8/10/92) -- Added embedded variables to the begin and end
                           headers.  Note: This eliminated the automatic
                           file age comment that used to be included for
                           newfile lists.
Version 1.6   (8/10/92) -- Added the /HA option to allow customized area
                           headers in the report.  This is an expansion
                           of an idea and code by David Powers.
Version 1.5   (8/09/92) -- Added the /HB and /HE options to allow
                           customized beginning and end text in the
                           report.  Added /? help option to print the
                           parameter list.  Added a clearer explanation
						   of /C's effect on area headers.
Version 1.4   (8/07/92) -- Added the /C option to include file list
                           comments.  Added code to recognize free
                           switches so they do not show up in the
                           descriptions.  Fixed error if first entry in
                           AREA.DAT did not include a file area.
Version 1.3   (8/06/92) -- Read the file list filename from AREA.DAT
                           instead of assuming FILES.BBS.
Version 1.2   (8/02/92) -- Added days header for new files list.
Version 1.1   (7/30/92) -- Added file age cutoff.
Version 1.0   (6/30/92) -- Original release.  Written in Borland C.

Large memory model
*/


#include <ctype.h>
#include <dos.h>
#include <fcntl.h>
#include <io.h>
#include <share.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys\stat.h>
#include <time.h>
#include <mstruct.h>  /*  Maximus structures by Scott J. Dudley  */


#define MAXPATH     128
#define MAXLINE     256
#define MAXDESC     4096


struct _arealist
	{
	char    name[40];
    char    info[80];
    char    path[80];
    char    filesbbs[80];
	sword   priv;
	dword   locks;
	byte    barricade;
	struct _arealist *next;
	};


typedef struct _arealist __arealist;
typedef struct _area __area;


void process_args(int argc, char *argv[]);
sword get_priv(char priv);
dword get_keys(char *key_list);
__arealist *read_areas(void);
void process_header(char type, char *hfile, char *name, char *info, FILE *out);
void process_entry(char *entry, char *path, char *name, char *info,
	  int *header, FILE *out);


char	areafile[MAXPATH] = {"area.dat"};
char    hareafile[MAXPATH] = {""};
char    hbeginfile[MAXPATH] = {""};
char    hendfile[MAXPATH] = {""};
char	outfile[MAXPATH] = {"flist.txt"};
sword   priv = NORMAL;
dword   keys = 0x00000000;
int     barricade = 0;
int     comment = 0;
long    days = -1;
int     offline = 0;
int     wrap = 2;
int     quiet = 0;

long    age = -1;


int main(int argc, char *argv[])
	{

	char        filespath[MAXPATH];

	int         header;

	FILE        *out;
	FILE        *files;

	__arealist  *areas;
	__arealist  *cur;
	__arealist  *prev;

    char        line[MAXDESC];


    printf("FLIST 2.1 -- Copyright 1992 by Bob Quinlan (8/26/92)\n\n");

	process_args(argc, argv);

	areas = read_areas();

	if ((out = _fsopen(outfile, "wt", SH_DENYWR)) == NULL)
		{
		fprintf(stderr, "Unable to open %s\n", outfile);
        exit(4);
		}

	if (hbeginfile[0] != '\0')
		process_header('b', hbeginfile, NULL, NULL, out);

	cur = areas;
	while (cur != NULL)
		{
		if ((priv >= (cur->priv)) && (((~keys)&(cur->locks)) == 0) &&
			  ((cur->barricade == 0) || (barricade)))
			{
            if (!quiet)
                printf("%s -- %s", cur->name, cur->info);
            strcpy(filespath, cur->filesbbs);
			if (filespath[0] == '\0')
				{
				strcpy(filespath, cur->path);
				strcat(filespath, "files.bbs");
				}
			header = 0;
			if ((files = _fsopen(filespath, "rt", SH_DENYNONE)) != NULL)
				{
                while (fgets(line, MAXDESC, files) != NULL)
					{
					process_entry(line, cur->path, cur->name, cur->info,
						  &header, out);
                    if (!quiet)
                        printf(".");
					}
				fclose(files);
				}
            if (!quiet)
                printf("\n");
            }
		/*  Increment to next link and free previous link  */
		prev = cur;
		cur = cur->next;
		free(prev);
		}

	if (hendfile[0] != '\0')
		process_header('e', hendfile, NULL, NULL, out);

	fclose(out);

	return 0;
	}


void process_args(int argc, char *argv[])
	{
	int i;

	for (i=1; i<argc; i++)
		{
		if ((argv[i][0] == '/') || (argv[i][0] == '-'))
			{
			switch (tolower(argv[i][1]))
				{
				case 'a':  /*  area.dat  */
					strncpy(areafile, argv[i]+2, MAXPATH);
					break;
				case 'b':  /*  include barricaded areas  */
					barricade = 1;
					break;
				case 'c':  /*  include file list comment lines  */
					comment = 1;
					break;
				case 'd':  /*  days (of file date)  */
					days = atol(argv[i]+2);
					age = time(NULL)-(days*24*60*60);
					break;
				case 'f':  /*  output file  */
					strncpy(outfile, argv[i]+2, MAXPATH);
					break;
				case 'h':  /*  headers  */
					switch(tolower(argv[i][2]))
						{
						case 'a':
							strncpy(hareafile, argv[i]+3, MAXPATH);
							break;
						case 'b':
							strncpy(hbeginfile, argv[i]+3, MAXPATH);
							break;
						case 'e':
							strncpy(hendfile, argv[i]+3, MAXPATH);
							break;
						default:
							fprintf(stderr, "Unknown switch: %s\n", argv[i]);
							exit(1);
							break;
						}
					break;
				case 'k':  /*  keys  */
					keys |= get_keys(argv[i]+2);
					break;
				case 'n':  /*  no-wrap  */
					wrap = 0;
					break;
				case 'o':  /*  offline files allowed  */
					offline = 1;
					break;
				case 'p':  /*  privilege  */
					priv = get_priv(argv[i][2]);
					if (priv < TWIT)
						{
						priv = TWIT;
						fprintf(stderr, "Invalid privilege: %c\n", argv[i][2]);
						}
					break;
                case 'q':  /*  quiet  */
                    quiet = 1;
                    break;
				case 't':  /*  truncate lines  */
					wrap = 1;
					break;
				default:
					fprintf(stderr, "Unknown switch: %s\n", argv[i]);
					exit(1);
					break;
				}
			}
		else
			{
			fprintf(stderr, "Not a switch: %s\n", argv[i]);
			exit(1);
			}
		}
	}


sword get_priv(char priv)
	{
	char    priv_name[] = {"tdlnwpfecash"};
	sword   priv_code[] =
				{
				TWIT,
				DISGRACE,
				LIMITED,
				NORMAL,
				WORTHY,
				PRIVIL,
				FAVORED,
				EXTRA,
				CLERK,
				ASSTSYSOP,
				SYSOP,
				HIDDEN
				};

	char    *where;

	where = strchr(priv_name, tolower(priv));
	if (where == NULL)
		return TWIT-1;

	return priv_code[(int)(where-priv_name)];
	}


dword get_keys(char *key_list)
	{
	char    *key;
	dword   mask = 0x00000000;
	int     keylen;
	int     i;

	key = strupr(strdup(key_list));

	keylen = strlen(key);
	for (i=0; i<keylen; i++)
		{
		if (key[i] == '*')
			{
			mask = 0xFFFFFFFF;
			break;
			}
		else if ((key[i] >= '1') && (key[i] <= '8'))
			mask |= 0x00000001L<<(key[i]-'1');
		else if ((key[i] >= 'A') && (key[i] <= 'X'))
			mask |= 0x00000100L<<(key[i]-'A');
		}

	free(key);

	return mask;
	}


__arealist *read_areas(void)
	{
	FILE        *areadat;
	__area      area;
	__arealist  *tmp;
	__arealist  *first;
	__arealist  *cur = NULL;
	dword       slen;
	dword       offset = 0;

	if ((areadat = _fsopen(areafile, "rb", SH_DENYNONE)) == NULL)
		{
		fprintf(stderr, "Unable to open %s\n", areafile);
        exit(2);
		}

	while (fread(&area, sizeof(__area), 1, areadat) > 0)
		{
		if (offset == 0)
			slen = (dword)(area.struct_len);
		if (area.filepath[0] != '\0')
			{
            if ((tmp = (__arealist *)malloc(sizeof(__arealist))) == NULL)
				{
				fprintf(stderr, "Out of memory!\n");
                exit(5);
				}
			if (cur == NULL)
				first = tmp;
			else
				cur->next = tmp;
			cur = tmp;
			strcpy(cur->name, area.name);
			strcpy(cur->info, area.fileinfo);
			strcpy(cur->path, area.filepath);
			strcpy(cur->filesbbs, area.filesbbs);
            cur->priv = area.filepriv;
            cur->locks = area.filelock;
            cur->barricade = (area.filebar[0] != '\0');
			}
		offset += slen;
        if (fseek(areadat, offset, SEEK_SET) != 0)
            break;
		}

	cur->next = NULL;
	fclose(areadat);

	return first;
	}


void process_header(char type, char *hfile, char *name, char *info, FILE *out)
	{
    FILE    *h;
    char    buf[MAXLINE];
    char    tbuf[MAXLINE];
    char    filler[MAXLINE];
    int     flen;
    char    *param;
    int     plen;
    int     fwidth;
    char    *stopper;
    char    c;
    int     j, k;

    if ((h = _fsopen(hfile, "rt", SH_DENYNONE)) == NULL)
        {
        fprintf(stderr, "Unable to open %s\n", hfile);
        exit(3);
        }
	/*  Read data from the header file  */
    while (fgets(buf, MAXLINE, h) != NULL)
        {
		/*  Process special replacement characters  */
        while ((param = strchr(buf, '%')) != NULL)
            {
            filler[0] = '\0';
            plen = 2;
			/*  Check the basic parameter type  */
			switch (tolower(param[1]))
				{
				case 'a':  /*  age  */
                    itoa(days, filler, 10);
					break;
				case 'd':  /*  date  */
                    _strdate(filler);
					break;
				case 't':  /*  time  */
                    _strtime(filler);
					break;
				}
			/*  Only allow name and information in area headers  */
			if (type == 'a')
				{
				switch (tolower(param[1]))
					{
					case 'n':  /*  name  */
                        strcpy(filler, name);
						break;
					case 'i':  /*  information  */
                        strcpy(filler, info);
						break;
					}
				}

			/*  If any replacements were found...  */
            if (filler[0] != '\0')
				{
				/*  Process width values if present  */
				c = tolower(param[plen]);
                if (strchr("lrc", c) != NULL)
					{
                    ++plen;
					/*  Convert the width value and find where it stops  */
					fwidth = (int)strtol(param+plen, &stopper, 10);
					plen += (int)(stopper-(param+plen));
                    /*  If the field is wider than the filler string...  */
                    if (fwidth >= MAXLINE)
                        {
                        fprintf(stderr, "Width %d too wide!  Ignored.\n",
                              fwidth);
                        }
					/*  If the field is wider than the current data...  */
                    else if (strlen(filler) < fwidth)
						{
						switch (c)
							{
							case 'l':  /*  left justify  */
								/*  Pad on the right  */
                                for (j=strlen(filler); j<fwidth; j++)
                                    filler[j] = ' ';
								break;
							case 'r':  /*  right justify  */
                                flen = strlen(filler);
								/*  Shift the string right  */
								for (j=1; j<=flen; j++)
                                    filler[fwidth-j] = filler[flen-j];
								/*  Pad on the left  */
                                for (j=0; j<fwidth-flen; j++)
                                    filler[j] = ' ';
								break;
							case 'c':  /*  center  */
                                flen = strlen(filler);
								/*  Pad on the right  */
                                k = (fwidth-flen+1)/2;
								for (j=1; j<=k; j++)
                                    filler[fwidth-j] = ' ';
								/*  Shift the string to center  */
								for (j=1; j<=flen; j++)
                                    filler[fwidth-k-j] = filler[flen-j];
								/*  Pad on the left  */
                                k = (fwidth-flen)/2;
								for (j=0; j<k; j++)
                                    filler[j] = ' ';
								break;
							}
						/*  Terminate the adjusted string  */
                        filler[fwidth] = '\0';
						}
					}

				/*  Process fill character if present  */
				if (tolower(param[plen]) == 'f')
					{
                    strset(filler, param[++plen]);
					++plen;
					}
				/*  Skip "no fill" parameter if present  */
				else if (strncmpi(param+plen, "nf", 2) == 0)
					{
                    plen += 2;
					}

				/*  Perform the replacment  */
                if ((strlen(buf)+strlen(filler)-plen) >= MAXLINE)
                    {
                    fprintf(stderr, "String overflow!\n");
                    exit(6);
                    }
				param[0] = '\0';
                strcpy(tbuf, buf);
                strcat(tbuf, filler);
                strcat(tbuf, param+plen);
                strcpy(buf, tbuf);
				}
			}

		/*  Output header  */
        fputs(buf, out);
		}

	fclose(h);
	}


void process_entry(char *entry, char *path, char *name, char *info,
	  int *header, FILE *out)
	{
	char    filepath[MAXPATH];
	int     file;

	long            fsize;
	struct ftime    ftimep;
	struct tm     	ftm;
	int             fdata;

    char    line[MAXDESC];
	char    fill[40];
	char    *descrip;
	char    *end;
	char    *ch;
	int     i;

	strcpy(line, entry);

	/*  Trim newline  */
	end = strchr(line, '\n');
	if (end != NULL)
		end[0] = '\0';
	else
		end = strchr(line, '\0');

	/*  Check for comment lines  */
	if (strchr(" -", line[0]) != NULL)
		{
		if (comment)
			{
			/*  Output area header if not already done  */
			if (!(*header))
				{
				if (hareafile[0] != '\0')
					process_header('a', hareafile, name, info, out);
                else
                    fprintf(out, "\n\n    Area %s -- %s\n\n", name, info);
                *header = 1;
				}

			/*  Output comment line  */
			fprintf(out, "%s\n", line);
			}
        }
    else
        {
		/*  Parse into filename and description, skipping switches  */
		descrip = line;
        do
			{
            ch = strpbrk(descrip, " \t");
            if (ch == NULL)
                descrip = end;
            else
                {
				ch[0] = '\0';
                i = strspn(++ch, " \t");
                descrip = ch+i;
                }
			} while (descrip[0] == '/');

		/*  Get file size and date  */
        strcpy(filepath, path);
		strcat(filepath, line);
        if ((file = sopen(filepath, O_RDONLY, SH_DENYNONE)) == -1)
            {
			if (offline)
                fdata = 0;
            else
				return;
			}
        else
            {
            fsize = filelength(file);
            getftime(file, &ftimep);
            close(file);
            fdata = 1;

            memset(&ftm, 0, sizeof(struct tm));
			ftm.tm_sec = ftimep.ft_tsec*2;
            ftm.tm_min = ftimep.ft_min;
			ftm.tm_hour = ftimep.ft_hour;
            ftm.tm_mday = ftimep.ft_day;
            ftm.tm_mon = ftimep.ft_month-1;
            ftm.tm_year = ftimep.ft_year+80;
            if (mktime(&ftm) < age)
				return;
			}

		/*  Output area header if not already done  */
		if (!(*header))
            {
			if (hareafile[0] != '\0')
                process_header('a', hareafile, name, info, out);
            else
                fprintf(out, "\n\n    Area %s -- %s\n\n", name, info);
			*header = 1;
            }

		/*  Output file entry  */
        if (fdata)
            {
			fprintf(out, "%-12s %7lu %2.2u-%2.2u-%2.2u", line,
                  fsize, ftimep.ft_month, ftimep.ft_day,
                  (ftimep.ft_year+80)%100);
            }
        else
            {
			fprintf(out, "%-12s                 ", line);
            }
		/*  Output description  */
        switch (wrap)
            {
			case 0:  /*  no-wrap  */
                fprintf(out, " %s\n", descrip);
                break;
			case 1:  /*  truncate  */
                fprintf(out, " %.49s\n", descrip);
                break;
			case 2:  /*  wrap  */
                strcpy(fill, " ");
                while (strlen(descrip) > 49)
                    {
                    for (i=49; i>20; i--)
                        {
                        if (strchr(" \t", descrip[i]) != NULL)
                            break;
                        }
					/*  If there are no breaks...  */
                    if (i == 20)
                        {
                        fprintf(out, "%s%.49s\n", fill, descrip);
                        descrip += 49;
                        }
                    else
                        {
                        descrip[i++] = '\0';
                        fprintf(out, "%s%.49s\n", fill, descrip);
                        descrip += i;
						while (strchr(" \t", descrip[0]) != NULL)
                            descrip++;
                        }
                    strcpy(fill, "                              ");
                    }
                fprintf(out, "%s%.49s\n", fill, descrip);
                break;
            }
        }
	}


