/*

FMTAG -- Force message area tagging of a specified area for the current user

Version 1.0   (1/02/93)
Written by Bob Quinlan of Austin, Texas, USA
Sysop of Red October at 512-834-2593 (1:382/111)

Copyright 1993 by Bob Quinlan

Compatible with Maximus 2.01


FMTAG forces a message area to be tagged for the current user.  It will
not do anything if the user does not already have a message tag record
in the file, so if the user has never tagged any areas it will fail.  It
requires you to supply the message area "number" for the area you want
tagged.  Here are the parameters you can specify:

    /Llastuserfile  LastUser file to use.  Defaults to LASTUSER.BBS in
                    the current directory.

    /Messagearea    Message area "number" to be tagged.  This must be
                    supplied.

    /Ttagfile       Message Tag file to use.  Defaults to MTAG.BBS in
                    the current directory.

Warning:  FMTAG does not check to see if the message area "number" you
give it is valid.  Be careful to specify a valid "number" or you will
add useless data to the MTAG.BBS file that Maximus will not remove.
(Checking for a valid "number" would take longer than everything else
put together, so it was left out in the interest of greater efficiency.)


FMTAG returns ERRORLEVEL 0 after a successful run.  ERRORLEVEL 1
indicates that the user does not yet have a message tag record in the
file.  ERRORLEVEL 2 is returned to indicate an error.

NOTICE:  You may use, copy, and distribute this program freely as long
as you insure that both the executable and the documentation (.DOC)
files are included in the distribution package.  The source code does
not need to be included.  You may modify this program and document, so
long as reasonable credit is given to the original author if a
substantial portion of the original remains intact.  The author is not
responsible for any losses which may occur either directly or indirectly
as a result of using this program.

This program uses the Maximus structures written by Scott J. Dudley.

HISTORY:
Version 1.0   (1/02/93) -- Original release.  Written in Borland C.

Large memory model
*/

#include <ctype.h>
#include <fcntl.h>
#include <io.h>
#include <share.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys\stat.h>
#include <mstruct.h>   /*  Maximus structures by Scott J. Dudley  */


#define MAXPATH     (128)
#define MAXAREA     (40)
#ifndef FALSE
    #define FALSE   0
    #define TRUE    1
#endif


main(int argc, char *argv[])
{
    struct  _usr        lastuser;
    struct  _tagdata    tagdata;

    char    lastuser_file[MAXPATH] = {"LASTUSER.BBS"};
    int     lastuser_fh;
    char    tag_file[MAXPATH] = {"MTAG.BBS"};
    int     tag_fh;
    char    area[MAXAREA+1] = {"\x1"};

    int     tagdata_len;

    int     match = FALSE;

    char    *ch;
    int     i, j, k;


    /***********/
    /*  FMTAG  */
    /***********/

    printf("FMTAG 1.0 -- Copyright 1992 by Bob Quinlan (1/02/93)\n");

    /*  Process switches  */
    for (i = 1; i < argc; i++)
    {
        switch (tolower(argv[i][1]))
        {
            case 'l':  /*  lastuser file  */
                strcpy(lastuser_file, argv[i]+2);
                break;
            case 'm':  /*  message area "number"  */
                /*  The first byte of area is the leading '\x1'  */
                strcpy(area+1, argv[i]+2);
                strupr(area+1);
                break;
            case 't':  /*  tag file  */
                strcpy(tag_file, argv[i]+2);
                break;
            default:
                fprintf(stderr, "Unknown switch: %s\n", argv[i]);
                break;
        }
    }

    /*  Check to see if a message area has been specified  */
    if (area[1] == '\0')
    {
        fprintf(stderr, "No area to tag.  You must specify a /m parameter.\n");
        exit(2);
    }

    /*  Open the lastuser file  */
    if ((lastuser_fh = sopen(lastuser_file, O_RDONLY | O_BINARY, SH_DENYNO))
        == -1)
    {
        fprintf(stderr, "Unable to open %s\n", lastuser_file);
        exit(2);
    }

    /*  Read the user record  */
    if (read(lastuser_fh, &lastuser, sizeof(struct _usr)) < sizeof(struct _usr))
    {
        fprintf(stderr, "Unable to read %s\n", lastuser_file);
        exit(2);
    }

    /*  Close the lastuser file  */
    close(lastuser_fh);

    /*  Open the tag file  */
    if ((tag_fh = sopen(tag_file, O_RDWR | O_BINARY, SH_DENYNO)) == -1)
    {
        fprintf(stderr, "Unable to open %s\n", tag_file);
        exit(2);
    }

    /*  Read the record size from the first record  */
    if (read(tag_fh, &tagdata_len, sizeof(int)) < sizeof(int))
    {
        fprintf(stderr, "Unable to read %s\n", tag_file);
        exit(2);
    }
    /*  If the record size is invalid assume the structure size  */
    if (tagdata_len <= 0)
    {
        tagdata_len = sizeof(struct _tagdata);
    }

    /*  Seek to the appropriate record of the tag file  */
    lseek(tag_fh, (long)lastuser.lastread_ptr*tagdata_len, SEEK_SET);
    /*  Read the current user's record from the tag file  */
    if (read(tag_fh, &tagdata, sizeof(struct _tagdata)) <
        sizeof(struct _tagdata))
    {
        printf("%s is not in %s\n", lastuser.name, tag_file);
        printf("%s not tagged.\n", area+1);
        exit(1);
    }

    /*  Check to be sure the record names match  */
    if (strcmpi(tagdata.name, lastuser.name) != 0)
    {
        printf("%s shows as \"%s\" in %s\n", lastuser.name, tagdata.name,
            tag_file);
        printf("%s not tagged.\n", area+1);
        exit(1);
    }

    /*  Scan through the tag data looking for the area to be forced  */
    ch = tagdata.areas;
    /*  Position to the next match  */
    while ((ch = strstr(ch, area)) != NULL)
    {
        /*  Position to the end of the match  */
        ch += strlen(area);
        /*  Check to see if the match is complete  */
        if ((ch[0] == '\x1') || (ch[0] == '\0'))
        {
            printf("%s already tagged.\n", area+1);
            match = TRUE;
        }
    }

    /*  If no match was found tag the forced area  */
    if (match == FALSE)
    {
        /*  Add the forced area to the end of the tag list data  */
        strcpy(tagdata.areas+strlen(tagdata.areas), area);
        /*  Lock the user's record of the tag file  */
        lock(tag_fh, (long)lastuser.lastread_ptr*tagdata_len,
            sizeof(struct _tagdata));
        /*  Seek to the user's record  */
        lseek(tag_fh, (long)lastuser.lastread_ptr*tagdata_len, SEEK_SET);
        /*  Write the new tag list  */
        write(tag_fh, &tagdata, sizeof(struct _tagdata));
        /*  Unlock the user's record  */
        unlock(tag_fh, (long)lastuser.lastread_ptr*tagdata_len,
            sizeof(struct _tagdata));
        printf("%s now tagged.\n", area+1);
    }

    /*  Close the tag file  */
    close(tag_fh);

    return 0;
}


