/*#define DEBUG*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <io.h>
#include <fcntl.h>
#include "prog.h"
#include "sstat.h"
#include "sstatp.h"


static int near MatchNN(NETADDR *n1, NETADDR *n2)
{
  return (n1->zone==n2->zone &&
          n1->net ==n2->net  &&
          n1->node==n2->node &&
          n1->point==n2->point);
}



static int near DoThisArea(char *tag)
{
  struct _arealist *al;
  
  for (al=sc.area; al; al=al->next)
    if (eqstri(tag, al->tag))
      return TRUE;
    
  return FALSE;
}

static int near DoThisNode(NETADDR *n)
{
  struct _nodelist *nl;
  
  for (nl=sc.node; nl; nl=nl->next)
    if (MatchNN(&nl->n, n))
      return TRUE;
    
  return FALSE;
}



/* Read all of the specified nodes in from the SQUISH.STA stats file */

static void near ReadArea(int fd, struct _ahlist *al, struct _areahdr *ah)
{
  struct _nodehdr nh;
  struct _stlist *sl;
  struct _nodtot *nt;


  al->in_msgs += ah->in_msgs;
  al->in_bytes += ah->in_bytes/1024;

  while (ah->n_nodes--)
  {
    if (read(fd, (char *)&nh, sizeof(struct _nodehdr)) != 
                                            sizeof(struct _nodehdr))
      return;
    
    #ifdef DEBUG
    printf("    Node         = %s\n", Address(&nh.node));
    printf("        OutMsgs  = %ld\n", nh.out_msgs);
    printf("        OutBytes = %ld\n", nh.out_bytes);
    #endif

    /* Only process the specified nodes */
      
    if (! DoThisNode(&nh.node))
      continue;
    
    for (sl=al->slist; sl; sl=sl->next)
      if (MatchNN(&sl->node, &nh.node))
        break;
      
    if (sl==NULL)
    {
      sl=smalloc(sizeof(struct _stlist));
      
      sl->next=al->slist;
      al->slist=sl;
    }

    /* Make sure that this node is already in the llist of node totals */

    for (nt=nodtot; nt; nt=nt->next)
      if (MatchNN(&nt->node, &nh.node))
        break;

    /* Not there, so add it */

    if (nt==NULL)
    {
      nt=smalloc(sizeof(struct _nodtot));

      nt->node=nh.node;

      nt->next=nodtot;
      nodtot=nt;
    }
    
    sl->node=nh.node;
    sl->out_msgs += nh.out_msgs;
    sl->out_bytes += nh.out_bytes/1024;
  }
}





static void near ParseStats(int fd)
{
  struct _areahdr ah;
  struct _ahlist *al;

  while (read(fd, (char *)&ah, sizeof(struct _areahdr))==
                                                sizeof(struct _areahdr))
  {
    #ifdef DEBUG
    printf("AREA     = %s\n", ah.tag);
    printf(" InMsgs  = %ld\n", ah.in_msgs);
    printf(" InBytes = %ld\n", ah.in_bytes);
    printf(" n_nodes = %d\n", ah.n_nodes);
    #endif

    for (al=ahlist; al; al=al->next)
      if (eqstri(ah.tag, al->tag))
        break;

    /* This area not found */

    if (al==NULL)
    {
      al=smalloc(sizeof(struct _ahlist));

      strcpy(al->tag, ah.tag);

      al->next=ahlist;
      ahlist=al;
    }

    ReadArea(fd, al, &ah);
  }
}





static int near Percent1(dword a, dword b)
{
  return (int)(a*100L/b);
}




static int near Percent2(dword a, dword b)
{
  return (int)((a*10000L/b) % 100L);
}

#define Percent(a, b) Percent1(a,b), Percent2(a,b)




static void near CalcTotals(dword *total_in_bytes, dword *total_in_msgs)
{
  struct _ahlist *al;
  struct _stlist *sl;

  *total_in_bytes=*total_in_msgs=0;

  for (al=ahlist; al; al=al->next)
  {
    if (! DoThisArea(al->tag))
      continue;
    
    al->total_out_bytes=al->total_out_msgs=0;

    for (sl=al->slist; sl; sl=sl->next)
    {
      al->total_out_bytes += sl->out_bytes;
      al->total_out_msgs  += sl->out_msgs;
    }

    if (al->total_out_bytes==0 && al->total_out_msgs==0)
      continue;

    *total_in_bytes += al->in_bytes;
    *total_in_msgs  += al->in_msgs;
  }
}




static void near CalculateStats(dword total_in_bytes, dword total_in_msgs)
{
  struct _ahlist *al;
  struct _stlist *sl;
  struct _nodtot *nt;

  for (al=ahlist; al; al=al->next)
  {
    if (! DoThisArea(al->tag))
      continue;

    /* Don't log areas with no output */

    printf("\nArea %s\n", al->tag);

    if (total_in_bytes==0 || total_in_msgs==0 ||
        al->total_out_bytes==0 || al->total_out_msgs==0)
    {
      printf("   NO SIGNIFICANT TRAFFIC\n");
      continue;
    }

    printf("  KBYTES IN: %8ld (%02d.%02d%% of total bytes in)\n",
           al->in_bytes,
           (int)(al->in_bytes*100/total_in_bytes),
           (int)((al->in_bytes*10000/total_in_bytes) % 100));

    printf("  MSGS IN:   %8ld (%02d.%02d%% of total msgs in)\n",
           al->in_msgs,
           (int)(al->in_msgs*100/total_in_msgs),
           (int)((al->in_msgs*10000/total_in_msgs) % 100));

    printf("  KBYTES OUT:%8ld\n\n", al->total_out_bytes);


    printf("   Node            KByteOut MsgOut %%KBytes %%  Msgs %%TKBCst %% TMCst\n");
    printf("   --------------- -------- ------ ------- ------- ------- -------\n");

    for (sl=al->slist; sl; sl=sl->next)
    {
      double area_percent_bytes, area_percent_msgs;

      area_percent_bytes=((double)sl->out_bytes/(double)al->total_out_bytes)*
                         ((double)al->in_bytes /(double)total_in_bytes)*
                          (double)100;

      area_percent_msgs =((double)sl->out_msgs/(double)al->total_out_msgs)*
                         ((double)al->in_msgs /(double)total_in_msgs)*
                          (double)100;

      printf("   %-15s %8ld %6ld %3d.%02d%% %3d.%02d%% %6.02f%% %6.02f%%\n",
             Address(&sl->node),
             sl->out_bytes,
             sl->out_msgs,
             Percent(sl->out_bytes, al->total_out_bytes),
             Percent(sl->out_msgs,  al->total_out_msgs),
             area_percent_bytes,
             area_percent_msgs);

      for (nt=nodtot; nt; nt=nt->next)
        if (MatchNN(&nt->node, &sl->node))
        {
          nt->total_percent_bytes += area_percent_bytes;
          nt->total_percent_msgs  += area_percent_msgs;
          break;
        }
    }
  }

  printf("\nNODE TOTALS:\n\n");

  printf("   Node             %Bytes % Msgs\n");
  printf("   ---------------- ------ ------\n");

  for (nt=nodtot; nt; nt=nt->next)
  {
    printf("   %-15s  %05.02f%% %05.02f%%\n",
           Address(&nt->node),
           nt->total_percent_bytes,
           nt->total_percent_msgs);
  }

}





static void near ParseConfigLine(char *line)
{
  static char *cfgdelim=" \t\n\r,";
  char *s;

  /* Strip off any comments */
  
  if ((s=strchr(line, ';')) != NULL)
    *s='\0';


  /* Grab a word from the config */
  
  s=strtok(line, cfgdelim);
  
  /* Nodes to track */
  
  if (s==NULL)
    return;

  if (eqstri(s, "track"))
  {
    struct _nodelist *node;
    NETADDR last;
  
    memset(&last, '\0', sizeof(NETADDR));


    /* Parse all net/node numbers off this line */
    
    while ((s=strtok(NULL, cfgdelim)) != NULL)
    {
      node=smalloc(sizeof(struct _nodelist));

      node->n=last;
      
      ParseNN(s, &node->n.zone, &node->n.net,
              &node->n.node, &node->n.point, FALSE);
      
      last=node->n;

      /* Append to linked list */
      
      node->next=sc.node;
      sc.node=node;
    }
  }
  else if (eqstri(s, "area"))
  {
    struct _arealist *area;
    
    while ((s=strtok(NULL, cfgdelim)) != NULL)
    {
      area=smalloc(sizeof(struct _arealist));
      
      area->tag=sstrdup(s);
      
      area->next=sc.area;
      sc.area=area;
    }
  }
  else
  {
    printf("Invalid keyword in config file: `%s'\n", s);
  }
}





static void near ParseConfig(char *cfg)
{
  FILE *fp;
  char line[PATHLEN];
  
  sc.node=NULL;
  sc.area=NULL;
  
  if (cfg==NULL)
    cfg="SSTAT.CFG";
  
  if ((fp=fopen(cfg, "r"))==NULL)
  {
    printf("Error opening `%s'!\n", cfg);
    exit(1);
  }
  
  while (fgets(line, PATHLEN, fp))
    ParseConfigLine(line);
  
  fclose(fp);
}





int _stdc main(int argc, char *argv[])
{
  dword total_in_bytes, total_in_msgs;
  int fd;

  NW(argc);
  NW(argv);
  
  ParseConfig(argv[1]);

  if ((fd=open("SQUISH.STA", O_RDONLY | O_BINARY))==-1)
  {
    printf("Error!  No statistics file to read!\n");
    return 1;
  }

  ParseStats(fd);
  close(fd);

  CalcTotals(&total_in_bytes, &total_in_msgs);
  CalculateStats(total_in_bytes, total_in_msgs);

  return 0;
}

void _fast NoMem(void)
{
  printf("Ran out of memory!\n");
  exit(1);
}

