/*****************************************************************************/
/*>                                                                         <*/
/*>                     Telegard Bulletin Board System                      <*/
/*>      Copyright 1988-1991 by Martin Pollard.  All rights reserved.       <*/
/*>                                                                         <*/
/*>                     Data Types and File Structures                      <*/
/*>                                                                         <*/
/*****************************************************************************/

#ifndef __TELEGARD_H
#define __TELEGARD_H

#define BBS_VER     "2.7"       /* current version number */

#define MAXBOARDS       256     /* max # of message bases */
#define MAXUBOARDS      256     /* max # of file bases */
#define MAXPROTOCOLS    120     /* max # of transfer protocols */
#define MAXEVENTS       10      /* max # of scheduled events */
#define MAXARCS         8       /* max # of archivers */
#define MAXMENUCMDS     50      /* max # of menu commands */
#define NUMVOTEAS       10      /* max # of voting question answers */
#define NUMVOTEQS       20      /* max # of voting questions */

#define ATNO            0       /* no anonymous posts allowed */
#define ATYES           1       /* anonymous posts are allowed */
#define ATFORCED        2       /* ALL posts are forced anonymous */
#define ATDEARABBY      3       /* "Dear Abby" message base */
#define ATANYNAME       4       /* users can post as ANY name they want */

#define MLOCAL          0       /* local message base */
#define MNETMAIL        1       /* NetMail message base */
#define MECHOMAIL       2       /* EchoMail message base */
#define MGROUPMAIL      3       /* GroupMail message base */

#define MPUBLIC         0       /* post public messages only */
#define MPRIVATE        1       /* post private messages only */

/*---------------------------------------------------------------------------*/

typedef unsigned char boolean;      /* Turbo Pascal "boolean" */
typedef unsigned char byte;         /* Turbo Pascal "byte" */
typedef signed short int integer;   /* Turbo Pascal "integer" */
typedef signed long int longint;    /* Turbo Pascal "longint" */
typedef unsigned char string;       /* Turbo Pascal "string" */
typedef unsigned short int word;    /* Turbo Pascal "word" */
typedef unsigned int bit;           /* shorthand for C bit fields */

/*---------------------------------------------------------------------------*/

typedef string acstring[21];        /* Access Control System string */
typedef byte acrq[4];               /* AR flags */
typedef byte clrs[2][10];           /* color records: */
                                    /* [0][x] = mono  */
                                    /* [1][x] = color */
typedef integer secrange[256];      /* security tables */

typedef byte cpackdatetime[6];              /* packed date/time */
typedef byte mzscanr[MAXBOARDS / 8];        /* msg base scan flags */
typedef byte fzscanr[MAXUBOARDS / 8];       /* file base scan flags */
typedef cpackdatetime mhireadr[MAXBOARDS];  /* last msg read table */

/*****************************************************************************/
/*                                                                           */
/*       The format of the packed date/time structure is as follows:         */
/*                                                                           */
/*                 Byte #0    |    Byte #1    |    Byte #2                   */
/*             ===============|===============|===============               */
/*             7 6 5 4 3 2 1 0|7 6 5 4 3 2 1 0|7 6 5 4 3 2 1 0               */
/*             `-----------' `-------------' `-----' `-------'               */
/*                RESERVED        Year        Month     Day                  */
/*                              (8 bits)    (4 bits) (5 bits)                */
/*                                                                           */
/*                 Byte #3    |    Byte #4    |    Byte #5                   */
/*             ===============|===============|===============               */
/*             7 6 5 4 3 2 1 0|7 6 5 4 3 2 1 0|7 6 5 4 3 2 1 0               */
/*             `-------' `---------' `---------' `-----------'               */
/*               Hour      Minute      Second    1/100 Seconds               */
/*             (5 bits)   (6 bits)    (6 bits)     (7 bits)                  */
/*                                                                           */
/*****************************************************************************/

typedef struct      /* user AC flags */
{
    bit rlogon : 1;         /* L - Restricted to one call a day */
    bit rchat : 1;          /* C - Can't page the SysOp */
    bit rvalidate : 1;      /* V - Posts marked unvalidated */
    bit ramsg : 1;          /* A - Can't change the AutoMessage */
    bit rpostan : 1;        /* * - Can't post anonymously */
    bit rpostpvt : 1;       /* E - Can't post private */
    bit rpostnet : 1;       /* N - Can't post NetMail */
    bit rpost : 1;          /* P - Can't post at all */
    bit rvoting : 1;        /* K - Can't vote */
    bit rmsg : 1;           /* M - Force mail deletion */
    bit rreserved : 1;      /* RESERVED */
    bit onekey : 1;         /* onekey input mode */
    bit avatar : 1;         /* user has AVATAR */
    bit pause : 1;          /* pause */
    bit novice : 1;         /* user is at novice help level */
    bit ansi : 1;           /* user has ANSI */
    bit color : 1;          /* user has color */
    bit alert : 1;          /* alert SysOp when user logs on */
    bit smw : 1;            /* short-message waiting for user */
    bit nomail : 1;         /* user mail-box is closed */
    bit fnodlratio : 1;     /* 1 - No UL/DL ratio */
    bit fnopostratio : 1;   /* 2 - No post/call ratio */
    bit fnofilepts : 1;     /* 3 - No file points checking */
    bit fnodeletion : 1;    /* 4 - Protection from deletion */
}
uflags;

/*---------------------------------------------------------------------------*/

typedef struct      /* *.MIX status flags */
{
    bit deleted : 1;        /* is message deleted? */
    bit unvalidated : 1;    /* is message unvalidated? */
    bit permanent : 1;      /* is message permanent? */
}
msgindexstatr;

typedef struct      /* *.MIX Fido message attributes */
{
    bit private : 1;        /* private */
    bit crash : 1;          /* crash mail */
    bit received : 1;       /* received */
    bit sent : 1;           /* sent */
    bit fattach : 1;        /* file attached */
    bit intransit : 1;      /* in-transit */
    bit orphan : 1;         /* orphaned */
    bit kill : 1;           /* kill after sending */
    bit local : 1;          /* local message */
    bit hold : 1;           /* hold for pickup */
    bit unused : 1;         /* **UNUSED** */
    bit freq : 1;           /* file request */
    bit rreq : 1;           /* return receipt request */
    bit receipt : 1;        /* return receipt message */
    bit areq : 1;           /* audit request */
    bit fureq : 1;          /* file update request */
}
fidoattrib;

typedef struct      /* *.MIX Fido message attributes (extended) */
{
    bit direct : 1;         /* direct to recipient */
    bit immediate : 1;      /* immediate delivery */
    bit delsent : 1;        /* delete when sent */
}
fidoxattrib;

typedef struct      /* *.MIX : Message index records */
{
    longint hdrptr;             /* message header pointer */
    integer isreplyto;          /* reply-to message number (-1=none) */
    cpackdatetime isreplytoid;  /* packed date/time of replied message */
    integer numreplies;         /* number of replies to THIS message */
    cpackdatetime msgdate;      /* message date/time (packed) */
    byte msgdowk;               /* message day of week (0=Sun..6=Sat) */
    fidoattrib attrib;          /* Fido msg attributes (see above) */
    fidoxattrib xattrib;        /* extended Fido attribs (see above) */
    msgindexstatr flags;        /* status flags (see above) */
}
msgindexrec;

typedef struct      /* *.BRD from/to information */
{
    byte anon;          /* anonymous type (0=no, 1/2=yes, 3="Abby", */
                        /*   4="Problemed Person", 5=any name */
    word usernum;       /* user number (0=none) */
    word zone;          /* zone (0=none) */
    word net;           /* net (0=none) */
    word node;          /* node (0=none) */
    word point;         /* point (0=none) */
    string name[37];    /* user name (all caps) if anon=3..5, */
                        /* otherwise null ('')                */
    string real[37];    /* user real name (all caps) */
    string alias[37];   /* user alias (all caps) */
}
fromtoinfo;

typedef struct      /* *.BRD : Message header records */
{
    longint msgptr;         /* pointer to message text */
    word msglength;         /* length of message text */
    word cost;              /* cost to send message (NetMail only) */
    fromtoinfo fromi;       /* from user information */
    fromtoinfo toi;         /* to user information */
    string title[72];       /* title of message */
    string datetime[20];    /* Net/Echo/Group original msg date */
}
mheaderrec;

/*---------------------------------------------------------------------------*/

typedef struct      /* *.DIR status flags */
{
    bit notval : 1;         /* if file is not validated */
    bit isrequest : 1;      /* if file is offline */
    bit resumelater : 1;    /* if file is resume-later */
}
filstat;

typedef struct      /* *.DIR : File records */
{
    string filename[13];    /* filename */
    string description[61]; /* file description */
    integer filepoints;     /* file points */
    integer nacc;           /* number DLs */
    byte res1;              /* RESERVED */
    integer blocks;         /* # 128 byte blks */
    integer owner;          /* ULer of file */
    string stowner[37];     /* ULer's name */
    string date[9];         /* date ULed */
    integer daten;          /* numeric date ULed       */
                            /* (# days since 01/01/85) */
    longint vpointer;       /* pointer to verbose descr (-1=none) */
    filstat filestat;       /* file status */
    byte res2[10];          /* RESERVED */
}
ulfrec;

/*---------------------------------------------------------------------------*/

typedef struct      /* *.MNU status flags */
{
    bit clrscrbefore : 1;   /* C: clear screen before menu display */
    bit dontcenter : 1;     /* D: don't center the menu titles! */
    bit nomenuprompt : 1;   /* N: no menu prompt whatsoever? */
    bit forcepause : 1;     /* P: force a pause before menu display? */
    bit autotime : 1;       /* T: is time displayed automatically? */
}
mnuflags;

typedef struct      /* *.MNU : Menu records */
{
    string menuname[3][101];    /* menu name */
    string directive[13];       /* help file displayed */
    string tutorial[13];        /* tutorial help file */
    string menuprompt[121];     /* menu prompt */
    acstring acs;               /* access requirements */
    string password[16];        /* password required */
    string fallback[9];         /* fallback menu */
    byte forcehelplevel;        /* forced help level for menu */
    byte gencols;               /* generic menus: # of columns */
    byte gcol[3];               /* generic menus: colors */
    mnuflags menuflags;         /* menu status variables */
}
menurec;

typedef struct      /* *.MNU command status flags */
{
    bit hidden : 1;     /* H: is command ALWAYS hidden? */
    bit unhidden : 1;   /* U: is command ALWAYS visible? */
}
cmdflags;

typedef struct      /* *.MNU : Command records */
{
    string ldesc[71];           /* long command description */
    string sdesc[36];           /* short command description */
    string ckeys[15];           /* command-execution keys */
    acstring acs;               /* access requirements */
    string cmdkeys[3];          /* command keys: type of command */
    string mstring[51];         /* MString: command data */
    cmdflags commandflags;      /* command status variables */
}
commandrec;

/*---------------------------------------------------------------------------*/

typedef struct      /* ZLOG.DAT : System log */
{
    string date[9];         /* entry date */
    integer userbaud[5];    /* new user logon BPS rates   */
                            /* (300,1200,2400,4800,9600+) */
    integer active;         /* % activity */
    integer calls;          /* # of calls */
    integer newusers;       /* # of new users */
    integer pubpost;        /* # of public posts */
    integer privpost;       /* # of private posts */
    integer netpost;        /* # of NetMail posts */
    integer criterr;        /* # of critical errors */
    integer uploads;        /* # of uploads */
    integer downloads;      /* # of downloads */
    longint uk;             /* amount ULed in K */
    longint dk;             /* amount DLed in K */
}
zlogrec;

/*---------------------------------------------------------------------------*/

typedef struct      /* BOARDS.DAT status flags */
{
    bit mbrealname : 1;     /* real names are forced */
    bit mbvisible : 1;      /* visible to users w/o access */
    bit mbfilter : 1;       /* filter ANSI/8-bit ASCII */
    bit mbstrip : 1;        /* strip center/title codes */
    bit mbaddtear : 1;      /* add tear/origin lines */
    bit mbtopstar : 1;      /* Top Star base (GroupMail only) */
    bit mbimmune : 1;       /* immune to "twitting" */
    bit mbinternet : 1;     /* special Internet (UUCP) handling */
}
mbflags;

typedef struct      /* BOARDS.DAT : Message base records */
{
    string name[41];        /* message base description */
    string filename[9];     /* BRD/MIX data filename */
    string msgpath[41];     /* messages pathname */
    string password[21];    /* base password */
    acstring acs;           /* access requirement */
    acstring postacs;       /* post access requirement */
    word maxmsgs;           /* max message count */
    byte anonymous;         /* anonymous type */
    mbflags mbstat;         /* message base status vars */
    byte mbtype;            /* message base type */
    byte mbpost;            /* message base posting type */
    string origin[51];      /* origin line */
    byte kludge_color;      /* color of kludge line */
    byte quote_color;       /* color of quoted line */
    byte tear_color;        /* color of tear line */
    byte origin_color;      /* color of origin line */
    word zone;              /* network address */
    word net;               /* (zone:net/node.point) */
    word node;
    word point;
}
boardrec;

/*---------------------------------------------------------------------------*/

typedef struct      /* CONFRENC.DAT : Conference data */
{
    struct
    {
        acstring acs;       /* access requirement */
        string name[41];    /* name of conference */
    }
    conference[27];
}
confrec;

/*---------------------------------------------------------------------------*/

typedef struct      /* EVENTS.DAT : Scheduled event records */
{
    boolean active;             /* whether active */
    string description[31];     /* event description (for logs) */
    string etype;               /* A:CS, C:hat, D:os call, E:xternal */
    string execdata[21];        /* errorlevel if "E", commandline if "D" */
    integer busytime;           /* off-hook time before; 0 if none */
    integer exectime;           /* time of execution */
    boolean busyduring;         /* busy phone DURING event? */
    integer duration;           /* length of time event takes */
    byte execdays;              /* bitwise execution days,    */
                                /* or day of month if monthly */
    boolean monthly;            /* monthly event? */
}
eventrec;

/*---------------------------------------------------------------------------*/

typedef struct      /* GFILES.DAT status flags */
{
    bit tbunhidden : 1;     /* whether *VISIBLE* to users w/o access */
    bit tbreserved1 : 1;    /* RESERVED FLAG #1 */
    bit tbreserved2 : 1;    /* RESERVED FLAG #2 */
}
tbflags;

typedef struct      /* GFILES.DAT : Text-file records */
{
    string title[41];   /* title */
    string filen[13];   /* filename */
    string gdate[9];    /* date of Tfile / Tfile base */
    integer gdaten;     /* numeric date for fast calculation */
    acstring acs;       /* access requirement */
    acstring ulacs;     /* upload to base access requirement */
    tbflags tbstat;     /* text-file base status vars */
    byte res[10];       /* RESERVED */
}
tfilerec;

/*---------------------------------------------------------------------------*/

typedef struct      /* LASTON.DAT : Last few callers records */
{
    longint callernum;      /* system caller number */
    string name[37];        /* user name of caller */
    integer number;         /* user number of caller */
    string citystate[31];   /* city/state of caller */
}
lcallers;

/*---------------------------------------------------------------------------*/

typedef struct      /* MACROS.LST : Keyboard macro records */
{
    string macro[4][241];
}
macrorec;

/*---------------------------------------------------------------------------*/

typedef struct      /* MODEM.DAT status flags */
{
    bit lockedport : 1;     /* COM port locked at constant rate */
    bit xonxoff : 1;        /* XON/XOFF (software) flow control */
    bit ctsrts : 1;         /* CTS/RTS (hardware) flow control */
}
modemflags;

typedef struct      /* MODEM.DAT : Modem configuration */
{
    word waitbaud;              /* wait bps */
    byte comport;               /* COM port number */
    string init[81];            /* initialization string */
    string answer[41];          /* answer string */
    string hangup[41];          /* hangup string */
    string offhook[41];         /* phone off-hook string */
    integer nocallinittime;     /* reinit modem after x mins of inactivity */
    word lockedportbaud;        /* bps rate to use when COM port locked */
    modemflags flags;           /* status flags (see above) */
    byte nocarrier;             /* NO CARRIER result code */
    byte resultcode[2][8];      /* CONNECT result codes:                    */
                                /* 300,1200,2400,4800,7200,9600,12000,14400 */
}
modemrec;

/*---------------------------------------------------------------------------*/

typedef struct      /* NAMES.LST : Sorted names listing */
{
    string name[37];    /* user name */
    integer number;     /* user number */
}
smalrec;

/*---------------------------------------------------------------------------*/

typedef struct      /* NETWORK.DAT status flags */
{
    bit nstrip : 1;         /* strip message codes? */
    bit naddtear : 1;       /* add tear/origin lines to Net/Group? */
    bit nholduser : 1;      /* hold all non-SysOp NetMail? */
    bit nhold : 1;          /* default NetMail to "hold"? */
    bit ncrash : 1;         /* default NetMail to "crash"? */
    bit nkill : 1;          /* default NetMail to "kill/sent"? */
}
netflags;

typedef struct      /* NETWORK.DAT : Network information */
{
    word zone;              /* network address       */
    word net;               /* (zone:net/node.point) */
    word node;
    word point;
    string origin[51];      /* origin line */
    byte kludge_color;      /* color of kludge line */
    byte quote_color;       /* color of quoted text */
    byte tear_color;        /* color of tear line */
    byte origin_color;      /* color of origin line */
    netflags flags;         /* status flags */
    string netpath[41];     /* NetMail path */
    string echopath[41];    /* EchoMail path */
    string grouppath[41];   /* GroupMail path */
    string nodepath[41];    /* nodelist path */
}
networkrec;

/*---------------------------------------------------------------------------*/

typedef struct      /* NODELIST.DAT status flags */
{
    bit hub : 1;        /* node is a net hub */
    bit host : 1;       /* node is a net host */
    bit region : 1;     /* node is region coord */
    bit zone : 1;       /* node is a zone coord */
    bit cm : 1;         /* runs continuous mail */
    bit ores1 : 1;      /* reserved for Opus */
    bit ores2 : 1;      /* reserved for Opus */
    bit ores3 : 1;      /* reserved for Opus */
    bit ores4 : 1;      /* reserved for Opus */
    bit ores5 : 1;      /* reserved for Opus */
    bit nores1 : 1;     /* reserved for non-Opus */
    bit nores2 : 1;     /* reserved for non-Opus */
    bit nores3 : 1;     /* reserved for non-Opus */
    bit nores4 : 1;     /* reserved for non-Opus */
    bit nores5 : 1;     /* reserved for non-Opus */
    bit nores6 : 1;     /* reserved for non-Opus */
}
nodeflags;

typedef struct      /* NODELIST.DAT modem type flags */
{
    bit hst : 1;    /* node uses a USRobotics HST modem */
    bit pep : 1;    /* node uses a Telebit PEP modem */
    bit v32 : 1;    /* node uses a V.32 modem */
    bit v32b : 1;   /* node uses a V.32bis modem */
    bit h96 : 1;    /* node uses a Hayes Express96 modem */
}
modemtypes;

typedef struct      /* NODELIST.DAT : Version 6 nodelist data */
{
    integer net;            /* net number */
    integer node;           /* node number */
    integer cost;           /* cost per minute to call */
    string name[34];        /* node name */
    string phone[40];       /* phone number */
    string city[30];        /* city and state */
    string password[8];     /* password */
    integer realcost;       /* phone company's charge */
    integer hubnode;        /* node # of this node's hub (0=none) */
    byte rate;              /* actual bps rate divided by 300 */
    modemtypes modem;       /* modem type codes */
    nodeflags flags;        /* set of flags */
    byte res[2];            /* RESERVED */
}
nodedatarec;

typedef struct      /* NODELIST.IDX : Version 6 nodelist index */
{
    integer node;       /* node number */
    integer net;        /* net number */
}
nodeindexrec;

/*---------------------------------------------------------------------------*/

typedef struct      /* PROTOCOL.DAT status flags */
{
    bit xbactive : 1;           /* is protocol active? */
    bit xbisbatch : 1;          /* is batch definition? */
    bit xbisresume : 1;         /* is resume definition? */
    bit xbxferokcode : 1;       /* does result code = xfer OK? */
}
xbflags;

typedef struct      /* PROTOCOL.DAT : File transfer protocols */
{
    xbflags xbstat;         /* protocol flags */
    string ckeys[15];       /* command keys */
    string descr[41];       /* description */
    acstring acs;           /* access string */
    string templog[26];     /* temp. log file */
    string uloadlog[26];    /* permanent log files */
    string dloadlog[26];
    string ulcmd[79];       /* UL/DL command lines */
    string dlcmd[79];
    string ulcode[6][7];    /* UL/DL codes */
    string dlcode[6][7];
    string envcmd[61];      /* environment setup cmd (B) */
    string dlflist[26];     /* DL file lists (B) */
    integer maxchrs;        /* max chrs in cmdline */
    integer logpf;          /* pos in log file for data (B) */
    integer logps;
    byte res[15];           /* RESERVED */
}
protrec;

/*---------------------------------------------------------------------------*/

typedef struct      /* SHORTMSG.DAT : One-line messages */
{
    string msg[161];    /* message */
    integer destin;     /* destination user # (-1=deleted) */
}
smr;

/*---------------------------------------------------------------------------*/

typedef struct      /* STATUS.DAT archive configuration records */
{
    boolean active;         /* whether this archive format is active */
    string ext[4];          /* 3-char file extension */
    string listline[26];    /* view archive cmdline ('' for internal) */
    string arcline[26];     /* compression cmdline */
    string unarcline[26];   /* de-compression cmdline */
    string testline[26];    /* integrity test cmdline ('' for *None*) */
    string cmtline[26];     /* comment cmdline ('' for *None*) */
    integer succlevel;      /* success ERRORLEVEL (-1=ignore results) */
}
filearcinforec;

typedef struct      /* STATUS.DAT : System configuration */
{
    string gfilepath[80];       /* GFILES path (data files) */
    string afilepath[80];       /* AFILES path (text files) */
    string menupath[80];        /* MENUS path (system menu files) */
    string trappath[80];        /* LOGS path (trap/chat/log files) */
    string msgpath[80];         /* MSGS path (msg base data files) */
    string tfilepath[80];       /* TFILES path (Tfiles base files) */
    string temppath[80];        /* TEMP path (temporary files) */

    string bbsname[81];         /* BBS name */
    string bbsphone[13];        /* BBS phone number */
    string sysopname[31];       /* SysOp's full name or alias */
    integer maxusers;           /* max number of users system can have */
    integer lowtime;            /* SysOp begin minute (in minutes) */
    integer hitime;             /* SysOp end time */
    integer dllowtime;          /* normal downloading hours */
    integer dlhitime;
    boolean shuttlelog;         /* is Shuttle Logon active? */
    boolean lock300;            /* lock out 300 baud? */
    string sysoppw[21];         /* SysOp PW */
    string newuserpw[21];       /* newuser PW (or NULL if none) */
    string shuttlepw[21];       /* Shuttle PW (if Shuttle active) */
    integer b300lowtime;        /* 300 baud calling hours */
    integer b300hitime;
    integer b300dllowtime;      /* 300 baud DL hours */
    integer b300dlhitime;
    boolean closedsystem;       /* DON'T allow new users? */
    boolean swapshell;          /* is swap shell function enabled? */
    integer eventwarningtime;   /* time before event warning */
    string tfiledate[9];        /* last date text-files were inserted */
    byte swapshelltype;         /* swap shell type (0=disk,1=EMS) */

    byte res1[23];              /* RESERVED SPACE #1 */

    acstring sop;               /* SysOp */
    acstring csop;              /* Co-SysOp */
    acstring msop;              /* Message SysOp */
    acstring fsop;              /* File SysOp */
    acstring spw;               /* SysOp PW at logon */
    acstring seepw;             /* see SysOp PWs remotely  */
    acstring normpubpost;       /* post public messages */
    acstring normprivpost;      /* post private messages */
    acstring anonpubread;       /* see who posted public anon */
    acstring anonprivread;      /* see who posted private anon */
    acstring anonpubpost;       /* post anon public messages */
    acstring anonprivpost;      /* post anon private messages */
    acstring seeunval;          /* see unvalidated files */
    acstring dlunval;           /* DL unvalidated files */
    acstring nodlratio;         /* no UL/DL ratio */
    acstring nopostratio;       /* no post/call ratio */
    acstring nofilepts;         /* no file points checking */
    acstring ulvalreq;          /* uploads require validation by SysOp */
    acstring normnetpost;       /* post NetMail messages */

    string fsecmd[41];          /* full screen editor command */

    byte res2[38];              /* RESERVED SPACE #2 */

    byte maxprivpost;           /* max private posts per call */
    byte maxnetpost;            /* max NetMail posts per call */
    byte maxpubpost;            /* max public posts per call */
    byte maxchat;               /* max chat-pages per call */

    byte res3[4];               /* RESERVED SPACE #3 */

    byte maxlogontries;         /* tries allowed for PW's at logon */
    byte bsdelay;               /* backspacing delay */
    byte sysopcolor;            /* SysOp color in chat mode */
    byte usercolor;             /* user color in chat mode */
    integer minspaceforpost;    /* minimum K drive space left to post */
    integer minspaceforupload;  /* minimum K drive space left to upload */
    byte backsysoplogs;         /* days to keep SYSOP##.LOG */
    byte wfcblanktime;          /* minutes after which to blank WFC menu */
    byte linelen;               /* default video line length */
    byte pagelen;               /* default video page length */

    byte res4[18];              /* RESERVED SPACE #4 */

    boolean specialfx;          /* WFC menu "special effects" */
    boolean logonlogoff;        /* enable LOGON.BAT & LOGOFF.BAT? */
    boolean allowalias;         /* allow alias's? (handles) */
    boolean phonepw;            /* use phone number password in logon? */
    boolean localsec;           /* is local security ON? */
    boolean localscreensec;     /* is local screen-security ON? */
    boolean globaltrap;         /* trap ALL USER'S activity? */
    boolean autochatopen;       /* does chat buffer auto-open? */
    boolean autominlogon;       /* Auto-Message in logon? */
    boolean bullinlogon;        /* bulletins in logon? */
    boolean lcallinlogon;       /* "Last Few Callers" list in logon? */
    boolean yourinfoinlogon;    /* "Your Info" in logon? */
    boolean multitask;          /* is BBS multitasking? */
    boolean offhooklocallogon;  /* take phone off-hook for local logons? */
    boolean forcevoting;        /* is manditory logon voting active? */
    boolean compressbases;      /* "compress" file/message base numbers? */
    boolean searchdup;          /* search for dup. filenames when UL? */
    byte slogtype;              /* SysOp log type: File/Printer/Both */
    boolean stripclog;          /* strip colors from SysOp log output? */
    integer newapp;             /* user# to send new user application to */
    integer guestuser;          /* user# of guest user account */
    integer timeoutbell;        /* minutes before time-out bell */
    integer timeout;            /* minutes before timeout (logoff) */
    boolean usewfclogo;         /* whether to use WFC menu logo */
    boolean useems;             /* attempt to use EMS for overlay */
    boolean usebios;            /* use ROM BIOS for local video output */
    boolean cgasnow;            /* suppress snow on CGA systems */

    byte res5[16];              /* RESERVED SPACE #5 */

    filearcinforec
        filearcinfo[MAXARCS];       /* archive specs */
    string filearccomment[3][81];   /* BBS comments for archives */

    boolean uldlratio;          /* are UL/DL ratios active? */
    boolean fileptratio;        /* is auto file-pt compensation active? */
    byte fileptcomp;            /* file point compensation ratio */
    byte fileptcompbasesize;    /* file point "base compensation size" */
    byte ulrefund;              /* percent time refund for ULs */
    byte tosysopdir;            /* "To SysOp" file base */
    boolean validateallfiles;   /* validate ALL FILES automatically? */
    string remdevice[11];       /* remote output device (GATEx,COMx,etc) */
    integer maxintemp;          /* max K allowed in TEMP\3\ */
    integer minresume;          /* min K to allow resume-later */
    byte maxdbatch;             /* max files in DL batch queue */
    byte maxubatch;             /* max files in UL batch queue */
    byte unlistfp;              /* # of file points for unlisted DLs */

    byte res6[21];              /* RESERVED SPACE #6 */

    longint newcredit;          /* newuser NetMail credit */
    longint autocredit;         /* auto-validation NetMail credit */
    byte newsl;                 /* newuser SL */
    byte newdsl;                /* newuser DSL */
    acrq newar;                 /* newuser AR */
    uflags newac;               /* newuser AC */
    integer newfp;              /* newuser file points */
    byte autosl;                /* auto-validation SL */
    byte autodsl;               /* auto-validation DSL */
    acrq autoar;                /* auto-validation AR */
    uflags autoac;              /* auto-validation AC */

    string allstartmenu[9];     /* logon menu to start ALL users on */

    byte res7[22];              /* RESERVED SPACE #7 */

    string batchfileext[4];     /* batch filename extension */
    string bulletprefix[9];     /* default bulletins filename prefix */

    byte res8[15];              /* RESERVED SPACE #8 */

    secrange timeallow;         /* time allowance */
    secrange callallow;         /* call allowance */
    secrange dlratio;           /* # ULs/# DLs ratios */
    secrange dlkratio;          /* DLk/ULk ratios */
    secrange postratio;         /* post/call ratios */

    string lastdate[9];         /* last system date */
    byte curwindow;             /* type of SysOp window currently in use */
    boolean istopwindow;        /* is SysOp window on top of screen? */
    longint callernum;          /* total number of callers */
    integer numusers;           /* number of users */
    zlogrec todayzlog;          /* TODAY's ZLOG record */
    integer postcredits;        /* file points/upload credit */
                                /* compensation for posts    */
    boolean rebootforevent;     /* reboot before events? */
    boolean watchdogdoor;       /* WatchDog active in doors? */
    boolean windowon;           /* is SysOp window on? */
    string swappath[80];        /* swap shell path */
    string arcpath[41];         /* archive utility path */
    string protpath[41];        /* protocol driver path */

    byte res9[35];              /* RESERVED SPACE #9 */

    word versionid;             /* version (eg 0x0270 = v2.7) */
}
systatrec;

/*---------------------------------------------------------------------------*/

typedef struct      /* STRING.DAT : User-definable strings */
{
    string ansiq[81];           /* "Do you want ANSI? " string */
    string note[2][81];         /* Logon notes (L #1-2) */
    string lprompt[81];         /* Logon prompt (L #3) */
    string echoc;               /* Echo char for PWs */
    string sysopin[81];         /* "SysOp In" (inside SysOp hours)*/
    string sysopout[81];        /* "SysOp Out" */
    string engage[81];          /* "Engage Chat" */
    string endchat[81];         /* "End Chat" */
    string wait[81];            /* "SysOp Working" */
    string pause[81];           /* "Pause" */
    string entermsg1[81];       /* "Enter Message" line #1 */
    string entermsg2[81];       /* "Enter Message" line #2 */
    string newscan1[81];        /* "NewScan begin" */
    string newscan2[81];        /* "NewScan done" */
    string scanmessage[81];     /* Message scanning prompt */
    string automsgt[81];        /* Auto-Message title */
    string autom;               /* Auto-Message border characters */

    string shelldos1[81];       /* " >> SysOp shelling to DOS ..." */
    string shelldos2[81];       /* " ... thank you for waiting." */
    string chatcall1[81];       /* "Paging SysOp, please wait..." */
    string chatcall2[81];       /* ">>><*><<<" */
    string guestline[81];       /* "To be a guest ... " */
    string namenotfound[81];    /* "Name NOT found in user list." */
    string bulletinline[81];    /* Bulletin line */
    string thanxvote[81];       /* "Thanks for voting" */

    string listline[81];        /* "List files - P to pause" */
    string newline[81];         /* "Search for new files -" */
    string searchline[81];      /* "Search all dirs for file mask" */
    string findline1[81];       /* "Search for descriptions... " */
    string findline2[81];       /* "Enter the string to search for.." */
    string downloadline[81];    /* "Download - " */
    string uploadline[81];      /* "Upload - " */
    string viewline[81];        /* "View interior files - " */
    string nofilepts[81];       /* "Insufficient file points." */
    string unbalance[81];       /* "Your UL/DL ratio is unbalanced" */

    string pninfo[81];          /* "P to Pause, N for next dir" */
    string gfnline1[81];        /* "[Enter]=All files" */
    string gfnline2[81];        /* "File mask: " */
    string batchadd[81];        /* "File added to batch queue." */

    string quotestr1[51];       /* message quote header #1 */
    string quotestr2[51];       /* message quote header #2 */

    string mtaskline[81];       /* "System Under Multi-Tasking..." */
}
fstringrec;

/*---------------------------------------------------------------------------*/

typedef struct      /* UPLOADS.DAT status flags */
{
    bit fbnoratio : 1;          /* if <No Ratio> active */
    bit fbunhidden : 1;         /* whether *VISIBLE* to users w/o access */
    bit fbdirdlpath : 1;        /* if *.DIR file stored in DLPATH */
    bit fbreserved1 : 1;        /* RESERVED FLAG #1 */
    bit fbusegifspecs : 1;      /* whether to use GifSpecs */
    bit fbreserved2 : 1;        /* RESERVED FLAG #2 */
}
fbflags;

typedef struct      /* UPLOADS.DAT : File base records */
{
    string name[41];        /* area description */
    string filename[13];    /* filename + ".DIR" */
    string dlpath[41];      /* download path */
    string ulpath[41];      /* upload path */
    integer maxfiles;       /* max files allowed (1-2000) */
    string password[21];    /* password required */
    byte arctype;           /* wanted archive type (1..max,0=inactive) */
    byte cmttype;           /* wanted comment type (1..3,0=inactive) */
    byte res1[2];           /* RESERVED SPACE #1 */
    fbflags fbstat;         /* file base status vars */
    acstring acs;           /* access requirements */
    acstring ulacs;         /* upload requirements */
    acstring nameacs;       /* see-names requirements */
    byte res2[10];          /* RESERVED SPACE #2 */
}
ulrec;

/*---------------------------------------------------------------------------*/

typedef struct      /* USER.LST status flags */
{
    bit lockedout : 1;          /* if locked out */
    bit udeleted : 1;           /* if deleted */
    bit trapactivity : 1;       /* if trapping users activity */
    bit trapseperate : 1;       /* if trap to seperate TRAP file */
    bit chatauto : 1;           /* if auto chat trapping */
    bit chatseperate : 1;       /* if seperate chat file to trap to */
    bit slogseperate : 1;       /* if seperate SysOp log */
    bit clsmsg : 1;             /* if clear-screen before message */
    bit avadjust : 1;           /* if AVATAR color adjust */
}
userflags;

typedef struct      /* USER.LST : User account records */
{

    /* configuration information */

    string name[37];            /* user name */
    string realname[37];        /* real name */
    string street[31];          /* mailing address */
    string citystate[31];       /* city, state */
    string zipcode[11];         /* zipcode */
    string ph[13];              /* user phone # */
    string bday[9];             /* user birthdate */
    string sex;                 /* user sex */

    string pw[21];              /* user password */
    string firston[9];          /* firston date */
    string laston[9];           /* laston date */

    string computer[31];        /* type of computer */
    string occupation[41];      /* occupation */
    string wherebbs[41];        /* BBS reference */
    string note[40];            /* SysOp note */

    string lockedfile[9];       /* lockout msg to print */
    clrs cols;                  /* user colors */
    byte linelen;               /* line length (# cols) */
    byte pagelen;               /* page length (# rows) */
    longint mpointer;           /* pointer to entry in MACRO.LST */
    string userstartmenu[9];    /* menu to start user out on */

    userflags flags;            /* flags (see above) */

    /* system access information */

    byte sl;                    /* SL */
    byte dsl;                   /* DSL */
    uflags ac;                  /* user flags */
    acrq ar;                    /* AR flags */

    /* message system information */

    byte lastmsg;               /* last message base # */
    integer pubpost;            /* # public posts */
    integer privpost;           /* # private posts */
    integer netpost;            /* # NetMail posts */
    integer forusr;             /* forward mail to user # */
    longint credit;             /* NetMail points (credit) */
    longint debit;              /* NetMail points (debit) */
    mhireadr mhiread;           /* NewScan high message pointers */
    mzscanr mzscan;             /* NewScan message bases */
    byte boardsysop[5];         /* msg board SysOp */

    /* file system information */

    byte lastfil;               /* last file base # */
    integer uploads;            /* # of ULs */
    integer downloads;          /* # of DLs */
    longint uk;                 /* amount ULed (kilobytes) */
    longint dk;                 /* amount DLed (kilobytes) */
    integer filepoints;         /* # of file points */
    byte flistopt;              /* type of file list type to use */
    fzscanr fzscan;             /* NewScan file bases */

    /* system usage information */

    longint ttimeon;            /* total mins spent on */
    integer loggedon;           /* # times logged on */
    integer tltoday;            /* # min left today */
    byte ontoday;               /* # times on today */
    byte illegal;               /* # illegal logon attempts */

    /* miscellaneous information */

    byte vote[20];              /* voting data */
    integer timebank;           /* # mins in Time Bank */
    integer timebankadd;        /* time added to timebank TODAY */
}
userrec;

/*---------------------------------------------------------------------------*/

typedef struct      /* VERBOSE.DAT : Verbose descriptions */
{
    string descr[4][51];
}
verbrec;

/*---------------------------------------------------------------------------*/

typedef struct      /* VOTING.DAT : Voting records */
{
    string question[80];    /* voting question */
    integer numa;           /* number of user's who have answered */
    struct
    {
        string ans[41];     /* answer description */
        integer numres;     /* # user's who picked this answer */
    }
    answ[NUMVOTEAS];
}
vdatar;

#endif

