/*
	SNEWS 1.91

	pccharst - routines to handle multiple character sets for IBM PCs


	Copyright (C) 1993  Daniel Fandrich
						<dan@fch.wimsey.bc.ca> or CompuServe 72365,306

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License, version 1, as
    published by the Free Software Foundation.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    See the file COPYING, which contains a copy of the GNU General
    Public License.


	Source is formatted with a tab size of 4.

 */


#include "defs.h"
#include "pccharst.h"
#include <dos.h>
#include <ctype.h>
#include <string.h>


#define DEFAULT_CP 437	/* mapping to use if active CP isn't supported */

#define NOCH '?'		/* character to show for those invalid in current code page */


enum code_pages code_page_table;

/* these names must match enum code_pages */
struct {int code_page_num; enum code_pages cptable;} code_page_list[] = {
	437, CP437,		/* US */
	850, CP850,		/* Multilingual/Latin I */

#if 0	/* these aren't in the table yet */
	852, CP852,		/* Slavic/Latin II */
	857, CP857,		/* Turkish/Latin V */
	860, CP860,		/* Portugal */
	861, CP861,		/* Iceland */
	862, CP862,		/* Hebrew */
	863, CP863,		/* Canadian - French */
	865, CP865,		/* Norway/Denmark */
	866, CP866,		/* Cyrillic */
	869, CP869,		/* Greek */
#endif
	0, CP_NOT_SUPP	/* end of list */
};

/* these names must match enum char_sets */
char *char_set_names[/*enum char_sets*/] = {
	"X-IBM437",		/* IBM code page 437 	code page 437 is best */
	"ISO-8859-1",	/* Latin 1				code page 850 is best */
	"ISO-8859-2",   /* Latin 2				code page 852 is best */
	"ISO-8859-3",	/* Latin 3				code page 850 is best? */
	"ISO-8859-4",	/* Latin 4				code page 850 is best? */
	"ISO-8859-9",	/* Latin 5				code page 857 is best */
#if 0	/* these aren't in the mapping table yet */
	"ISO-8859-5",	/* Latin/Cyrillic		code page 866 is best */
	"ISO-8859-6",	/* Latin/Arabic */
	"ISO-8859-7",	/* Latin/Greek          code page 869 is best */
	"ISO-8859-8",	/* Latin/Hebrew */
	"ISO-8859-10",	/* Latin 6				code page 852 is best? */
#endif
	"US-ASCII",		/* (must be last in table) equivalent to ISO-8859-1 */
	NULL
};


/* cptable[code_pages][char_sets][character code] */
char cptable[2 /*8*/][6/*12*/][0x80] = {

{ /* code page 437 mappings */

/* Map from code page 437 to code page 437 */
{
0x80,0x81,0x82,0x83,0x84,0x85,0x86,0x87,
0x88,0x89,0x8A,0x8B,0x8C,0x8D,0x8E,0x8F,
0x90,0x91,0x92,0x93,0x94,0x95,0x96,0x97,
0x98,0x99,0x9A,0x9B,0x9C,0x9D,0x9E,0x9F,
0xA0,0xA1,0xA2,0xA3,0xA4,0xA5,0xA6,0xA7,
0xA8,0xA9,0xAA,0xAB,0xAC,0xAD,0xAE,0xAF,
0xB0,0xB1,0xB2,0xB3,0xB4,0xB5,0xB6,0xB7,
0xB8,0xB9,0xBA,0xBB,0xBC,0xBD,0xBE,0xBF,
0xC0,0xC1,0xC2,0xC3,0xC4,0xC5,0xC6,0xC7,
0xC8,0xC9,0xCA,0xCB,0xCC,0xCD,0xCE,0xCF,
0xD0,0xD1,0xD2,0xD3,0xD4,0xD5,0xD6,0xD7,
0xD8,0xD9,0xDA,0xDB,0xDC,0xDD,0xDE,0xDF,
0xE0,0xE1,0xE2,0xE3,0xE4,0xE5,0xE6,0xE7,
0xE8,0xE9,0xEA,0xEB,0xEC,0xED,0xEE,0xEF,
0xF0,0xF1,0xF2,0xF3,0xF4,0xF5,0xF6,0xF7,
0xF8,0xF9,0xFA,0xFB,0xFC,0xFD,0xFE,0xFF
},

/* Map from ISO 8859-1 to code page 437 */
{
/* 128 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 136 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 144 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 152 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 160 */	0xFF,0xAD,0x9B,0x9C,0x0F,0x9D,0x7C,0x15,
/* 168 */	'"', '@', 0xA6,0xAE,0xAA,0xC4,NOCH,'-',
/* 176 */	0xF8,0xF1,0xFD,'3', 0x27,0xE6,0x14,0xFA,
/* 184 */	',', '1', 0xA7,0xAF,0xAC,0xAB,NOCH,0xA8,
/* 192 */	'A', 'A', 'A', 'A', 0x8E,0x8F,0x92,0x80,
/* 200 */   'E', 0x90,'E', 'E', 'I', 'I', 'I', 'I',
/* 208 */	'D', 0xA5,'O', 'O', 'O', 'O', 0x99,'X',
/* 216 */	0xE9,'U', 'U', 'U', 0x9A,'Y', NOCH,0xE1,
/* 224 */	0x85,0xA0,0x83,'a', 0x84,0x86,0x91,0x87,
/* 232 */	0x8A,0x82,0x88,0x89,0x8D,0xA1,0x8C,0x8B,
/* 240 */	NOCH,0xA4,0x95,0xA2,0x93,'o', 0x94,0xF6,
/* 248 */	0xED,0x97,0xA3,0x96,0x81,'y', NOCH,0x98
},

/* Map from ISO 8859-2 to code page 437 */
{
/* 128 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 136 */   NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 144 */   NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 152 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 160 */	0xFF,'A', NOCH,'L', 0x0F,'L', 'S', 0x15,
/* 168 */	'"', 'S', 'S', 'T', 'Z', 0xC4,'Z', 'Z',
/* 176 */	0xF8,'a', NOCH,'l', 0x27,'l', 's', NOCH,
/* 184 */	',', 's', 's', 't', 'z', '"', 'z', 'z',
/* 192 */	'R', 'A', 'A', 'A', 0x8E,'L', 'C', 0x80,
/* 200 */   'C', 0x90,'E', 'E', 'E', 'I', 'I', 'D',
/* 208 */	'D', 'N', 'N', 'O', 'O', 'O', 0x99,'X',
/* 216 */	'R', 'U', 'U', 'U', 0x9A,'Y', 'T', 0xE1,
/* 224 */	'r', 0xA0,0x83,'a', 0x84,'l', 'c', 0x87,
/* 232 */	'c', 0x82,'e', 0x89,'e', 0xA1,0x8C,'d',
/* 240 */	0xEB,'n', 'n', 0xA2,0x93,'o', 0x94,0xF6,
/* 248 */	'r', 'u', 0xA3,'u', 0x81,'y', 't', 0xF9
},

/* Map from ISO 8859-3 to code page 437 */
{
/* 128 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 136 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 144 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 152 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 160 */	0xFF,'H', NOCH,0x9C,0x0F,NOCH,'H', 0x15,
/* 168 */	'"', 'I', 'S', 'G', 'J', 0xC4,NOCH,'Z',
/* 176 */	0xF8,'h', 0xFD,'3', 0x27,0xE6,'h', 0xFA,
/* 184 */	',', 'i', 'd', 'g', 'j', 0xAB,NOCH,'z',
/* 192 */	'A', 'A', 'A', NOCH,0x8E,'C', 'C', 0x80,
/* 200 */   'E', 0x90,'E', 'E', 'I', 'I', 'I', 'I',
/* 208 */	NOCH,0xA5,'O', 'O', 'O', 'G', 0x99,'X',
/* 216 */	'G', 'U', 'U', 'U', 0x9A,'U', 'S', 0xE1,
/* 224 */	0x85,0xA0,0x83,NOCH,0x84,'c', 'c', 0x87,
/* 232 */	0x8A,0x82,0x88,0x89,0x8D,0xA1,0x8C,0x8B,
/* 240 */	NOCH,0xA4,0x95,0xA2,0x93,'g', 0x94,0xF6,
/* 248 */	'g', 0x97,0xA3,0x96,0x81,'u', 's', 0xF9
},

/* WHAT IS GREENLANDIC K (162) AND LAPPISH ENG (189,191)? */
/* Map from ISO 8859-4 to code page 437 */
{
/* 128 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 136 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 144 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 152 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 160 */	0xFF,'A', 'k', 'R', 0x0F,'I', 'L', 0x15,
/* 168 */	'"', 'S', 'E', 'G', 'T', 0xC4,'Z', '-',
/* 176 */	0xF8,'a', NOCH,'r', 0x27,'i', 'l', NOCH,
/* 184 */	',', 's', 'e', 'g', 't', NOCH,'z', NOCH,
/* 192 */	'A', 'A', 'A', 'A', 0x8E,0x8F,0x92,'I',
/* 200 */   'C', 0x90,'E', 'E', 'E', 'I', 'I', 'I',
/* 208 */	'D', 'N', 'O', 'K', 'O', 'O', 0x99,'X',
/* 216 */	0xE9,'U', 'U', 'U', 0x9A,'U', 'U', 0xE1,
/* 224 */	'a', 0xA0,0x83,'a', 0x84,0x86,0x91,'i',
/* 232 */	'c', 0x82,'e', 0x89,'e', 0xA1,0x8C,'i',
/* 240 */	0xEB,'n', 'o', 'k', 0x93,'o', 0x94,0xF6,
/* 248 */	0xED,'u', 0xA3,0x96,0x81,'u', 'u', 0xF9
},

/* Map from ISO 8859-9 to code page 437 */
{
/* 128 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 136 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 144 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 152 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 160 */	0xFF,0xAD,0x9B,0x9C,0x0F,0x9D,0x7C,0x15,
/* 168 */	'"', '@', 0xA6,0xAE,0xAA,0xC4,NOCH,'-',
/* 176 */	0xF8,0xF1,0xFD,'3', 0x27,0xE6,0x14,0xFA,
/* 184 */	',', '1', 0xA7,0xAF,0xAC,0xAB,NOCH,0xA8,
/* 192 */	'A', 'A', 'A', 'A', 0x8E,0x8F,0x92,0x80,
/* 200 */   'E', 0x90,'E', 'E', 'I', 'I', 'I', 'I',
/* 208 */	'G', 0xA5,'O', 'O', 'O', 'O', 0x99,'X',
/* 216 */	0xE9,'U', 'U', 'U', 0x9A,'I', 'S', 0xE1,
/* 224 */	0x85,0xA0,0x83,'a', 0x84,0x86,0x91,0x87,
/* 232 */	0x8A,0x82,0x88,0x89,0x8D,0xA1,0x8C,0x8B,
/* 240 */	'g', 0xA4,0x95,0xA2,0x93,'o', 0x94,0xF6,
/* 248 */	0xED,0x97,0xA3,0x96,0x81,'i', 's', 0x98
}
},


{ /* code page 850 mappings */

/* Map from code page 437 to code page 850 */
{
0x80,0x81,0x82,0x83,0x84,0x85,0x86,0x87,
0x88,0x89,0x8A,0x8B,0x8C,0x8D,0x8E,0x8F,
0x90,0x91,0x92,0x93,0x94,0x95,0x96,0x97,
0x98,0x99,0x9A,0xBD,0x9C,0xBE,NOCH,0x9F,
0xA0,0xA1,0xA2,0xA3,0xA4,0xA5,0xA6,0xA7,
0xA8,0xDA,0xAA,0xAB,0xAC,0xAD,0xAE,0xAF,
0xB0,0xB1,0xB2,0xB3,0xB4,0xB4,0xB9,0xBB,
0xBF,0xB9,0xBA,0xBB,0xBC,0xD9,0xBC,0xBF,
0xC0,0xC1,0xC2,0xC3,0xC4,0xC5,0xC3,0xCC,
0xC8,0xC9,0xCA,0xCB,0xCC,0xCD,0xCE,0xCA,
0xC1,0xCB,0xC2,0xC8,0xC0,0xDA,0xC9,0xC3,
0xC5,0xD9,0xDA,0xDB,0xDC,0xDB,0xDB,0xDF,
NOCH,0xE1,NOCH,NOCH,NOCH,NOCH,0xE6,NOCH,
NOCH,NOCH,NOCH,NOCH,0x1D,NOCH,NOCH,NOCH,
NOCH,0xF1,0x10,0x11,'?', 0xA8,0xF6,0x7E,
0xF8,0xFA,0xFA,0x1F,'n' ,0xFD,0xFE,0xFF
},

/* Map from ISO 8859-1 to code page 850 */
{
/* 128 */   NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 136 */   NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 144 */   NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 152 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 160 */	0xFF,0xAD,0xBD,0x9C,0xCF,0xBE,0xDD,0xF5,
/* 168 */	0xF9,0xB8,0xA6,0xAE,0xAA,0xC4,0xA9,0xEE,
/* 176 */	0xF8,0xF1,0xFD,0xFC,0xEF,0xE6,0xF4,0xFA,
/* 184 */	0xF7,0xFB,0xA7,0xAF,0xAC,0xAB,0xF3,0xA8,
/* 192 */	0xB7,0xB5,0xB6,0xC7,0x8E,0x8F,0x92,0x80,
/* 200 */   0xD4,0x90,0xD2,0xD3,0xDE,0xD6,0xD7,0xD8,
/* 208 */	0xD1,0xA5,0xE3,0xE0,0xE2,0xE5,0x99,0x9E,
/* 216 */	0x9D,0xEB,0xE9,0xEA,0x9A,0xED,0xE8,0xE1,
/* 224 */	0x85,0xA0,0x83,0xC6,0x84,0x86,0x91,0x87,
/* 232 */	0x8A,0x82,0x88,0x89,0x8D,0xA1,0x8C,0x8B,
/* 240 */	0xD0,0xA4,0x95,0xA2,0x93,0xE4,0x94,0xF6,
/* 248 */	0x9B,0x97,0xA3,0x96,0x81,0xEC,0xE7,0x98
},

/* IS CARON (183) 0xF7? */
/* Map from ISO 8859-2 to code page 850 */
{
/* 128 */   NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 136 */   NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 144 */   NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 152 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 160 */	0xFF,'A', NOCH,'L', 0xCF,'L', 'S', 0xF5,
/* 168 */	0xF9,'S', 'S', 'T', 'Z', 0xC4,'Z', 'Z',
/* 176 */	0xF8,'a', NOCH,'l', 0xEF,'l', 's', NOCH,
/* 184 */	0xF7,'s', 's', 't', 'z', '"', 'z', 'z',
/* 192 */	'R', 0xB5,0xB6,'A', 0x8E,'L', 'C', 0x80,
/* 200 */   'C', 0x90,'E', 0xD3,'E', 0xD6,0xD7,'D',
/* 208 */	0xD1,'N', 'N', 0xE0,0xE2,'O', 0x99,0x9E,
/* 216 */	'R', 'U', 0xE9,'U', 0x9A,0xED,'T', 0xE1,
/* 224 */	'r', 0xA0,0x83,'a', 0x84,'l', 'c', 0x87,
/* 232 */	'c', 0x82,'e', 0x89,'e', 0xA1,0x8C,'d',
/* 240 */	0xD0,'n', 'n', 0xA2,0x93,'o', 0x94,0xF6,
/* 248 */	'r', 'u', 0xA3,'u', 0x81,0xEC,'t', 0xFA
},

/* Map from ISO 8859-3 to code page 850 */
{
/* 128 */   NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 136 */   NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 144 */   NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 152 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 160 */	0xFF,'H', NOCH,0x9C,0xCF,NOCH,'H', 0xF5,
/* 168 */	0xF9,'I', 'S', 'G', 'J', 0xC4,NOCH,'Z',
/* 176 */	0xF8,'h', 0xFD,0xFC,0xEF,0xE6,'h', 0xFA,
/* 184 */	0xF7,'i', 's', 'g', 'j', 0xAB,NOCH,'z',
/* 192 */	0xB7,0xB5,0xB6,NOCH,0x8E,'C', 'C', 0x80,
/* 200 */   0xD4,0x90,0xD2,0xD3,0xDE,0xD6,0xD7,0xD8,
/* 208 */	NOCH,0xA5,0xE3,0xE0,0xE2,'G', 0x99,0x9E,
/* 216 */	'G', 0xEB,0xE9,0xEA,0x9A,'U', 'S', 0xE1,
/* 224 */	0x85,0xA0,0x83,NOCH,0x84,'c', 'c', 0x87,
/* 232 */	0x8A,0x82,0x88,0x89,0x8D,0xA1,0x8C,0x8B,
/* 240 */	NOCH,0xA4,0x95,0xA2,0x93,'g', 0x94,0xF6,
/* 248 */	'g', 0x97,0xA3,0x96,0x81,'u', 's', 0xFA
},

/* Map from ISO 8859-4 to code page 850 */
{
/* 128 */   NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 136 */   NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 144 */   NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 152 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 160 */	0xFF,'A', 'k', 'R', 0xCF,'I', 'L', 0xF5,
/* 168 */	0xF9,'S', 'E', 'G', 'T', 0xC4,'Z', 0xEE,
/* 176 */	0xF8,'a', NOCH,'r', 0xEF,'i', 'l', NOCH,
/* 184 */	0xF7,'s', 'e', 'g', 't', NOCH,'z', NOCH,
/* 192 */	'A', 0xB5,0xB6,0xC7,0x8E,0x8F,0x92,'I',
/* 200 */   'C', 0x90,'E', 0xD3,'E', 0xD6,0xD7,'I',
/* 208 */	0xD1,'N', 'O', 'K', 0xE2,0xE5,0x99,0x9E,
/* 216 */	0x9D,'U', 0xE9,0xEA,0x9A,'U', 'U', 0xE1,
/* 224 */	'a', 0xA0,0x83,0xC6,0x84,0x86,0x91,'i',
/* 232 */	'c', 0x82,'e', 0x89,'e', 0xA1,0x8C,'i',
/* 240 */	0xD0,'n', 'o', 'k', 0x93,0xE4,0x94,0xF6,
/* 248 */	0x9B,'u', 0xA3,0x96,0x81,'u', 'u', 0xFA
},

/* Map from ISO 8859-9 to code page 850 */
{
/* 128 */   NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 136 */   NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 144 */   NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 152 */	NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,NOCH,
/* 160 */	0xFF,0xAD,0xBD,0x9C,0xCF,0xBE,0xDD,0xF5,
/* 168 */	0xF9,0xB8,0xA6,0xAE,0xAA,0xC4,0xA9,0xEE,
/* 176 */	0xF8,0xF1,0xFD,0xFC,0xEF,0xE6,0xF4,0xFA,
/* 184 */	0xF7,0xFB,0xA7,0xAF,0xAC,0xAB,0xF3,0xA8,
/* 192 */	0xB7,0xB5,0xB6,0xC7,0x8E,0x8F,0x92,0x80,
/* 200 */   0xD4,0x90,0xD2,0xD3,0xDE,0xD6,0xD7,0xD8,
/* 208 */	'G', 0xA5,0xE3,0xE0,0xE2,0xE5,0x99,0x9E,
/* 216 */	0x9D,0xEB,0xE9,0xEA,0x9A,'I', 'S', 0xE1,
/* 224 */	0x85,0xA0,0x83,0xC6,0x84,0x86,0x91,0x87,
/* 232 */	0x8A,0x82,0x88,0x89,0x8D,0xA1,0x8C,0x8B,
/* 240 */	'g', 0xA4,0x95,0xA2,0x93,0xE4,0x94,0xF6,
/* 248 */	0x9B,0x97,0xA3,0x96,0x81,'i', 's', 0x98
}
}
};

char base64[0x80] = {
/* 0x00 */	-1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,
/* 0x08 */	-1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,
/* 0x10 */	-1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,
/* 0x18 */	-1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,
/* 0x20 */	-1,   -1,   -1,   -1,   -1,   -1,   -1,   -1,
/* 0x28 */	-1,   -1,   -1,   0x3E, -1,   -1,   -1,   0x3F,
/* 0x30 */	0x34, 0x35, 0x36, 0x37, 0x38, 0x39, 0x3A, 0x3B,
/* 0x38 */	0x3C, 0x3D, -1,   -1,   -1,   -1,   -1,   -1,
/* 0x40 */	-1,   0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06,
/* 0x48 */	0x07, 0x08, 0x09, 0x0A, 0x0B, 0x0C, 0x0D, 0x0E,
/* 0x50 */	0x0F, 0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16,
/* 0x58 */	0x17, 0x18, 0x19, -1,   -1,   -1,   -1,   -1,
/* 0x60 */	-1,   0x1A, 0x1B, 0x1C, 0x1D, 0x1E, 0x1F, 0x20,
/* 0x68 */	0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27, 0x28,
/* 0x70 */	0x29, 0x2A, 0x2B, 0x2C, 0x2D, 0x2E, 0x2F, 0x30,
/* 0x78 */	0x31, 0x32, 0x33, -1,   -1,   -1,   -1,   -1
};

/*------------------------- get current code page --------------------------*/
int get_code_page(void)
{
#ifdef __OS2__
	ULONG CpList[1];
	ULONG CpSize;

	if (DosQueryCp(sizeof(CpList), CpList, &CpSize))
		return DEFAULT_CP;

	return (int) CpList[0];

#else /* __MSDOS__ */

	union REGS inregs, outregs;

	inregs.x.ax = 0x6601;
	intdos(&inregs, &outregs);
	if (outregs.x.cflag)
		return DEFAULT_CP;	/* no code page support -- use default */

	return outregs.x.bx;

#endif
}

/*----------------- return code page enum based on page --------------------*/
enum code_pages select_cp_table(int code_page_num)
{
	int index;

	for (index = 0; code_page_list[index].code_page_num; ++index)
		if (code_page_list[index].code_page_num == code_page_num)
			break;
	return index;
}

/*---------------- return character set enum based on name -----------------*/
enum char_sets select_char_set(char *name)
{
	int index;

	for (index = 0; char_set_names[index]; ++index)
		if (stricmp(char_set_names[index], name) == 0)
			return index;
	return DEFAULT_CHAR_SET;		/* default character set */
}

/*---------- translate MIME character to the code page character -----------*/
unsigned char translate_char(unsigned char inchar, enum char_sets cset)
{
/* US_ASCII isn't in the tables, so choose ISO-8859-1 which will do the job */
	if (cset == US_ASCII)
		cset = ISO_8859_1;
	return (inchar < 0x80) ? inchar :
		((code_page_table == CP_NOT_SUPP) ? inchar : cptable[code_page_table][cset][inchar & 0x7F]);
}

/*------- translate line of MIME characters to code page characters --------*/
char *translate_line(char *origline, enum char_sets cset)
{
	static char out_line[81];
	int out_index;

	for (out_index = 0; (out_index < sizeof(out_line) - 1) &&
		 ((out_line[out_index++] = translate_char(*origline++, cset)) != 0) ;)
		;
	out_line[out_index] = 0;
	return out_line;
}

/*------------ Translate 4 bytes of base64 to 3 bytes of binary ------------*/
static int decode_base64(char *base, char *bin)
{
	bin[0] = (base64[base[0]] << 2) | (base64[base[1]] >> 4);
	bin[1] = (base64[base[1]] << 4) | (base64[base[2]] >> 2);
	bin[2] = (base64[base[2]] << 6) | (base64[base[3]]);
	return    (base64[base[0]] < 0) ? 0			/* only with a bad line */
			: (base64[base[1]] < 0) ? 0			/* only with a bad line */
			: (base64[base[2]] < 0) ? 1
			: (base64[base[3]] < 0) ? 2
			: 3;
}

/*--------- translate RFC 1342 header line to code page characters ----------*/
/* e.g.  From: =?ISO-8859-1?Q?Patrik_F=E4ltstr=F6m?= <paf@nada.kth.se> */
/* e.g.  Subject: =?ISO-8859-1?B?SWYgeW91IGNhbiByZWFkIHRoaXMgeW8=?= */
char *translate_header(char *origline, enum char_sets cset)
{
	static char out_line[81];
	char *outp = out_line;
	char char_set_name[80];
	enum char_sets line_char_set;
	char *hex_code = "12";
	char hex_value;
	char encoding;
	int num_conv, i;

	while (*origline) {
		if ((*origline == '=') && (*(origline+1) == '?') && strstr(origline+2, "?=")) {

			/* detected RFC 1342 header line -- parse it */
			strtok(strcpy(char_set_name, origline += 2), "?\n\r");
			line_char_set = select_char_set(char_set_name);

			/* point to encoding method character */
			origline = 1 + strchr(origline, '?');
			encoding = tolower(*origline);

			/* point to first character of encoded text */
			origline = 1 + strchr(origline, '?');

			if (encoding == 'q')				/* quoted-printable encoding */
				while (*origline) {		/* scan quoted-printable characters */

					/* end of special portion of header? */
					if ((*origline == '?') && (*(origline+1) == '=')) {
						if (*(origline += 2) == ' ')
							++origline;
						break;
					}

					/* special character */
					if (*origline == '=') {
						hex_code[0] = *++origline;
						hex_code[1] = *++origline;
						++origline;
						sscanf(hex_code, "%x", &hex_value);
						*outp++ = translate_char(hex_value, line_char_set);

					/* encoded space */
					} else if (*origline == '_') {
						*outp++ = ' ';
						++origline;

					/* normal text */
					} else
						*outp++ = translate_char(*origline++, cset);

				} /* while */


			else if (encoding == 'b')			/* base64 encoding */
				while (*origline) {
					/* end of special portion of header? */
					if ((*origline == '?') && (*(origline+1) == '=')) {
						if (*(origline += 2) == ' ')
							++origline;
						break;
					}
					num_conv = decode_base64(origline, outp);
					for (i = 0; i < num_conv; ++i)
						outp[i] = translate_char(outp[i], line_char_set);
					outp += num_conv;
					origline += 4;
				} /* while */

		} else
			*outp++ = translate_char(*origline++, cset);

	} /* while */
	*outp = '\0';

	return out_line;
}
