/***********************************************************************
*$Header:   J:/gedcom/gedlib/vcs/gedmrk.c_v   1.0   05 Aug 1992 19:00:40   fhdodj  $
*
*$config$="/K! /L/* /R* /Mgedmem.c"
*!global paths!
*   gedcom\library\gedmem.c
*   gedcom\all\gedmem.c
*!end!
*
*   FILE NAME: GEDMEM.C
*
*   DESCRIPTION:
*       This file contains functions for memory management.
*
*   ROUTINES:
*
*
*$Log:   J:/gedcom/gedlib/vcs/gedmrk.c_v  $
 * 
 *    Rev 1.0   05 Aug 1992 19:00:40   fhdodj
 * Initial revision.
 * 
 *    Rev 1.6   26 Mar 1992 10:31:34   fhdodj
 * Made ged_check_mem a macro.
 * 
 *    Rev 1.5   25 Mar 1992 15:52:36   fhdodj
 * Added the function ged_pool_size(). Changed ged_check_mem to be a little
 * faster.
 * 
 *    Rev 1.4   21 Oct 1991 10:10:48   fhdodj
 * Changed char to byte.
 * 
 *    Rev 1.3   28 Jun 1991 14:52:22   fhdkrf
 * Add keywords for PolyDoc
 * 
 *    Rev 1.2   18 Jun 1991 08:11:54   odj
 * Fixed typo
 * 
 *    Rev 1.1   18 Jun 1991 07:54:42   odj
 * Changed chunk chain so that new chunks are added to the front of the list
 * instead of the end.  Added calls to ged_error() where chunk header has
 * been overwritten, or out of memory.
 * 
 *    Rev 1.0   20 Dec 1990 09:06:26   odj
 * Initial revision.
***********************************************************************/
#include "gedcom.h"

#ifdef NON_ANSI
#define max(a, b) ((a) < (b) ? (b) : (a))
#endif

/********************************************************************
*	 FILE: GEDMEM.C
*
*        DESCRIPTION:
*            This is a proposed starting point for a standardized
*            approach to dynamic memory management. Possible issues
*            to be discussed are:
*                        Extended Memory
*                        Record Caching
*                        Machine Independence
*                        Others
*
********************************************************************/

/********************************************************************
*!name!
*    ged_check_mem()
*!1!
*
*       NAME: ged_check_mem
*
*       DESCRIPTION:
*           Checks the current pool for corrupted chunk pointers.
*           If a memory violation is found, a message is displayed,
*           and the program exits.
*
*       RETURN VALUE:
*           None.
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
#define GED_CHECK_MEM()							\
    {			/*!end!*/ 					\
    if (ged_get_pool())							\
        {								\
        CHUNK *tmpChunk = ged_get_pool()->firstChunk;				\
        byte *bufSize;							\
									\
        while (tmpChunk)						\
            {								\
            if (((bufSize = tmpChunk->bufPtr + tmpChunk->size) !=	\
                                             tmpChunk->endPtr) ||	\
                ((bufSize - tmpChunk->free)  != tmpChunk->curPtr))	\
                ged_error(1, NULL);					\
            tmpChunk = tmpChunk->nextChunk;				\
            }								\
	}								\
    else								\
        return(NULL);							\
    }
/**/
/********************************************************************
*
*       NAME: ged_mark_mem
*
*       DESCRIPTION:
*           Sets a mark in the current pool.
*
*       RETURN VALUE:
*           A pointer to the beginning of the requested memory or
*           NULL if there is no pool.
*
********************************************************************/
CHUNKMARKER * ged_mark_mem(size)
    unsigned long size;
    {
    CHUNK *tempChunk;
    CHUNKMARKER	*markAddress;

        GED_CHECK_MEM()
	tempChunk = ged_get_chunk(size);
        if (!tempChunk)
            return(NULL);
        markAddress = (CHUNKMARKER *)tempChunk->curPtr;
        markAddress->nextMarker = (CHUNKMARKER *) NULL;
        markAddress->markedChunk = tempChunk;
        markAddress->saveFree = tempChunk->free;
	markAddress->free = tempChunk->free - sizeof(CHUNKMARKER);
        markAddress->savePtr = (byte *)tempChunk->curPtr;
        markAddress->readPtr = markAddress->writePtr =
                                      (byte *)(markAddress + 1);
        markAddress->read = 0;
        markAddress->size = size;
        tempChunk->free = 0;
        tempChunk->curPtr = tempChunk->endPtr;
        return(markAddress);
    }

/********************************************************************
*
*       NAME: ged_release_marked_chunk
*
*       DESCRIPTION:
*           Removes the mark in the chunk.
*
*       RETURN VALUE:
*           the address of the next marked chunk.
*
********************************************************************/
CHUNKMARKER *ged_release_marked_chunk(markAddress)
    CHUNKMARKER *markAddress;
    {
    CHUNK *tempChunk = markAddress->markedChunk;

        tempChunk->free = markAddress->saveFree;
        tempChunk->curPtr = markAddress->savePtr;
        markAddress = markAddress->nextMarker;
	return(markAddress);
    }

/********************************************************************
*
*       NAME: ged_unmark_mem
*
*       DESCRIPTION:
*           Removes the mark in the current pool.
*
*       RETURN VALUE:
*           None.
*
********************************************************************/
CHUNKMARKER * ged_unmark_mem(CHUNKMARKER * markAddress)
    {
    POOL  *curPool = ged_get_pool();

        GED_CHECK_MEM();
        while (markAddress)
            markAddress = ged_release_marked_chunk(markAddress);
	return(markAddress);
    }

/********************************************************************
*
*       NAME: ged_get_marked_size
*
*       DESCRIPTION:
*           Removes the mark in the current pool.
*
*       RETURN VALUE:
*           None.
*
********************************************************************/
int ged_get_marked_size(CHUNKMARKER * markAddress)
    {
    int size = 0;

        while (markAddress)
            {
            size += (markAddress->saveFree - markAddress->free -
	             sizeof(CHUNKMARKER));
            markAddress = markAddress->nextMarker;
            }
        return(size);
    }

/********************************************************************
*
*       NAME: ged_get_num_chunks
*
*       DESCRIPTION:
*           Removes the mark in the current pool.
*
*       RETURN VALUE:
*           None.
*
********************************************************************/
int ged_get_num_chunks(CHUNKMARKER * markAddress)
    {
    int num = 0;

        while (markAddress)
            {
            markAddress = markAddress->nextMarker;
            num++;
	    }
        return(num);
    }

/********************************************************************
*
*       NAME: ged_put_byte_mem
*
*       DESCRIPTION:
*           Puts a byte in the next position of the marked memory
*
*       RETURN VALUE:
*           A pointer to the beginning of the requested memory or
*           NULL if there is no pool.
*
********************************************************************/
CHUNKMARKER * ged_put_byte_mem(character, markAddress)
    byte character;
    CHUNKMARKER **markAddress;
    {

        GED_CHECK_MEM();
        (*markAddress)->free--;
        if ((*markAddress)->free == 0)
            {
	    CHUNKMARKER *tempMark = ged_mark_mem((*markAddress)->size);
            if (!tempMark)
	        return(NULL);
            tempMark->nextMarker = *markAddress;
	    *markAddress = tempMark;
            (*markAddress)->free--;
            }
        *(*markAddress)->writePtr = character;
        (*markAddress)->writePtr += 1;
	return(*markAddress);
    }
/********************************************************************
*
*       NAME: ged_remove_byte_mem
*
*       DESCRIPTION:
*           Puts a byte in the next position of the marked memory
*
*       RETURN VALUE:
*           A pointer to the beginning of the requested memory or
*           NULL if there is no pool.
*
********************************************************************/
byte ged_remove_byte_mem(markAddress)
    CHUNKMARKER **markAddress;
    {
    byte character;

        if (!*markAddress)
            return((byte)NULL);
        (*markAddress)->free++;
        if ((*markAddress)->free > (*markAddress)->saveFree)
            {
	    (*markAddress)->free--;
            *markAddress = ged_release_marked_chunk(markAddress);
            return(ged_remove_byte_mem(markAddress));
            }
        character = *(*markAddress)->writePtr;
        (*markAddress)->writePtr -= 1;
	return(character);
    }
/**/
/********************************************************************
*
*       NAME: ged_get_byte_mem
*
*       DESCRIPTION:
*           Gets a byte from the next position of the marked memory
*
*       RETURN VALUE:
*           A pointer to the beginning of the requested memory or
*           NULL if there is no pool.
*
********************************************************************/
byte ged_get_byte_mem(markAddress)
    CHUNKMARKER **markAddress;
    {
    byte character;
    CHUNKMARKER *tempMark = *markAddress;
    CHUNKMARKER *prevMark = (CHUNKMARKER *)NULL;

	while (tempMark->nextMarker)
	    {
	    prevMark = tempMark;
	    tempMark = tempMark->nextMarker;
	    }
	tempMark->read++;
	if (tempMark->read >= tempMark->saveFree)
	    {
	    ged_release_marked_chunk(tempMark);
	    tempMark = prevMark;
	    return(ged_get_byte_mem(&tempMark));
            }
        character = *tempMark->readPtr;
        tempMark->readPtr += 1;
        return(character);
    }
    
byte *ged_copy_marked_mem(markAddress)
    CHUNKMARKER *markAddress;
    {
    int size = ged_get_marked_size(markAddress);
    byte *gedBuf = ged_alloc_pool(size);
    int x = 0;
    byte *tmp = gedBuf;
    int chunks = ged_get_num_chunks(markAddress);

       if (chunks < 2)
           return(markAddress->readPtr);
       for (x = 0; x < size; x++)
           *tmp++ = ged_get_byte_mem(&markAddress);
        return(gedBuf);
    }

