/***********************************************************************
*$Header:   J:/gedcom/gedlib/vcs/gednode.c_v   1.6   25 Mar 1992 15:42:06   fhdodj  $
*
*$config$="/K! /L/* /R* /Mgednode.c"
*!global paths!
*   gedcom\library\gednode.c
*   gedcom\all\gednode.c
*!end!
*
*   FILE NAME: GEDNODE.C
*
*   DESCRIPTION:
*       This file contains functions for manipulating nodes in trees.
*
*   ROUTINES:
*
*
*$Log:   J:/gedcom/gedlib/vcs/gednode.c_v  $
 * 
 *    Rev 1.6   25 Mar 1992 15:42:06   fhdodj
 * Added functions ged_move_to_child(), ged_move_to_sibling(),
 * ged_remove_siblings(), ged_remove_children().
 * 
 *    Rev 1.5   23 Oct 1991 09:12:24   fhdodj
 * Made fix to ged_remove_node for the case where the node to be removed is the 
 * a first child, with no children, and which has siblings.
 * 
 *    Rev 1.4   21 Oct 1991 10:09:28   fhdodj
 * Changed char to byte.
 * 
 *    Rev 1.3   28 Jun 1991 14:57:20   fhdkrf
 * Added keywords for PolyDoc
 * 
 *    Rev 1.2   21 Jun 1991 13:40:52   fhdodj
 * Fixed ged_connect_child and ged_connect_sibling.  They were attaching
 * the node in the n+3 and n+2 positions respectively instead of the nth 
 * position.
 * 
 *    Rev 1.1   17 Jun 1991 16:19:32   odj
 * Added check in ged_prune for top and forestFlag.
 * 
 *    Rev 1.0   20 Dec 1990 09:07:34   odj
 * Initial revision.
***********************************************************************/
#include "gedcom.h"

/********************************************************************
*!name!
*    ged_connect_child()
*!1!
*
*        NAME: ged_connect_child
*
*        DESCRIPTION:
*            Connect the child node to the parent node as sibling
*            number pos to parent->child. If pos is negative, then
*            as the last sibling.
*
*        RETURN VALUE:
*            The child node, or NULL if error.
*
*        CALLS: ged_connect_sibling
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
NODE * ged_connect_child(parent, child, pos)
    NODE *parent;
    NODE *child;
    int   pos;
    {			/*!end!*/
    NODE *tempChild = child;
    NODE *parChild;

        if (!parent)
            return(NULL);
        if (parent->child && (pos != 1)) /* parent already has children   */
            ged_connect_sibling(parent->child, child, pos - 1);
        else        /* parent has no children                           */
            {
	    if (pos == 1)
	        {
		parChild = parent->child;
		}
            parent->child = child;
            if (child)
                {
                child->parent = parent;
                while (tempChild->sibling)
                    {
                    tempChild = tempChild->sibling;
                    tempChild->parent = parent;
                    }
                if ( (pos == 1) && tempChild )
		    tempChild->sibling = parChild;
                }
            }
        return(child);
    }
/**/
/********************************************************************
*!name!
*    ged_connect_sibling()
*!1!
*
*        NAME: ged_connect_sibling
*
*        DESCRIPTION:
*            Connect sibling2 as a sibling to sibling1
*            in position pos. If pos is negative, then as the
*            last sibling.
*
*        RETURN VALUE:
*            Sibling2.
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
NODE * ged_connect_sibling(sibling1, sibling2, pos)
    NODE *sibling1;
    NODE *sibling2;
    int   pos;
    {			/*!end!*/
    NODE *temp_sibling2;

        if (!sibling1)
            return(sibling2);

        /* find sibling pos           */
        while (--pos && sibling1->sibling)
            sibling1 = sibling1->sibling;
        if (sibling2)
            {
            temp_sibling2 = sibling2;
            while (sibling2->sibling)
                {
                sibling2->parent = sibling1->parent;
                sibling2 = sibling2->sibling;
                }
            sibling2->sibling = sibling1->sibling;
            sibling1->sibling = temp_sibling2;
            sibling2->parent = sibling1->parent;
            }
        else
            sibling1->sibling = (NODE *)NULL;
        return(sibling2);
    }
/**/
/***************************************************************************
*!name!
*    ged_promote()
*!1!
*
*        NAME:   ged_promote
*
*        DESCRIPTION:
*            Will move node and its children to the parents position.
*            If the father does not exist then nothing is done.
*
*        RETURN VALUE:
*            none.
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
***************************************************************************/
void ged_promote(node)
    NODE *node;
    {			/*!end!*/
     register NODE *child;
     NODE          *parent = NULL,
                   *parChild,
                   *prevChild;

        if (node && node->parent && node->parent->child)
            {
            parent   = node->parent;
            child    = node->child;
            parChild = node->parent->child;
            }
        if (parent)
            {
            parent->line = node->line;
            parent->length = node->length;
            parent->child = child;
            if (child)
                {
                child->parent = parent;
                while (child->sibling)
                    {
                    child = child->sibling;
                    child->parent = parent;
                    }
                /* Link up child to parents old siblings */
                child->sibling = parChild;
                while(child->sibling != node)
                    child = child->sibling;
                child->sibling = node->sibling;
                }
            else if (parChild == node)
	        {
		parent->child = node->sibling;
		}
            else if (parChild)  /* The tree is invalid if false */
	        {
		while ((parChild != node) && parChild->sibling)
		    {
		    prevChild = parChild;
                    parChild = parChild->sibling;
		    }
		prevChild->sibling = node->sibling;
		}
            }
    }
/**/
/********************************************************************
*!name!
*    ged_prune_tree()
*!1!
*
*        NAME: ged_prune_tree
*
*        DESCRIPTION:
*            Removes all nodes which do not have a value and do not
*            have any children.
*
*        RETURN VALUE:
*            The root node of the new tree.
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
NODE * ged_prune_tree(tree, forestFlag)
    NODE *tree;
    int   forestFlag;
    {			/*!end!*/
    NODE *root = tree;
    NODE *temp;
    NODE *top = tree;

        while (tree)
            {
            while (tree->child)
                tree = tree->child;
            if ((*(ged_get_value(tree)) == '\0') || 
                (ged_get_value(tree) == NULL))
                {
                if (!tree->parent)
                     return(NULL);
                temp = tree;
                tree = tree->parent;
                ged_remove_subtree(temp);
                }
            else
                {
                while(!tree->sibling && (tree != top))
                    tree = tree->parent;
                if ((tree == top) && forestFlag)
                    top = top->sibling;
                else if (tree == top)
                    break;
                tree = tree->sibling;
                }
            }
        return(root);
    }
/**/
/*****************************************************************************
*!name!
*    ged_remove_node()
*!1!
*
*        NAME:   ged_remove_node
*
*        DESCRIPTION:
*            Remove node and splice in children all at node's present position.
*            the node is not removed but the child's node is removed
*            and the line from the child's node is moved to node's.
*            A side effect is node being the last inserted sibling.
*
*        RETURN VALUE:
*            none.
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
******************************************************************************/
void ged_remove_node(node)
    NODE *node;
    {			/*!end!*/
    NODE *sib;
    NODE *prev_sib;
    register NODE *nextChild = NULL;
    register NODE *frstGrndKids;
        
        if (node)
	    {
            sib = node->sibling;
	    nextChild = node->child;
	    }
        if (nextChild && node)
            {
            /* Transfer first child data to current data handle node */
            node->line = nextChild->line;
            node->length = nextChild->length;
            node->child = nextChild->child;                       
            
            /** Now update the first childs childrens parent pointers **/
            frstGrndKids = nextChild->child;
            /* Fix the Parent pointers of the node which was move above */
            while (frstGrndKids)
                {          
                frstGrndKids->parent = nextChild->parent;
                frstGrndKids = frstGrndKids->sibling;
                }
            nextChild = nextChild->sibling;
            /* Link up the siblings */
            while (nextChild)
                {
                node->sibling = nextChild;
                nextChild->parent = node->parent;
                node = nextChild;
                nextChild = nextChild->sibling;
                }
            /* Link in the remaining siblings */
            node->sibling = sib;
            }
        else if (node)
            {
            prev_sib = ged_get_previous_sibling(node);
	    if (prev_sib)
	        prev_sib->sibling = sib;
            if (node->parent && (node->parent->child == node))
	        node->parent->child = sib;
            node->line    = NULL;
            node->length  = (unsigned int) 0;
	    node->child   = NULL;
	    node->sibling = NULL;
	    node->parent  = NULL;
            }
    }
/**/
/********************************************************************
*!name!
*    ged_remove_subtree()
*!1!
*
*        NAME: ged_remove_subtree
*
*        DESCRIPTION:
*            Deletes subcontext from the tree.
*
*        RETURN VALUE:
*            The root of the new tree.
*
*        NOTE: ged_remove_subtree does not release the memory used
*              by the removed node.
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
NODE * ged_remove_subtree(node)
    NODE *node;
    {			/*!end!*/
    NODE *temp;

        /* if it doesn't exist or is the root return NULL          */
        if (!node || !node->parent || !node->parent->child)
            return(NULL);
        temp = node->parent->child;
        
        /* otherwise remove the subtree from the tree                    */
        if (temp != node)
            {
            while (temp->sibling != node)
                temp = temp->sibling;
            temp->sibling = node->sibling;
            }
        else
            node->parent->child = node->sibling;
        temp = node;
        while (node->parent)
            node = node->parent;
        temp->parent = NULL;
        temp->sibling = NULL;
        return(node);
    }

/********************************************************************
*!name!
*    ged_move_to_child()
*!1!
*
*        NAME: ged_move_to_child
*
*        DESCRIPTION:
*            Removes subcontext from the tree and reattaches it at the
*            specified point as sibling number pos to the first child.
*            If pos is negative, then as the last sibling.
*
*        RETURN VALUE:
*            The child node, or NULL if error.
*
*        CALLED BY: ged_connect_child, ged_remove_node.
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
NODE * ged_move_to_child(node, parent, pos)
    NODE *node;
    NODE *parent;
    int   pos;
    {			/*!end!*/
        ged_remove_subtree(node);
	return(ged_connect_child(parent, node, pos));
    }
/**/

/********************************************************************
*!name!
*    ged_move_to_sibling()
*!1!
*
*        NAME: ged_move_to_sibling
*
*        DESCRIPTION:
*            Removes subcontext from the tree and reattaches it at the
*            specified point as sibling number pos to node. If pos is
*            negative, then as the last sibling.
*
*        RETURN VALUE:
*            node.
*
*        CALLED BY: ged_connect_sibling, ged_remove_node.
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
NODE * ged_move_to_sibling(node, sibling, pos)
    NODE *node;
    NODE *sibling;
    int   pos;
    {			/*!end!*/
        ged_remove_subtree(node);
	return(ged_connect_sibling(sibling, node, pos));
    }
/**/
/********************************************************************
*!name!
*    ged_remove_children()
*!1!
*
*        NAME: ged_remove_children
*
*        DESCRIPTION:
*            Removes all of the nodes children from the tree.
*
*        RETURN VALUE:
*            A pointer to the first node of the removed sibling chain.
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
NODE * ged_remove_children(node)
    NODE *node;
    {			/*!end!*/
    NODE *temp_node = ged_get_child(node);

        if (node)
            node->child = NULL;
        return(temp_node);
    }
/**/
/********************************************************************
*!name!
*    ged_remove_siblings()
*!1!
*
*        NAME: ged_remove_siblings
*
*        DESCRIPTION:
*            Removes all of the nodes siblings from the tree.
*
*        RETURN VALUE:
*            A pointer to the first node of the removed sibling chain.
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
NODE * ged_remove_siblings(node)
    NODE *node;
    {			/*!end!*/
    NODE *temp_node = ged_get_sibling(node);

        if (node)
            node->sibling = NULL;
        return(temp_node);
    }
