/***********************************************************************
*$Header:   J:/gedcom/gedlib/vcs/gedread.c_v   1.1   05 Aug 1992 19:24:48   fhdodj  $
*
*$config$="/K! /L/* /R* /Mgedread.c"
*!global paths!
*   gedcom\library\gedread.c
*   gedcom\all\gedread.c
*!end!
*
*   FILE NAME: GEDREAD.C
*
*   DESCRIPTION:
*       This file contains functions for reading from files.
*
*   ROUTINES:
*
*
*$Log:   J:/gedcom/gedlib/vcs/gedread.c_v  $
 * 
 *    Rev 1.1   05 Aug 1992 19:24:48   fhdodj
 * Changed fgets back to using fgetc, and split out read functions that allocate
 * their own memory to the file gedrdmem.c
 * 
 *    Rev 1.0   28 Jul 1992 09:34:36   fhdodj
 * Initial revision.
 * 
 *    Rev 1.12   03 Mar 1992 09:44:54   fhdodj
 * Replaced fpos with -bytesRead in NON_ANSII part of ged_read_line.
 * 
 *    Rev 1.11   03 Mar 1992 09:35:14   fhdodj
 * Removed fflush and ftell from file.
 * 
 *    Rev 1.10   21 Oct 1991 09:06:44   fhdodj
 * Changed char to byte.
 *
 *    Rev 1.9   09 Aug 1991 10:24:24   fhdodj
 * Changed condition in ged_read_tree so that LAST_RECORD is now returned.
 * 
 *    Rev 1.8   26 Jul 1991 13:31:26   fhdodj
 * Fixed status return from ged_read_line and ged_read_level.  Didn't return
 * LAST_RECORD correctly.
 * 
 *    Rev 1.7   28 Jun 1991 14:08:10   fhdkrf
 * Added keywords for PolyDoc
 * 
 *    Rev 1.6   25 Jun 1991 09:04:14   fhdodj
 * Fixed functions ged_connect_child and ged_connect_sibling.  They were 
 * attaching nodes in the wrong place when given an argument other than -1.
 * 
 *    Rev 1.5   18 Jun 1991 13:39:56   odj
 * Added check in ged_read_line so we dont get an error for invalid character
 * when there is a carriage return or newline.
 * 
 *    Rev 1.4   18 Jun 1991 11:08:12   odj
 * Changed error number ranges.
 *
 *    Rev 1.3   17 Jun 1991 16:03:00   odj
 * Added IO status checks.  Added calls to ged_error() function.  Also added
 * checking in ged_read_line function for invalid tags and cross references
 * being read.  Changed arguments to ged_read_level() function so that status is
 * now being checked in this function also.
 * 
 *    Rev 1.2   25 Apr 1991 10:06:18   odj
 * changed ged_fprintf() to output a carriage return linefeed insteadof just a 
 * linefeed.
 * 
 *    Rev 1.1   14 Feb 1991 15:02:42   odj
 * void is not allowed by some non ansi compilers (e.g. the CYBER). so I 
 * changed void to be char for non ansi compilers.
 * 
 *    Rev 1.0   20 Dec 1990 08:59:48   odj
 * Initial revision.
***********************************************************************/
#include "gedcom.h"

void skip_this_level(FILE *, char *);

/********************************************************************
*!name!
*    ged_find_level()
*!1!
*
*        NAME: ged_find_level
*
*        DESCRIPTION:
*            Get the next level number from the given gedcom buffer.
*
*        RETURN VALUE:
*            The level number of the next line. Will return zero if
*            there is no next level number.
*
*         CALLED BY: ged_to_tree
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
int ged_find_level(ged)
    byte **ged;
    {               /*!end!*/
    register int   level = 0;
    register byte *ptr = *ged;

	   /* advance the pointer until we find a number                   */
	   while (!isdigit(*ptr) && (*ptr != '\0'))
		  ptr++;
	   while (isdigit(*ptr)) /* get the number                         */
		  level = (level * 10) + (int)(*ptr++ - '0');
	   *ged = ptr;
	   return(level);
    }
/**/
/********************************************************************
*!name!
*    ged_find_line()
*!1!
*
*        NAME: ged_find_line
*
*        DESCRIPTION:
*            Get the next line from a gedcom buffer. status will be
*            set to MORE if a NULL char is not found after the line,
*            or LAST if a NULL char is found at the end of the line.
*            If there is more in the buffer, status is set to MORE.
*            Otherwise status is set to LAST.
*
*        RETURN VALUE:
*            The next line of the GEDCOM record.
*
*         CALLED BY: ged_to_tree
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
byte * ged_find_line(ged, status)
    byte **ged;
    int  *status;
    {               /*!end!*/
    register byte *line = *ged;
    byte          *start = line;

	   while ((*line == ' ') || (*line == 9))
		  line++;
	   start = line--;

	   /* Advance the pointer until we are at the end of the line */
	   while ((*line != '\n') && (*line != '\r') && (*line != '\0'))
		  line++;
	   if (*line != '\0')  /* if not end of the file add NULL     */
		  {
		  *status = MORE;
		  *line++ = '\0';
		  }
	   else
		  *status = LAST;
	   *ged = line;
	   return(start);
    }
/**/
/********************************************************************
*!name!
*    ged_read_level()
*!1!
*
*        NAME: ged_read_level
*
*        DESCRIPTION:
*            Get the next level from a gedcom file.
*
*        RETURN VALUE:
*            The next level. Zero will be returned if there is none.
*
*         CALLED BY: ged_read_rec
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
int ged_read_level(fp, gedBuf, limit, startLevel, status, bytesRead)
    register FILE *fp;
    byte          **gedBuf;
    long          *limit;
    short          startLevel;
    short         *status;
    int           *bytesRead;
    {               /*!end!*/
    int  level = 0;
    int  levelRead = 0;
    int  c;
    long startLimit = *limit;

	   /* advance until we find a number or find EOF */
	   while (!isdigit(c = fgetc(fp)) && (c != '+') && (c != EOF))
		  {
		  *bytesRead += 1;
		  if (!isspace(c) && (c != 26) && (c != '\n') && (c != '\r'))
			 ged_error(2060, NULL);
	    }
	  *bytesRead += 1;
	  if (ferror(fp))
		 ged_error(101, NULL);

	   /* get the number          */
	   while ((isdigit(c)) || (c == '+'))  /* addded chk for +level */
		  {
		  levelRead = 1;
		  if (c == '+')    /* added ck for + level and set level to 19 */
		    level = 19;
		  else
		    {
		    level = (level * 10) + (int)(c - '0');
		    **gedBuf = (byte)c;
		    *gedBuf += 1;
		    *limit -= 1;
		    }
		  if (*limit < 1)
			 return(-99);
		  *bytesRead += 1;
		  c = fgetc(fp);
		  if (ferror(fp))
			 ged_error(102, NULL);
		  }
	   if ((startLimit != *limit) && (level <= startLevel) && (levelRead))
		  {
#ifdef NON_ANSI
		  fseek(fp, -*bytesRead, 1);
#else
		  fseek(fp, -*bytesRead, SEEK_CUR);
#endif
		  if (ferror(fp))
			 ged_error(108, NULL);
		  *gedBuf -= 1;
		  *limit += 1;
		  while(isdigit(**gedBuf))
			 {
			 **gedBuf = '\0';
			 *gedBuf -= 1;
			 *limit += 1;
			 }
		  }
	   if (c == EOF)
		  {
		  **gedBuf = '\0';
		  *gedBuf += 1;
		  *limit -= 1;
		  *status = LAST_RECORD;
		  }
	   else
		  *status = MORE_RECORDS;
	   if (*limit < 1)
		  *status = REC_TOO_LONG;
	   if ( ( (startLimit == *limit) || (startLimit == (*limit + 1)) ) &&
		   !levelRead )
		  *status = END_OF_FILE;
	   return(level);
    }
/**/
/********************************************************************
*!name!
*    ged_read_line()
*!1!
*
*        NAME: ged_read_line
*
*        DESCRIPTION:
*            Get the next line from a gedcom file. status
*            REC_TOO_LONG is given when no more space is
*            available in the input buffer.  LAST_RECORD is
*            given when EOF is encountered.  MORE_RECORDS is
*            given when there are more records available to read.
*            END_OF_FILE is returned if EOF is encountered before
*            anything is read.
*
*        RETURN VALUE:
*            The next line.
*
*         CALLED BY: ged_read_rec
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
void ged_read_line(fp, gedBuf, limit, status, bytesRead)
    register FILE     *fp;
    byte     **gedBuf;
    long     *limit;
    short    *status;
    int      *bytesRead;
    {               /*!end!*/
    register byte *line = *gedBuf;
    long startLimit = *limit;
    int c = fgetc(fp);
    short hasCrossRef = 0;
    short inTag = 0;
    short len = 0;

	   if (ferror(fp))
		  ged_error(103, NULL);
	   *bytesRead += 1;

	   /* Advance the pointer until we are past white space         */
	   while ((c == '\t') || (c == ' '))
		  {
		  (*limit)--;
		  if (*limit < 1)
			 break;

		  *line++ = (byte)c;
		  *bytesRead += 1;
		  c = fgetc(fp);
		  if (ferror(fp))
			 ged_error(104, NULL);
		  }

	   if (c == '@')
		  {
		  hasCrossRef = 1;
		  (*limit)--;

		  if (*limit > 0)
			 {
			 *line++ = (byte)c;
			 *bytesRead += 1;
			 c = fgetc(fp);
			 if (ferror(fp))
				ged_error(105, NULL);
			 }
		  }
	   else
		  inTag = 1;

	   /* Advance the pointer until we are at the end of the line         */
	   while ((c != EOF))
		  {
		  (*limit)--;
		  if (*limit < 1)
			 break;

		  if (hasCrossRef)
			 {
			 if (c == '@')
				{
				hasCrossRef = 0;

				/* Advance the pointer past any tabs and spaces */
				while ((c == '\t') || (c == ' '))
				    {
				    (*limit)--;
				    if (*limit < 1)
				    break;

				    *line++ = (byte)c;
				    *bytesRead += 1;
				    c = fgetc(fp);
				    if (ferror(fp))
					   ged_error(106, NULL);
				    }
				inTag = 1;
				len = 0;
				}
			 else
				{
				if ((c == '\n') || (c == '\r'))
				    break;
				len++;
				if ((c < 32) || (c > 127))
				    ged_error(201, NULL);
				if (len > MAX_CROSSREF_LEN)
				    ged_error(202, NULL);
				}
			 }
		  else if (inTag)
			 {
			 if (isspace(c))
				inTag = 0;
			 else
				{
				if ((c == '\n') || (c == '\r'))
				    break;
				len++;
				if ((c < 32) || (c > 127))
				    ged_error(203, NULL);
				if (len > MAX_TAG_LEN)
				    ged_error(204, NULL);
				}
			 }

		  *line++ = (byte)c;
		  if ((c == '\n') || (c == '\r'))
			 break;
		  *bytesRead += 1;
		  c = fgetc(fp);
		  if (ferror(fp))
			 ged_error(107, NULL);
		  }
	   if (hasCrossRef)
		  ged_error(205, NULL);
	   while (!isdigit(c = fgetc(fp)) && (c != '+') && (c != EOF))
		  {
		  *bytesRead += 1;
		  if (!isspace(c) && (c != 26) && (c != '\n') && (c != '\r'))
			 ged_error(206, NULL);
		  }
	   ungetc(c, fp);
//	   fseek(fp, -1L, SEEK_SET);
//           clearerr(fp);
	   if (c != EOF)
		  *status = MORE_RECORDS;
	   else
		  *status = LAST_RECORD;
	   if (*limit < 1)
		  *status = REC_TOO_LONG;
	   if ( (startLimit == *limit) || (startLimit == (*limit + 1)) )
		  *status = END_OF_FILE;
	   *line = '\0';
	   *gedBuf = line;
    }

/***********************************************************************
*
*     this routine was added to skip past the UDE template by skipping
*     all gedcom lines subordinate to the level number passed.
*
*     called by ged_read_partial_rec
*************************************************************************/

void skip_this_level(fp, endString)
    FILE     *fp;
    char     *endString;
    {               /*!end!*/
    int limit = 500;
    long  bytesRead = 0;
    char anyBuf[501];

	bytesRead = ftell(fp);
	while (fgets(anyBuf, limit, fp) != NULL)
	    {
	    if (strstr(anyBuf, endString))
	      {
	      fseek(fp, bytesRead, SEEK_SET);
	      break;
	      }
	    bytesRead = ftell(fp);
	    }
   }
/**/

/********************************************************************
*!name!
*    ged_read_rec()
*!1!
*
*        NAME: ged_read_rec
*
*        DESCRIPTION:
*            Read GEDCOM file. stop when EOF or the starting level is
*            encountered. status REC_TOO_LONG is given when no more
*            space is available in the input buffer.  LAST_RECORD is
*            given when EOF is encountered.  MORE_RECORDS is given
*            when there are more records available to read.
*
*        RETURN VALUE:
*            Status contains a value telling if EOF was
*            encountered, there are more records, it was the last
*            record, or there is not enough space in gedBuf. a
*            pointer will be returned to the position in the buffer
*            after the last character read.
*
*        CALLS: ged_read_line, ged_read_level
*
*        CALLED BY: ged_read_tree
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
byte * ged_read_rec(fp, gedBuf, limit, status)
    FILE     *fp;
    byte     *gedBuf;
    long      limit;
    short     *status;
    {               /*!end!*/
    int   level, prev;
    int   startLev = 0;
    byte *bufPtr;
    int   bytesRead = 0;

	   if (!fp)
		  return(NULL);
	   if (!gedBuf || (limit <= 0))
		  {
		  *status = REC_TOO_LONG;
		  return(NULL);
		  }
	   *status = -1;
	   prev = startLev = ged_read_level(fp, &gedBuf, &limit, -1, status, &bytesRead);
	   ged_read_line(fp, &gedBuf, &limit, status, &bytesRead);
	   bufPtr = gedBuf;
	   bytesRead = 0;
	   while ((*status != REC_TOO_LONG) && (*status != LAST_RECORD) &&
			((level = ged_read_level(fp, &gedBuf, &limit, startLev, status, &bytesRead))
							> startLev) &&
			(*status != REC_TOO_LONG))
		  {
		  if (prev < level)  /* need to go down a level          */
			 if ((level - prev) != 1)  /* not next child of prev      */
				{  
				*status = MISSING_LEVEL;
				gedBuf = bufPtr;
				}
		  ged_read_line(fp, &gedBuf, &limit, status, &bytesRead);
		  bytesRead = 0;
		  if ((prev < level) && ((level - prev) != 1))
			 {
			 gedBuf = bufPtr;
			 }
		  else
			 prev = level;
		  bufPtr = gedBuf;
		  }
	   if (*gedBuf != '\0')
		  *gedBuf++ = '\0';
	   if (limit < 1)
		  *status = REC_TOO_LONG;
	   return(gedBuf);
    }
/**/
/********************************************************************
*!name!
*    ged_read_partial_rec()
*!1!
*
*        NAME: ged_read_partial_rec
*
*        DESCRIPTION:
*            Read GEDCOM file. stop when EOF or endLevel and endTag is
*            encountered. status REC_TOO_LONG is given when no more
*            space is available in the input buffer.  LAST_RECORD is
*            given when EOF is encountered.  MORE_RECORDS is given
*            when there are more records available to read.
*
*        RETURN VALUE:
*            Status contains a value telling if EOF was
*            encountered, there are more records, it was the last
*            record, or there is not enough space in gedBuf. a
*            pointer will be returned to the position in the buffer
*            after the last character read.
*
*        CALLS: ged_read_line, ged_read_level
*
*        CALLED BY: ged_read_tree
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
byte * ged_read_partial_rec(fp, gedBuf, limit, status, endLevel, endTag)
    FILE     *fp;
    byte     *gedBuf;
    long      limit;
    short    *status;
    int       endLevel;
    byte     *endTag;
    {               /*!end!*/
    int   level, prev;
    byte *bufPtr;
    char endString[10];
    NODE tmpNode;
    int  done = 0;
    int  bytesRead = 0;

	if (!fp)
	    return(NULL);
	if (!gedBuf || (limit <= 0))
	    {
	    *status = REC_TOO_LONG;
	    return(NULL);
	    }

//	sprintf(endString,"%c %s\0", level, endTag);
	 strcpy(endString, "2 UDER");
	*status = -1;
	tmpNode.parent  = NULL;
	tmpNode.child   = NULL;
	tmpNode.sibling = NULL;
	   prev = ged_read_level(fp, &gedBuf, &limit, -1, status, &bytesRead);
	   ged_read_line(fp, &gedBuf, &limit, status, &bytesRead);
	bufPtr = gedBuf;
	   bytesRead = 0;
	while (!done && (*status != REC_TOO_LONG) &&
		  (*status != LAST_RECORD) &&
		  (level = ged_read_level(fp, &gedBuf, &limit, 0, status, &bytesRead)) &&
		  (*status != REC_TOO_LONG))
	    {
	    if (level == 19)
	       {
	       level = prev;
	       gedBuf = bufPtr;
	       skip_this_level (fp, endString);
	       }
	   else
	      {
	      if (prev < level)  /* need to go down a level          */
		if ((level - prev) != 1)  /* not next child of prev      */
		    {
		    *status = MISSING_LEVEL;
		    gedBuf = bufPtr;
		    }
	       tmpNode.line = gedBuf;
	       ged_read_line(fp, &gedBuf, &limit, status, &bytesRead);
	       }
	    if ((level == endLevel) && (!ged_match(ged_get_tag(&tmpNode), endTag)))
		{
#ifdef NON_ANSI
		   fseek(fp, -(bytesRead + 1), 1);
#else
		   fseek(fp, -(bytesRead + 1), SEEK_CUR);
#endif
		done = 1;
			 gedBuf = bufPtr;
		}
		  if ((prev < level) && ((level - prev) != 1))
			 {
			 gedBuf = bufPtr;
			 }
		  else
			 prev = level;
		  bufPtr = gedBuf;
		  bytesRead = 0;
		  }
	   if (*gedBuf != '\0')
		  *gedBuf++ = '\0';
	   if (limit < 1)
		  *status = REC_TOO_LONG;
	   return(gedBuf);
    }
/**/
/********************************************************************
*!name!
*    ged_read_tree()
*!1!
*
*        NAME: ged_read_tree
*
*        DESCRIPTION:
*            reads the GEDCOM file and then makes a tree from it.
*
*        RETURN VALUE:
*            A pointer to the root node of the tree. Status will
*            contain a value telling if EOF was encountered, there
*            are more records, it was the last record, or there is
*            not enough space in gedBuf.
*
*        CALLS: ged_read_rec, ged_to_tree
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
NODE * ged_read_tree(fp, gedBuf, limit, status)
    FILE    *fp;
    byte    *gedBuf;
    unsigned long  limit;
    short   *status;
    {               /*!end!*/
    int stat;

	ged_read_rec(fp, gedBuf, limit, status);
	return(ged_to_tree(&gedBuf, &stat));
    }
NODE * ged_read_partial_tree(fp, gedBuf, limit, status, endLevel, endTag)
    FILE    *fp;
    byte    *gedBuf;
    unsigned long  limit;
    short   *status;
    int      endLevel;
    byte *endTag;
    {               /*!end!*/
    int stat;

	   ged_read_partial_rec(fp, gedBuf, limit, status, endLevel, endTag);
	return(ged_to_tree(&gedBuf, &stat));
    }
/**/
/********************************************************************
*!name!
*    ged_to_tree()
*!1!
*
*        NAME: ged_to_tree
*
*        DESCRIPTION:
*            Given an input buffer containing GEDCOM data, this
*            function will return a tree corresponding to the
*            data. This function does not need a record starting
*            at zero, but will read from the starting level til
*            it gets to a corresponding or less level.
*
*        RETURN VALUE:
*            The root node of the resultant tree, or NULL if
*            invalid GEDCOM is found.
*
*        CALLS: ged_clone_node, ged_find_level, ged_find_line,
*               ged_connect_child, ged_connect_sibling.
*
*        CALLED BY: ged_read_tree
*
*   CALLS: !/see()!
*
*!0!
*SYNOPSIS:
*
*!-1!
********************************************************************/
NODE * ged_to_tree(ged, status)
    byte **ged;
    int   *status;
    {               /*!end!*/
    int level, prev, start;
    NODE *node, *root;

	   start = prev = ged_find_level(ged);
	   node = root = ged_clone_node(ged_find_line(ged, status), NULL, NULL);
	   if (!node)
		  return(NULL);
	   while ((level = ged_find_level(ged)) > start)
		  {

		  /* need to go down a level                                   */
		  if (prev < level)
			 {
			 if ((level - prev) != 1)
				return(NULL);/* not next child of prev         */
			 node = ged_connect_child(node,
				  ged_clone_node(ged_find_line(ged, status), NULL, NULL),
						   -1);
			 node = ged_get_last_sibling(node);
			 if (!node)
				return(NULL);
			 }

		  /* is either a sibling or a sibling of a parent           */
		  else
			 {
			 while (node && (prev-- > level)) /* get to the correct level */
				node = node->parent;
			 node = ged_connect_sibling(node,
				  ged_clone_node(ged_find_line(ged, status), NULL, NULL),
						  -1);
			 if (!node)
				return(NULL);
			 }
		  prev = level;
		  }
	   return(root);
    }

