/*  $Id: text.c,v 1.1 1993/07/13 10:11:40 anjo Exp $
 *  
 *  File	text.c
 *  Part of	ChessBase utilities file format (CBUFF)
 *  Author	Anjo Anjewierden, anjo@swi.psy.uva.nl
 *  Purpose	Printing text line by line
 *  Works with	GNU CC 2.4.5
 *  
 *  Notice	Copyright (c) 1993  Anjo Anjewierden
 *  
 *  History	04/07/93  (Created)
 *  		11/07/93  (Last modified)
 */ 


/*------------------------------------------------------------
 *  Directives
 *------------------------------------------------------------*/

#include "cbuff.h"


/*------------------------------------------------------------
 *  Initialisation
 *------------------------------------------------------------*/

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node newTextBuffer
@deftypefun TextBuffer newTextBuffer (FILE *@var{fd}, int @var{columns})
Creates a new text-buffer associated with file @var{fd} and with a
capacity of @var{columns} characters.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

TextBuffer
newTextBuffer(FILE *fd, int columns)
{ TextBuffer tb;

  tb = alloc(sizeof(struct textbuffer));
  
  tb->file = fd;
  tb->columns = columns;
  tb->maxColumns = columns + 100;
  tb->line = alloc(sizeof(char) * (tb->maxColumns+1));
  tb->current = tb->line;
  tb->count = 0;
  tb->lastSpace = -1;
  
  return tb;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node freeTextBuffer
@deftypefun void freeTextBuffer (TextBuffer @var{tb})
First prints any remaining text in @var{tb} (using @code{flushTextBuffer})
and then reclaims the memory associated with text-buffer @var{tb}.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
freeTextBuffer(TextBuffer tb)
{ if (tb->count > 0)
    flushTextBuffer(tb);
  unalloc(tb->line);
  unalloc(tb);
}


/*------------------------------------------------------------
 *  Adding text
 *------------------------------------------------------------*/

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node stringTextBuffer
@deftypefun void stringTextBuffer (TextBuffer @var{tb}, char *@var{s})
Appends the string @var{s} to the text-buffer @var{tb}.  If this overflows
the text-buffer, the current line will be written to the file associated with
the text-buffer (broken at a white space character).
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
stringTextBuffer(TextBuffer tb, char *s)
{ char *t = tb->current;
  int count = tb->count;

  while (*s)
  { *t = '\0';
    if (count >= tb->columns)
    { if (isspace(*s))			/* No need for special actions */
      { flushTextBuffer(tb);
	count = tb->count;
	t = tb->current;
	tb->lastSpace = -1;
	s++;
	continue;
      }

      if (tb->lastSpace != -1)		/* Break at last space seen */
      { tb->line[tb->lastSpace] = '\0';
	flushTextBuffer(tb);
	strcpy(tb->line, &tb->line[tb->lastSpace+1]);
	count = tb->count = strlen(tb->line);
	t = tb->current = &tb->line[count];
	tb->lastSpace = -1;
	continue;
      }

					/* Line too long */
      *t++ = *s++;
      count++;
      if (count > tb->maxColumns)
      { fprintf(stderr, "** Internal error: Extremely long line (%d chars)\n",
		count);
	exit(1);
      }
      continue;
    }

    if (*s == '\n')
    { flushTextBuffer(tb);
      count = tb->count;
      t = tb->current;
      s++;
      continue;
    }

    if (isspace(*s))
      tb->lastSpace = count;

    if (*s == '\t')
    { *t++ = '\t';
      count = (count+8)/8;
      s++;
      continue;
    }

    *t++ = *s++;
    count++;
  }
  *t = '\0';
  tb->count = count;
  tb->current = t;
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node stringSpaceTextBuffer
@deftypefun void stringSpaceTextBuffer (TextBuffer @var{tb}, char *@var{s})
Similar to @code{stringTextBuffer} but ensures that there is a space
character at the end of @var{tb} before appending the string @var{s}.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
stringSpaceTextBuffer(TextBuffer tb, char *s)
{ if (tb->count > 0 && !isspace(*tb->current))
    stringTextBuffer(tb, " ");
  stringTextBuffer(tb, s);
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node stringNewlineTextBuffer
@deftypefun void stringNewlineTextBuffer (TextBuffer @var{tb}, char *@var{s})
Similar to @code{stringTextBuffer} but ensures that string @var{s}
starts on a new line.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
stringNewlineTextBuffer(TextBuffer tb, char *s)
{ if (tb->count > 0)
    stringTextBuffer(tb, "\n");
  stringTextBuffer(tb, s);
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node stringRawTextBuffer
@deftypefun void stringRawTextBuffer (TextBuffer @var{tb}, char *@var{s})
First flushes the text-buffer (writing any remaining text to the
file) and then appends string @var{s} to the file as well.  This
is useful for text that has to be put on a single line that would
otherwise overflow the line.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
stringRawTextBuffer(TextBuffer tb, char *s)
{ if (tb->count)
    flushTextBuffer(tb);
  fprintf(tb->file, "%s\n", s);
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node formatTextBuffer
@deftypefun void formatTextBuffer (TextBuffer @var{tb}, char *@var{format}, @dots{})
@code{formatTextBuffer} is similar @code{printf} but operates on
text-buffer @var{tb}.  An example is:
@example
@{ TextBuffer tb;
  int count;

  ...
  formatTextBuffer(tb, "Count %d lines", count);
@}
@end example
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
formatTextBuffer(TextBuffer tb, char *format, ...)
{ va_list args;
  char buf[MAX_LINE_SIZE+1];

  va_start(args, format);
  vsprintf(buf, format, args);
  stringTextBuffer(tb, buf);
  va_end(args);
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node formatSpaceTextBuffer
@deftypefun void formatSpaceTextBuffer (TextBuffer @var{tb}, char *@var{format}, @dots{})
Similar to @code{formatTextBuffer} but ensures a space character precedes
the output.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
formatSpaceTextBuffer(TextBuffer tb, char *format, ...)
{ va_list args;
  char buf[MAX_LINE_SIZE+1];

  va_start(args, format);
  vsprintf(buf, format, args);
  stringSpaceTextBuffer(tb, buf);
  va_end(args);
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node formatNewlineTextBuffer
@deftypefun void formatNewlineTextBuffer (TextBuffer @var{tb}, char *@var{format}, @dots{})
Similar to @code{formatTextBuffer} but ensures output starts on a newline.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
formatNewlineTextBuffer(TextBuffer tb, char *format, ...)
{ va_list args;
  char buf[MAX_LINE_SIZE+1];

  va_start(args, format);
  vsprintf(buf, format, args);
  stringNewlineTextBuffer(tb, buf);
  va_end(args);
}


/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node formatRawTextBuffer
@deftypefun void formatRawTextBuffer (TextBuffer @var{tb}, char *@var{format}, @dots{})
Similar to @code{stringRawTextBuffer}, but using @code{printf} style
arguments.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
formatRawTextBuffer(TextBuffer tb, char *format, ...)
{ va_list args;
  char buf[MAX_LINE_SIZE+1];

  va_start(args, format);
  vsprintf(buf, format, args);
  stringRawTextBuffer(tb, buf);
  va_end(args);
}


/*------------------------------------------------------------
 *  Private functions
 *------------------------------------------------------------*/

/* - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
@node flushTextBuffer
@deftypefun void flushTextBuffer (TextBuffer @var{tb})
Flushes the contents of the text-buffer @var{tb} to the associated
file.  This function is not normally used by applications.
@end deftypefun
- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - */

void
flushTextBuffer(TextBuffer tb)
{ fprintf(tb->file, "%s\n", tb->line);
  tb->count = 0;
  tb->current = tb->line;
  tb->line[0] = '\0';
}
