/****************************************************************************\

\****************************************************************************/
#include "fginput.h"
#include "internal.h"


/****************************************************************************\

	INTERNAL FUNCTIONS

\****************************************************************************/
static int  pollbox(const char *);
static void move_right(void);
static int  move_left(void);
static void shift_left(void);
static void shift_right(void);
static void end(void);
static void insert(void);
static void strip_spaces(void);
static void Outblanks(int);
static void setcursor(CURSORTYPE);

static void showchar(int);
static void draw_cursor(void);

extern void (*_print)(char*,int);


/****************************************************************************\

	VARIABLE DECLARATIONS

\****************************************************************************/
static char _TempString[MAXCHARS+1];	//Holds currently editing string. (95)max chars
static char _Blank[2]=" ";
static int  _StringX;
static int  _FieldLen;
static int  _Left;
static int  _Right;
static int  _Top;
static int  _Bottom;
static int  _InCursor=INSCURSOR;

extern int  _Defined;
extern int  _VMode;
extern int  _VType;
extern int  _VLines;


/****************************************************************************\


\****************************************************************************/
/****************************************************************************\

	int input_line(char *line, const char *mask)

\****************************************************************************/
int input_line(char *line, const char *mask)
{

	int test;

	if ((test=setup(line,mask))!=0) return(test);

	/* Display old value highlighted */
	if (strlen(line)>0) {

		if (_VType==GRAPHICS) {
			fg_setcolor(BackHighlightcolor);
			fg_rect(_Left+3,_Left+1+strlen(line)*8,_Top+3,_Bottom-3);
			fg_setcolor(TextHighlightcolor);
			fg_move(_Left+3,_Bottom-3);
		}
		else {
			fg_setattr(TextHighlightcolor,BackHighlightcolor,0);
			fg_locate(_Top,_Left);
		}
		_print(_TempString,strlen(_TempString));

	}
	else
		showstring(CURSOR);


	/* Poll box */
	test=pollbox(mask);

	/* Copy new string to *line if CR pressed */
	if (test!=IN_ESCAPE) {
		strip_spaces();
		strcpy(line,_TempString);
	}
	else
		strcpy(_TempString,line);

	_StringX=0;
	showstring(NOCURSOR);

	return(test);

}

/****************************************************************************\

	int setup(char *line, const char *mask)

			Sets up global variables and draws the input box

\****************************************************************************/
int setup(char *line, const char *mask)
{

	/* Check parameters */
	if (!_Defined) return(IN_NOT_DEFINED);
	if (_VMode==NOTSUPPORTED) return(IN_NOT_SUPPORTED);
	_FieldLen=strlen(mask);
	if (_FieldLen>MAXCHARS) return(IN_MASK_TOO_LONG);
	if (strlen(line)>MAXCHARS) return(IN_BUFFER_TOO_LONG);

	_StringX=0;

	/* Set Text Size */
	if (TextSize!=8&&TextSize!=14&&TextSize!=16) TextSize=8;
	if (_VMode==COLOR16b&&TextSize==8) TextSize=14;
	fg_fontsize(TextSize);

	/* Set coordinates */
	if (_VType==GRAPHICS) {
		_Left  =fg_getxpos();
		_Right =_Left+4+_FieldLen*8;
		_Bottom=fg_getypos();
		_Top   =_Bottom-5-TextSize;
	}
	else {
		fg_where(&_Top,&_Left);
		_Right =_Left+_FieldLen;
		_Bottom=_Top;
	}

	if (_Right>fg_getmaxx()) return(IN_BOX_TOO_BIG);

	memset(_TempString,NULL,MAXCHARS+1);	//Null out _TempString;
	strcpy(_TempString,line);					//Copy current string to _TempString

	/* Draw input box */
	if (_VType==GRAPHICS) {
		fg_setcolor(Backcolor);
		fg_rect(_Left,_Right,_Top,_Bottom);
		fg_setcolor(BorderTcolor);
		fg_move(_Left,_Top);
		fg_draw(_Right,_Top);
		fg_setcolor(BorderRcolor);
		fg_draw(_Right,_Bottom);
		fg_setcolor(BorderBcolor);
		fg_draw(_Left,_Bottom);
		fg_setcolor(BorderLcolor);
		fg_draw(_Left,_Top);
	}
	else {
		fg_setattr(0,Backcolor,0);
		fg_locate(_Top,_Left);
		Outblanks(_FieldLen);
	}

	return(IN_SUCCESS);

}

/****************************************************************************\

	void showstring(int cursor)

			Displays string from _StringX offset to EOS

\****************************************************************************/
void showstring(int cursor)
{

	int   offset=_Left+3+_StringX*8;
	char *ptr=_TempString+_StringX;

	if (_VType==GRAPHICS) {
		fg_setcolor(Backcolor);
		fg_rect(offset,_Right-1,_Top+1,_Bottom-1);

		if (cursor) draw_cursor();

		fg_setcolor(TextNormalcolor);
		fg_move(offset,_Bottom-3);
		_print(ptr,strlen(ptr));

		if (cursor) {
			fg_setcolor(TextHighlightcolor);
			fg_move(offset,_Bottom-3);
			_print(ptr,1);
		}
	}
	else {

		fg_setattr(TextNormalcolor,Backcolor,0);
		fg_locate(_Top,_Left+_StringX);
		Outblanks(_FieldLen-_StringX);

		fg_setcolor(TextNormalcolor);
		fg_locate(_Top,_Left+_StringX);
		_print(ptr,strlen(ptr));
	}

}

/****************************************************************************\

	static void showchar(int highlight)

			Displays _TempString[_StringX] either highlighted or not.

\****************************************************************************/
static void showchar(int highlight)
{

	int fore;
	int offset;

	if (highlight) { fore=TextHighlightcolor; }
	else           { fore=TextNormalcolor;    }

	if (_VType==GRAPHICS) {
		offset=_Left+3+_StringX*8;
		fg_setcolor(Backcolor);
		fg_rect(offset,offset+7,_Top+3,_Bottom-2);
		if (highlight) draw_cursor();
		fg_setcolor(fore);
		fg_move(offset,_Bottom-3);
	}
	else {
		fg_setattr(fore,Backcolor,0);
		fg_locate(_Top,_Left+_StringX);
	}
	_print(_TempString+_StringX,1);

}

/****************************************************************************\

	static void draw_cursor(void)

			Draws the graphics mode cursor.

\****************************************************************************/
static void draw_cursor(void)
{

	fg_setcolor(Cursorcolor);
	if (_InCursor==OVRCURSOR) fg_rect(_Left+3+_StringX*8,_Left+_StringX*8+8,_Top+3,_Bottom-2);
	else fg_rect(_Left+3+_StringX*8,_Left+_StringX*8+8, _Bottom-3,_Bottom-2);

}

/****************************************************************************\

	static int pollbox(const char *mask)

\****************************************************************************/
static int pollbox(const char *mask)
{

	int ch;
	_StringX=0;

	/* Read initial key press */
	fg_locate(_Top,_Left+_StringX);
	ch=wait_key_masked(mask[_StringX]);

	if (ch>31&&ch<256)	//Was key a non-editing key?
		memset(_TempString,NULL,MAXCHARS+1);
	showstring(CURSOR);

	do {

		switch(ch) {
			case ESC:		return(IN_ESCAPE);
			case CR:    	return(IN_SUCCESS);
			case BS:    	if (move_left()) shift_left();				break;
			case RARROW:	move_right();										break;
			case LARROW:	move_left();										break;
			case UARROW:	if (Polled_Mode) return(IN_UARROW);			break;
			case DARROW:	if (Polled_Mode) return(IN_DARROW);			break;
			case TAB:		if (Polled_Mode) return(IN_TAB);				break;
			case SHIFT_TAB:if (Polled_Mode) return(IN_SHIFT_TAB);		break;
			case DEL:		shift_left();										break;
			case HOME:		showchar(0); _StringX=0; showchar(1);		break;
			case END:		end();												break;
			case INS:		insert();											break;
			default:
				if (_InCursor==INSCURSOR) shift_right();
				_TempString[_StringX]=ch;
				move_right();
				break;
		}

		fg_locate(_Top,_Left+_StringX);
		ch=wait_key_masked(mask[_StringX]);

	}while(1);

}

/****************************************************************************\

	static void move_right(void)

			Moves the cursor 1 position to the right if possible.

\****************************************************************************/
static void move_right(void)
{

	if (_TempString[_StringX]==NULL)
		_TempString[_StringX]=' ';

	if (_StringX+1>=_FieldLen) return;
	showchar(0);	//Unhighlight last position
	_StringX++;
	showchar(1);	//Highlight new position

}

/****************************************************************************\

	static int move_left(void)

			Moves the cursor 1 position to the left if possible.

\****************************************************************************/
static int move_left(void)
{

	if (_StringX-1<0) return (0); //Couldn't move left

	showchar(0);	//Unhighlight last position
	_StringX--;
	showchar(1);   //Highlight new position
	return(1);		//Could move left

}

/****************************************************************************\

	static void shift_left(void)

			Shifts the string from _StringX to the left 1 position

\****************************************************************************/
static void shift_left(void)
{

	register int x=_StringX;
	while(_TempString[x]!=NULL) {
		_TempString[x]=_TempString[x+1];
		x++;
	}

	memset(_TempString+x,NULL,_FieldLen-x);
	showstring(CURSOR);

}

/****************************************************************************\

	static void shift_right(void)

			Shifts the string from _StringX to the right 1 position

\****************************************************************************/
static void shift_right(void)
{

	register int x;
	for(x=_FieldLen-1; x>_StringX; x--)
		_TempString[x]=_TempString[x-1];
	showstring(CURSOR);

}

/****************************************************************************\

	static void end(void)

			Moves the cursor to the end of the string

\****************************************************************************/
static void end(void)
{

	strip_spaces();								//Remove trailing spaces
	showchar(0);									//Unhighlight old character
	_StringX=strlen(_TempString);				//Move cursor
	if (_StringX==_FieldLen)					//Are we at the end of the box?
		_StringX--;									//then move back one
	showchar(1);									//Highlight new character

}

/****************************************************************************\

	static void insert(void)

			Toggles between inserting and overstriking

\****************************************************************************/
static void insert(void)
{

	if (_InCursor==INSCURSOR) {
		_InCursor=OVRCURSOR;
		if (_VType==TEXT) setcursor(CURSOR_SOLID);
		else showchar(1);
	}
	else {
		_InCursor=INSCURSOR;
		if (_VType==TEXT) setcursor(CURSOR_NORMAL);
		else showchar(1);
	}

}

/****************************************************************************\

	static void strip_spaces(void)

			Removes trailing spaces from _TempString[]

\****************************************************************************/
static void strip_spaces(void)
{

	/* Strip trailing spaces */
	int space=-1;
	register int x=0;

	while (_TempString[x]!=NULL) {
		if (_TempString[x]==' '&&space<0)							space=x;
		else if (_TempString[x]!=NULL&&_TempString[x]!=' ')	space=-1;
		x++;
	}
	if (space>=0) _TempString[space]=NULL;

}

/****************************************************************************\

	static void Outblanks(int number)

			Outputs number of spaces to screen.

\****************************************************************************/
static void Outblanks(int number)
{
	while (number-->0) _print(_Blank,1);
}

/****************************************************************************\

	static void setcursor(CURSORTYPE type)

			Changes the text mode cursor form.

\****************************************************************************/
static void setcursor(CURSORTYPE type)
{

	int s,e;

	switch (type) {

		case CURSOR_NONE:
			s=0x20,e=0;
			break;
		case CURSOR_SOLID:
			if (_VLines==43) 	s=0,e=8;
			else              s=0,e=_VType==TEXT ? 12:7;
			break;
		default:
			if (_VType==TEXT)	s=11,e=12;
			else              s=6,e=7;
			break;

	}//switch cursortype

	_CH=s;
	_CL=e;
	_AH=1;
	asm INT 0x10;

}

