// JET.CPP
// Copyright (C) 1993, BIGGUN International
// Sample UPJOY Driven(TM) program demonstrating how to interpret
// UPJOY messages as well as how button combinations work
// Updated: 12/01/93
// Tab width = 4

#include <windows.h>
#include "jet.h"
#include "upjoymsg.h"

#define AXIS_COUNT		4
#define SWITCHED_OFF	0xff
#define ALL_AXES_OFF	0xffffffff

#define CANNON			0

#define TRIGGER       	1
#define TARGET_SELECT	2
#define WEAPON_SELECT	3
#define AIR_BRAKES		4 

#define BUTTON_COUNT	8

#define	DOWN	  		1
#define UP        		0

#define START			1
#define STOP			0

#define VIEW_FRONT		5
#define VIEW_RIGHT		6
#define WATCH_YOUR_SIX	7
#define VIEW_LEFT		8


typedef union 
	{
	DWORD PackedPositions;
	BYTE Position[ AXIS_COUNT ];
	} JOYPOS;


class JOY
	{
	HWND hWnd;
	UINT SelectedWeapon;
	BOOL ButtonStates[ BUTTON_COUNT + 1 ];	// button IDs are 1 based

	void TargetSelect();
	void ViewSelect(UINT View, BOOL ViewButtonDown);
	void WeaponSelect();
	void CannonSelect();
    void Fire(BOOL StartFire);
    void DropFlares();
	void ExtendAirbrakes();
	void RetrieveAirbrakes();

	public:
	void DispatchPosition(DWORD PackedPositions);
	void DispatchButtonUp(UINT JoyButtonID);
	void DispatchButtonDown(UINT JoyButtonID);
	void SetAppWindow(HWND AppWnd) { hWnd = AppWnd; }
	} Joy;



extern "C" LRESULT WINAPI _export WndProc(HWND hWnd, UINT message,
	WPARAM wParam, LPARAM lParam);

#pragma argsused
int PASCAL WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance,
	LPSTR lpszCmdLine, int nCmdShow)
	{
	static char szAppName[] = "JET";
	WNDCLASS    wndclass;

	// Based on Charles Petzold's HEXCALC program from Programming Windows
	if (!hPrevInstance)
		{
		wndclass.style          = CS_HREDRAW | CS_VREDRAW;
		wndclass.lpfnWndProc    = WndProc;
    	wndclass.cbClsExtra     = 0;
        wndclass.cbWndExtra     = DLGWINDOWEXTRA;
        wndclass.hInstance      = hInstance;
        wndclass.hIcon          = LoadIcon (hInstance, szAppName);
        wndclass.hCursor        = LoadCursor (NULL, IDC_ARROW);
		wndclass.hbrBackground  = (HBRUSH) (COLOR_WINDOW + 1);
        wndclass.lpszMenuName   = NULL;
        wndclass.lpszClassName  = szAppName;

        RegisterClass (&wndclass);
    	}

	HWND hWnd = CreateDialog(hInstance, szAppName, 0, NULL);
	if (hWnd == NULL)
		return 0;

	Joy.SetAppWindow(hWnd);

	ShowWindow(hWnd, nCmdShow);

	MSG msg;

	while (GetMessage(&msg, NULL, 0, 0))
    	{
        TranslateMessage(&msg);
        DispatchMessage(&msg);
		}

	return msg.wParam;
	}


void JOY::DispatchPosition(DWORD PackedPositions)
	{
	// PackedPosition contains four axis positions. X1 is in the least
	// significant byte and Y2 is in the highest byte -- 0xY2X2Y1X1.
	// If an axis has a value of 0xff it is OFF, else it has a value
	// in the range 0 through 100.

	// NOTE: UPJOY uses the Cartesian XY coordinate system i.e.
	// Stick right (X1) and rudder right (X2) result in higher readings.
	// Stick forward (Y1) and throttle open (Y2) result in higher
	// readings. This differs from game adapter hardware logic.

	// For a digital-style game like PACMAN, an axis is defined as centered
	// when: 63 > AXIS > 37. This leaves a Dead Zone of 25%.
	// A player can adjust stick sensitivity to his or her liking via Mapper.

	// For a flight simulator, space simulator, or auto racing game,
	// THERE IS NO SUCH THING AS A DEAD ZONE. See the UPJOY online help
	// for details.

	static JOYPOS Yesterday = { ALL_AXES_OFF };	// Initially we assume no axes
												// are available
	static char AxisNames[][ 3 ] = { "X1", "Y1", "X2", "Y2" };

	static char ScratchBuf[ 7 ];

	JOYPOS Today;

	Today.PackedPositions = PackedPositions;

	UINT CurrentPosition;

    // Loop to unpack the four packed axes
	for (int AxIdx = 0; AxIdx < AXIS_COUNT; AxIdx++)
		{
		CurrentPosition = Today.Position[ AxIdx ];

		if (CurrentPosition != Yesterday.Position[ AxIdx ])	// New position?
			{
			HWND hPosition = GetDlgItem(hWnd, AxIdx + IDT_AXIS_DISPLAY);

			if (CurrentPosition == SWITCHED_OFF)
				{
				// When here, the axis was probably unplugged because
				// it was previously valid.
				wsprintf(ScratchBuf, "%s:OFF", (LPSTR) AxisNames[ AxIdx ]);
				SetDlgItemText(hWnd, AxIdx + IDT_AXIS_DISPLAY, ScratchBuf);
				EnableWindow(hPosition, FALSE);	// Gray the text
				}
			else
				{
				if (Yesterday.Position[ AxIdx ] == SWITCHED_OFF)
					{
					// We've just received news that we can start
					// using this axis again.
					EnableWindow(hPosition, TRUE);	// Ungray text
					}

				wsprintf(ScratchBuf, "%s:%d", (LPSTR) AxisNames[ AxIdx ],
						CurrentPosition);
				SetDlgItemText(hWnd, AxIdx + IDT_AXIS_DISPLAY, ScratchBuf);
				}
			}
		}
	Yesterday = Today;	// Save reference for future use
	}


void JOY::TargetSelect()
	{
	static char* Targets[] = { "MIG-29", "SU-27", "Window", "NO TARGET" };
	static UINT SelectedTarget;

	if (SelectedTarget > 3)
		SelectedTarget = 0;

	SetDlgItemText(hWnd, IDT_SELECTED_TARGET, (LPCSTR) Targets[ SelectedTarget ]);

	SelectedTarget++;
	}



void JOY::ViewSelect(UINT View, BOOL ViewButtonDown)
	{
	static char* Views[] = { "45 Up", "Right", "Rear", "Left" };
	if (ViewButtonDown)
		{
		SetDlgItemText(hWnd, IDT_SELECTED_VIEW, (LPCSTR) Views[ View - VIEW_FRONT ]);
		}
	else
		{
		SetDlgItemText(hWnd, IDT_SELECTED_VIEW, (LPCSTR) "Front");
		}
	}


void JOY::WeaponSelect()
	{
	static char* Weapons[] = { "Laser", "Phaser", "Cannon" };

	if (SelectedWeapon > 2)
		SelectedWeapon = 0;

	SetDlgItemText(hWnd, IDT_SELECTED_WEAPON, (LPCSTR) Weapons[ SelectedWeapon]);

	SelectedWeapon++;
	}



void JOY::CannonSelect()
	{
	SetDlgItemText(hWnd, IDT_SELECTED_WEAPON, (LPCSTR) "Cannon");
	SelectedWeapon = CANNON;
	}



void JOY::Fire(BOOL StartFire)
	{
	CheckDlgButton(hWnd, IDB_FIRING, StartFire);
    }



void JOY::DropFlares()
	{
	FlashWindow(hWnd, TRUE);
	}



void JOY::ExtendAirbrakes()
	{
	CheckDlgButton(hWnd, IDB_AIRBRAKES, TRUE);
	}



void JOY::RetrieveAirbrakes()
	{
	CheckDlgButton(hWnd, IDB_AIRBRAKES, FALSE);
	}



void JOY::DispatchButtonDown(UINT JoyButtonID)
	{
	// JET Ignores button IDs above 8, but see online help, UPJOY.HLP for
	// information on using buttons 9 through 16
	if (JoyButtonID > BUTTON_COUNT)
		return;

    ButtonStates[ JoyButtonID ] = DOWN;

	if (JoyButtonID == TRIGGER)
		{
		if (ButtonStates[ TARGET_SELECT ] == DOWN)
			DropFlares();
		else if (ButtonStates[ WEAPON_SELECT ] == DOWN)
        	{
			CannonSelect();	// Guns Quick!
			Fire(START);
            }
		else if (ButtonStates[ AIR_BRAKES ] == DOWN)
			RetrieveAirbrakes();
		else
			Fire(START);
		}
	else
    	{
		switch(JoyButtonID)
			{
			case TARGET_SELECT:
				TargetSelect();
				break;

			case WEAPON_SELECT:
				WeaponSelect();
				break;

			case AIR_BRAKES:
				ExtendAirbrakes();
				break;

			case VIEW_FRONT:
			case VIEW_RIGHT:
			case WATCH_YOUR_SIX:
			case VIEW_LEFT:
				ViewSelect(JoyButtonID, DOWN);
				break;

			default:
				break;
			}
		}
	HWND hButton = GetDlgItem (hWnd, JoyButtonID + IDB_BUTTONS);

	if (hButton != NULL)
		SendMessage(hButton, BM_SETSTATE, TRUE, 0);
	}



void JOY::DispatchButtonUp(UINT JoyButtonID)
	{
	// JET Ignores button IDs above 8, but see online help, UPJOY.HLP for
	// information on using buttons 9 through 16
	if (JoyButtonID > BUTTON_COUNT)
    	return;

	ButtonStates[ JoyButtonID ] = UP;
	if (JoyButtonID == TRIGGER)
		Fire(STOP);					// Trigger was released
	else if (JoyButtonID >= VIEW_FRONT && JoyButtonID <= VIEW_LEFT)
		ViewSelect(JoyButtonID, UP);// Coolly is centered, so select front view

	HWND hButton = GetDlgItem (hWnd, JoyButtonID + IDB_BUTTONS);
	if (hButton != NULL)
		SendMessage(hButton, BM_SETSTATE, FALSE, 0);
	}



LRESULT WINAPI _export WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
	{
	static HDRVR DriversLicense;

	switch (message)
		{
		case JOYPOSITION:
			// wParam is undefined, and reserved.
			// lParam contains four packed axis positions.
			Joy.DispatchPosition(lParam);
			return 0;

		case JOYBUTTONDOWN:
			// wParam is undefined, and reserved.
		  	// The HIWORD of lParam is undefined and reserved -- DON'T use it
			// in calculations.
			// LOWORD(lParam) contains the Button ID
			Joy.DispatchButtonDown(LOWORD(lParam));
			return 0;

		case JOYBUTTONUP:
			// wParam is undefined, and reserved
		  	// The HIWORD of lParam is reserved for future use -- the value is
			// undefined, so DON'T use it.
			// LOWORD(lParam) contains the Button ID
			Joy.DispatchButtonUp(LOWORD(lParam));
			return 0;

		case WM_CREATE:
			// UPJOY is installed in the [Drivers] section of SYSTEM.INI
			// Example:
			// [Drivers]
			// UPJOY=UPJOY.DRV
			// other=other.drv
			// etc...
			// The NULL in OpenDriver refers to the installable [Drivers] section
			// The HIWORD(lParam) in OpenDriver must be set to 0 for spec version
			// coupling
			for ( ;; )
				{
            	DriversLicense = OpenDriver("UPJOY", NULL, MAKELPARAM(hWnd, 0));
				// See online help UPJOY.HLP for a discussion of parallel keyboard
				// and joystick support.

				// No need to check DriversLicense's validity here -- that is done
				// just before the CloseDriver() call. If DriversLicense is NULL
				// a game simply won't receive UPJOY messages. Don't interrogate
				// a player when DriversLicense is NULL. If (!DriversLicense)
				// it means that the user won't be playing with a joystick.

				// The exception to the rule is a game that only works with a
				// joystick (joystick required). Such a game should display a 
				// message that it couldn't get an UPJOY DriversLicense and then
				// Retry or Quit

				if (DriversLicense != NULL)
					return 0;	// JET got a DriversLicense

				if (MessageBox(hWnd, "JET couldn't get an UPJOY DriversLicense", NULL,
					MB_RETRYCANCEL | MB_ICONSTOP) != IDRETRY)
					{
					return -1L;	// Stop window creation
					}
				// else Retry
				}


		case WM_ENDSESSION:
			if (wParam == FALSE)
				break;
			// else wParam = TRUE which means this session is ending so fall through
		case WM_DESTROY:
			if (DriversLicense != NULL)
				{
				// Destroy DriversLicense before window is destroyed
				CloseDriver(DriversLicense, 0L, 0L);
				DriversLicense = NULL;
				}
			PostQuitMessage(0);
			return 0;
		}
	 return DefWindowProc(hWnd, message, wParam, lParam);
     }


