
/* file RIP.CPP */

/********************************************************
 *                                                      *
 *            C++ interface for QRIP/TSR                *
 *              by Shane Hathaway, 1994                 *
 *                                                      *
 *       Absolutely public domain - modify and          *
 *         distribute this C++ module freely!           *
 *                                                      *
 ********************************************************/

#ifndef _RIP_H_
#include "rip.h"
#endif


qrip::qrip(void)
{
  call_qrip(0xf0);
  if (regs.x.ax == 0x9142 && regs.x.dx != 0)
    qrip_installed = 1;
  else qrip_installed = 0;
}


unsigned short qrip::call_qrip(char funcno)
{
  if (qrip_installed || funcno >= 0xf0)
  {
    regs.h.ah = 0xac;
    regs.h.al = funcno;
    regs.x.di = 0x1092;
    int86(0x2f, &regs, &regs);
  }
  return regs.x.bx;
}


void rip_blockinfo::fillblock(union REGS regs)
{
  protocol = regs.h.ah & 0x07;
  uploading = (regs.h.ah & 0x08) ? 1 : 0;
  filetype = regs.h.al;
  filename = (char far *)MK_FP(regs.x.dx, regs.x.bx);
}


void rip_module::set_qrip_str(char *s, int fn)
{
  int l;
  unsigned short sg, fs;
  char far *qrip_str;

  call_qrip(fn ? 0x08 : 0x07);
  sg = regs.x.dx;
  fs = regs.x.bx;

  movedata(FP_SEG(s), FP_OFF(s), sg, fs, 80);
  qrip_str = (char far *)MK_FP(sg, fs);
  qrip_str[79] = 0;
  l = strlen(s) - 1;
  if (s[l] != '\\') 
  {
    qrip_str[min(78, l)] = '\\';
    qrip_str[min(79, l)] = 0;
  }
}


unsigned char rip_module::ret_key(void)
{
  unsigned c = regs.x.ax;

  str_waiting = regs.x.bx;
  if (c >= 0x0100)
  {
    cue_change(c - 0x0100);
    c = 0;
  }
  return (c & 0xff);
}


void rip_module::outs(char far *s)
{
  // This member function isn't inline because it's likely to be used often.

  regs.x.bx = FP_OFF(s);
  regs.x.dx = FP_SEG(s);
  call_qrip(5);
}


unsigned char rip_module::outs1(char far *s)
{
  regs.x.bx = FP_OFF(s);
  regs.x.dx = FP_SEG(s);
  call_qrip(10);
  return ret_key();
}


char far *rip_module::getkeystr(int &len)
{
  call_qrip(0x0c);
  switch(regs.x.ax)
  {
  case 0: // Nothing to send.
  case 2: // Special code still waiting.
    len = 0;
    return NULL;
  case 1:
    len = regs.x.cx;
    return (char far *)MK_FP(regs.x.dx, regs.x.bx);
  }
}


void rip_module::protect_write(int x, int y, char *s)
{
  regs.x.cx = (x & 0xff) | (y << 8);
  regs.x.bx = FP_OFF(s);
  regs.x.dx = FP_SEG(s);
  call_qrip(0x0e);
}


void rip_module::protect_scroll(int x1, int y1, int x2, int y2, int lines)
{
  regs.x.bx = (x1 & 0xff) | (y1 << 8);
  regs.x.dx = (x2 & 0xff) | (y2 << 8);
  regs.x.cx = lines;
  call_qrip(0x0f);
}


void rip_modes::cue_change(int i)
{
  if (i < 0x10)
  {
    switch(i)
    {
    case 0x00:
      // ignore - just wait until a key is pressed.
      break;
    case 0x01:
      middlebutton();
      break;
    case 0x02:
      cue_blockmode();
      break;
    case 0x03:
      rightbutton();
      break;
    case 0x04:
      statbar = 1;
      statbar_on();
      break;
    case 0x05:
      statbar = 0;
      statbar_off();
      break;
    case 0x06:
      vt102 = 1;
      vt102_on();
      break;
    case 0x07:
      vt102 = 0;
      vt102_off();
      break;
    case 0x08:
      doorway = 1;
      doorway_on();
      break;
    case 0x09:
      doorway = 0;
      doorway_off();
      break;
    case 0x0a:
      hotkeys = 1;
      hotkeys_on();
      break;
    case 0x0b:
      hotkeys = 0;
      hotkeys_off();
      break;
    case 0x0c:
      tabkey = 1;
      tabkey_on();
      break;
    case 0x0d:
      tabkey = 0;
      tabkey_off();
      break;
    case 0x0e:
      do_xoff();
      break;
    case 0x0f:
      do_xon();
      break;
    }
  }
  else if (i < 0x1a)
    execute_app(i - 0x10);
}


int farcharbuffer::setup_buffer(int sz)
{
  buf = (char far *)farrealloc(buf, sz + 1);
  if (buf)
    size = sz;
  else
    size = 0;
  head = 0;
  return (buf ? 1 : 0);
}


void farcharbuffer::push(char c)
{
  int i;

  if (!size) return;
  if (head + 1 >= size) return;
  buf[head++] = c;
}


char far *farcharbuffer::popstr(void)
{
  int c;

  if (!size || !head) return NULL;
  buf[head] = 0;
  head = 0;
  return buf;
}


char farcharbuffer::popchr(void)
{
  // farcharbuffer is optimized for popping a whole string at a time.
  // This function has to move the entire buffer each time a character
  // is popped this way.
  // Someone may wish to improve this.

  int c, i;

  // Another problem: can't use memmove() because it doesn't use far
  // pointers in small data memory models.  Change a few lines here
  // and use memmove() instead if you intend to always use a large data
  // model.

  if (head < 1) return 0;
  c = *buf;
  for (i = 1; i < head; i++) buf[i - 1] = buf[i];
  head--;
  return c;
}


void buffered_rip::outstr(char *s)
{
  while (*s)
    outchr(*s++);
}


void buffered_rip::outchr(char c)
{
  if (chrsout.full())
    update(0);
  chrsout.push(c);
}


void buffered_rip::update(int minchrs)
{
  int w, c;

  if (!qrip_resident()) return; // Don't bother!
  c = -1;
  w = (chrsout.holding() > minchrs);
  if (w)
  {
    if (kbhit())
      c = rip_module::outs1(chrsout.popstr());
    else
      rip_module::outs(chrsout.popstr());
  }
  if (c < 0) c = rip_module::getkey();
  if (c > 0) keysin.push(c);
  if (strwait())
  {
    int len;
    char far *s;
    
    len = 0;
    s = rip_module::getkeystr(len);
    while (len)
    {
      keysin.push(*s++);
      len--;
    }
  }
  if (lowlevel_kbhit())
  {
    unsigned short cc;

    c = lowlevel_getch();
    cc = c;
    if (!cc)
    {
      c = lowlevel_getch();
      cc = ((unsigned short)c) << 8;
      switch(cc)
      {
      case UPKEY:
      case DOWNKEY:
      case RIGHTKEY:
      case LEFTKEY:
      case HOMEKEY:
      case ENDKEY:
        if (query_hotkeys())
          rip_module::sendkey(cc);
        else goto no_send_qrip;
        break;
      case S_TABKEY:
        if (query_hotkeys() || query_tabkey())
          rip_module::sendkey(cc);
        else goto no_send_qrip;
        break;
      default:
      no_send_qrip:
        keysin.push(0);
        keysin.push(c);
        break;
      }
    }
    else switch(c)
    {
    case TABKEY:
      if (query_hotkeys() || query_tabkey())
      {
        rip_module::sendkey(cc);
        break;
      }
      // no break;
    default:
      if (query_hotkeys())
        rip_module::sendkey(cc);
      else keysin.push(c);
    }
  }
}


term_emu::term_emu(void)
{
  qrip q;

  rdetect = q.qrip_resident();
}


void term_emu::outstr(char *s)
{
  if (rdetect) buffered_rip::outstr(s);
  else ansi_outstr(s);
}


void term_emu::outchr(char c)
{
  if (rdetect) buffered_rip::outchr(c);
  else ansi_outchr(c);
}


void term_emu::clrscrn(void)
{
  if (rdetect) buffered_rip::graphics_on();
  else ansi_clrscrn();
}


#ifdef TESTMODE
  // #define TESTMODE before compiling to take a simple look
  // at how the basic outchr() and getch() functions work.

int main(int argc, char *argv[])
{
  int c;

  term_emu t;

  t.clrscrn();
  t.outstr("Type in anything you like.  If QRIP/TSR is installed,\r\n"
           "output will be piped through the TSR and you'll be able to see any\r\n"
           "RIP graphics you type in.  Push Ctrl-Z to end.\r\n");

  c = 0;
  while (c != 0x1a)
  {
    c = 0;
    if (t.kbhit()) c = t.getch();
    if (c) t.outchr(c);
    t.update(0);
  }

  t.graphics_off();
  t.hangup();

  return 0;
}

#endif
