/**************************************************************************
*  File:        3DOTS.CPP          Copyright (c) 1994 by Matthew Hildebrand
*
*  Purpose:     Demonstrate TCG 0.00's ability to display many colours.
**************************************************************************/

#include <conio.h>
#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include "..\include\tcg.h"
#include "..\include\vcoord.h"

#define QUIT_IF_ESCAPE      if (getKey() == 27)  \
                              exit(EXIT_SUCCESS)

void setup(void);
void manyColoursDemo(void);
int getKey(void);



/**************************************************************************
*  Function:    main
*
*  Purpose:     Program entry point.
*
*  Entry:       argc = Number of parameters - 1.
*               argv = Array containing the actual parameters.
*
*  Exit:        N/A
**************************************************************************/

void main(void)
{
  int ch;

  //*** Warn users about program execution time
  clrscr();
  printf("NOTICE:\n\n"
         "Due to the large number of floating-point calculations performed\n"
         "by this program, it can take up to several minutes to execute\n"
         "on a machine without a math coprocessor.  Note that pressing a\n"
         "key during program execution will abort.\n\n"
         "Press Y to continue, or any other key to abort...");
  ch = getKey();
  if (ch!='y' && ch!='Y')
  {
    printf("\n\n");
    exit(EXIT_SUCCESS);
  }

  //*** Set things up
  setup();

  //*** Run the demo
  manyColoursDemo();
}



/**************************************************************************
*  Function:    setup
*
*  Purpose:     Load a driver, and do other such setup things.
*
*  Entry:       N/A
*
*  Exit:        If an error occurs, the program will exit from within this
*               function.
**************************************************************************/

void setup(void)
{
  char *filename;

  //*** Load a suitable graphics driver
  filename = detectGraph();
  if (filename == NULL)
  {
    printf("No truecolour video mode detected; aborting.\n\n"
           "Try installing a VESA BIOS extensions TSR, such as UNIVESA\n"
           "(in TCG's VESABE directory) or the TSR (if any) which came\n"
           "with your video card.  VESA BIOS extensions version 1.1 or\n"
           "higher are required.\n\n");
    exit(EXIT_FAILURE);
  }
  switch (loadGraphDriver(filename))
  {
    case TCG_SUCCESS:                       // success
      break;
    case TCG_OPEN_ERR:                      // file not found
      printf("Error opening driver file %s; aborting.\n\n"
             "Please ensure that the TCGDRIVERS environment variable has been set to\n"
             "point to the TCG drivers directory; refer to TCGDEMO.DOC for details.\n\n",
             filename);
      exit(EXIT_FAILURE);
    case TCG_FORMAT_ERR:                    // not a TCG driver
      printf("Format error processing file %s; aborting.\n\n", filename);
      exit(EXIT_FAILURE);
    case TCG_ALLOC_ERR:                     // out of memory
      printf("Insufficient memory; aborting.\n\n");
      exit(EXIT_FAILURE);
    case TCG_FILE_ERR:                      // file read error
      printf("Error reading file %s; aborting.\n\n", filename);
      exit(EXIT_FAILURE);
    default:                                // who knows?
      printf("Unknown error loading driver %s; aborting.\n\n", filename);
      exit(EXIT_FAILURE);
  }
  atexit(unloadGraphDriver);                // unload driver on exit

  //*** Switch to graphics mode
  if (!initGraphics())                      // initialize graphics mode
  {
    printf("Unable to initialize graphics hardware; aborting.\n\n");
    exit(EXIT_FAILURE);
  }
  atexit(deInitGraphics);                   // restore text mode

  //*** Seed the random number generator
  randomize();
}



/**************************************************************************
*  Function:    manyColoursDemo
*
*  Purpose:     Demonstrate the number of colours available with truecolour
*               graphics.
*
*  Entry:       N/A
*
*  Exit:        N/A
**************************************************************************/

void manyColoursDemo(void)
{
  VirtualCoord virtScreen;
  int rCenterX, rCenterY, gCenterX, gCenterY, bCenterX, bCenterY;
  float rDist, gDist, bDist;
  float rVal, gVal, bVal;
  Colour colour;
  int countX, countY;

  //*** Determine the coordinates of seed points
  virtScreen.virtParams(100, 100);          // specify virtual size
  virtScreen.realParams(OUTMAXX, OUTMAXY);  // specify real size
  virtScreen.realCoords(50, 20, (unsigned*)&rCenterX, (unsigned*)&rCenterY);
  virtScreen.realCoords(20, 80, (unsigned*)&gCenterX, (unsigned*)&gCenterY);
  virtScreen.realCoords(80, 80, (unsigned*)&bCenterX, (unsigned*)&bCenterY);

  //*** Draw the pixels, one at a time
  clearGraphics(30, 30, 30);
  for (countY=0; countY<=OUTMAXY&&!kbhit(); countY++)
    for (countX=0; countX<=OUTMAXX; countX++)
    {
      rDist = hypot(fabs(countX-rCenterX), fabs(countY-rCenterY));
      if (rDist != 0.0)
      {
        rVal = 25500.0 / rDist;
        if (rVal > 255.0)
          rVal = 255.0;
      }
      else
        rVal = 255.0;
      gDist = hypot(fabs(countX-gCenterX), fabs(countY-gCenterY));
      if (gDist != 0.0)
      {
        gVal = 25500.0 / gDist;
        if (gVal > 255.0)
          gVal = 255.0;
      }
      else
        gVal = 255.0;
      bDist = hypot(fabs(countX-bCenterX), fabs(countY-bCenterY));
      if (bDist != 0.0)
      {
        bVal = 25500.0 / bDist;
        if (bVal > 255.0)
          bVal = 255.0;
      }
      else
        bVal = 255.0;

      colour.r = (unsigned char)rVal;
      colour.g = (unsigned char)gVal;
      colour.b = (unsigned char)bVal;
      putPixel(countX, countY, &colour);
    }

  QUIT_IF_ESCAPE;
}



/**************************************************************************
*  Function:    getKey
*
*  Purpose:     Get a character from the keyboard.  This function differs
*               from getch() in that it can correctly handle extended keys
*               such as the function keys.
*
*  Entry:       N/A
*
*  Exit:        Returns the keycode of the key pressed.
**************************************************************************/

int getKey(void)
{
  int ch;

  ch = getch();                             // get a byte
  if (!(ch & 0xFF))                         // is it zero?
    ch = getch() << 8;                      // yes, get another byte
  return (ch);                              // return the keycode
}
