/* Copyright (C) 1993, 1994 Aladdin Enterprises.  All rights reserved.
  
  This file is part of Aladdin Ghostscript.
  
  Aladdin Ghostscript is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Aladdin Ghostscript Free Public
  License (the "License") for full details.
  
  Every copy of Aladdin Ghostscript must include a copy of the License,
  normally in a plain ASCII text file named PUBLIC.  The License grants you
  the right to copy, modify and redistribute Aladdin Ghostscript, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notice and this notice be preserved on
  all copies.
*/

/* gsdparam.c */
/* Default device parameters for Ghostscript library */
#include "memory_.h"			/* for memcpy */
#include "gx.h"
#include "gserrors.h"
#include "gsparam.h"
#include "gxdevice.h"
#include "gxfixed.h"

/* Get the device parameters. */
int
gs_getdeviceparams(gx_device *dev, gs_param_list *plist)
{	gx_device_set_procs(dev);
	fill_dev_proc(dev, get_params, gx_default_get_params);
	return (*dev_proc(dev, get_params))(dev, plist);
}

/* Get standard parameters. */
int
gx_default_get_params(gx_device *dev, gs_param_list *plist)
{
	int code;

	/* Standard page device parameters: */

	gs_param_string dns;
	float HWResolution[2];
	gs_param_float_array psa, ibba, hwra, ma;
#define set_param_array(a, d, s)\
  (a.data = d, a.size = s, a.persistent = false);

	/* Non-standard parameters: */

	int colors = dev->color_info.num_components;
	int depth = dev->color_info.depth;
	int GrayValues = dev->color_info.max_gray + 1;
	int HWSize[2];
	  gs_param_int_array hwsa;
	float InitialMatrix[6];
	  gs_param_float_array ima;

	/* Fill in page device parameters. */

	param_string_from_string(dns, dev->dname);
	HWResolution[0] = dev->x_pixels_per_inch;
	HWResolution[1] = dev->y_pixels_per_inch;
	set_param_array(hwra, HWResolution, 2);
	set_param_array(psa, dev->page.page_size, 2);
	set_param_array(ibba, dev->page.imaging_bbox, 4);
	set_param_array(ma, dev->page.margins, 2);

	/* Fill in non-standard parameters. */

	HWSize[0] = dev->width;
	HWSize[1] = dev->height;
	set_param_array(hwsa, HWSize, 2);
	fill_dev_proc(dev, get_initial_matrix, gx_default_get_initial_matrix);
	{	gs_matrix mat;
		(*dev_proc(dev, get_initial_matrix))(dev, &mat);
		InitialMatrix[0] = mat.xx;
		InitialMatrix[1] = mat.xy;
		InitialMatrix[2] = mat.yx;
		InitialMatrix[3] = mat.yy;
		InitialMatrix[4] = mat.tx;
		InitialMatrix[5] = mat.ty;
	}
	set_param_array(ima, InitialMatrix, 6);

	/* Transmit the values. */

	if (
			/* Standard parameters */

	     (code = param_write_name(plist, "OutputDevice", &dns)) < 0 ||
	     (code = param_write_float_array(plist, "HWResolution", &hwra)) < 0 ||
	     (code = param_write_float_array(plist, "PageSize", &psa)) < 0 ||
	     (code = (dev->page.imaging_bbox_set ?
		      param_write_float_array(plist, "ImagingBBox", &ibba) :
		      param_write_null(plist, "ImagingBBox"))) < 0 ||
	     (code = param_write_float_array(plist, "Margins", &ma)) < 0 ||
	     (code = (dev->page.orientation_set ?
		      param_write_int(plist, "Orientation", &dev->page.orientation) :
		      param_write_null(plist, "Orientation"))) < 0 ||

			/* Non-standard parameters */

	     (code = param_write_int_array(plist, "HWSize", &hwsa)) < 0 ||
	     (code = param_write_float_array(plist, "InitialMatrix", &ima)) < 0 ||
	     (code = param_write_string(plist, "Name", &dns)) < 0 ||
	     (code = param_write_int(plist, "Colors", &colors)) < 0 ||
	     (code = param_write_int(plist, "HWBitsPerPixel", &depth)) < 0 ||
	     (code = param_write_int(plist, "GrayValues", &GrayValues)) < 0 ||
	     (code = param_write_long(plist, "PageCount", &dev->page_count)) < 0

	   )
		return code;

	/* Fill in color information. */

	if ( colors > 1 )
	{	int RGBValues = dev->color_info.max_rgb + 1;
		long ColorValues = 1L << depth;
		if ( (code = param_write_int(plist, "RedValues", &RGBValues)) < 0 ||
		     (code = param_write_int(plist, "GreenValues", &RGBValues)) < 0 ||
		     (code = param_write_int(plist, "BlueValues", &RGBValues)) < 0 ||
		     (code = param_write_long(plist, "ColorValues", &ColorValues)) < 0
		   )
			return code;
	}

	if ( depth <= 8 && colors <= 3 )
	{	byte palette[3 << 8];
		byte *p = palette;
		gs_param_string hwcms;
		gx_color_value rgb[3];
		gx_color_index i;
		if ( palette == 0 )
			return_error(gs_error_VMerror);
		for ( i = 0; (i >> depth) == 0; i++ )
		{	int j;
			(*dev_proc(dev, map_color_rgb))(dev, i, rgb);
			for ( j = 0; j < colors; j++ )
				*p++ = gx_color_value_to_byte(rgb[j]);
		}
		hwcms.data = palette, hwcms.size = colors << depth, hwcms.persistent = false;
		if ( (code = param_write_string(plist, "HWColorMap", &hwcms)) < 0 )
			return code;
	}

	return 0;
}

/* Set the device parameters. */
/* If the device was open and the put_params procedure closed it, */
/* return 1; otherwise, return 0 or an error code as usual. */
int
gs_putdeviceparams(gx_device *dev, gs_param_list *plist)
{	bool was_open = dev->is_open;
	int code;
	fill_dev_proc(dev, put_params, gx_default_put_params);
	code = (*dev_proc(dev, put_params))(dev, plist);
	return (code < 0 ? code : was_open && !dev->is_open ? 1 : code);
}

/* Set standard parameters. */
/* Note that setting the size or resolution closes the device. */
/* We break out a procedure that only sets the parameters, */
/* without closing or reopening the device, for the benefit of */
/* window devices that need to do the latter themselves. */
int
gx_default_put_params_only(gx_device *dev, gs_param_list *plist)
{	int ecode = 0, rcode = 0, code;
	gs_param_int_array hwsa;
	gs_param_float_array hwra, psa;
	float page_width, page_height;

	/* We check the parameters in the order HWResolution, HWSize, */
	/* PageSize so that PageSize supersedes HWSize and so that */
	/* HWSize and PageSize each have the resolution available to */
	/* compute the other one. */

	switch ( code = param_read_float_array(plist, "HWResolution", &hwra) )
	{
	default:
		ecode = code;
	case 1:
		break;
	case 0:
		if ( hwra.size != 2 )
			ecode = gs_error_typecheck;
#define ap hwra.data
		else if ( ap[0] <= 0 || ap[1] <= 0 )
			ecode = gs_error_rangecheck;
		else
		  {	dev->x_pixels_per_inch = ap[0];
			dev->y_pixels_per_inch = ap[1];
#undef ap
			rcode = 1;
			break;
		  }
		param_signal_error(plist, "HWResolution", ecode);
	}

	switch ( code = param_read_int_array(plist, "HWSize", &hwsa) )
	{
	default:
		ecode = code;
	case 1:
		break;
	case 0:
		if ( hwsa.size != 2 )
			ecode = gs_error_typecheck;
#define ap hwsa.data
		else if ( ap[0] <= 0 || ap[1] <= 0 )
			ecode = gs_error_rangecheck;
#define max_coord (max_fixed / fixed_1)
#if max_coord < max_int
		else if ( ap[0] > max_coord || ap[1] > max_coord )
			ecode = gs_error_limitcheck;
#endif
#undef max_coord
		else
		  {	dev->width = ap[0];
			dev->height = ap[1];
			dev->page.page_size[0] =
			  ap[0] * 72 / dev->x_pixels_per_inch;
			dev->page.page_size[1] =
			  ap[1] * 72 / dev->y_pixels_per_inch;
#undef ap
			rcode = 1;
			break;
		  }
		param_signal_error(plist, "HWSize", ecode);
	}

	switch ( code = param_read_float_array(plist, "PageSize", &psa) )
	{
	default:
		ecode = code;
	case 1:
		break;
	case 0:
		if ( psa.size != 2 )
			ecode = gs_error_typecheck;
#define ap psa.data
		else if ( (page_width = ap[0] * dev->x_pixels_per_inch / 72) <= 0 ||
			  (page_height = ap[1] * dev->y_pixels_per_inch / 72) <= 0
			)
			ecode = gs_error_rangecheck;
#define max_coord (max_fixed / fixed_1)
#if max_coord < max_int
		else if ( page_width > max_coord || page_height > max_coord )
			ecode = gs_error_limitcheck;
#endif
#undef max_coord
		else
		  {	dev->width = page_width;
			dev->height = page_height;
			dev->page.page_size[0] = ap[0];
			dev->page.page_size[1] = ap[1];
#undef ap
			rcode = 1;
			break;
		  }
		param_signal_error(plist, "HWSize", ecode);
	}

	return (ecode != 0 ? ecode : rcode);
}
int
gx_default_put_params(gx_device *dev, gs_param_list *plist)
{	gx_device temp_dev;
	int code;
	temp_dev = *dev;
	code = gx_default_put_params_only(&temp_dev, plist);
	if ( code < 0 )
		return code;
	/* Close the device; gs_putdeviceparams will reopen it. */
	if ( dev->is_open && code )
	{	int ccode = gs_closedevice(dev);
		if ( ccode < 0 ) return ccode;
	}
	dev->x_pixels_per_inch = temp_dev.x_pixels_per_inch;
	dev->y_pixels_per_inch = temp_dev.y_pixels_per_inch;
	dev->width = temp_dev.width;
	dev->height = temp_dev.height;
	dev->page = temp_dev.page;
	return code;
}
