/* Copyright (C) 1991, 1992, 1993, 1994 Aladdin Enterprises.  All rights reserved.
  
  This file is part of Aladdin Ghostscript.
  
  Aladdin Ghostscript is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Aladdin Ghostscript Free Public
  License (the "License") for full details.
  
  Every copy of Aladdin Ghostscript must include a copy of the License,
  normally in a plain ASCII text file named PUBLIC.  The License grants you
  the right to copy, modify and redistribute Aladdin Ghostscript, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notice and this notice be preserved on
  all copies.
*/

/* zfilter2.c */
/* Additional filter creation */
#include "memory_.h"
#include "ghost.h"
#include "errors.h"
#include "oper.h"
#include "gsstruct.h"
#include "ialloc.h"
#include "idict.h"
#include "idparam.h"
#include "strimpl.h"
#include "sfilter.h"
#include "sbwbs.h"
#include "sbhc.h"
#include "scfx.h"
#include "slzwx.h"
#include "ifilter.h"

/* Import the Level 2 scanner extensions. */
extern const stream_template _ds *scan_ascii85_template;
extern const stream_template s_A85D_template;

/* Initialize the Level 2 scanner for ASCII85 strings. */
private void
zfilter2_init(void)
{	scan_ascii85_template = &s_A85D_template;
}

/* ================ Standard PostScript filters ================ */

/* ------ ASCII85 filters ------ */

/* <target> .filter_ASCII85Encode <file> */
int
zA85E(os_ptr op)
{	return filter_write(op, 0, &s_A85E_template, NULL, false);
}

/* <source> .filter_ASCII85Decode <file> */
int
zA85D(os_ptr op)
{	return filter_read(op, 0, &s_A85D_template, NULL, false);
}

/* ------ CCITTFax filters ------ */

/* Common setup for encoding and decoding filters. */
private int
cf_setup(os_ptr op, stream_CF_state *pcfs)
{	int code;
	if ( (code = dict_bool_param(op, "Uncompressed", false,
				     &pcfs->Uncompressed)) < 0 ||
	     (code = dict_int_param(op, "K", -9999, 9999, 0,
				    &pcfs->K)) < 0 ||
	     (code = dict_bool_param(op, "EndOfLine", false,
				     &pcfs->EndOfLine)) < 0 ||
	     (code = dict_bool_param(op, "EncodedByteAlign", false,
				     &pcfs->EncodedByteAlign)) < 0 ||
	     (code = dict_int_param(op, "Columns", 0, 9999, 1728,
				    &pcfs->Columns)) < 0 ||
	     (code = dict_int_param(op, "Rows", 0, 9999, 0,
				    &pcfs->Rows)) < 0 ||
	     (code = dict_bool_param(op, "EndOfBlock", true,
				     &pcfs->EndOfBlock)) < 0 ||
	     (code = dict_bool_param(op, "BlackIs1", false,
				     &pcfs->BlackIs1)) < 0 ||
	     (code = dict_int_param(op, "DamagedRowsBeforeError", 0, 9999,
				    0, &pcfs->DamagedRowsBeforeError)) < 0 ||
	     (code = dict_bool_param(op, "FirstBitLowOrder", false,
				     &pcfs->FirstBitLowOrder)) < 0
	   )
		return code;
	pcfs->raster = (pcfs->Columns + 7) >> 3;
	return 0;
}

/* <target> <dict> .filter_CCITTFaxEncode <file> */
int
zCFE(os_ptr op)
{	stream_CFE_state cfs;
	int code;
	check_type(*op, t_dictionary);
	check_dict_read(*op);
	code = cf_setup(op, (stream_CF_state *)&cfs);
	if ( code < 0 )
	  return code;
	return filter_write(op, 1, &s_CFE_template, (stream_state *)&cfs, false);
}

/* <source> <dict> .filter_CCITTFaxDecode <file> */
/* <source> .filter_CCITTFaxDecode <file> */
int
zCFD(os_ptr op)
{	os_ptr dop;
	int npop;
	stream_CFD_state cfs;
	int code;
	if ( r_has_type(op, t_dictionary) )
	{	check_dict_read(*op);
		dop = op, npop = 1;
	}
	else
		dop = 0, npop = 0;
	code = cf_setup(dop, (stream_CF_state *)&cfs);
	if ( code < 0 )
	  return code;
	return filter_read(op, npop, &s_CFD_template, (stream_state *)&cfs, false);
}

/* ------ Generalized LZW/GIF filters ------ */

/* <target> .filter_LZWEncode <file> */
int
zLZWE(os_ptr op)
{	return filter_write(op, 0, &s_LZWE_template, NULL, false);
}

/* <source> .filter_LZWDecode <file> */
/* <source> <dict> .filter_LZWDecode <file> */
int
zLZWD(os_ptr op)
{	stream_LZW_state lzs;
	os_ptr dop;
	int code;
	int npop;
	if ( r_has_type(op, t_dictionary) )
	{	check_dict_read(*op);
		dop = op, npop = 1;
	}
	else
		dop = 0, npop = 0;
	if ( (code = dict_int_param(dop, "InitialCodeLength", 2, 11, 8,
				    &lzs.InitialCodeLength)) < 0 ||
	     (code = dict_bool_param(dop, "FirstBitLowOrder", false,
				     &lzs.FirstBitLowOrder)) < 0 ||
	     (code = dict_bool_param(dop, "BlockData", false,
				     &lzs.BlockData)) < 0 ||
	     (code = dict_int_param(dop, "EarlyChange", 0, 1, 1,
				    &lzs.EarlyChange)) < 0
	   )
		return code;
	return filter_read(op, npop, &s_LZWD_template, (stream_state *)&lzs, false);
}

/* ================ Non-standard filters ================ */

/* ------ Bounded Huffman code filters ------ */

/* Common setup for encoding and decoding filters */
private int
bhc_setup(os_ptr op, stream_BHC_state *pbhcs)
{	int code;
	int num_counts;
	int data[max_hc_length + 1 + 256];
	uint dsize;
	int i;
	uint num_values, accum;
	ushort *counts;
	ushort *values;
	if ( (code = dict_bool_param(op, "FirstBitLowOrder", false,
				     &pbhcs->FirstBitLowOrder)) < 0 ||
	     (code = dict_int_param(op, "MaxCodeLength", 1, max_hc_length,
				    max_hc_length, &num_counts)) < 0 ||
	     (code = dict_int_array_param(op, "Tables", countof(data),
					  data)) <= 0
	   )
	  return (code < 0 ? code : gs_note_error(e_rangecheck));
	dsize = code;
	if ( dsize <= num_counts )
	  return_error(e_rangecheck);
	for ( i = 0, num_values = 0, accum = 0; i <= num_counts;
	      i++, accum <<= 1
	    )
	  {	uint count = data[i];
		num_values += count;
		accum += count;
	  }
	if ( dsize != num_counts + 1 + num_values ||
	     accum != 1 << (num_counts + 1)
	   )
	  return_error(e_rangecheck);
	for ( ; i < num_counts + 1 + num_values; i++ )
	  {	uint value = data[i];
		if ( value >= num_values )
		  return_error(e_rangecheck);
	  }
	pbhcs->definition.counts = counts =
	  (ushort *)ialloc_byte_array(num_counts + 1, sizeof(ushort),
				      "bhc_setup(counts)");
	pbhcs->definition.values = values =
	  (ushort *)ialloc_byte_array(num_values, sizeof(ushort),
				      "bhc_setup(values)");
	if ( counts == 0 || values == 0 )
	  {	ifree_object(values, "bhc_setup(values)");
		ifree_object(counts, "bhc_setup(counts)");
		return_error(e_VMerror);
	  }
	for ( i = 0; i <= num_counts; i++ )
	  counts[i] = data[i];
	pbhcs->definition.counts = counts;
	pbhcs->definition.num_counts = num_counts;
	for ( i = 0; i < num_values; i++ )
	  values[i] = data[i + num_counts + 1];
	pbhcs->definition.values = values;
	pbhcs->definition.num_values = num_values;
	return 0;
}

/* <target> <dict> .filter_BoundedHuffmanEncode <file> */
int
zBHCE(os_ptr op)
{	stream_BHCE_state bhcs;
	int code;
	check_type(*op, t_dictionary);
	check_dict_read(*op);
	code = bhc_setup(op, (stream_BHC_state *)&bhcs);
	if ( code < 0 )
	  return code;
	return filter_write(op, 1, &s_BHCE_template, (stream_state *)&bhcs,
			    false);
}

/* <source> <dict> .filter_BoundedHuffmanDecode <file> */
int
zBHCD(os_ptr op)
{	stream_BHCD_state bhcs;
	int code;
	code = bhc_setup(op, (stream_BHC_state *)&bhcs);
	if ( code < 0 )
	  return code;
	return filter_read(op, 1, &s_BHCD_template, (stream_state *)&bhcs,
			   false);
}

/* ------ Burrows/Wheeler block sorting filters ------ */

/* Common setup for encoding and decoding filters */
private int
bwbs_setup(os_ptr op, stream_BWBS_state *pbwbss)
{	int code;
	if ( (code = dict_int_param(op, "BlockSize",
				    1, max_int / sizeof(int) - 10, 16384,
				    &pbwbss->BlockSize)) < 0
	   )
		return code;
	return 0;
}

/* <target> <dict> .filter_BWBlockSortEncode <file> */
int
zBWBSE(os_ptr op)
{	stream_BWBSE_state bwbss;
	int code;
	check_type(*op, t_dictionary);
	check_dict_read(*op);
	code = bwbs_setup(op, (stream_BWBS_state *)&bwbss);
	if ( code < 0 )
	  return code;
	return filter_write(op, 1, &s_BWBSE_template, (stream_state *)&bwbss,
			    false);
}

/* <source> <dict> .filter_BWBlockSortDecode <file> */
int
zBWBSD(os_ptr op)
{	stream_BWBSD_state bwbss;
	int code;
	code = bwbs_setup(op, (stream_BWBS_state *)&bwbss);
	if ( code < 0 )
	  return code;
	return filter_read(op, 1, &s_BWBSD_template, (stream_state *)&bwbss,
			   false);
}

/* ------ Move-to-front filters ------ */

/* <target> .filter_MoveToFrontEncode <filter> */
int
zMTFE(os_ptr op)
{	return filter_write(op, 0, &s_MTFE_template, NULL, false);
}

/* <source> .filter_MoveToFrontDecode <file> */
int
zMTFD(os_ptr op)
{	return filter_read(op, 0, &s_MTFD_template, NULL, false);
}

/* ================ Initialization procedure ================ */

op_def zfilter2_op_defs[] = {
		/* Standard filters */
	{"1.filter_ASCII85Encode", zA85E},
	{"1.filter_ASCII85Decode", zA85D},
	{"2.filter_CCITTFaxEncode", zCFE},
	{"2.filter_CCITTFaxDecode", zCFD},
	{"1.filter_LZWDecode", zLZWD},
	{"1.filter_LZWEncode", zLZWE},
		/* Non-standard filters */
	{"2.filter_BoundedHuffmanDecode", zBHCD},
	{"2.filter_BoundedHuffmanEncode", zBHCE},
	{"2.filter_BWBlockSortEncode", zBWBSE},
	{"2.filter_BWBlockSortDecode", zBWBSD},
	{"1.filter_MoveToFrontEncode", zMTFE},
	{"1.filter_MoveToFrontDecode", zMTFD},
	op_def_end(zfilter2_init)
};
