/*
**  --- MODEM_IO.C ---
**
**  1) Call ModemSendTo(), ModemWaitFor(), etc.
**  2) Call ModemDriver() until IDLE is returned.
**  3) Call ModemResult() to get result (TRUE/FALSE)
**
*/

#include <string.h>
#include <ctype.h>
#include "windows.h"
#include "pcl4w.h"
#include "ascii.h"
#include "modem_io.h"
#include "term_io.h"
#include "paint.h"
#include "line.h"
#include "constants.h"

extern HWND hMainWnd;
extern int  OnLineFlag;

#define MAX_BUF_BYTES 64


struct
{int Port;                /* port # */
 int Status;              /* Function (changed to IDLE when done) */
 int Function;            /* Function (SEND_TO, WAIT_FOR, etc. ) */
 int Result;              /* Result of last function (status must be IDLE) */
 long TimeMark;           /* Time wait till */
 long Pace;               /* char pace in ms (SendTo) */
 int Case;                /* case sensitive if TRUE (WaitFor) */
 int Index;               /* index into buffer */
 int Length;              /* length of buffer */
 char Buffer[1+MAX_BUF_BYTES];   /* data buffer */
} ModemData = {0,IDLE,IDLE,TRUE,0L,0,0,0,0,0,0};

/*** PRIVATE Functions ***/

/* hangup modem 'phone' */

static int DoHangup(void)
{long Time;
 Time = GetCurrentTime();
 switch(ModemData.Index)
  {case 1:
     /* wait 1 second */
     if(Time >= ModemData.TimeMark + 1000L)
       {ModemData.Index = 2;
        ModemData.TimeMark = Time;
       }
     break;
   case 2:
   case 3:
   case 4:
     /* wait 1/4 second */
     if(Time >= ModemData.TimeMark + 250L)
       {/* send a '+' */
        PutChar(ModemData.Port,'+');
        ModemData.Index++;
        ModemData.TimeMark = Time;
       }
     break;
   case 5:
     /* wait 1 second */
     if(Time >= ModemData.TimeMark + 1000L)
       {ModemData.Status = IDLE;
        return IDLE;
       }
     break;
  }
 return RUNNING;
}

/* wait for quiet */

static int DoQuiet(void)
{int Code;
 /* character(s) incoming ? */
 while(TRUE)
   {Code = GetChar(ModemData.Port);
    if(Code<0) break;
    else
      {/* echo char */
       DisplayChar((char)Code);
       ModemData.TimeMark = GetCurrentTime();
      }
    }
 /* enough silence ? */
 if(GetCurrentTime() >= ModemData.TimeMark)
   {ModemData.Status = IDLE;
    ModemData.Result = TRUE;
    return IDLE;
   }
 return RUNNING;
}

static char FetchChar(void)
{int i;
 char c;
 if(ModemData.Status==IDLE) return 0;
 /* fetch next char in buffer */
 c = ModemData.Buffer[ModemData.Index++];
 if(ModemData.Index==ModemData.Length)
   {/* idle after this call */
    ModemData.Status = IDLE;
    ModemData.Result = TRUE;
   }
  return c;
 }

/* send string to modem */

static int DoSendTo(void)
{int i;
 char c;
 int Code;
 int Delta = 0;
 /* echo any previous chars */
 PortEcho(ModemData.Port,250);
 /* time to send next character ? */
 if(GetCurrentTime() < ModemData.TimeMark) return RUNNING;
 /* send next character */
 c = FetchChar();
 switch(c)
   {case '^':
      c = FetchChar() - '@';
      break;
    case '!':
      /* replace ! with carriage return */
      c = CR;
      break;
    case '~':
      /* delay 1/2 second */
      Delta = 500;
      c = ' ';
      break;
    case ' ':
      /* delay 1/4 second */
      Delta = 250;
      break;
   } /* end switch */
 /* transmit as 7 bit ASCII character */
 PutChar(ModemData.Port,(char)(0x7f & c));
 ModemData.TimeMark = GetCurrentTime() + (long)Delta + (long)ModemData.Pace;
 return RUNNING;
} /* end SendToModem */

/* wait for 'String' from modem */

static int DoWaitFor(void)
{int i, k;
 int Code;
 int Pass;
 char c1, c2;
 char *Ptr;
 char Temp[80];
 /* use up all chars in serial input buffer */
 for(Pass=0;Pass<80;Pass++)
   {/* out of time ? */
    if(GetCurrentTime() > ModemData.TimeMark)
      {/* we are out of time */
       ModemData.Status = IDLE;
       ModemData.Result = FALSE;
       return IDLE;
      }
    /* get next character */
    Code = GetChar(ModemData.Port);
    if(Code<=-1) return RUNNING;
    /* Code >= 0 */
    DisplayChar((char)Code);
    c1 = ModemData.Buffer[ModemData.Index++];
    c2 = (char)Code;
    /* case sensitive ? */
    if(!ModemData.Case)
      {c1 = toupper(c1);
       c2 = toupper(c2);
      }
    /* does char match ? */
    if(c1==c2)
      {if(ModemData.Index==ModemData.Length)
        {/* we matched ! */
         ModemData.Status = IDLE;
         ModemData.Result = TRUE;
         return IDLE;
        }
      }
    else ModemData.Index = 0;
   } /* end for */
 return RUNNING;
}

/*** PUBLIC Functions ***/

/* execute next step: call until returns IDLE */

int ModemDriver(void)
{int i;
 switch(ModemData.Status)
   {case IDLE:
      return IDLE;
    case SEND_TO:
      return( DoSendTo() );
    case WAIT_FOR:
      return ( DoWaitFor() );
    case QUIET:
      return ( DoQuiet() );
    case HANGUP:
      return ( DoHangup() );
   }
 return RUNNING;
}

/* start: send string to modem */

int ModemSendTo(int Port,float Pace,char *String)
{int i;
 int Length;
 /* already running */
 if(ModemData.Status!=IDLE) return(FALSE);
 ModemData.Port = Port;
 ModemData.Status = SEND_TO;
 ModemData.Function = SEND_TO;
 ModemData.Pace = (long) (1000.0 * Pace);
 ModemData.Result = TRUE;
 Length = strlen(String);
 if(Length>=MAX_BUF_BYTES) Length = MAX_BUF_BYTES;
 ModemData.Length = Length;
 for(i=0;i<Length;i++) ModemData.Buffer[i] = String[i];
 ModemData.Buffer[Length] = '\0';
 ModemData.Index = 0;
 ModemData.TimeMark = GetCurrentTime() + ModemData.Pace;
 return TRUE;
} /* end ModemSendTo */

/* wait for continuous silence */

int ModemQuiet(int Port,float QuietTime)
{int i;
 int Length;
 /* already running */
 if(ModemData.Status!=IDLE) return(FALSE);
 ModemData.Port = Port;
 ModemData.Status = QUIET;
 ModemData.Function = QUIET;
 ModemData.Result = TRUE;
 ModemData.TimeMark = GetCurrentTime() + (long)(1000.0*QuietTime);
 return TRUE;
}

/* start: wait for string from modem */

int ModemWaitFor(int Port,float MaxTime,int Case,char *String)
{int i;
 int Length;
 /* already running */
 if(ModemData.Status!=IDLE) return(FALSE);
 ModemData.Port = Port;
 ModemData.Status = WAIT_FOR;
 ModemData.Function = WAIT_FOR;
 ModemData.Result = TRUE;
 ModemData.Case = Case;
 ModemData.Result = FALSE;
 Length = strlen(String);
 if(Length>=MAX_BUF_BYTES) Length = MAX_BUF_BYTES;
 ModemData.Length = Length;
 for(i=0;i<Length;i++) ModemData.Buffer[i] = String[i];
 ModemData.Buffer[Length] = '\0';
 ModemData.Index = 0;
 ModemData.TimeMark = GetCurrentTime() + (long)(1000.0*MaxTime);
 return TRUE;
} /* end ModemSendTo */

/* get result of last operation */

int ModemResult(void)
{/* get result flag */
 return ModemData.Result;
}

int ModemFunction(void)
{/* get last function */
 return ModemData.Function;
}

/* hangup modem */

int ModemHangup(int Port)
{int i;
 /* ignore current function, if any */
 ModemData.Status = HANGUP;
 ModemData.Function = HANGUP;
 ModemData.Result = TRUE;
 ModemData.TimeMark = GetCurrentTime();
 /* use INDEX as state variable in DoHangup() */
 ModemData.Index = 1;
 return TRUE;
}
