/*
     Library PMIO: A library for stdio/pcio on a PM screen
     Copyright (C) 1994  Colin Jensen
     
     This program is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published by
     the Free Software Foundation; either version 2 of the License, or
     (at your option) any later version.
     
     This program is distributed in the hope that it will be useful,
     but WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
     GNU General Public License for more details.
     
     You should have received a copy of the GNU General Public License
     along with this program; if not, write to the Free Software
     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

     Contact addresses:
	Colin Jensen
	email: cjensen@netcom.com
	US mail: 4902 Esguerra Terrace, Fremont CA, 94555
*/
#include <string.h>

#include "pmio.h"
#include "pmio1.h"

TextView::TextView()
: x(1), y(1), attr ((WHITE << 4) | BLACK), wxmin (0), wymin (0)
{
  wxmax = width () - 1;
  wymax = height () - 1;
  memset (text, ' ', sizeof (text));
  memset (text_attr, attr, sizeof (text_attr));
}

void TextView::gettextinfo (struct text_info *tinfo)
{
  tinfo->winleft = wxmin+1;
  tinfo->winright = wxmax+1;
  tinfo->wintop = wymin+1;
  tinfo->winbottom = wymax+1;
  tinfo->attribute = attr;
  tinfo->screenheight = height ();
  tinfo->screenwidth = width ();
  tinfo->curx = x;
  tinfo->cury = y;
}

void TextView::settextinfo (const struct text_info *tinfo)
{
  set_window (tinfo->winleft, tinfo->wintop,
	      tinfo->winright, tinfo->winbottom);
  attr = tinfo->attribute;
  gotoxy (tinfo->curx, tinfo->cury);
  /* Should validate x and y! */
  /* Ignore height and width fields */
}


void TextView::set_window (int awxmin, int awymin, int awxmax, int awymax)
{
  if (1 <= awxmin && awxmin <= width ()
      && 1 <= awxmax && awxmax <= width ()
      && 1 <= awymin && awymin <= height ()
      && 1 <= awymax && awymax <= height ()
      && awxmin <= awxmax
      && awymin <= awymax)
    {
      wxmin = awxmin-1;
      wxmax = awxmax-1;
      wymin = awymin-1;
      wymax = awymax-1;
    }
}

char TextView::getdirt (int y, char *min, char *max) 
{
  if (!dirty_flags[y])
    return 0;
  dirty_flags[y] = 0;
  if (max)
    *max = dirty_max[y];
  if (min)
    *min = dirty_min[y];
  dirty_max[y] = 0;
  dirty_min[y] = width () - 1;
  if (max && *max >= width ())
    *max = width ();
  if (min && *min < 0)
    *min = 0;
  return 1;
}

void TextView::dirty (int y, int min, int max) 
{ 
  y = wymin + y - 1;
  min--; max--;
  dirty_flags[y] = 1; 
  if (min < dirty_min[y])
    dirty_min[y] = min;
  if (max > dirty_max[y])
    dirty_max[y] = max;
}

void TextView::scroll (int count)
{
  if (count == 0)
    return;
  else if (count > height () - 1
	   || -count > height () - 1)
    {
      // Scroll entire contents off screen
      memset (text, ' ', sizeof (text));
      memset (text_attr, attr, sizeof (text_attr));
      for (int i = 0; i < height (); i++)
	dirty (i);
    }
  else if (count > 0)
    {
      int n = count * width ();  // Number of bytes destroyed
      int lp = height () - count; // Number of lines preserved
      int n2 = height () * width () - n; // Number of bytes preserved
      /* Move saved text */
      memcpy (text,        text        + n,     n2);
      memcpy (text_attr,   text_attr   + n,     n2);
      memcpy (dirty_flags, dirty_flags + count, lp);
      memcpy (dirty_max,   dirty_max   + count, lp);
      memcpy (dirty_min,   dirty_min   + count, lp);
      /* "Clear" out blanked area */
      memset (text        + n2, ' ',      n);
      memset (text_attr   + n2, attr,     n);
      memset (dirty_flags + lp, 1,        count);
      memset (dirty_max   + lp, width (), count);
      memset (dirty_min   + lp, 1,        count);
    }
  else
    {
      int ld = (-count);  // Number of lines destroyed
      int n = (-count) * width ();  // Number of bytes destroyed
      int lp = height () + count; // Number of lines preserved
      int n2 = height () * width () - n; // Number of bytes preserved
      /* Move saved text */
      memcpy (text        + n,  text,        n2);
      memcpy (text_attr   + n,  text_attr,   n2);
      memcpy (dirty_flags + ld, dirty_flags, lp);
      memcpy (dirty_max   + ld, dirty_max,   lp);
      memcpy (dirty_min   + ld, dirty_min,   lp);
      /* "Clear" out blanked area */
      memset (text,        ' ',      n);
      memset (text_attr,   attr,     n);
      memset (dirty_flags, 1,        ld);
      memset (dirty_max  , width (), ld);
      memset (dirty_min  , 1,        ld);
    }

  if (physical_scroll (count))
    return;

  int h = height ();
  for (h = 1; h <= height (); h++)
    dirty (h);
}

void TextView::clrscr ()
{
  memset (text, ' ', sizeof (text));
  memset (text_attr, attr, sizeof (text_attr));
  x = y = 0;
  for (int n = 1; n <= height (); n++)
    dirty (n);
}

void TextView::write_stdio (int count, const char *data)
{
  while (count)
    {
      switch (*data)
	{
	case '\r':
	  cr ();
	  count--; data++;
	  break;
	case '\n':
	  clreol ();
	  cr ();
	  nl ();
	  count--; data++;
	  break;
	case '\t':
	  write (8 - (x & 0x07), "        ");
	  break;
	default:
	  int n;
	  for (n=1; n < count && strchr ("\r\n\t", data[n]) == 0; n++)
	    ;
	  write (n, data);
	  count -= n;
	  data += n;
	}
    }
}


