// ========================================================================
//  TODB LIBRARY
//    kspersis.h
//
//    KSPersistent class
//
//    Version: see TODB.H file
//
//    Copyright 1993 Christian Thrien
//    All rights reserved
// ========================================================================

#ifndef _KLPERSIS_H
#define _KLPERSIS_H

#include "fblock.h"
#include "boolean.h"
#include <stdio.h>

class ErrReporter;

enum KSPersistentError
{
    KSP_FHEAD
};

class KSPersistent
{
public:
    virtual ~KSPersistent();

    // -- called from derived class's constructor
    void LoadObject( ObjAdr oa = 0 );
    void LoadFirstBlock( ObjAdr oa = 0 );

    // -- called from derived class's destructor
    virtual void SaveObject();

    // -- class interface methods for modifying database
    Boolean AddObject();
    void ChangeObject();
    void DeleteObject();
    ObjAdr ObjectAddress() const;

    // error control
    static void SetErrOut( ErrReporter * er );

protected:
    KSPersistent( TODB & db, int cid );

    // -- derived class methods for reading members
    void ReadObject( void * buf, int length );
    void ReadObject( String & str );
    void ReadObject( unsigned int & i );
    void ReadObject( unsigned long & l );

    // -- derived class methods for writing members
    void WriteObject( void * buf, int length );
    void WriteObject( String & str );
    void WriteObject( unsigned int i );
    void WriteObject( unsigned long l );

    // -- specialized class members for reading and writing
    KeyNbr GetCount();
    void AppendIf( KeyNbr count, long pos, KeyType k );

    // -- provided by derived class
    virtual void Write() = 0;
    virtual void Read() = 0;

private:
    ObjHeader objhdr_;
    BlockNbr objaddress_;      // first block address for this object
    TODB & todb_;	       // database for this object
    Block * firstblock_;       // first block for reading/writing
    Block * curblock_;	       // current block for reading/writing
    int offset_;	       // current char position
    Boolean changed_;	       // true if user changed the object
    Boolean deleted_;	       // true if user deleted the object
    Boolean newobject_;	       // true if user is adding the object
    BlockNbr lastblock_;

    void ObjectOut();
    void ReadFirstBlock();
    void UpdateFirstBlockHeader();

    // -- error control
    static ErrReporter * ErrOut;
    static void ErrorHandler( KSPersistentError err );
};

inline KSPersistent::~KSPersistent()
{
    delete firstblock_;
}

inline void KSPersistent::ChangeObject()
{
    changed_ = True;
}

inline void KSPersistent::DeleteObject()
{
    deleted_ = True;
}

inline ObjAdr KSPersistent::ObjectAddress() const
{
    return objaddress_;
}

inline void KSPersistent::ReadObject( unsigned int & i )
{
    ReadObject( &i, sizeof(int) );
}

inline void KSPersistent::ReadObject( unsigned long & l )
{
    ReadObject( &l, sizeof(long) );
}

inline void KSPersistent::WriteObject( unsigned int i )
{
    WriteObject( &i, sizeof(int) );
}

inline void KSPersistent::WriteObject( unsigned long l )
{
    WriteObject( &l, sizeof(long) );
}

#endif // _KLPERSIS_H
