// ========================================================================
//  TODB LIBRARY
//    schset.h
//
//    SearchSet class
//
//    Version: see TODB.H file
//
//    Copyright 1993 Christian Thrien
//    All rights reserved
// ========================================================================

#ifndef _SCHSET_H
#define _SCHSET_H

#include "ptodb.h"
#include "keyset.h"

class TODB;
class TIndex;
class KSet;
class ErrReporter;

static const int SEARCHSET_MATCH	 =  1;
static const int SEARCHSET_NO_MATCH	 =  0;
static const int SEARCHSET_PATTERN_VALID =  0;	// valid pattern
static const int SEARCHSET_PATTERN_ES	 = -1;	// literal escape at end of pattern
static const int SEARCHSET_PATTERN_RANGE = -2;	// incorrect range in [..] construct
static const int SEARCHSET_PATTERN_CLOSE = -3;	// no end bracket in [..] construct
static const int SEARCHSET_PATTERN_EMPTY = -4;	// [..] contstruct is empty

enum SearchSetError
{
    SSE_ALLOC,
    SSE_DB
};

class SearchSet
{
public:
    // constructor
    SearchSet( TODB * db );
    SearchSet( const SearchSet & ss );

    // destructor
    ~SearchSet();

    // number of elements
    KeyNbr Card() const;

    // assignment
    SearchSet & operator = ( const SearchSet & ss );

    // sets operators:

    // union (3 operands)
    SearchSet operator * ( const SearchSet & ss );
    // union (2 operands)
    void operator *= ( const SearchSet & ss );

    // intersection (3 operands)
    SearchSet operator + ( const SearchSet & ss );
    // intersection (2 operands)
    void operator += ( const SearchSet & ss );

    // difference (3 operands)
    SearchSet operator - ( const SearchSet & ss );
    // difference (2 operands)
    void operator -= ( const SearchSet & ss );

    // sets comparison:
    // same sets?
    int operator == ( const SearchSet & ss ) const;
    // different sets?
    int operator != ( const SearchSet & ss ) const;

    // test
    // returns 1 if pattern has any special wildcard characters, 0 otherwise
    int isPattern( char * pattern ) const;
    // check if pattern is valid
    int isValidPattern( char * pattern ) const;

    // search functions
    int FindExact( IdType cid, IdType idxid, char * buf);
    int FindAll( IdType cid, IdType idxid );
    int FindPattern( IdType cid, IdType idxid, char * pattern );

    // navigation (a returned value of 0 marks the end)
    KeyType First();
    KeyType Next();

    // error control
    static void SetErrOut( ErrReporter * er );

protected:
    SearchSet();

private:
    TODB * todb_;
    TIndex * btree_;
    KSet * set_;
    KeySet * pSet_;
    char * entry_;
    IdType curcid_;
    IdType curidxid_;
    ObjAdr addr_;

    void SetUnion( KSet & a, KSet & b, KSet & c );
    void SetIntersection( KSet & a, KSet & b, KSet & c );
    void SetDifference( KSet & a, KSet & b, KSet & c );
    void MakePrefix( IdType cid, IdType idxid );
    IdType ToCID();
    IdType ToIDXID();

    int PosFirst();
    int PosNext();

    // error control
    static ErrReporter * ErrOut;
    static void ErrorHandler( SearchSetError err );
};

inline void SearchSet::operator *= ( const SearchSet & ss )
{
    *this = *this * ss;
}

inline void SearchSet::operator += ( const SearchSet & ss )
{
    *this = *this + ss;
}

inline void SearchSet::operator -= ( const SearchSet & ss )
{
    *this = *this - ss;
}

inline KeyType SearchSet::First()
{
    return set_->First();
}

inline KeyType SearchSet::Next()
{
    return set_->Next();
}

#endif // _SCHSET_H
