/* Wild.C -- independently compiled WildCard WILD.OBJ package */

/* Version 1.1 copyright 1993 by Roedy Green
 * Canadian Mind Products
 * #601 - 1330 Burrard
 * Vancouver, BC Canada  V6Z 2B8
 * (604) 685-8412
 * May be used freely for any purpose but military.
 * Mike Greve of Switzerland had a very large hand
 * in writing this code. */

/* compile separately with the same model as the rest of your program,
 * then link in with your program. The Compact or Large memory models
 * will let you handle more strings, but Huge will not buy you any more
 * capacity.  See Wild.Pri for documenation on how this all works. */

#include <stdlib.h> /* toupper */
#include <stdio.h>  /* fopen fread */
#include <string.h> /* strcmp strcpy memset */
#include <dos.h>    /* _dos_findnext _find_t */
#include <direct.h> /* directory getcwd */
#include "wild.h"   /* enumerations, prototypes */

/* T Y P E S */

typedef struct ENTRY /* What one string entry in a basket looks
		      * like */
{
  struct ENTRY	 *PrevEntrySameHash;	 /* pointer to prev entry hashing to same value */
  BYTE		  StringLen;		 /* length of string */
  char		  AString[FILENAME_MAX]; /* the String itself --
					  * variable length, null-terminated. */
}		ENTRY;

typedef struct BASKET
{			      /* basket structure containing text string
			       * buffers */
  struct BASKET  *PrevBasket; /* enchained to prev basket */
  ENTRY 	  Entries[1];
			      /* variable number of variable-length
			       * string entries */

}		BASKET;

/* C O N S T A N T S */

#define   EOS '\0'
/* end of string marker */

#define   BasketChunk  8192
/* number of bytes to allocate to a string basket */

#define   PointersChunk  2048
/* number of extra pointer entries to allocate per time. These will be 2
 * or 4 bytes each, representing 4096 or 8092 bytes */

#define HashPrime   149 /* size of hash table. The bigger this is,
			 * the fewer collisions we get and the
			 * faster is our search becomes.  Works
			 * best if this is a prime number.  Up to a point of
			 * diminishing returns, the speed in proportional
			 * to the size of this number. */

#define CLEAR(block)	 (void) memset(&block, 0, sizeof(block))

/* V A R I A B L E S */

local ENTRY   *HashTable[HashPrime]; /* the HashTable is just a list of
				       * pointers to the most recent string entries
				       * in various baskets, that hash to the
				       * corresponding value. */

local char    **fargv; /* pointer to array of ptrs to filename
			* strings we have in our baskets. The
			* pointers point to the strings, not
			* housekeeping ENTRY in front of the string. */

local BASKET   *BasketStart; /* Pointer to current Basket. At head of
			      * current Basket, is pointer to previous
			      * Basket, so we can free them all when we
			      * are done. When a basket gets full we
			      * allocate another. */

local char     *BasketEnd; /* A basket is dynamically allocated
			    * containing ENTRYs with variable length,
			    * counted, ASCIIZ filename strings strung
			    * one after the other, representing files
			    * matching WildCards we want to keep.
			    * BasketEnd points to free space after
			    * any strings in the current basket. */

local char     *BasketMars; /* points to area just past the end of the
			     * string basket region we have allocated.
			     * If ever we start a string here or above
			     * we COULD be in trouble. */

local WORD	fargc; /* Count of how many file names are in all
			* the baskets right now.  Sometimes this
			* count includes the nullified, deleted
			* strings. However we clean those out
			* after each negative wildcard is
			* processed. */

local WORD	fargcMars; /* value just past the end of what we have
			    * allocated for pointers.  If had room
			    * for 2000 pointers, this would be 2000 */

local WORD	CRCTable[256]; /* used to compute CRC functions */

local WORD	CurrentDirLen; /* not including the EOS, but including
				* trailing \ */

local WORD	CurrentMomLen; /* not including the EOS, but including
				* trailing \. See CurrentMom below. */

local BOOL	NegativeWildCard; /* is the wildcard currently being
				   * processed negative? */

local BOOL	SlashS; /* Should this wildcard be handled with /S
			 * processing to handle subdirs? */

local BOOL	PrevSlashS; /* Was there a /S?	Are we processing all
			     * subdirs as well as files? */

local BOOL	FineFilter[64]; /* whether to accept each of 64 possible
				 * attributes.	Built by invoking the
				 * CallBackAttr function with each of the
				 * 64 possible attibutes. */

local ATTRIB	CoarseFilter = 0;
/* Attribute mask to grossly filter out files never wanted. DOS behaves a
 * little strangely when fed a coarse filter.  It completely ignores the
 * read-only and archive bits. If you ask for DIRs, you get normal files
 * too. But, if you ask for Vol, you DON'T get anything but VOL. It is
 * all bizarrely asymmetric. We ignore this, and carry on as if DOS were
 * consistent.	It comes out ok, in the wash, since the FineFilter
 * filters the mess DOS gives us. */

/* attribute filter for getting at subdirs */

local char	CurrentDrive[3]; /* C: */

local char	CurrentDir[_MAX_DIR]; /* e.g. \MYMOM\MYDIR\ lead and
				       * trail backslash. Root is just
				       * one backslash */

local char	CurrentMom[_MAX_DIR]; /* e.g. \MYMOM\, mother dir of
				       * current dir. If is no mom, is
				       * null string. */

local char	PrevWildCard[_MAX_DIR]; /* Tidied version of previous
					 * wildcard from command line.
					 * This is used to allow wildcards
					 * without dirs to use the
					 * previous directory. */

/* P R O T O T Y P E S	in alphabetical order */

/* Wild_AttribPatternMatch, Wild_Free and Wild are defined in Wild.H */

local void     *Wild_Alloc(void *OldHandle, WORD Size);
local void	Wild_EnsureRoom(void);
local void	Wild_Expand(char *WildCard);
local ENTRY    *Wild_FindEntry(ENTRY * Entry, WORD Hash);
local void	Wild_GetCurrentDir(void);
local void	Wild_Graft(char *WildCard, char *FileName);
local void	Wild_GraftLevel(char *SubWildCard, char *WildCard, char *Level);
local WORD	Wild_HashCRC(ENTRY * Entry);
local void	Wild_InitCRCTable(void);
local void	Wild_PrepareFineFilter(BOOL CallBackAttr(ATTRIB Attrib));
local void	Wild_ProcessHit(char *WildCard, char *Hitname);
local void	Wild_ProcessHits(char *WildCard);
local void	Wild_ProcessSubs(char *WildCard);
local void	Wild_Prune(void);
local void	Wild_Shrink(void);
local void	Wild_Tidy(char *WildCard);


/* F U N C T I O N    D E F I N I T I O N S  bottom up order */

/* ************************************** */

local void	     *Wild_Alloc(void *OldHandle, size_t Size)
/* ensure (re)alloc succeeded */
    {
    void	   *Handle;
    if (OldHandle)
      {
      if (Handle = realloc(OldHandle, Size))
	return (Handle);
      }
    else
      {
      if (Handle = malloc(Size))
	return (Handle);
      }

    printf("\nAllocation error: Not enough RAM for WildCard processing."
      "\nAt least %u more bytes needed.\n", Size);
    exit(3);
    }

/* ************************************** */

extern BOOL	Wild_AttribPatternMatch(char Wanted[6], ATTRIB Attrib)
/* true if Attrib matches the given string pattern Wanted string is of
 * form: ADVSHR A-archive D-directory V-volume S-system H-hidden
 * R-read-only
 * "ADVSHR"
 * "?00+1-" <-- example
 * 1=must be 1
 * 0=must be 0
 * +=must have some 1s
 * -=must have some 0s */

    {
    BOOL	    Bit,
      NeedPlus = FALSE,  /* no + seen yet, so no need for 1s in + slots */
      NeedMinus = FALSE; /* ditto for minus */
    int 	    i;
    WORD	    PlusCount = 0,
      MinusCount = 0;

    /* work right to left comparing the Wanted char string with the Attrib bits */
    for (i = 5; i >= 0; i--)
      {
      Bit = (BOOL) (((WORD) Attrib >> (5 - i)) & 1);
      switch (Wanted[i])
	{
	case '1':
	  if (!Bit)
	    return (FALSE);
	  break;
	case '0':
	  if (Bit)
	    return (FALSE);
	  break;
	case '+':
	  NeedPlus = TRUE;
	  if (Bit)
	    PlusCount++;
	  break;
	case '-':
	  NeedMinus = TRUE;
	  if (!Bit)
	    MinusCount++;
	  break;
	case '?':
	  break;
	default:
	  printf(
	    "\n Programmer error: Invalid WildCard attribute specifier string: %s.\n",
	    Wanted);
	  exit(99);
	} /* end switch */
      } /* end for */
    if (NeedPlus && (PlusCount == 0))
      return (FALSE);
    if (NeedMinus && (MinusCount == 0))
      return (FALSE);
    return (TRUE);
    }

/* ************************************** */

local void	Wild_PrepareFineFilter(BOOL CallBackAttr(ATTRIB Attrib))
/* Prepare FineFilter so we don't have to keep invoking the CallBackAttr
 * filter function.  Also prepare CoarseFilter -- special types of files
 * EVER wanted.  We can use CoarseFilter to get DOS to do some gross filtering. */
    {
    ATTRIB	    Attrib;
    for (Attrib = 0; Attrib < 64; Attrib++)
      {
      if (FineFilter[Attrib] = CallBackAttr(Attrib))
	CoarseFilter |= Attrib;
      }
    }

/* ************************************** */

local void	Wild_GrowPointers(void)
/* We have run out of room for our collection of filename pointers.
 * Expand the size of our array */
    {
    /* Grow the array of pointers by 2048 elts */
    fargcMars = fargc + PointersChunk;
    fargv = (char **) Wild_Alloc(fargv, fargcMars * sizeof(char *));
    }

/* ************************************** */

local void	Wild_GrowStrings(void)
/* We have run out of room for our collection of filename strings.
 * Create yet another basket of strings. */

    {
    BASKET	   *OldBasketStart = BasketStart;

    /* HashTable stays the same size */
    /* allocate another 8K basket of strings */
    BasketStart = (BASKET *) Wild_Alloc(NULL, BasketChunk);

    /* Store pointer to prev basket at the head of this new basket. Leave
     * it NULL if this is the first basket. */
    BasketStart->PrevBasket = OldBasketStart;

    BasketEnd = (char *) &BasketStart->Entries[0];

    BasketMars = BasketEnd + BasketChunk - sizeof(ENTRY);
    /* if get out to this point, we should consider the basket filled */
    }

/* ************************************** */

local void	Wild_EnsureRoom(void)
/* ensure there is room to add at least one more filename string to our
 * basket */
    {
    if (fargc >= fargcMars)
      Wild_GrowPointers();
    if (BasketEnd >= BasketMars)
      Wild_GrowStrings();
    }

/* ************************************** */

local void	Wild_Tidy(char *WildCard)

/* Tidy up the WildCard to make it as short as possible.
 *
 * There are three sources of information:
 *
 * 1. The CurrentDirectory.  Calculated once.
 *
 * 2. The WildCard spec from the "commandline".  Calculated once
 * per commandline item.
 *
 * 3. The filename without directory info that findnext hands us
 * back. We don't yet have #3.
 *
 * We are processing #2 here.
 *
 * We want to figure out how to write the wildcard drive and
 * directory as compactly as possible, since we will have to
 * prefix it to every file we find.
 *
 * There are four tricks we can use:
 *
 * 1. drop drive if it is the current drive.
 *
 * 2. drop directory if it is the current directory.
 *    e.g. \MOM\ME\ to nothing.
 *
 * 3. convert \GRANNY\MOM\ME\SON\ to SON\
 *
 * 4. convert \GRANNY\MOM\BROTHER to ..\BROTHER
 *
 */
    {
    char	    Drive[_MAX_DRIVE];
    char	    Dir[_MAX_DIR];
    char	   *DirPtr = Dir;
    char	    Fname[_MAX_FNAME];
    char	    Ext[_MAX_EXT];
    char	    Path[FILENAME_MAX];

    /* tidy up name to get rid of complex .\ ..\ etc. */
    (void) _fullpath(Path, WildCard, FILENAME_MAX);

    (void) strupr(Path);
    /* want all upper case for fast compare */

    _splitpath(Path, Drive, Dir, Fname, Ext);

    /* now work on shortening the name.  Chop drive if poss. */

    if (strncmp(Drive, CurrentDrive, 2) == 0)
      {
      *Drive = EOS;

      /* Drive matches, now if dir is current dir, it too can be removed.
       * CurrentDir has a trailing \
       * Following code also handles \MOM\ME\ to nothing.  It comes out in the wash. */

      /* Try to convert \GRANNY\MOM\ME\SON\ to SON\ */
      if (strncmp(Dir, CurrentDir, CurrentDirLen) == 0)
	{
	DirPtr += CurrentDirLen;
	/* Cut off the prefixing \GRANNY\MOM\ME\ */
	}

      /* Otherwise try converting convert \GRANNY\MOM\BROTHER\ to ..\BROTHER\
       * Don't bother unless it would shorten the name. */
      else if ((CurrentMomLen > 3) &&
	(strncmp(Dir, CurrentMom, CurrentMomLen) == 0))
	{
	DirPtr += CurrentMomLen - 3;
	strncpy(DirPtr, "..\\", 3);
	/* Cut off the prefixing MOM, replace with ..\ */
	}
      }
    /* glue pieces back together again, back on top of original input */
    _makepath(WildCard, Drive, DirPtr, Fname, Ext);
    return;
    }

/* ************************************** */

local void	Wild_Graft(char *WildCard, char *FileName)
/* Graft the FileName onto the end of Wildcard, replacing old filename in
 * Wildcard, if any. */
    {
    char	    Drive[_MAX_DRIVE];
    char	    Dir[_MAX_DIR];

    _splitpath(WildCard, Drive, Dir, NULL, NULL);
    _makepath(WildCard, Drive, Dir, FileName, NULL);
    }

/* ************************************** */

local void	Wild_InitCRCTable(void)
/* build lookup table to help compute the CRC function.  Only needs be called once. */
    {
    WORD	    i;
    WORD	    crc;

#define Poly 0x8408
    /* we use 16-bit CCITT polynomial */

    for (i = 0; i < 256; i++)
      {
      crc = (WORD) i;
      /* ^ is bitwise XOR */
      crc = (crc >> 1) ^ ((crc & 1) ? Poly : 0);
      crc = (crc >> 1) ^ ((crc & 1) ? Poly : 0);
      crc = (crc >> 1) ^ ((crc & 1) ? Poly : 0);
      crc = (crc >> 1) ^ ((crc & 1) ? Poly : 0);
      crc = (crc >> 1) ^ ((crc & 1) ? Poly : 0);
      crc = (crc >> 1) ^ ((crc & 1) ? Poly : 0);
      crc = (crc >> 1) ^ ((crc & 1) ? Poly : 0);
      crc = (crc >> 1) ^ ((crc & 1) ? Poly : 0);

      CRCTable[i] = crc;
      }
    }

/* ************************************** */

local WORD	Wild_HashCRC(ENTRY * Entry)
/* Calculate a hash CRC-style checksum on a counted string. len is the
 * unsigned length byte at the head of the counted string. Note: Length
 * byte is included in the CRC, but not trailing EOS. */
    {
    WORD	    len = (Entry->StringLen) + 1;
    /* length of string to hash, plus len byte */
    BYTE *	    c	= (BYTE *) &Entry->StringLen;
    /* character of the string, start with len byte */
    WORD	    crc = 0;

    while (len--)
      crc = (crc >> 8) ^ CRCTable[(crc & 0xff) ^ *c++];
    /* ^ is bitwise XOR */

    return (crc % HashPrime);
    }

/* ************************************** */

ENTRY * Wild_FindEntry(ENTRY * LookFor, WORD Hash)

    { /* LookFor is the string we are looking for.
       * Hash is its CRC hash modulo the
       * HashPrime. */

    ENTRY	   *fp; /* pointer used to chase the hash chain. */
    WORD	   len = LookFor->StringLen;
    for (fp = HashTable[Hash]; fp; fp = fp->PrevEntrySameHash)
      { /* chase the chain of prev pointers till
	 * we find a match, or we run off the end,
	 * indicated by a null PrevEntrySameHash. */
      if ( fp->StringLen == len &&
	!strncmp(fp->AString, LookFor->AString, len))
	break;
      }
    return (fp);
    }

/* ************************************** */

local void	Wild_ProcessHit(char *WildCard, char *Hitname)
/* FindNext has found us a file-type match. Process one file found as a
 * result of a positive or negative wildcard.
 *
 * Pointers to entries are stored in a HashTable to speed up searches for
 * existing files.
 *
 * First test if the file is already present.  If it is, and
 * NegativeWildCard == TRUE, we clear the first byte in the string and
 * exit. Otherwise we add it to the list of filenames and to HashTable.
 *
 * WildCard has been previously tidied to as compact a representation as possible.
 */

    {
    ENTRY	   *Entry,
		   *OldEntry;
    WORD	    Hash;

    Entry = (ENTRY *) BasketEnd; /* point to free space after existing
				  * strings in basket */

    /* copy the directory and tack on the file name.
     * We don't need to call Wild_Tidy, since the WildCard directory name has
     * already been compacted with Wild_Tidy. */
    Wild_Graft(strcpy(Entry->AString, WildCard), Hitname);

    /* save away length of string.  We don't necessarily permanently
     * allocate this space to this string. We might just abandon it in the upcoming code. */
    Entry->StringLen = (BYTE) strlen(Entry->AString);

    /* check if it's already in the basket */
    Hash = Wild_HashCRC(Entry);
    if (NegativeWildCard)
      {
      /* was negative */
      if (OldEntry = Wild_FindEntry(Entry, Hash))
	{			    /* delete the string. fargc pointer to it
				     * will be removed later in Wild_Prune */
	OldEntry->StringLen = 0;    /* yes, clear length byte	*/
	OldEntry->AString[0] = EOS; /* and first char of string.
				     * We leave this on the hash chain,
				     * so we need to ensure it does not
				     * get picked up as a match. */
	}

      else { /* negative wildcard, but name not found, so nothing to do */ }
      }
    else
      {
      /* was positive Wildcard */

      if (OldEntry = Wild_FindEntry(Entry, Hash))
	return; /* it's in there already - nothing more to
		 * do */

      /* Positive Wildcard was not in there already.  Add it */

      Entry->PrevEntrySameHash = HashTable[Hash]; /* link it into the hash table */

      HashTable[Hash] = Entry;

      fargv[fargc++] = Entry->AString; /* add it to our list of pointers */

      BasketEnd += Entry->StringLen + sizeof(Entry->PrevEntrySameHash) + sizeof(Entry->StringLen) +1;
      /* adjust end of basket to account for stuff we added. */
      /* account for string, prev pointer, length byte, EOS */

      Wild_EnsureRoom(); /* make sure we have enough room for next
			  * one */
      }
    }

/* ************************************** */

local void	Wild_ProcessHits(char *WildCard)
/* Process file-type hits for this wildcard.  Here we handle both
 * positive and negative WildCards.  We could improve negative WildCard
 * processing by not looking at the disk with _dos_findnext, and instead
 * just examine the strings in our baskets to see if any match the
 * negative WildCard. However, that would mean duplicating the pattern
 * matching code inside _dos_findnext.	Wildcard has been tidied previously
 * to give us as compact a representation as possible. */

    {
    struct _find_t  Hit; /* a file found by wildcard search, what
			  * _dos_findnext returns.  Since we don't
			  * call this routine recursively, it is ok
			  * to allocate Hit on the stack */

    if (WildCard && WildCard[0]) /* not null */
      if (_dos_findfirst(WildCard, CoarseFilter, &Hit) == 0)
	do
	  {
	  if (FineFilter[Hit.attrib])
	    Wild_ProcessHit(WildCard, Hit.name);
	  } while (_dos_findnext(&Hit) == 0);
    }

/* ************************************** */

local void Wild_GraftLevel(char *SubWildCard, char *WildCard, char *Level)

/* insert Level into WildCard forming SubWildCard, so we can recurse one
 * level deeper into the directory structure.  Note Dir will usually
 * have lead and trail backslash. */
    {
    char	    Drive[_MAX_DRIVE];
    char	    Dir[_MAX_DIR];
    char	    Fname[_MAX_FNAME];
    char	    Ext[_MAX_EXT];

    _splitpath(WildCard, Drive, Dir, Fname, Ext);
    /* Dir will usually have lead and trail backslash */

    strcat(Dir, Level);
    /* Insert the extra level of directory we need to search */

    _makepath(SubWildCard, Drive, Dir, Fname, Ext);
    /* Glue pieces back together again */

    Wild_Tidy(SubWildCard);
    /* encode as compactly as possible. */
    }

/* ************************************** */

local void	Wild_ProcessSubs(char *WildCard)
/* We recursively process all WildCard-matching ordinary files and all
 * subdirs of this dir.  SlashS type processing is in effect. Since we
 * are processing recursively we use malloc rather than the stack to avoid
 * overflowing it. */

    {
    struct _find_t *SubHit
    = (struct _find_t *) Wild_Alloc(NULL, sizeof(struct _find_t));
    /* a subdir found by _dos_findnext */

    char	   *SubWildCard
    = (char *) Wild_Alloc(NULL, FILENAME_MAX);
    /* e.g. /MyDir/MySub/*.* to search for any subdirs and also
     * /MyDir/MySub/*.H to process WildCard-matching files in those
     * subdirs.  The reason we can use this variable for dual purpose is
     * that _dos_findfirst encodes everything it needs for _dos_findnext
     * in SubHit so we can recycle SubWildCard. */

    Wild_ProcessHits(WildCard); /* process ordinary files first */

    strcpy(SubWildCard, WildCard);
    Wild_Graft( SubWildCard, "*.*"); /* ignore file part when searching
				     * for dirs */

    /* process only Subdirs, including hidden, system etc. subdirs */
    if (_dos_findfirst(SubWildCard, (_A_SUBDIR | _A_ARCH |
      _A_RDONLY | _A_HIDDEN |
      _A_SYSTEM), SubHit) == 0)
      do
	{
	/* ignore all hits but true subdirs */
	if ((SubHit->attrib & _A_SUBDIR) && SubHit->name[0] != '.')
	  {
	  Wild_GraftLevel(SubWildCard, WildCard, SubHit->name);
	  Wild_ProcessSubs(SubWildCard);
	  }
	}
      while (_dos_findnext(SubHit) == 0);

    free(SubWildCard);
    free(SubHit);

    }

/* ************************************** */

local void	Wild_Shrink(void)
/* We can now finally shrink our results, just pointers, but not
 * strings, to minimum size.  We strip down as best we can, ready for
 * handing the strings over to the user.  We would like to leave him as
 * much free RAM as possible. c.f. Wild_Prune.
 *
 * Ideally we would shrink each basket of strings too, removing deleted
 * strings, and changing all the pointers to them. */
    {

    /* shrink the array of string pointers to the precise needed size. */
    fargcMars = fargc;
    fargv = (char **) Wild_Alloc(fargv, max(fargc * sizeof(char *), 1));

    }

/* ************************************** */

local void	Wild_Prune(void)
/* Squeeze out files removed by neg wildcards, just pointers but not
 * strings.  We could rebuild the HashTable, since some entries will be
 * gone.  However, that effort would almost certainly not be worth it
 * since we are unlikely to search again for the filenames we just removed.
 * c.f. Wild_Shrink. */

    {
    WORD	    i,
      j;
    /* if all pointers are non-null, this does nothing */

    /* squeeze up the pointers */
    for (i = 0, j = 0; i < fargc; i++)
      if (fargv[i] && *fargv[i]) /* pointer must not be null.  String must
				  * not be null */
	fargv[j++] = fargv[i];
    fargc = j;

    /* Ideally we would squeeze out the strings too and adjust all the
     * pointers. */
    }

/* ************************************** */

local void	Wild_Expand(char *UserWildCard)
/* Expand given pos or neg WildCard into separate filenames and put them
 * in the Basket.  WildCard may also have an immediately trailing /S.
 * If next parm, separated by a space, is a /S, the caller, Wild,
 * has already set SlashS TRUE. */

    {

    WORD	    len; /* working variable, length of Wildcard */

    char	    WildCard[FILENAME_MAX];
    /* make our own copy we can meddle with */

    if (!UserWildCard[0])
      return; /* nothing to process */

    /* handle leading minus */
    if (NegativeWildCard = (BOOL) (*UserWildCard == '-'))
      UserWildCard++; /* negative WildCard, bypass the - */

    strcpy(WildCard, UserWildCard);

    if (WildCard[0] == EOS)
      return; /* nothing to process */

    /* handle immediately trailing /S */
    /* SlashS may already be set TRUE if succeeding parm, separated by space in a /S */
    if ((len = strlen(WildCard)) >= 2)
      /* check for /S on the end. */
      if (stricmp(WildCard + len - 2, "/s") == 0)
	{
	SlashS = TRUE;
	/* chop the /S off */
	WildCard[len - 2] = EOS;
	}

    if (!WildCard[0])
      return; /* nothing to process */

    /* handle using prev dir as default for successor */
    if (!strpbrk(WildCard, ":\\"))

      { /* We found no : or \ so there was no
	 * drive or dir. Give it the same
	 * drive/dir as prev WildCard since no dir
	 * specified. Also give it same /S,
	 * however don't inherit the negative
	 * sign.  Note we do this BEFORE Wild_Tidy */

      Wild_Graft(PrevWildCard, WildCard);
      strcpy(WildCard, PrevWildCard);
      SlashS |= PrevSlashS; /* If previous wildcard had /S, simulate another, but only
			       when no drive or dir specified on this wildcard. */
      }

    Wild_Tidy(WildCard); /* encode the wildcard as compactly as
			  * possible */

    strcpy(PrevWildCard, WildCard); /* save for next WildCard */

    if ((PrevSlashS = SlashS))
      Wild_ProcessSubs(WildCard);
    else
      Wild_ProcessHits(WildCard);

    if (NegativeWildCard)
      Wild_Prune(); /* throw out pointers to deleted strings,
		     * after each negative wildcard processed. */
    }

/* ************************************** */

local void	Wild_GetCurrentDir(void)

/* get FullCWD. Break it into pieces CurrentDrive, CurrentDir, and
 *  CurrentMom */
    {
    char	   *End; /* used to chop Mom to proper length */
    char	    FullCWD[_MAX_DIR + 4];
    /* fully qualified current working directory Where we start out and
     * where we end up.  Might be on any drive, any directory. */

    (void) getcwd(FullCWD, _MAX_DIR);
    /* FullCWD is all upper case already. Has no trail backslash. */

    /* SplitPath wants a filename, not a dirname */
    strcat(FullCWD, "\\*.*");
    _splitpath(FullCWD, CurrentDrive, CurrentDir, NULL, NULL);
    /* this gets us CurrentDrive and CurrentDir */
    /* CurrentDir will have a trailing backslash */
    CurrentDirLen = strlen(CurrentDir);

    strcpy(CurrentMom, CurrentDir);
    /* chop of the trailing \ */
    CurrentMom[CurrentDirLen - 1] = EOS;

    if (End = strrchr(CurrentMom, '\\'))
      { /* is a mom */
      *(++End) = EOS;
      }
    else
      { /* is no mom */
      CurrentMom[0] = EOS;
      }
    CurrentMomLen = strlen(CurrentMom);
    }

/* ************************************** */

extern void	Wild_Free(void)

/* free up all the dynamic storage allocated for WildCard expansions */
    {
    BASKET	   *OldBasketStart;

    while (BasketStart)
      { /* chase basket chain, newest first,
	 * freeing all baskets */
      OldBasketStart = BasketStart->PrevBasket;
      /* pointer at head of Basket will let us find prev basket */
      free(BasketStart);
      BasketStart = OldBasketStart;
      }
    free(fargv); /* free all the pointers */
    }

/* ************************************** */

extern void	Wild(BOOL CallBackAttr(ATTRIB Attrib),
			   void CallBackProcess(WORD fargc, char **fargv),
				     WORD wargc,
				     char **wargv)

/* This is the top banana routine, the one the user calls to start
 * wildcard processing.  The user provides this routine with a call-back
 * function to define the attributes he likes, a call-back function to
 * process the resulting filename pointers (fargv), and a array of
 * pointers, wargv, to a list of wildcards to process. wargc counts how
 * many wildcards there are to expand. */

    {
    WORD	    i,
      j; /* used to scoot over the set of WildCards
	  * the user provides */

    /* prepare FineFilter so we don't have to keep invoking the
     * user's slow CallBackAttr filter function. */
    Wild_PrepareFineFilter(CallBackAttr);

    PrevWildCard[0] = EOS; /* no prev WildCard at this point */
    PrevSlashS = FALSE;    /* no prev WildCard */

    CLEAR(HashTable);

    Wild_InitCRCTable();
    /* build table to compute CRC hash functions */

    BasketStart = NULL;
    Wild_GrowStrings();

    /* allocate the array of pointers to the strings */
    fargc = 0;
    /* NOTE: fargc counts our internally generated list of expanded file
     * names.  wargc, the parameter, counts how many wildcards we were
     * handed to process.  When we are done we will give OUR fargv list
     * back to the user */
    Wild_GrowPointers();

    Wild_GetCurrentDir();

    for (i = 0; i < wargc; i++)
      /* expand each non-null WildCard in turn */
      /* We process wargv[0] as well.  If user is passing us argv, it is
       * up to him to bypass the progname in the argv[0] */
      {

      /* determine if next non null parm is a /S */
      SlashS = FALSE;
      for (j = i + 1; j < wargc && wargv[j] == NULL; j++)
	{ ; }
      if (j < wargc && stricmp(wargv[j], "/s")==0)
	SlashS=TRUE;

      /* expand the wildcard */
      if (wargv[i])
	Wild_Expand(wargv[i]);
      }

    Wild_Shrink();
    /* shrink RAM usage back to the minumum */

    CallBackProcess(fargc, fargv);
    /* allow user to process the array of pointers to the strings.  We
     * don't free them.  The user might. That's all folks! */
    }

/* ************************************** */
/* -30- end Wild.C */
