/*
    File......: GT_Browse.prg
    Author....: Martin Bryant
    BBS.......: The Dark Knight Returns
    Net/Node..: 050/069
    User Name.: Martin Bryant
    Date......: 12/02/93
    Revision..: 1.0

    This is an original work by Martin Bryant and is placed
    in the public domain.

    Modification history:
    ---------------------

    Rev 1.0 12/02/93
    PD Revision.
*/

/*  $DOC$
 *  $FUNCNAME$
 *      GT_BROWSE()
 *  $CATEGORY$
 *      General
 *  $ONELINER$
 *      Browse a DBF with for and while clauses.
 *  $SYNTAX$
 *      GT_Browse(<aFields>,<aTitles>,[<cHead>],[<nTop>], ;
 *          [<nLeft>],[<nBottom>],[<nRight>],[<bFind>], ;
 *          [<bFor>],[<nFreeze>],[<bKeyEx>],[<bWaitFunc>], ;
 *          [<cBox>],[<lScreen>]) -> lSelected
 *  $ARGUMENTS$
 *      <aFields> is an array of code blocks defining the
 *      information to be displayed.
 *
 *      <aTitles> is an array of strings to display as
 *      column titles.
 *
 *      <cHead> is a title for the Browse.
 *
 *      <nTop>,<nLeft>,<nBottom>,<nRight> are the table
 *      corners.
 *
 *      <bFind> is the while clause code block to compare
 *      against the index key.
 *
 *      <bFor> is the for condition code block.
 *
 *      <nFreeze> is the number of columns to freeze from
 *      the left.
 *
 *      <bKeyEx> is a code block containing the code for
 *      handling key exceptions. Should return .T. if an
 *      item has been selected, otherwise .F..
 *
 *      <bWaitFunc> is the code block defining the 'Waiting
 *      For a Key Press' function.
 *
 *      <cBox> box lines definition.
 *
 *      <lScreen> Should the screen be saved and restored ?
 *  $RETURNS$
 *      lSelected
 *  $DESCRIPTION$
 *      Browse a DBF with for and while clauses.
 *  $EXAMPLES$
 *  $SEEALSO$
 *
 *  $INCLUDE$
 *
 *  $END$
 */

#include "GT_LIB.ch"

FUNCTION GT_Browse(aFields,aTitles,cHead,nTop,nLeft, ;
    nBottom,nRight,bFind,bFor,nFreeze,bKeyEx,bWaitFunc, ;
    cBox,lRestScreen)

LOCAL aStdKeys := { ;
    K_DOWN, ;
    K_UP, ;
    K_PGDN, ;
    K_PGUP, ;
    K_CTRL_PGUP, ;
    K_CTRL_PGDN, ;
    K_RIGHT, ;
    K_LEFT, ;
    K_HOME, ;
    K_END, ;
    K_CTRL_LEFT, ;
    K_CTRL_RIGHT, ;
    K_CTRL_HOME, ;
    K_CTRL_END }

LOCAL aStdFuncs := { ;
    { | oTBCol | oTBCol:Down() }, ;
    { | oTBCol | oTBCol:Up() }, ;
    { | oTBCol | oTBCol:PageDown() }, ;
    { | oTBCol | oTBCol:PageUp() }, ;
    { | oTBCol | oTBCol:GoTop() }, ;
    { | oTBCol | oTBCol:GoBottom() }, ;
    { | oTBCol | oTBCol:Right() }, ;
    { | oTBCol | oTBCol:Left() }, ;
    { | oTBCol | oTBCol:Home() }, ;
    { | oTBCol | oTBCol:End() }, ;
    { | oTBCol | oTBCol:PanLeft() }, ;
    { | oTBCol | oTBCol:PanRight() }, ;
    { | oTBCol | oTBCol:PanHome() }, ;
    { | oTBCol | oTBCol:PanEnd() } }

//  Define index CB
LOCAL bIndex := GT_IndexBlock(0)
LOCAL bWhile := { | | .T. }
LOCAL cRestore := ''
LOCAL cSearch := ''
LOCAL lFault := .F.
LOCAL lAllRecords := .T.
LOCAL nCursor := GT_NoCursor()
LOCAL nIndex := INDEXORD()
LOCAL nKey := 0
LOCAL nLocation := 0
LOCAL nRecord := 0
LOCAL oBrowse := NIL
LOCAL uIndexkey := NIL

Default aFields To {}
Default aTitles To {}
Default cHead To ''
Default nTop To 03
Default nLeft To 02
Default nBottom To MAXROW() - 04
Default nRight To MAXCOL() - 02
Default bFind To NIL
Default bFor To NIL
Default nFreeze To 0
Default bKeyEx To { | | LASTKEY() = K_ENTER }
Default bWaitFunc To { | | INKEY(0) }
Default cBox To BOX_DD
Default lRestScreen To .F.

//  Pressed Enter too soon !
IF NEXTKEY() = K_ENTER
    KEYBOARD ''
ENDIF

//  find OK ?
IF bFind = NIL
    // All
    bWhile := { || .T. }
ELSE
    // Compare find to index
    bWhile := &("{ || " + INDEXKEY() + " = '" + EVAL(bFind) + "' }")
    lAllRecords := .F.
ENDIF

IF bFor = NIL
    bFor := { | | .T. }
ELSE
    lAllRecords := .F.
ENDIF

//  Restore screen
IF lRestScreen
    cRestore := SAVESCREEN(00,00,MAXROW(),MAXCOL())
ENDIF

//  Make new browse object
oBrowse := TBROWSENEW(nTop+1,nLeft+1,nBottom-1,nRight-1)

// Define Colours
oBrowse:colorspec := SETCOLOR()

// Define Column lines
oBrowse:headsep := REPLICATE(SUBSTR(cBox,2,1),3)

//  Move Blocks
oBrowse:gobottomblock := { | | GT_GoBottom(bWhile,bFor,bFind) }
oBrowse:gotopblock := { | | GT_GoTop(bWhile,bFor,bFind) }
oBrowse:skipblock := { | records | GT_Skip(bWhile,bFor,records,.F.) }

//  Add columns
AEVAL(aFields, { | data, elem | ;
    data := TBCOLUMNNEW(aTitles[elem],data) ;
    , IF(VALTYPE(EVAL(data:block))=='N' ;
        ,data:colorBlock := { | x | IF(x>0,{1,2},IF(x<0,{5,2},{1,2})) } ;
        ,NIL) ;
    ,data:defcolor := {1,2} ;
    ,oBrowse:AddColumn(data) },1,LEN(aFields))

//  Clear and draw box
GT_Window(nTop,nLeft,nBottom,nRight,cBox,SETCOLOR(),cHead)

//  Fix cols ?
IF nFreeze > 0
    // Fix columns
    oBrowse:Freeze := nFreeze
ENDIF

DO WHILE .T.

    // Stabilize the display
    DISPBEGIN()
    DO WHILE ( !oBrowse:Stabilize() )
        nKey := INKEY()
        IF ( nKey != 0 )
            EXIT
        ENDIF
    ENDDO
    DISPEND()

    IF ( oBrowse:Stable )
        // Display is Stable
        nKey := EVAL(bWaitFunc)
    ENDIF

    IF .NOT. GT_IsData(nKey)
        cSearch := ''
    ENDIF

    // Process key
    nLocation := ASCAN(aStdKeys,nKey)

    DO CASE
    CASE nLocation > 0
        // Standard Key function
        EVAL(aStdFuncs[nLocation],oBrowse)

    CASE (nKey == K_ESC )
        // Exit
        EXIT

    CASE IF(nKey<0,SETKEY(nKey)!=NIL,.F.)
        // Other function key
        EVAL(SETKEY(nKey))
        // Force redisplay of current row
        oBrowse:RefreshCurrent()

    CASE GT_IsData(nKey) .AND. .NOT. EMPTY(INDEXKEY())
        // Add to search string
        cSearch += UPPER(CHR(nKey))

        // Find ?
        nRecord := RECNO()
        DBSEEK(cSearch)
        IF EOF()
            DBGOBOTTOM()
        ENDIF
        lFault := .T.
        DO WHILE EVAL(bWhile) .AND. .NOT. EOF()
            IF EVAL(bFor)
                lFault := .F.
                EXIT
            ENDIF
            DBSKIP(1)
        ENDDO

        // Refresh screen or go back
        IF .NOT. lFault
            nRecord := RECNO()
            oBrowse:GoTop()
            DBGOTO(nRecord)
            oBrowse:RefreshAll()
        ELSE
            DBGOTO(nRecord)
        ENDIF

    OTHERWISE
        // Key exception
        DO WHILE ( !oBrowse:Stabilize() ) ; ENDDO

        // Save old index and record
        uIndexkey := EVAL(bIndex)
        nRecord := RECNO()

        IF EVAL(bKeyEx,oBrowse)
            // One selected
            EXIT
        ENDIF

        // Changed index ?
        IF INDEXORD() != nIndex
            bIndex := GT_IndexBlock(0)
            nIndex := INDEXORD()
            nRecord := -1
        ENDIF

        //  Still visible ?
        IF EOF() .OR. .NOT. (EVAL(bWhile) .AND. EVAL(bFor))
            // Find a visible one and ensure refreshall
            oBrowse:Gotop()
        ENDIF

        // Refresh all or one ?
        IF !(uIndexkey == EVAL(bIndex)) .OR. RECNO() != nRecord
            oBrowse:RefreshAll()
        ELSE
            oBrowse:RefreshCurrent()
        ENDIF

    ENDCASE

ENDDO

//  restore old cursor
SETCURSOR(nCursor)

//  Restore Screen
IF lRestScreen
    RESTSCREEN(00,00,MAXROW(),MAXCOL(),cRestore)
ENDIF

/*
    End of GT_Browse()
*/
RETURN(nKey != K_ESC) // 

