/*
 * Copyright (C) 1992, 1995 Stephen A. Wood.  All rights reserved.
 * 
 * This file is part of PAF C-Tools.
 * 
 * These programs are free software; you can redistribute them and/or modify
 * them under the terms of the GNU General Public License as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.
 * 
 * These programs are distributed in the hope that they will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
 * or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License along with
 * this program; if not, write to the Free Software Foundation, Inc., 675
 * Mass Ave, Cambridge, MA 02139, USA. The GNU General Public License can be
 * found in the file LICENSE.
 * 
 * The author of PAF C-Tools is Stephen A. Wood.  He may be reached on internet
 * at the address saw@cegaf.gov, or via mail at 328 Dominion Drive, Newport
 * News, VA 23602.
 * 
 * ------------
 * 
 * Dump all the notes in a PAF database to standard output. The output may
 * directed to a file, which can then be edited with any text editor.  The
 * edited notes can then be used to update the PAF database with the program
 * nupdate.
 */

#include <stdio.h>
/* #include <time.h> */
#include "pafsubs.h"

#define NLINE_MLEN 79
int dumpall_flag=FALSE;         /* Dump entries with no notes too */
int appendpaf_flag=FALSE;       /* Appends PAF fields to end of notes */
char *append_prefix="";         /* Put this string in front of appended info */

void printnotes(RECORD_PTR rin);
char *convert_name(char *s, RECORD_PTR * ptrs);
char *convert_place(char *s, RECORD_PTR *ptrs);
char *convert_date(char *s, LONGDATE packed_date);

main(int argc, char *argv[])
{
  RECORD_PTR      rin;
  RECORD_PTR      rin_max;
  NAMADD_REC      namadd;
  PAF_FILE       *indiv_file;
  int             anum;
  char *path="";                        /* Path for files */
  int bad_syntax=FALSE;
  
  while(argc-- > 1 && !bad_syntax){
    argv++;
    if(argv[0][0] != '-'){
      path = *argv;
    } else
      switch(argv[0][1]){
      case 'e': case 'E':
	dumpall_flag = TRUE;
	break;
      case 'a': case 'A':
	appendpaf_flag = TRUE;
	dumpall_flag = TRUE;
	if(argv[0][2] != '\0') append_prefix = argv[0]+2;
	break;
      case 0:
      case 'h': case 'H':
      default:
	bad_syntax = TRUE;
	break;
      }
  }
  if(bad_syntax) {
    fputs("usage: notedump [path] [-a[PREFIX]] [-e]\n",stderr);
    fputs("  Writes the notes from the PAF database in the current directory\n",stderr);
    fputs("  (or the directory \"path\") to the standard output in ASCII form\n",stderr);
    fputs("\n  -a appends PAF info to end of each individuals notes\n",stderr);
    fputs("    PREFIX is a optional string written in front appended info\n"
	  ,stderr);
    fputs("  -e dumps every individual, even if no notes\n",stderr);
    fputs("  [path] specifies what directory to find the PAF database\n",stderr);
    exit(1);
  }

  if(paf_open_name(path,'r') || paf_open_indiv(path,'r') || paf_open_marr(path,'r')
     || paf_open_notes(path,'r') || paf_open_namadd(path,'r')) {
    fprintf(stderr,"Failed to open PAF database in directory %s\n",path);
    exit(1);
  }

  indiv_file = get_paf_filed(INDIV);
  rin_max = indiv_file->nrec;
  
  get_namadd_rec((RECORD_PTR) 1, &namadd);
  printf("%% %s\n", namadd.name);
  printf("%% %s\n", namadd.addr1);
  printf("%% %s\n", namadd.addr2);
  printf("%% %s\n", namadd.addr3);
  printf("%% %s\n", namadd.phone);
  printf("%%\n");
  
  for (rin = 1; rin <= rin_max; rin++) {
    printnotes(rin);
  }
  return (0);
}

void 
  printnotes(RECORD_PTR rin)
{
  INDIV_REC       indiv;
  NOTE_REC        pad;
  int             i;
  char            s1[100],s2[100];
  
  get_indiv_rec(rin, &indiv);
  
  /* Only proceed if individual is not deleted and has a note. */
  if (indiv.sex != 'D' && (indiv.notepad != 0 || dumpall_flag)) {
    printf("%%%% [%d]", rin);
    convert_name(s1, indiv.names);
    printf(" %s ", s1);
    printf(" %%%%");
    /* First note will insert newline */
    {
      int             llen;
      int             newline;
      RECORD_PTR      npad;
      llen = 0;
      newline = 0;
      npad = indiv.notepad;
      while (npad != 0) {       /* Chain through notepads */
	/* printf("[%d]",npad); */
	get_note_rec(npad, &pad);
	npad = pad.next_pad;
	for (i = 0; i < 254; i++) {
	  if (pad.notelines[i] == 0)
	    if (llen == 0) {
	      llen = 1;
	    } else if (llen == 1) {
	      llen = 0;
	      break;    /* Zero length line terminates notes */
	    } else {
	      llen = 0;
	      putchar('\n');
	    }
	  else if (llen == 0 && pad.notelines[i] == 1) {
	    putchar('\n');      /* Start of new note */
	    newline = 1;
	    llen++;
	  } else {
	    putchar(pad.notelines[i]);
	    llen++;
	  }
	}
      }
      if (!newline)
	putchar('\n');  /* Just in case note was blank */
    }
    if (appendpaf_flag) {
      RECORD_PTR next_marriage, srin;
      MARR_REC marr;
      INDIV_REC spouse;
	
      if(*append_prefix != '\0') printf("%s ",append_prefix);
      printf("%s [%.10s]\n", s1,indiv.idnum);/* s is the name */

      convert_date(s1,indiv.birthdate);
      convert_place(s2,indiv.birthplace);
      if(s1[0] != 0 || s2[0] != 0) printf("Born: %s %s\n",s1,s2);

      convert_date(s1,indiv.christdate);
      convert_place(s2,indiv.christplace);
      if(s1[0] != 0 || s2[0] != 0) printf("Christened: %s %s\n",s1,s2);

      convert_date(s1,indiv.deathdate);
      convert_place(s2,indiv.deathplace);
      if(s1[0] != 0 || s2[0] != 0) printf("Died: %s %s\n",s1,s2);

      convert_date(s1,indiv.burialdate);
      convert_place(s2,indiv.burialplace);
      if(s1[0] != 0 || s2[0] != 0) printf("Burried: %s %s\n",s1,s2);

      next_marriage = indiv.ownmarriage;
      if(next_marriage != 0) printf("Marriages:\n");
      while(next_marriage != 0){
	get_marr_rec(next_marriage,&marr);
	if(indiv.sex == 'M'){
	  next_marriage = marr.mnext_husband;
	  srin = marr.wife;
	} else {
	  next_marriage = marr.mnext_wife;
	  srin = marr.husband;
	}
	if(srin != 0) {
	  get_indiv_rec(srin,&spouse);
	  convert_name(s1, spouse.names);
	  printf(" %s [%.10s], ", s1,spouse.idnum);
	} else {
	  printf(" Unknown, ");
	}
	convert_date(s1,marr.marriagedate);
	convert_place(s2,marr.marriageplace);
	printf("%s %s",s1,s2);
	if(marr.divorce=='Y') {
	  printf(" Divorced\n");
	}
	printf("\n");
      }
    }
    printf("%%%%--------------------------------------------------------------\n");
  }
  return;
}
char *convert_name(char *s, RECORD_PTR * ptrs)
{
  int             i;
  NAME_REC        namrec;
  s[0] = 0;
  
/*  {int flag;
   flag = 0;
   for(i=0; i<5; i++){
     if(ptrs[i] > 864) flag = 1;
   }
   if(flag) printf("Bad nam ids: %d %d %d %d %d\n",ptrs[0],ptrs[1],ptrs[2],ptrs[3]
	  ,ptrs[4]);
 }*/
  for (i = 1; i < 4; i++) {     /* Print three given names */
    if (ptrs[i] != 0) {
      get_name_rec(ptrs[i], &namrec);
      strcat(s, namrec.name);
      strcat(s, " ");
    }
  }
  if (ptrs[0] != 0) {           /* Print last name */
    get_name_rec(ptrs[0], &namrec);
    strcat(s, namrec.name);
  } else {
    strcat(s, "????");
  }
  if (ptrs[4] != 0) {           /* Print title */
    get_name_rec(ptrs[4], &namrec);
    strcat(s, ", ");
    strcat(s, namrec.name);
  }
  return(s);
}
char *convert_place(char *s, RECORD_PTR *ptrs)
{
  int i;
  NAME_REC namrec;
  int empty;

  s[0] = 0;
  empty = TRUE;
  for(i=0; i<4; i++) {
    if(ptrs[i] != 0) {
      get_name_rec(ptrs[i], &namrec);
      strcat(s, namrec.name);
      empty = FALSE;
    }
    if(i<3) strcat(s,",");
  }
  i=strlen(s);
  while(s[--i] == ',') s[i] = 0;
/*  if(empty) s[0]=0;*/
  return(s);
}

char *monthnames[]={"NOMONTH","January","February","March","April",
		      "May","June","July","August","September",
		      "October","November","December","SEENOTES",
		      "SUBMITTED","CLEARED","UNCLEARED","COMPLETED",
		      "CANCELLED","STILLBORN","INFANT","DNS/CAN",
		      "DNS","BIC","CHILD"};

char *convert_date(char *s,LONGDATE packed_date)
{
  int year,month,day,dual;
  int i;
  
  if(packed_date == 0){
    s[0] = 0;
  } else {
    switch((packed_date & 0x30000)>>16)
      {
      case 0:
	strcpy(s,"BEF ");
	i = 4;
	break;
      case 1:
	strcpy(s,"ABT ");
	i = 4;
	break;
      case 2:
	s[0] = 0;
	i = 0;
	break;
      case 3:
	strcpy(s,"AFT ");
	i = 4;
	break;
      }
    year = (packed_date & 0xFF) << 4;
    year += (packed_date >> 12) & 0xF;
    month = ((packed_date & 0xF00) >> 7);
    month += ((packed_date & 0x800000) >>23);
    day = ((packed_date & 0x7C0000) >> 18);
    if(month==0&&day==0){
      if(year!=0) {
	if(day==0){
	  sprintf(s+i,"%d",year);
	}else{
	  sprintf(s+i,"%d %d",day,year);
	}
      }
    }else{
      if(day==0){
	sprintf(s+i,"%s %d",monthnames[month],year);
      }else{
	sprintf(s+i,"%s %d, %d",monthnames[month],day,year);
      }
    }
    if((dual=((packed_date & 0xFF000000)>>24)) != 0){
      i = strlen(s);
      sprintf(s+i,"/%d",year+dual);
    }
  }
}
