/*****************************************************************************
*       The Graphics Engine version 1.41                                     *
*                                                                            *
*       The Graphics Engine code and documentation are                       *
*       Copyright (c) 1993-1995 by Matthew Hildebrand; all rights reserved.  *
*                                                                            *
*       Unauthorised usage or modification of any or all of The Graphics     *
*       Engine is strictly prohibited.                                       *
*****************************************************************************/

#include <alloc.h>
#include <stdio.h>
#include <string.h>
#include "..\include\tge.h"
#include "..\include\varfont.h"

extern "C" FILE *TGE_openFile(char *filename, char *assumeExt, char *envVar, char *mode);



/**************************************************************************
*  Function:    VariableFont::VariableFont
*
*  Purpose:     Class constructor.
*
*  Entry:       N/A
*
*  Exit:        N/A
**************************************************************************/

VariableFont::VariableFont(void)
{
  //*** Initialize private data
  characterData = NULL;                     // signal no font loaded
  maxWide = maxHigh = 0;                    // greatest width and height
  spaceBetweenChars = 1;                    // default spacing:  1 pixel
  charBuf = NULL;                           // no character buffer
}



/**************************************************************************
*  Function:    VariableFont::~VariableFont
*
*  Purpose:     Class destructor.
*
*  Entry:       N/A
*
*  Exit:        N/A
**************************************************************************/

VariableFont::~VariableFont(void)
{
  //*** Free the character data if a font has been loaded
  if (characterData != NULL)
    farfree(characterData);
  if (charBuf != NULL)
    farfree(charBuf);
}



/**************************************************************************
*  Function:    VariableFont::load
*
*  Purpose:     Load a font and prepare it for use.
*
*  Entry:       filename = Filename of the font to load.
*
*  Exit:        Returns 0 on error.
**************************************************************************/

int VariableFont::load(char *filename)
{
  FILE *fontFile;
  char signature[8];
  unsigned long infoSize, fileOffset, maxImageSize=0L;
  unsigned count, ch;
  FontCharInfo huge *curPtr;
  unsigned status;
  char huge *addr;

  //*** Free the character data if a font has been loaded
  if (characterData != NULL)
    farfree(characterData);

  //*** Try to open the file
  if ((fontFile=TGE_openFile(filename,".fnt","TGEFONTS","rb")) == NULL)
    return (0);

  //*** Verify the font file signature
  if (!fread(signature,8,1,fontFile))       // read in the signature
    return (0);
  fileOffset = 8;                           // adjust file offset
  if (memcmp(signature, "TGEFONT2", 8))     // verify it
    return (0);

  //*** Skip past the font comment block, which ends with a NULL character
  do
  {
    if ((ch=fgetc(fontFile)) == EOF)        // read a character
      return (0);
    fileOffset++;                           // adjust file offset
  }
  while (ch != 0x00);                       // loop until a NULL byte is read

  //*** Read in the font palette
  if (!fread(fontPalette,768,1,fontFile))
    return (0);
  fileOffset += 768;                        // adjust file offset

  //*** Allocate memory for the font's character information
  fseek(fontFile, 0L, SEEK_END);            // seek to end-of-file
  infoSize = ftell(fontFile) - fileOffset;  // get information size
  fseek(fontFile, fileOffset, SEEK_SET);    // seek to old offset
  curPtr = (FontCharInfo huge*)farmalloc(infoSize); // allocate memory
  characterData = curPtr;                   // store address in class data

  //*** Read in the font data
  if (infoSize <= 0xFFFF)                   // is block <= 64 Kb in size?
  {                                         // yes, handle it the fast way
    if (!fread(characterData, (size_t)infoSize, 1, fontFile))  // read block
    {
      farfree(characterData);               // free memory block
      return (0);                           // return error code
    }
  }
  else                                      // block > 64 Kb, use slow method
  {
    status = 1;                             // initialize status flag
    addr = (char huge*) characterData;      // initialize current address
    while (infoSize)                        // loop until all data read
    {
      //*** Read a block of data, maximum 64 Kb at a time
      if (!fread((void*)addr, infoSize>=0xFFFF?0xFFFF:(size_t)infoSize, 1, fontFile))
      {                                     // handle the error
        farfree(characterData);             // free memory block
        status = 0;                         // error code for return
        break;                              // break out of loop
      }
      addr += infoSize>=0xFFFF ? 0xFFFF : (size_t)infoSize; // update address
      infoSize -= infoSize>=0xFFFF ? 0xFFFF : infoSize;     // update bytes
    }
    return (status);                        // return status flag
  }
  fclose(fontFile);                         // close font file

  //*** Initialize the FontCharInfo array
  for (count=0; count<256; count++)
  {
    //*** Set up the character's FontFileInfo entry
    characters[count].offsetFromTop = curPtr->offsetFromTop;
    characters[count].wide = curPtr->wide;
    characters[count].deep = curPtr->deep;
    if (curPtr->wide!=0 && curPtr->deep!=0) // address if character exists
      characters[count].bitmap = (void far*) (((char*)curPtr)+sizeof(short));
    else                                    // NULL if character doesn't exist
      characters[count].bitmap = NULL;

    //*** If this character's bitmap is the biggest so far, note its size
    if (imageSizeDim(curPtr->wide,curPtr->deep) > maxImageSize)
      maxImageSize = imageSizeDim(curPtr->wide, curPtr->deep);

    //*** Check if character is the widest or tallest so far
    if (characters[count].wide > maxWide)
      maxWide = characters[count].wide;
    if (characters[count].deep+characters[count].offsetFromTop > maxHigh)
      maxHigh = characters[count].deep + characters[count].offsetFromTop;

    //*** Point curPtr to the next character
    (char*)curPtr += characters[count].wide*characters[count].deep + 3*sizeof(short);
  }

  //*** Allocate a temporary buffer for characters
  if ((charBuf=farmalloc(maxImageSize)) == NULL)
  {
    farfree(characterData);                 // free characterd data
    characterData = NULL;                   // signal data freed
    return (0);                             // return error code
  }

  //*** Set the font colours to closely match the active palette
  matchColours();

  //*** Return success code
  return (1);
}



/**************************************************************************
*  Function:    VariableFont::put
*
*  Purpose:     Put a single character on the screen.
*
*  Entry:       x = X-coordinate at which to write the character.
*               y = Y-coordinate at which to write the character.
*               ch = Character to write.
*
*  Exit:        N/A
**************************************************************************/

void VariableFont::put(int x, int y, char ch)
{
  unsigned wide, deep, xCount, yCount;
  unsigned char huge *inPtr, *outPtr;

  //*** If the character doesn't exist, return immediately
  if (characters[ch].bitmap == NULL)
    return;

  //*** Set up the temporary bitmap's dimensions
  wide = imageWidth(characters[ch].bitmap);   // get width
  deep = imageHeight(characters[ch].bitmap);  // get height
  imageWidth(charBuf) = wide;                 // store width
  imageHeight(charBuf) = deep;                // store height

  //*** Generate the temporary bitmap using the colour translation table
  inPtr = (unsigned char huge*)characters[ch].bitmap + 4;
  outPtr = (unsigned char huge*)charBuf + 4;
  for (yCount=0; yCount<deep; yCount++)
    for (xCount=0; xCount<wide; xCount++,inPtr++,outPtr++)
    {
      if (*inPtr != 0)                        // if not transparent,
        *outPtr = colourTranslate[*inPtr];    // set the pixel
      else
        *outPtr = 0;                          // else keep it transparent
    }

  //*** Draw the character
  putImageInv(x, y+characters[ch].offsetFromTop, charBuf);
}



/**************************************************************************
*  Function:    VariableFont::put
*
*  Purpose:     Put a string on the screen.
*
*  Entry:       x = X-coordinate at which to write the string.
*               y = Y-coordinate at which to write the string.
*               string = String to write.
*
*  Exit:        N/A
**************************************************************************/

void VariableFont::put(int x, int y, char *string)
{
  unsigned char ch;

  //*** Draw the characters, one by one
  while ((ch=*string) != 0x00)              // repeat until end of string
  {
    //*** Draw the character if it exists
    if (characters[ch].bitmap != NULL)
    {
      put(x, y, ch);                                // draw the bitmap
      x += characters[ch].wide + spaceBetweenChars; // update x
    }

    //*** Skip to the next character
    string++;
  }
}


 
/**************************************************************************
*  Function:    VariableFont::width
*
*  Purpose:     Return the width, in pixels, of a string.
*
*  Entry:       string = String of which to find the width.
*
*  Exit:        Returns the width of the character in pixels,
*               accounting for blank space between characters.
**************************************************************************/

unsigned VariableFont::width(char *string)
{
  unsigned curWidth;
  unsigned char ch;

  //*** Find the width one character at a time
  curWidth = 0;                             // 0 pixels wide to begin
  while ((ch=*string) != 0x00)              // repeat until end of string
  {
    curWidth += characters[ch].wide;        // add character width
    curWidth += spaceBetweenChars;          // allow for space
    string++;                               // skip to the next character
  }
  curWidth -= spaceBetweenChars;            // remove extra spacing

  //*** Return the width
  return (curWidth);
}



/**************************************************************************
*  Function:    VariableFont::height
*
*  Purpose:     Return the height, in pixels, of a string.
*
*  Entry:       string = String of which to find the height.
*
*  Exit:        Returns the height of the string in pixels.
**************************************************************************/

unsigned VariableFont::height(char *string)
{
  unsigned curHeight, maxHeight=0;
  unsigned char ch;

  //*** Find the tallest character one character at a time
  curHeight = 0;                            // 0 pixels tall to begin
  while ((ch=*string) != 0x00)              // repeat until end of string
  {
    curHeight = characters[ch].deep + characters[ch].offsetFromTop;
    if (curHeight > maxHeight)              // is this character taller?
      maxHeight = curHeight;                // yes, new maximum height
    string++;                               // skip to the next character
  }

  //*** Return the height
  return (maxHeight);
}



/**************************************************************************
*  Function:    VariableFont::setColours
*
*  Purpose:     Set the font colours to closely match the active palette.
*
*  Entry:       N/A
*
*  Exit:        N/A
**************************************************************************/

void VariableFont::matchColours(void)
{
  unsigned char hasBeenFound[256];
  int charCount, xCount, yCount;
  unsigned char huge *curPtr;
  unsigned short wide, deep;
  unsigned char curPixel;

  //*** Initialize the hasBeenFound array
  for (charCount=0; charCount<256; charCount++)
    hasBeenFound[charCount] = 0;            // not found yet

  //*** Change each character's colours to match the current palette
  for (charCount=0; charCount<256; charCount++)
  {
    //*** Prepare for accessing character information
    wide = characters[charCount].wide;      // get character width
    deep = characters[charCount].deep;      // get character depth
    curPtr = (unsigned char huge*)characters[charCount].bitmap; // bitmap
    curPtr += 2 * sizeof(unsigned short);   // skip past dimension info

    //*** Actually change the pixel (will skip if character not there)
    for (yCount=0; yCount<deep; yCount++)
    {
      for (xCount=0; xCount<wide; xCount++,curPtr++)
      {
        curPixel = *curPtr;                 // get the pixel
        if (hasBeenFound[curPixel] == 0)    // find the pixel if necessary
        {
          hasBeenFound[curPixel] = 1;       // signal pixel has been found
          colourTranslate[curPixel] =       // get the closest colour
                colourCloseToX(fontPalette[curPixel*3],
                               fontPalette[curPixel*3+1],
                               fontPalette[curPixel*3+2], 0);
        }
      }
    }
  }
}



/**************************************************************************
*  Function:    VariableFont::palette
*
*  Purpose:     Set a new font palette.  The font colours will be changed
*               to match, as closely as possible, the colours of the new
*               palette.
*
*  Entry:       palette = Pointer to the new font palette.
*
*  Exit:        N/A
**************************************************************************/

void VariableFont::palette(void *palette)
{
  if (palette != fontPalette)
    memcpy(fontPalette, palette, 768);
  matchColours();
}



/**************************************************************************
*  Function:    VariableFont::palette
*
*  Purpose:     Set a single colour in the font palette.
*
*  Entry:       palReg = Colour number (0..255) to change.
*               red = New red value.
*               green = New green value.
*               blue = New blue value.
*
*  Exit:        N/A
**************************************************************************/

void VariableFont::palette(unsigned char palReg, unsigned char red,
        unsigned char green, unsigned char blue)
{
  ((unsigned char*)fontPalette)[palReg*3] = red;      // red component
  ((unsigned char*)fontPalette)[palReg*3+1] = green;  // green component
  ((unsigned char*)fontPalette)[palReg*3+2] = blue;   // blue component
  matchColours();                                     // update colours
}



/**************************************************************************
*  Function:    VariableFont::palette
*
*  Purpose:     Get a single colour from the font palette.
*
*  Entry:       palReg = Colour number (0..255) to get.
*               red = Address of red value buffer.
*               green = Address of green value buffer.
*               blue = Address of blue value buffer.
*
*  Exit:        N/A
**************************************************************************/

void VariableFont::palette(unsigned char palReg, unsigned char *red,
        unsigned char *green, unsigned char *blue)
{
  *red = ((unsigned char*)fontPalette)[palReg*3];     // red component
  *green = ((unsigned char*)fontPalette)[palReg*3+1]; // green component
  *blue = ((unsigned char*)fontPalette)[palReg*3+2];  // blue component
}
