/*****************************************************************************
   MODULE: rerr.c
  PURPOSE: recio error functions
COPYRIGHT: (C) 1994-1995, William Pierpoint
 COMPILER: Borland C Version 3.1
       OS: MSDOS Version 6.2
  VERSION: 2.12
  RELEASE: January 29, 1995
*****************************************************************************/

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "recio.h"

extern void _rsetexitfn(REC *rp);

/* private macros */
#define rflags(rp)       ((rp)->r_flags)

/* private module variables */
static void (*_r_errfn)(REC *) = NULL; /* ptr to error function */

/****************************************************************************/
char *                       /* returns pointer to field string             */
    _rerrs(                  /* get field string if errfn() cleared error   */
        REC *rp,             /* record pointer                              */
        int  errnum)         /* error number                                */
/****************************************************************************/
{
    char *fldp=NULL;         /* pointer to field string */

    if (!rseterr(rp, errnum)) fldp = rflds(rp);
    return fldp;
}

/****************************************************************************/
void                           /* returns nothing                           */
    rseterrfn(                 /* registers a callback error function       */
        void(*rerrfn)(REC *rp))/* pointer to error function                 */
/****************************************************************************/
{
    _r_errfn = rerrfn;         /* point to callback function */
    _rsetexitfn(recin);        /* register exit function */
}

/****************************************************************************/
void                         /* returns nothing                             */
    rclearerr(               /* clears eof and error indicators             */
        REC *rp)             /* record pointer                              */
/****************************************************************************/
{
    if (risvalid(rp)) {
        errno = 0;
        rflags(rp) &= ~_R_EOF;
        rflags(rp) &= ~(_R_ERR * _R_ERRMAX);
    } else {
        rseterr(NULL, EINVAL);
    }
}

/****************************************************************************/
int                          /* returns error number (0=no error)           */
    rerror(                  /* gets error number for record stream         */
        REC *rp)             /* record pointer                              */
/****************************************************************************/
{
    int errnum;              /* return error number */

    if (risvalid(rp)) {
         errnum = rflags(rp) / _R_ERR;
         errnum &= _R_ERRMAX;
    } else {
        errnum = rseterr(NULL, EINVAL);
    }
    return errnum;
}

/****************************************************************************/
int                          /* returns possibly modified error number      */
    rseterr(                 /* sets error number and calls error function  */
        REC *rp,             /* record pointer                              */
        int errnum)          /* error number                                */
/****************************************************************************/
{
    /* if valid record pointer and stream error number clear */
    if (risvalid(rp)) {
        if (errnum > _R_ERRMAX) errnum = R_EINVAL;
        if (!rerror(rp)) {

            /* set error number on stream */
            rflags(rp) |= _R_ERR * errnum;

            /* invoke callback error function */
            if (_r_errfn) _r_errfn(rp);

            /* find out if errfn() cleared error */
            errnum = rerror(rp);
        }

    /* else invalid record pointer */
    } else {

        /* set global error number */
        errno = errnum;

        /* invoke callback error function */
        if (_r_errfn) _r_errfn(rp);

        /* find out if errfn() cleared error */
        errnum = errno;
    }

    /* if errfn cleared error, 0 returned */
    return errnum;
}

/****************************************************************************/
char *                       /* returns error message                       */
    rstrerror(               /* gets error message for rerror number        */
        int errnum)          /* error number                                */
/****************************************************************************/
{
    switch (errnum) {
    case 0:         return ("no error");
    case R_EINVAL:  return ("invalid argument");
    case R_EINVDAT: return ("invalid data");
    case R_EINVMOD: return ("invalid mode");
    case R_EMISDAT: return ("missing data");
    case R_ENOMEM:  return ("out of memory");
    case R_ENOPUT:  return ("unable to put data");
    case R_ERANGE:  return ("data out of range");
    }
    return ("unknown error");
}

/****************************************************************************/
char *                       /* returns error message                       */
    rerrstr(                 /* gets error message for record stream        */
        REC *rp)             /* record pointer                              */
/****************************************************************************/
{
    return (risvalid(rp) ? rstrerror(rerror(rp)) : rstrerror(R_EINVAL));
}

/****************************************************************************/
void                         /* returns nothing                             */
    rinit(                   /* initialize recio library                    */
     void(*rerrfn)(REC *rp), /* pointer to error function                   */
     void(*rwarnfn)(REC *rp))/* pointer to warning function                 */
/****************************************************************************/
{
    rseterrfn(rerrfn);
    rsetwarnfn(rwarnfn);
}
