/*** MODEM_IO.C ***/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "pcl4c.h"
#include "ascii.h"
#include "modem_io.h"
#include "term_io.h"
#include "win_io.h"

#define FALSE 0
#define TRUE !FALSE
#define ONE_SECOND 18

/* NOTE: Requires AT COMMAND SET for all functions in this file */

static int  Debug = FALSE;

static int  LastPort;         /* last port referenced */

static char MatchString[80];  /* ModemWaitFor() match string */
static int  MatchLength= 0;   /* string length */
static int  MatchCount = 0;   /* # sub-strings */
static struct
  {char *Start;               /* ptr to 1st char of string */
   char *Ptr;                 /* working ptr */
  } MatchList[10];

char Temp[120];

/*** PRIVATE functions ***/

static int BreakTest(void)
{/* User BREAK ? */
 if(SioBrkKey()||SioKeyPress())
    {WinPutString(SCR_WIN,"User BREAK\n");
     return(TRUE);
    }
 return(FALSE);
}

void MatchInit(char *S)
{int  i;
 char C;
 char *Ptr;
 MatchCount = 0;
 strncpy(MatchString,S,80);
 MatchLength = strlen(MatchString);
 Ptr = MatchString;
 MatchList[MatchCount].Start = Ptr;
 MatchList[MatchCount++].Ptr = Ptr;
 while(*Ptr)
   {if(*Ptr=='|')
      {/* mark start of next string */
       MatchList[MatchCount].Start = Ptr + 1;
       MatchList[MatchCount++].Ptr = Ptr + 1;
      }
    Ptr++;
   }
}

void MatchUpper(void)
{int i;
 char *Ptr;
 Ptr = MatchString;
 for(i=0;i<MatchLength;i++)
   {*Ptr = toupper(*Ptr);
    Ptr++;
   }
}

int MatchChar(char C)
{int  i;
 char *Ptr;
 char *Start;
 /* consider each string in turn */
 for(i=0;i<MatchCount;i++)
   {Ptr = MatchList[i].Ptr;
    Start = MatchList[i].Start;
    if(*Ptr==C)
      {/* char C matches */
#if 0
sprintf(Temp,"<%c:%d:%d>",C,i,(Ptr-Start));
WinPutString(SCR_WIN,Temp);
#endif
       Ptr++;
       if((*Ptr=='|')||(*Ptr=='\0'))
         {MatchList[i].Ptr = Start;
#if 0
sprintf(Temp,"<<%d>>",i);
WinPutString(SCR_WIN,Temp);
#endif
          return i;
         }
       else MatchList[i].Ptr = Ptr;
      }
    else
      {/* char C does NOT match */
       MatchList[i].Ptr = Start;
       /* look again if was not 1st char  */
       if(Ptr!=Start) i--;
      }
   }
 return -1;
}

/*** PUBLIC functions ***/

/* echos incoming to screen */

void ModemEcho(int Port,int Echo)
{int rc;
 long Time;
 Time = SioTimer();
 while(SioTimer() < Time+(long)Echo)
   {rc = CharGet(Port,1);
    if(rc>=0) WinPutChar(SCR_WIN,(char)rc);
   }
}

/* send string to modem & get echo */

int ModemSendTo(
  int  Port,       /* port to talk to */
  int  Pace,       /* inter-char delay */
  char *String)    /* string to send to modem */
{int i, rc;
 char c;
 int Code;
 long Time;
 if(Debug)
   {sprintf(Temp," [Sending '%s'] ",String);
    WinPutString(SCR_WIN,Temp);
   }
 for(i=0;i<strlen(String);)
    {/* User BREAK ? */
     if(BreakTest()) return(FALSE);
     /* delay <Pace> tics */
     if(Pace>0) SioDelay(Pace);
     /* fetch character */
     c = String[i++];
     switch(c)
        {case '^':
            /* next char is control char */
            c = String[i++] - '@';
            break;
         case '!':
            /* replace ! with carriage return */
            c = CR;
            break;
         case '~':
            /* delay 1/2 second */
            SioDelay(ONE_SECOND/2);
            c = ' ';
            break;
         case ' ':
            /* delay 1/4 second */
            SioDelay(ONE_SECOND/4);
            c = ' ';
            break;
        } /* end switch */
     /* transmit as 7 bit ASCII character */
     CharPut(Port,(char)(0x7f & c));
    }
 return(TRUE);
} /* end SendTo */

/* wait for 'String' */

/* NOTES:
**  (1)  Will return NULL if no match, else '0' if 1st, '2' if 2nd, etc.
**       where String = "<1st substr>|<2nd substr>| ..."
**  (2)  Example call: ModemWaitFor(COM1,180,FALSE,"more ?|Menu:");
*/

char ModemWaitFor(
  int  Port,       /* Port to talk to */
  int  Tics,       /* wait in tics for string */
  int  CaseFlag,   /* TRUE = case sensitive compares */
  char *String)    /* string to wait for */
{int i, k;
 int rc;
 char C;
 int Code;
 long Time;
 /* wait for string */
 Time = SioTimer();
 MatchInit(String);
 if(!CaseFlag) MatchUpper();
 if(Debug)
   {sprintf(Temp," [Awaiting '%s'] ",String);
    WinPutString(SCR_WIN,Temp);
   }
 while(SioTimer()<(Time+(long)Tics))
   {/* User BREAK ? */
    if(BreakTest()) return(FALSE);
    /* wait for next character */
    Code = CharGet(Port,1);
    if(Code<-1) return(FALSE);
    if(Code>=0)
      {/* echo char */
       WinPutChar(SCR_WIN,(char)Code);
       /* case sensitive ? */
       if(CaseFlag) C = (char)Code;
       else C = toupper( (char)Code );
       /* does char match ? */
       rc = MatchChar(C);
       if(rc>=0) return ('0' + rc);
      }
   } /* end for(i) */
 return 0;
}

/* enter command state */

/* NOTE: assumes escape char = '+' & guard time = 1 sec */

void ModemCmdState(int Port)
{int i;
 /* delay a bit over 1 second */
 SioDelay(ONE_SECOND+ONE_SECOND/4);
 /* send Escape Code exactly 3 times */
 for(i=0;i<3;i++)
    {CharPut(Port,'+');
     SioDelay(ONE_SECOND/4);
    }
 /* delay again */
 SioDelay(ONE_SECOND+ONE_SECOND/4);
} /* end ModemCmdState */

/* hangup phone (in command state) */

void ModemHangup(int Port)
{/* enter command state */
 ModemCmdState(Port);
 /* hangup ! */
 ModemSendTo(Port,4,"!AT!");
 ModemEcho(Port,10);
 ModemSendTo(Port,4,"ATH0!");
} /* end Hangup */

/* wait for continuous quiet (no incoming serial data) */

int ModemQuiet(
  int  Port,       /* Port to talk to */
  int  Tics)       /* # tics quiet required */
{int i;
 int Code;
 long CharTime;
 /* set up */
 CharTime = SioTimer();
 while(1)
   {/* User BREAK ? */
    if(BreakTest())
      {
       return(FALSE);
      }
    /* wait for next character */
    Code = CharGet(Port,1);
    if(Code<-1) return(FALSE);
    if(Code>=0)
      {CharTime = SioTimer();
       WinPutChar(SCR_WIN,(char)Code);
      }
    else
      {/* ==-1, timed out */
       if(SioTimer() >= CharTime+(long)Tics) return TRUE;
      }
   } /* end while */
}

void ModemDump(void)
{int i;
 sprintf(Temp,"WAITFOR: Count=%d Length=%d \n",
   MatchCount,MatchLength);
 WinPutString(SCR_WIN,Temp);
 sprintf(Temp,"   String= [0x%x] '%s'\n",MatchString,MatchString);
 WinPutString(SCR_WIN,Temp);
 for(i=0;i<10;i++)
   {if(MatchList[i].Start==NULL) break;
    sprintf(Temp,"   Start=0x%x Ptr=0x%x\n",MatchList[i].Start,MatchList[i].Ptr);
    WinPutString(SCR_WIN,Temp);
   }
}

void ModemDebug(void)
{Debug = TRUE;
 WinPutString(SCR_WIN,"[DEBUG]");
}
