/**************************************************************************/
/* This REXX command file contains examples of I/O on a variable, EBCDIC  */
/* Micro Focus indexed (VSAM) file.  It does several different types of   */
/* I/O, including:                                                        */
/*                                                                        */
/*     1.  Open for input                                                 */
/*         Read sequentially through a file                               */
/*         Close                                                          */
/*                                                                        */
/*     2.  Open for input                                                 */
/*         Position on a record                                           */
/*         Read sequentially through the remainder of the file            */
/*         Close                                                          */
/*                                                                        */
/*     3.  Open for I/O                                                   */
/*         Write a new record                                             */
/*         Close                                                          */
/*                                                                        */
/*     4.  Open for input                                                 */
/*         Do a random read                                               */
/*         Close                                                          */
/*                                                                        */
/*     5.  Open for I/O                                                   */
/*         Rewrite an existing record                                     */
/*         Close                                                          */
/*                                                                        */
/*     6.  Open for input                                                 */
/*         Do a random read                                               */
/*         Close                                                          */
/*                                                                        */
/*     7.  Open for I/O                                                   */
/*         Delete an existing record                                      */
/*         Close                                                          */
/*                                                                        */
/* Notes:  When you read a variable record, the 5 byte record length is   */
/*         appended to the beginning of the returned record. For example: */
/*                                                                        */
/*           00090Taunya  9876 3rd St.    Dallas      TX                  */
/*                                                                        */
/*         When you write or rewrite a variable record, the first 5 bytes */
/*         of the record must be the length of the record.  For example:  */
/*                                                                        */
/*           VSAMRecord = 'Taunya  9876 3rd St.    Dallas      TX'        */
/*           VSAMRecord = right(length(VSAMRecord), 5, '0') || VSAMRecord */
/*                                                                        */
/*         This command file is ready to run.  It does I/O on             */
/*         EBCVARBL.DAT, which is an EBCDIC, variable block, Micro Focus  */
/*         indexed file.  This file is included in the INDEXIO.ZIP file.  */
/*         If this file is not in the current directory, change the       */
/*         VSAMFileName variable below to include the path.  For example: */
/*                                                                        */
/*             VSAMFileName = 'C:\FILES\EBCVARBL.DAT'                     */
/*                                                                        */
/*         This program opens the file, does the I/O, then closes the     */
/*         file for each example.  This is done for illustrative purposes */
/*         only.  Normally, you would open the file once, do all of the   */
/*         I/O, then close the file.  Opening the file once would be much */
/*         faster since the open operation takes a lot longer than any of */
/*         the other I/O operations.                                      */
/*                                                                        */
/*         The examples in this command file set the ComArea prior to     */
/*         performing I/O.  The ComArea tells the I/O program what type   */
/*         of I/O you want to perform (open, read sequential, read        */
/*         random, etc).  Here is an explanation of the ComArea fields:   */
/*                                                                        */
/*             Byte 1 - Call Code                                         */
/*               0 - Open for input                                       */
/*               Q - Open for I/O                                         */
/*               2 - Random read                                          */
/*               4 - Write or Rewrite                                     */
/*               5 - Rewrite or Write                                     */
/*               E - Erase                                                */
/*               P - Position (greater than or equal to key)              */
/*               1 - Read sequential (from beginning of file or last pos) */
/*               9 - Close                                                */
/*                                                                        */
/*             Byte 2 - Return Code                                       */
/*               0 - Request successfully processed                       */
/*               9 - Attempting to read past end of file on seq read      */
/*               B - Key definition block for the file is too big         */
/*               F - File name not found                                  */
/*               G - Error reading file header                            */
/*               I - Invalid info passed in the ComArea                   */
/*               N - Record not found on random read or erase             */
/*               O - Error opening file name                              */
/*               E - Other error                                          */
/*                                                                        */
/*             Bytes 3 through 102 (100 bytes total) - File Name          */
/*                                                                        */
/*             Byte 103 - F for fixed, or V for variable                  */
/*                                                                        */
/*             Bytes 104 through 203 (100 bytes total) - Record Key       */
/*                                                                        */
/*             Byte 204 - Y for EBCDIC file, anything else is EBCDIC file */
/*                                                                        */
/*         The 4 call code (write new) and 5 (rewrite) are almost         */
/*         interchangeable.  You can use either one to write a new record */
/*         or to rewrite a record.  The 4 call code will first try to     */
/*         write a new record.  If it fails, it will rewrite the record.  */
/*         The 5 call code will first try to rewrite the record.  If it   */
/*         fails, it will write a new record.  So you can use either      */
/*         call code to write or rewrite, however, performance will be    */
/*         a little better if you know whether the record already exists  */
/*         and use the appropriate call code.                             */
/*                                                                        */
/*         The P call code positions on the record that is greater than   */
/*         or equal to the key that you pass in.  Not only does it        */
/*         position on the record, it also returns the record.  A         */
/*         position changes the sequential pointer within the file.       */
/*         After doing a position, sequential reads return the records    */
/*         that come after the record that was positioned on.             */
/**************************************************************************/

'@ECHO OFF'

VSAMFileName = 'EBCVARBL.DAT'
VSAMFileName = LEFT(VSAMFileName, 100)

CALL L2_INIT_AND_LOAD_CALLDLL

CALL L2_READ_SEQUENTIAL

CALL L2_POSITION_AND_READ_SEQUENTIAL

CALL L2_WRITE

CALL L2_READ_RANDOM

CALL L2_REWRITE

CALL L2_READ_RANDOM

CALL L2_ERASE

CALL L2_UNLOAD_CALLDLL

EXIT 0

/********************************LEVEL 2 CODE********************************/

L2_INIT_AND_LOAD_CALLDLL:
    CALL L3_LOAD_REXXUTIL

    CALL L3_CHECK_IF_EBCVARBL_FILE_EXISTS

    CALL L3_LOAD_CALLDLL
L2_INIT_AND_LOAD_CALLDLL_EXIT:
    RETURN

L2_READ_SEQUENTIAL:
    CALL L3_DISPLAY_READ_SEQUENTIAL_MESSAGE

    CALL L3_OPEN_VSAM_FILE_FOR_INPUT

    CALL L3_READ_SEQUENTIALLY_THROUGH_VSAM_FILE

    CALL L3_CLOSE_FILE
L2_READ_SEQUENTIAL_EXIT:
    RETURN

L2_POSITION_AND_READ_SEQUENTIAL:
    CALL L3_DISPLAY_POSITION_AND_READ_MESSAGE

    CALL L3_OPEN_VSAM_FILE_FOR_INPUT

    CALL L3_POSITION_ON_JOHN

    CALL L3_READ_SEQUENTIALLY_THROUGH_VSAM_FILE

    CALL L3_CLOSE_FILE
L2_POSITION_AND_READ_SEQUENTIAL_EXIT:
    RETURN

L2_WRITE:
    CALL L3_DISPLAY_WRITE_MESSAGE

    CALL L3_OPEN_VSAM_FILE_FOR_INPUT_OUTPUT

    CALL L3_WRITE_A_RECORD_TO_VSAM_FILE

    CALL L3_CLOSE_FILE
L2_WRITE_EXIT:
    RETURN

L2_READ_RANDOM:
    CALL L3_DISPLAY_READ_RANDOM_MESSAGE

    CALL L3_OPEN_VSAM_FILE_FOR_INPUT

    CALL L3_READ_A_RECORD_FROM_VSAM_FILE

    CALL L3_CLOSE_FILE
L2_READ_RANDOM_EXIT:
    RETURN

L2_REWRITE:
    CALL L3_DISPLAY_REWRITE_MESSAGE

    CALL L3_OPEN_VSAM_FILE_FOR_INPUT_OUTPUT

    CALL L3_REWRITE_A_RECORD_TO_VSAM_FILE

    CALL L3_CLOSE_FILE
L2_REWRITE_EXIT:
    RETURN

L2_ERASE:
    CALL L3_DISPLAY_ERASE_RANDOM_MESSAGE

    CALL L3_OPEN_VSAM_FILE_FOR_INPUT_OUTPUT

    CALL L3_ERASE_A_RECORD_FROM_VSAM_FILE

    CALL L3_CLOSE_FILE
L2_ERASE_EXIT:
    RETURN

L2_UNLOAD_CALLDLL:
    call CallDLLDropFuncs
L2_UNLOAD_CALLDLL_EXIT:
    RETURN

/********************************LEVEL 3 CODE********************************/

L3_LOAD_REXXUTIL:
    IF RXFUNCQUERY('SYSLOADFUNCS') THEN DO
        CALL RXFUNCADD 'SYSLOADFUNCS', 'REXXUTIL', 'SYSLOADFUNCS'
        CALL SYSLOADFUNCS
    END
L3_LOAD_REXXUTIL_EXIT:
    RETURN

L3_CHECK_IF_EBCVARBL_FILE_EXISTS:
    call sysfiletree VSAMFileName,'file','f'
    if file.0 = 0 then do
        say
        say strip(VSAMFileName) 'does not exits.  This file is included in'
        say 'INDEXIO.ZIP.' strip(VSAMFileName) 'must be in the current directory,'
        say 'or you can change the "VSAMFileName" variable in this command'
        say 'file to include the path where' strip(VSAMFileName) 'resides.  For'
        say 'example:'
        say
        say '    VSAMFileName = C:\FILES\EBCFILXED.DAT'
        exit 9
    end
L3_CHECK_IF_EBCVARBL_FILE_EXISTS_EXIT:
    RETURN

L3_LOAD_CALLDLL:
    if RXFUNCQUERY('CallDLLLoadFuncs') then do
        call RXFUNCADD 'CallDLLLoadFuncs', 'CALLDLL', 'CallDLLLoadFuncs'
        call CallDLLLoadFuncs
    end
L3_LOAD_CALLDLL_EXIT:
    RETURN

L3_DISPLAY_READ_SEQUENTIAL_MESSAGE:
    say
    say 'Hit enter to read' strip(VSAMFileName) 'sequentially.'

    parse pull junk
L3_DISPLAY_READ_SEQUENTIAL_MESSAGE_EXIT:
    RETURN

L3_OPEN_VSAM_FILE_FOR_INPUT:
    ComArea = '00'VSAMFileName'V'
    call CallDLL 'INDEXIO', 'ComArea', 'VSAMRecord'

    ReturnCode = substr(ComArea, 2, 1)
    if ReturnCode <> 0 then do
        say 'There was an error opening' strip(VSAMFileName)'.  Aborting...'
        exit 9
    end
L3_OPEN_VSAM_FILE_FOR_INPUT_EXIT:
    RETURN

L3_READ_SEQUENTIALLY_THROUGH_VSAM_FILE:
    key = left('', 100)
    ComArea = '10'VSAMFileName'V'key'Y'
    VSAMRecord = left('', 99)

    do while ReturnCode = 0
        call CallDLL 'INDEXIO', 'ComArea', 'VSAMRecord'

        ReturnCode = substr(ComArea, 2, 1)
        if ReturnCode = 0 then
            say strip(VSAMRecord)
    end
L3_READ_SEQUENTIALLY_THROUGH_VSAM_FILE_EXIT:
    RETURN

L3_READ_A_RECORD_FROM_VSAM_FILE:
    key = left('Taunya', 100)
    ComArea = '20'VSAMFileName'V'key'Y'
    VSAMRecord = left('', 99)

    call CallDLL 'INDEXIO', 'ComArea', 'VSAMRecord'
    ReturnCode = substr(ComArea, 2, 1)
    if ReturnCode = 0 then
        say strip(VSAMRecord)
L3_READ_A_RECORD_FROM_VSAM_FILE_EXIT:
    RETURN

L3_CLOSE_FILE:
    ComArea = '90'VSAMFileName'V'
    call CallDLL 'INDEXIO', 'ComArea', 'VSAMRecord'

    ReturnCode = substr(ComArea, 2, 1)

    if ReturnCode <> 0 then do
        SAY 'There was an error closing' strip(VSAMFileName)'.  Aborting...'
        exit 9
    end
L3_CLOSE_FILE_EXIT:
    RETURN

L3_DISPLAY_POSITION_AND_READ_MESSAGE:
    say
    say 'Hit enter to read' strip(VSAMFileName) 'sequentially, starting with John.'

    parse pull junk
L3_DISPLAY_POSITION_AND_READ_MESSAGE_EXIT:
    RETURN

L3_POSITION_ON_JOHN:
    key = left('John', 100)
    ComArea = 'P0'VSAMFileName'V'key'Y'
    VSAMRecord = left('', 99)

    call CallDLL 'INDEXIO', 'ComArea', 'VSAMRecord'
    ReturnCode = substr(ComArea, 2, 1)
    if ReturnCode = 0 then
        say strip(VSAMRecord)
L3_POSITION_ON_JOHN_EXIT:
    RETURN

L3_DISPLAY_WRITE_MESSAGE:
    say
    say 'Hit enter to add Taunya to' strip(VSAMFileName)

    parse pull junk
L3_DISPLAY_WRITE_MESSAGE_EXIT:
    RETURN

L3_OPEN_VSAM_FILE_FOR_INPUT_OUTPUT:
    ComArea = 'Q0'VSAMFileName'V'
    call CallDLL 'INDEXIO', 'ComArea', 'VSAMRecord'

    ReturnCode = substr(ComArea, 2, 1)
    if ReturnCode <> 0 then do
        say 'There was an error opening' strip(VSAMFileName)'.  Aborting...'
        exit 9
    end
L3_OPEN_VSAM_FILE_FOR_INPUT_OUTPUT_EXIT:
    RETURN

L3_WRITE_A_RECORD_TO_VSAM_FILE:
    key = left('Taunya', 100)
    ComArea = '40'VSAMFileName'V'key'Y'
    VSAMRecord = 'Taunya  9876 3rd St.    Dallas      TX'
    VSAMRecord = right(length(VSAMRecord), 5, '0') || VSAMRecord

    call CallDLL 'INDEXIO', 'ComArea', 'VSAMRecord'
    ReturnCode = substr(ComArea, 2, 1)
    if ReturnCode = 0 then
        say 'Taunya was successfully added to' strip(VSAMFileName)
    else do
        say 'There was an error writing to' VSAMFile'.  Aborting...'
        exit 9
    end
L3_WRITE_A_RECORD_TO_VSAM_FILE_EXIT:
    RETURN

L3_DISPLAY_READ_RANDOM_MESSAGE:
    say
    say 'Hit enter to read the record for Taunya from' strip(VSAMFileName)

    parse pull junk
L3_DISPLAY_READ_RANDOM_MESSAGE_EXIT:
    RETURN

L3_DISPLAY_REWRITE_MESSAGE:
    say
    say 'Hit enter to update Taunya in' strip(VSAMFileName)

    parse pull junk
L3_DISPLAY_REWRITE_MESSAGE_EXIT:
    RETURN

L3_REWRITE_A_RECORD_TO_VSAM_FILE:
    key = left('Taunya', 100)
    ComArea = '50'VSAMFileName'V'key'Y'
    VSAMRecord = 'Taunya  1234 Rosemont   Baltimore   MD'
    VSAMRecord = right(length(VSAMRecord), 5, '0') || VSAMRecord

    call CallDLL 'INDEXIO', 'ComArea', 'VSAMRecord'
    ReturnCode = substr(ComArea, 2, 1)
    if ReturnCode = 0 then
        say 'Taunya was successfully updated in' strip(VSAMFileName)
    else do
        say 'There was an error rewriting to' strip(VSAMFile)'.  Aborting...'
        exit 9
    end
L3_REWRITE_A_RECORD_TO_VSAM_FILE_EXIT:
    RETURN

L3_DISPLAY_ERASE_RANDOM_MESSAGE:
    say
    say 'Hit enter to erase the record for Taunya from' strip(VSAMFileName)

    parse pull junk
L3_DISPLAY_ERASE_RANDOM_MESSAGE_EXIT:
    RETURN

L3_ERASE_A_RECORD_FROM_VSAM_FILE:
    key = left('Taunya', 100)
    ComArea = 'E0'VSAMFileName'V'key'Y'
    VSAMRecord = left('', 99)

    call CallDLL 'INDEXIO', 'ComArea', 'VSAMRecord'
    ReturnCode = substr(ComArea, 2, 1)
    if ReturnCode = 0 then
        say 'Taunya was successfully erased from' strip(VSAMFileName)
    else do
        say 'There was an error erasing Taunya from' strip(VSAMFile)'.  Aborting...'
        exit 9
    end
L3_ERASE_A_RECORD_FROM_VSAM_FILE_EXIT:
    RETURN


