/*****************************************************************
*
* PROJECT:        Swiss Perfect 3
*
*                 Copyright (C) 1996 Robert Rozycki
*                 SwissPerfect General Public Licence
*
* FILE:           sp3rrpar.cpp
*
* DESCRIPTION:    Performs Round-Robin pairing.
*
* AUTHOR:         Robert Rozycki     rozycki@perth.dialix.oz.au
*
* PORTABILITY:    ANSI C++
*
* $Revision: 1.1 $
*
* $Date: 1996/02/19 11:24:24 $
*
*******************************************************************/

#include <stdlib.h>
#include <string.h>
#include "sp3rrpar.h"

RoundRobinPair_c::RoundRobinPair_c()
{
}

/////////////////////////////////////////////////////////////////
//
//  Interface function.
//  Parameters: round_table   - pointer to a buffer for pairing
//                              (must be allocated by the caller)
//              players_count - number of players to be paired
//              round         - number of round to be paired
//

int RoundRobinPair_c::GetPairing( int *round_table, int players_count,
                                  int round, int multi )
{
   int tables, i, ind, trn_rounds;
   int odd_player = FALSE;

   tables = players_count >> 1;
   if( players_count & 0x01 )
   {
      odd_player = TRUE;
      tables++;
   }
   trn_rounds = tables * 2 - 1; 

// in case its a multiround tournament
   round = ((round-1) % trn_rounds) + 1;
   pCurrRound = new RRPair_t[ tables+1 ];
   if( pCurrRound == NULL )
     return -1;
   memset( pCurrRound, 0, ( tables+1 ) * sizeof( RRPair_t ) );

   GenerateRound( round, tables, odd_player, multi );

// Load pairing to the interface buffer.
// We could have do it with a sipmly memcpy but as we internally use
// a structure and externally an int vector we could hit some
// problems with compiler packing.

   ind = 0;
   for( i = 1; i <= tables; i++ )
   {
      round_table[ind] = pCurrRound[i].white;
      ind++;
      round_table[ind] = pCurrRound[i].black;
      ind++;
   }
   delete pCurrRound;
   return 0;
}


//============= Sub-class for the Rutsch pairing system ============//

Rutsch_c::Rutsch_c( ) : RoundRobinPair_c( )
{
}


/////////////////////////////////////////////////////////////////
//
//  Fixes colours in the pairing table
//

void Rutsch_c::AdjustPairing( int round, int tables, int multi )
{
   int i, tmp;
   boolean swap, odd_round;

   odd_round = round & 0x01;
   multi &= 0x01;

   if( pCurrRound[ tables ].black != 0 )
   {
      if( odd_round ^ multi )
      {
         tmp = pCurrRound[tables].white;
         pCurrRound[tables].white = pCurrRound[tables].black;
         pCurrRound[tables].black = tmp;
      }
   }

   if( multi & 0x01 )
      swap = FALSE;
   else
      swap = TRUE;

   for( i = tables-1; i >= 1; i -- )
   {
      if( swap )
      {
         tmp = pCurrRound[i].white;
         pCurrRound[i].white = pCurrRound[i].black;
         pCurrRound[i].black = tmp;
         swap = FALSE;
      }
      else
         swap = TRUE;
   }
}

/////////////////////////////////////////////////////////////////
//
//  Generates pairing table for a given round
//

void Rutsch_c::GenerateRound( int round, int tables, boolean odd_player,
                              int multi )
{
   int i, pos, places, from, to;

   places = tables << 1;

// initial insert point
   pos = (round-1) % (places-1) + 1;
      pos++;
   if( pos >= places )
      pos++;

   if( !odd_player )
   {
      from = 2;
      to = places;
   }
   else
   {
      from = 1;
      to = places - 1;
   }
   for( i = from; i <= to; i++ )
   {
   // skip black on last table
      if( pos == places )
         pos++;
      if( pos <= tables )
         pCurrRound[ tables - pos + 1 ].white = i;
      else
         if( pos <= places )
            pCurrRound[ pos - tables ].black = i;
         else
            if( pos <= places + tables )
               pCurrRound[ tables - (pos % places) + 1 ].white = i;
            else
               pCurrRound[ pos - (places+tables) ].black = i;
      pos++;
   }

// fix last pair
   if( odd_player )
      pCurrRound[ tables ].black = 0;
   else
   {
      pCurrRound[ tables ].black = pCurrRound[ tables ].white;
      pCurrRound[ tables ].white = 1;
   }
   AdjustPairing( round, tables, multi );
}

//============= Sub-class for the standard pairing system ============//

StandardRR_c::StandardRR_c( ) : RoundRobinPair_c( )
{
}

/////////////////////////////////////////////////////////////////
//
//  Generates pairing table for a given round
//

void StandardRR_c::GenerateRound( int round, int tables, boolean odd_player,
                                  int multi )
{
   int i, odd_round, bot_ind, top_ind;
   int places, bot_half_beg, top_half_beg;
   int player, top_player, bot_player;

   places = tables << 1;

   top_half_beg = (round+1) / 2 + 1;
   bot_half_beg = (round) / 2 + tables + 1;

   odd_round = round & 0x01;
   top_player = top_half_beg;
   bot_player = bot_half_beg;

   if( odd_round )
   {
      top_ind = 1;
      bot_ind = tables-1;
   }
   else
   {
      top_ind = tables-1;
      bot_ind = 1;
   }

   if( !odd_player )
   {
      top_ind++;
      bot_ind++;
   }

   for( i = 1; i < tables; i++ )
   {
      if( top_player == places )
         top_player = 1;
      if( bot_player == places )
         bot_player = 1;

      if( odd_round )
      {
         pCurrRound[ top_ind ].white = top_player;
         top_ind++;
         top_player++;
         pCurrRound[ bot_ind ].black = bot_player;
         bot_ind--;
         bot_player++;
      }
      else
      {
         pCurrRound[ top_ind ].black = top_player;
         top_ind--;
         top_player++;
         pCurrRound[ bot_ind ].white = bot_player;
         bot_ind++;
         bot_player++;
      }
   }

// if it is an even multi swap all
   if( ( multi & 0x01 ) == 0 )
   {
       for( i = 1; i <= tables; i++ )
       {
           player = pCurrRound[ i ].white;
           pCurrRound[ i ].white = pCurrRound[ i ].black;
           pCurrRound[ i ].black = player;
       }
   }
   FixLastPair( tables, round, odd_player, multi );
}


/////////////////////////////////////////////////////////////////
//
//  Puts correct players on the last table (or one player
//  if the number of players is odd)
//

void StandardRR_c::FixLastPair( int tables, int round, boolean odd_player,
                                int multi )
{
   int last_player, player, places;
   boolean odd_round;

   places = tables << 1;
   odd_round = round & 0x01;
   multi &= 0x01;

   if( odd_round )
   {
      last_player = round / 2 + 1;
      if( odd_player )
      {
         pCurrRound[tables].white = last_player;
         pCurrRound[tables].black = 0;
      }
      else
      {
         pCurrRound[1].white = last_player;
         pCurrRound[1].black = places;
      }
   }
   else
   {
      last_player = round / 2 + tables;
      if( odd_player )
      {
         pCurrRound[tables].white = last_player;
         pCurrRound[tables].black = 0;
      }
      else
      {
         pCurrRound[1].black = last_player;
         pCurrRound[1].white = places;
      }
   }

   if( !odd_player && !multi )
   {
      player = pCurrRound[ 1 ].white;
      pCurrRound[ 1 ].white = pCurrRound[ 1 ].black;
      pCurrRound[ 1 ].black = player;
   }
}
