/*****************************************************************
*
* PROJECT:        Swiss Perfect 3
*
*                 Copyright (C) 1996 Robert Rozycki
*                 SwissPerfect General Public Licence
*
* FILE:           sp3rtg.cpp
*
* DESCRIPTION:    Calculates ELO ratings for Swiss and Round-Robin
*                 tournaments.
*                 For details see documentation.
*
* AUTHOR:         Robert Rozycki   rozycki@perth.dialix.oz.au
*
* PORTABILITY:    ANSI C++
*
* $Revision: 1.1 $
*
* $Date: 1996/03/30 11:10:54 $
*
* $Log: sp3rtg.cpp $
// Revision 1.1  1996/03/30  11:10:54  robert
// Initial revision
//
*
*******************************************************************/

#include <stdlib.h>
#include <string.h>
#include "l_types.h"
#include "sp3util.h"
#include "sp3rtg.h"


int RtgExpectedScore[] = {
   3,  10, 17, 25, 32,
   39, 46, 53, 61, 68,
   76, 83, 91, 98, 106,
   113,121,129,137,145,
   153,162,170,179,188,
   197,206,215,225,235,
   245,256,267,278,290,
   302,315,328,344,357,
   374,391,411,432,456,
   484,517,559,619,735,
   10000};

int RtgBrackets[] = {
   0,  7,  14, 21, 29,
   36, 43, 50, 57, 65,
   72, 80, 87, 95, 102,
   110,117,125,133,141,
   149,158,166,175,184,
   193,202,211,220,230,
   240,251,262,273,284,
   296,309,322,336,351,
   366,383,401,422,444,
   470,501,538,589,677,
	999};


/////////////////////////////////////////////////////////////////
//
// Object Ratings_c provides functionality necessary to calculate
// performance rating and related values
//
// A derived object can use Ratings_c's function and define calculate()
// override its virtual functions
//
// NOTE: As in all Swiss Perfect modules game results and total scores
//       are given in a doubled values to avoid floating point
//       calculations.
//       For example a win in a single game is represented as 2
//       and a draw as 1.  This fact is used implicitly in
//       some calculations (for example calculating percentage
//       score


Ratings_c::Ratings_c( )
{
    Options.rtg_sys          = 0;      // not used in this implementation
    Options.max_diff_dn      = 350;
    Options.max_diff_up      = 350;
    Options.low_thresh       = 2000;
    Options.low_women_thresh = 2000;
    Options.played_only      = 1;
    Options.ignore_forfeits  = 1;
}

Ratings_c::~Ratings_c()
{}


///////////////////////////////////////////////////////////////////
//
// Calculates rating summary for one player given a vector
// of results
//
RtgSummary_t Ratings_c::CalcFromResults( int own_rtg, RtgRes_t *results,
                                        int count, int incl )
{
   RtgSummary_t  rtg_summ;

   PrepRtgInfo( rtg_summ, own_rtg, results, count, incl );
   rtg_summ = Calculate( rtg_summ );
   return rtg_summ;
}


////////////////////////////////////////////////////////////////
//
// Calculates rating summary for one player.
// The following fields in 'rtg_summ' must be assigned correct
// values before calling this function:
//     'own_rtg',  'aver_rtg', 'games', 'score'
// The following fields will be calculated:
//     'exp_score', 'perf_rtg', 'gain'
//
RtgSummary_t Ratings_c::Calculate( RtgSummary_t& rtg_summ )
{
   int delta, rtg_diff;
   int exp_score;

   rtg_diff = rtg_summ.own_rtg - rtg_summ.aver_rtg;
   if( rtg_summ.own_rtg > 0 )
   {
      exp_score = ExpectedScore( rtg_diff );
      exp_score = (exp_score * rtg_summ.games);
   }
   else
   {
      exp_score = 0;
   }

   rtg_summ.exp_score = exp_score;

   if( rtg_summ.own_rtg > 0 )
   {
      rtg_summ.perf_rtg = rtg_summ.aver_rtg +
                          PerformDelta( rtg_summ.score, rtg_summ.games );
      delta = (rtg_summ.score * 50 ) - exp_score;
      rtg_summ.gain = uIntDiv( delta, 10, ROUND_NEAREST );
   }
   else
   {
      rtg_summ.perf_rtg = UnratedPerformance( rtg_summ );
      rtg_summ.gain = 0;
   }
   return rtg_summ;
};


///////////////////////////////////////////////////////////////
//
// Calculates performance rating for an unrated player.
// Updates rtg_summ record with calculated value.
//
int Ratings_c::UnratedPerformance( RtgSummary_t& rtg_summ )
{
   int pluses, perf_rtg;

   if( rtg_summ.score <= rtg_summ.games )
   {
      perf_rtg = rtg_summ.aver_rtg + PerformDelta( rtg_summ.score, rtg_summ.games );
   }
   else
   {
      pluses = rtg_summ.score - rtg_summ.games;
      perf_rtg = rtg_summ.aver_rtg + uIntDiv( 25*pluses, 2, ROUND_NEAREST );
   }
   return perf_rtg;
}


/////////////////////////////////////////////////////////////////
//
//  Prepares rating calculations.
//  Updates the following fields in RtgSummary_t:
//     score, games, aver_rtg, rtg_sum.
//  NOTE:  This function used for SWISS only.
//
void Ratings_c::PrepRtgInfo( RtgSummary_t& summ, int own_rtg,
                             RtgRes_t *results, int count, int incl )
{
   int i, div_by;
   RtgRes_t *score;

   memset( &summ, 0, sizeof( RtgSummary_t ) );
   if( own_rtg < Options.low_thresh )
   {
      own_rtg = 0;
      summ.rated = FALSE;
   }
   else
   {
      summ.rated = TRUE;
   }
   summ.own_rtg = own_rtg;
   for( i = 0; i < count; i++ )
   {
      score = results + i;
    // ignore if below Options.low_thresh
      if( score->oppon_rtg < Options.low_thresh )
         continue;
      if( AddGameResult( score, summ.score, summ.games ) == TRUE )
      {
         if( summ.own_rtg > 0 )
         {
            score->oppon_rtg = AdjustOpponRating( summ.own_rtg,
                                                  score->oppon_rtg );
         }
         summ.rtg_sum += score->games * score->oppon_rtg;
      }
   }
   div_by = summ.games;
   if( incl && summ.own_rtg != 0 )
   {
      summ.rtg_sum += summ.own_rtg;
      div_by += 1;
   }
   if( div_by == 0 )
      summ.aver_rtg = 0;
   else
      summ.aver_rtg = uIntDiv( summ.rtg_sum, div_by, ROUND_NEAREST );
}


/////////////////////////////////////////////////////////////////
//
//  Tests if the game was valid for rating purposes.
//  It is rather trivial here but can be overridden if, say,
//  a rating system allows taking forfeits into calcualations.
//
int Ratings_c::ValidGame( RtgRes_t *res )
{
   if( res->type == NORMAL_PLAYED )
      return TRUE;
   else
      return FALSE;
}

/////////////////////////////////////////////////////////////////
//
//  Modifies 'score' and 'games' according to 'res'
//
int Ratings_c::AddGameResult( RtgRes_t *res, int& score, int& games )
{
   if( res->games == 0 )
       res->games = 1;
   if( ValidGame( res ) )
   {
      games += res->games;
      score += res->score;
      return TRUE;
   }
   else
      return FALSE;
}


/////////////////////////////////////////////////////////////////
//
//  Adjusts opponent's rating if the differnece is too big.
//  Modifies 'res' record accordingly.
//
int Ratings_c::AdjustOpponRating( int own_rtg, int oppon_rtg )
{
   if( own_rtg > oppon_rtg + Options.max_diff_dn )
      oppon_rtg = own_rtg - Options.max_diff_dn;
   else
      if( own_rtg < oppon_rtg - Options.max_diff_up )
         oppon_rtg = own_rtg + Options.max_diff_up;
   return oppon_rtg;
}

////////////////////////////////////////////////////////////////
//
// Obtains performance delta based on the percentage score.
// Deltas coresponding to particular scores are stored in a table.
// If score < 50%, delta is negative
//
int Ratings_c::PerformDelta( int score, int games )
{
   int perform, percent_score;

   if( games == 0 )
      percent_score = 50;
   else
      percent_score = uIntDiv( ( score * 50 ), games, ROUND_NEAREST );

   if( percent_score >= 50 )
      perform =  RtgBrackets[ percent_score-50 ];
   else
      perform = -RtgBrackets[ 50-percent_score ];
   return( perform );
};


///////////////////////////////////////////////////////////
//
// Given a rating difference finds expected score in percents
//
int Ratings_c::ExpectedScore( int rtg_diff )
{
   int i, exp_score, abs_diff;
   int found;

   abs_diff = abs( rtg_diff );
   found = FALSE;
   for( i = 0; i <= 50; i++ )
   {
      if( RtgExpectedScore[i] >= abs_diff )
      {
         if( rtg_diff < 0 )
            exp_score = 50-i;
         else
            exp_score = 50+i;
         found = TRUE;
         break;
      }
   }
   if( found == FALSE )
   {
      if( rtg_diff < 0 )
         exp_score = 0;
      else
         exp_score = 100;
   }
   return( exp_score );
};


/***********************************************************************

 Object RoRoRatings_c provides functionality necessary to calculate
 performance ratings and related values in Round-Robin tournaments.

 Calculating rating for round-robins is identical to Swiss if all
 players are rated but very different if there are unrated players.

 In such a case:
 1) unrated players who score 0 are deleted from calculations.
 2) average rating for the tournament based on rated players' scores
    is calculated and assumed to be the average rating for unrated
    players.
 3) performance ratings for unrated players are calculated and
    assumed to be those players' pre-tournament ratings.
 4) average ratings for rated players are calculated according to
    the special formula.
 5) performance rating and other values are calculated as in Swiss.

***********************************************************************/

RoRoRatings_c::RoRoRatings_c( )
{}

RoRoRatings_c::~RoRoRatings_c()
{}

////////////////////////////////////////////////////////////////
//
// Calculates rating summary for each player.
// Pointer to a cross table is passed as a parameter.
// Summ is pre-initialised to hold own ratings (own_rtg)
//
void RoRoRatings_c::CalcAllFromResults( RtgSummary_t *summ,
                                       RtgRes_t **results,
                                       int players, int rounds )
{
   int i, j, pass, second_pass_reqd, games;
   int *bad_players;

   bad_players = new int[ players+1 ];
   for( i = 0; i <= players; i++ )
      bad_players[i] = FALSE;

// calculate scores & games
   pass = 1;
   second_pass_reqd = TRUE;
   while( pass <= 2 && second_pass_reqd == TRUE )
   {
      second_pass_reqd = FALSE;
      for( i = 0; i < players; i++ )
      {
         summ[i].score = 0;
         summ[i].games = 0;
         if( summ[i].own_rtg > 0 )
            summ[i].rated = TRUE;
         else
            summ[i].rated = FALSE;
         for( j = 0; j < rounds; j++ )
         {
            if( bad_players[ results[i][j].oppon_no ] == FALSE )
               AddGameResult( &(results[i][j]), summ[i].score, summ[i].games );
         }
         if( summ[i].own_rtg == 0 && summ[i].score == 0 )
         {
            bad_players[ i ] = TRUE;
            second_pass_reqd = TRUE;
         }
      }
      pass++;
   };
   delete bad_players;
   for( i = 0; i < players; i++ )
   {
      if( summ[i].rated == FALSE )
         summ[i].aver_rtg = AverRtgForUnrated( summ, *(results+i),
                                               rounds, games );
   }
//   AverRtgForAllRated( summ, players );
   AverRtgForAllRated( summ, results, rounds, players );
   CalculateAll( summ, players, FALSE );
}

void RoRoRatings_c::GetAversIn( RtgSummary_t *summ, int players )
{
   int i, aver_for_unrated;

   aver_for_unrated = AverRtgForUnrated( summ, players );

   RoRoTrnAver = aver_for_unrated;

// put ratings for unrated and clear aver_rtg for others
   for( i = 0; i < players; i++ )
   {
      summ[i].aver_rtg = 0;
      if( summ[i].own_rtg == 0 )
      {
         summ[i].rated = FALSE;
         if( summ[i].score > 0 )
         {
            summ[i].aver_rtg = aver_for_unrated;
            summ[i].perf_rtg = UnratedPerformance( summ[i] );
            summ[i].own_rtg = summ[i].perf_rtg;
         }
      }
      else
      {
         summ[i].rated = TRUE;
      }
   }
}

////////////////////////////////////////////////////////////////
//
// Calculates rating summary for each player.
//
void RoRoRatings_c::CalculateAll( RtgSummary_t *summ, int players,
                                  int avers_in )
{
   int i;

   for( i = 0; i < players; i++ )
   {
      if( summ[i].own_rtg < Options.low_thresh )
         summ[i].own_rtg = 0;
   }

   for( i = 0; i < players; i++ )
   {
      summ[i].perf_rtg = 0;
   }

   if( !avers_in )
   {
      GetAversIn( summ, players );

   // Get average ratings for rated
      AverRtgForAllRated( summ, players );
   }

// second pass
   for( i = 0; i < players; i++ )
   {
      if( summ[i].rated == FALSE )
         summ[i].own_rtg = UnratedPerformance( summ[i] );
   }

   for( i = 0; i < players; i++ )
      Calculate( summ[i] );
}


//////////////////////////////////////////////////////////////////////
//
//  Calculates tournament average for unrated players given total
//  scores.
//
int RoRoRatings_c::AverRtgForUnrated( RtgSummary_t *summ, int players )
{
   long sum = 0;
   int i, delta, aver_delta, rated = 0, counted = 0;
   long aver_rtg, tot_delta = 0;

   for( i = 0; i < players; i++ )
   {
      counted++;
      if( summ[i].own_rtg == 0 )
      {
         if( summ[i].score == 0 )
            counted--;
         continue;
      }

      rated++;
      sum += summ[i].own_rtg;
      delta = PerformDelta( summ[i].score, summ[i].games );
      tot_delta += delta;
   }
   aver_rtg   = uIntDiv( sum, rated, ROUND_NEAREST );
   aver_delta = uIntDiv( tot_delta, rated, ROUND_NEAREST );
   aver_rtg   -= uIntDiv( aver_delta * (counted-1), counted, ROUND_NEAREST );
   return aver_rtg;
}

//////////////////////////////////////////////////////////////////////
//
//  Calculates tournament average for unrated players given all
//  results.
//
int RoRoRatings_c::AverRtgForUnrated( RtgSummary_t *summ, RtgRes_t *results,
                                      int rounds, int& games, int own_rtg )
{
   long sum = 0;
   int i, delta, aver_delta, rated = 0, counted = 0;
   int oppon;
   long aver_rtg, tot_delta = 0;

   if( own_rtg > 0 )
   {
      sum = own_rtg;
      counted++;
      rated++;
   }
   for( i = 0; i < rounds; i++ )
   {
      if( (oppon = results[i].oppon_no) == 0 || !ValidGame( &results[i] ) )
         continue;
      counted++;
      oppon--;  // normalise
      if( summ[oppon].own_rtg == 0 )
      {
         if( summ[oppon].score == 0 )
            counted--;
         continue;
      }
      rated++;
      sum += summ[oppon].own_rtg;
      delta = PerformDelta( summ[oppon].score, summ[oppon].games );
      tot_delta += delta;
   }
   aver_rtg   = uIntDiv( sum, rated, ROUND_NEAREST );
   aver_delta = uIntDiv( tot_delta, rated, ROUND_NEAREST );
   aver_rtg   -= uIntDiv( aver_delta * (counted-1), counted, ROUND_NEAREST );
   games = counted;
   return aver_rtg;
}


////////////////////////////////////////////////////////////////
//
// Calculates average opposition ratings for rated players
// given total scores
//
void RoRoRatings_c::AverRtgForAllRated( RtgSummary_t *summ, int players )
{
   long sum = 0;
   int i, j, adj, own_rtg, oppon_rtg, oppon_count;

   for( i = 0; i < players; i++ )
   {
      if( summ[i].rated == FALSE )
         continue;
      own_rtg = summ[i].own_rtg;
      sum = 0;
      oppon_count = 0;

      for( j = 0; j < players; j++ )
      {
      // ignore own rating
         if( i == j )
            continue;
         oppon_rtg = AdjustOpponRating( own_rtg, summ[j].own_rtg );
         sum += oppon_rtg;
         if( oppon_rtg > 0 )
            oppon_count++;
      }
      if( oppon_count > 0 )
         summ[i].aver_rtg = uIntDiv( sum, oppon_count, ROUND_NEAREST );
      else
         summ[i].aver_rtg = 0;

     summ[i].aver_rtg = uIntDiv(RoRoTrnAver * players - summ[i].own_rtg,
                            players-1, ROUND_NEAREST );
   }

// rating adjustments
   for( i = 0; i < players; i++ )
   {
      own_rtg = summ[i].own_rtg;
      sum = 0;

      for( j = 0; j < players; j++ )
      {
      // ignore own rating
         if( i == j )
            continue;
         oppon_rtg = AdjustOpponRating( own_rtg, summ[j].own_rtg );
         adj = uIntDiv( oppon_rtg-summ[j].own_rtg, players-1, ROUND_NEAREST );
         summ[i].aver_rtg += adj;
      }
   }
}

////////////////////////////////////////////////////////////////
//
// Calculates average opposition ratings for rated players
// given all results
//
void RoRoRatings_c::AverRtgForAllRated( RtgSummary_t *summ,
                    RtgRes_t **results, int rounds, int players )
{
   int i, adj, oppon_rtg, games;

   for( i = 0; i < players; i++ )
   {
      if( summ[i].rated == TRUE )
      {
         summ[i].aver_rtg = AverRtgForUnrated( summ, *(results+i), rounds,
                                               games, summ[i].own_rtg );
      // remove own rating
         adj = uIntDiv( oppon_rtg-summ[i].own_rtg, games, ROUND_NEAREST );
         summ[i].aver_rtg += adj;
      }
   }
}
