/*****************************************************************
*
* PROJECT:        Rating Module Tester
*
*                 Copyright (C) 1996 Robert Rozycki
*                 SwissPerfect General Public Licence
*
* FILE:           sprtg.cpp
*
* DESCRIPTION:
*
* AUTHOR:         Robert Rozycki   rozycki@perth.dialix.oz.au
*
* PORTABILITY:    ANSI C++
*
* $Revision: 1.1 $
*
* $Date: 1996/03/30 11:07:47 $
*
*******************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "l_types.h"
#include "trn_data.h"
#include "sp3rtg.h"

#define SPRTG_VERSION "SPRTG 1.00.0, Copyright (C) 1996 Robert Rozycki"

enum { WARNING_MSG, ERROR_MSG };

/* local functions */

static void ProcessTrn( TrnData_c *trn_data,  RtgSummary_t *rtg_table,
                        int players_count, int trn_type, int data_mode,
                        int multi_games );
static void PrintUsage();
static int  ReadData( FILE *file, TrnData_c * );
static void BuildResultsVec( TrnData_c *, int player, int round,
                            RtgRes_t *, int multi_games );
static void PrintRtgSummary( unsigned int no, RtgSummary_t * );
static void UpdateRtgOptions( TrnData_c *trn_data, Ratings_c *ratings );
static int  HandleDataError( LineStat_t& line_stat, int line_no );
static void PrintError( int line_no, int msg_type, char *text );


typedef struct {
  char  name[ MAX_PARAM_NAME ];
  int   *int_val;
  char **str_val;
} ParamStorage_t;

RtgOptions_t  RtgOptions;

ParamStorage_t ParamStorage[] = {
  { "LowThresh", &RtgOptions.low_thresh, NULL },
  { "MaxDiffUp", &RtgOptions.max_diff_up, NULL },
  { "MaxDiffDn", &RtgOptions.max_diff_dn, NULL },
};

#define PARAM_COUNT  ( sizeof(ParamStorage)/sizeof(ParamStorage[0]) )

/////////////////////////////////////////////////////////////////
//
//  main()
//
int main( int argc, char *argv[] )
{
   char *data_file;
   FILE *file;
   int i, players_count;
   int data_mode, trn_type, multi_games;
   RtgSummary_t   *rtg_table;
   TrnData_c  *trn_data;

// Read parameters and configure execution
   if( argc < 2 )
   {
      PrintUsage();
      return 0;
   }

   for( i = 1; i < argc; i++ )
   {
      if( argv[i][0] != '-' )
         continue;
      switch( argv[i][1] )
      {
         case 'f':
            data_file = strdup( argv[i+1] );
            i++;
            break;
         default :
            continue;
      }
   }

   file = fopen( data_file, "r" );
   if( file == NULL )
      return( -1 );

// set default options and tournament type
   RtgOptions.max_diff_dn      = 350;
   RtgOptions.max_diff_up      = 350;
   RtgOptions.low_thresh       = 2000;
   RtgOptions.low_women_thresh = 2000;
   RtgOptions.played_only      = 1;
   RtgOptions.ignore_forfeits  = 1;
   trn_type = 0;   // Swiss as default
   multi_games = 1;

   while( TRUE )
   {
      trn_data = new TrnData_c();
      if( trn_data == NULL )
      {
         printf( "Out of memory" );
         return( 0 );
      }
      ReadData( file, trn_data );
      data_mode = trn_data->GetDataMode();

      trn_data->SetTrnType( trn_type );
      trn_type = trn_data->GetTrnType();

      trn_data->SetMultiGames( multi_games );
      multi_games = trn_data->GetMultiGames();

      players_count = trn_data->GetPlayersCount();
      if( players_count <= 0 )
         return 0;
      rtg_table = new RtgSummary_t[ players_count + 1 ];
      ProcessTrn( trn_data, rtg_table, players_count, trn_type, data_mode,
                  multi_games );

      for( i = 0; i <= players_count; i++ )
         PrintRtgSummary( i, rtg_table );
      delete rtg_table;
      delete trn_data;
   }
   return 0;
}


/////////////////////////////////////////////////////////////////
//
//  Creates and calls appropriate objects to perform rating
//  calculations
//
static void ProcessTrn( TrnData_c *trn_data, RtgSummary_t *rtg_table,
                        int players_count, int trn_type, int data_mode,
                        int multi_games )
{
   int i, count, own_rtg;
   RtgRes_t       *scores;
   RtgRes_t      **cross;
   Ratings_c      *Rtg;
   RoRoRatings_c  *RoRoRtg;

   if( trn_type == TRNTYP_ROUND_ROBIN )
   {
      RoRoRtg = new RoRoRatings_c();
      UpdateRtgOptions( trn_data, RoRoRtg );
      Rtg = NULL;
   }
   else
   {
      Rtg = new Ratings_c();
      UpdateRtgOptions( trn_data, Rtg );
      RoRoRtg = NULL;
   }

   if( data_mode == RESULTS_DATA )
   {
      count   = trn_data->GetRoundsPlayed();
      scores  = new RtgRes_t[ count ];
      cross   = (RtgRes_t **)malloc( sizeof( RtgRes_t *) * (players_count+1) );
      if( trn_type == TRNTYP_ROUND_ROBIN )
      {
         for( i = 0; i <= players_count; i++ )
            cross[i] = new RtgRes_t[players_count+1];
         for( i = 0; i < players_count; i++ )
         {
            memset( &rtg_table[i], 0, sizeof( RtgSummary_t ) );
            rtg_table[i].own_rtg = trn_data->GetPlayerRtg( i+1, 1 );
            BuildResultsVec( trn_data, i+1, count, cross[i], multi_games );
         }
         RoRoRtg->CalcAllFromResults( rtg_table, cross, players_count,
                                      players_count-1 );
      }
      else     // SWISS
      {
         for( i = 0; i < players_count; i++ )
         {
            own_rtg = trn_data->GetPlayerRtg( i+1, 1 );
            BuildResultsVec( trn_data, i+1, count, scores, multi_games );
            rtg_table[i] = Rtg->CalcFromResults( own_rtg, scores, count, 0 );
         }
      }
   }
   else     // data_mode == TOT_SCORES
   {
      for( i = 0; i < players_count; i++ )
      {
         rtg_table[i].own_rtg = trn_data->GetPlayerRtg( i+1, 1 );
         rtg_table[i].score   = trn_data->GetPlayerTotScore( i+1 );
         rtg_table[i].games   = trn_data->GetPlayerGames( i+1 );
      }
      if( trn_type == TRNTYP_ROUND_ROBIN )
      {
         RoRoRtg->CalculateAll( rtg_table, players_count );
      }
      else
      {
         for( i = 0; i < players_count; i++ )
         {
            rtg_table[i].aver_rtg = trn_data->GetAverRtg( i+1, 1 );
            if( rtg_table[i].own_rtg < RtgOptions.low_thresh )
            {
               rtg_table[i].own_rtg = 0;
               rtg_table[i].rated = FALSE;
            }
            else
            {
               rtg_table[i].rated = TRUE;
            }
            Rtg->Calculate( rtg_table[i] );
         }
      }
   }

   delete Rtg;
   delete RoRoRtg;
}

/////////////////////////////////////////////////////////////////
//
//  Checks if any parameters have been read from the data file.
//  Updates those parameters in the Ratings module.
//  This function should be called after reading in data from
//  the data file and before calling the Ratings object to
//  process it.
//
void UpdateRtgOptions( TrnData_c *trn_data, Ratings_c *ratings )
{
  int i;
  ParamStorage_t *par;

  par = &ParamStorage[0];
  for( i = 0; i < (int)PARAM_COUNT; i++ )
  {
     trn_data->GetParamValue( par[i].name, par[i].int_val, par[i].str_val );
  }
  ratings->SetOptions( RtgOptions );
}

//////////////////////////////////////////////////////////////////////
//
//   Obtains player's results from the TrnData object and builds
//   a results vector required by the Ratings class interface.
//
static void BuildResultsVec( TrnData_c *trn_data, int player, int round,
                            RtgRes_t *scores, int multi_games )
{
   int i;
   Res_t res;

   for( i = 0; i < round; i++ )
   {
      res = trn_data->GetPlayerResult( player, i+1 );
      scores[ i ].oppon_no  = res.oppon;
      scores[ i ].oppon_rtg = trn_data->GetPlayerRtg( res.oppon, 1 );
      scores[ i ].score     = res.score;
      scores[ i ].type      = res.type;
      scores[ i ].games     = multi_games;
   }
}


//////////////////////////////////////////////////////////////////////
//
//   Controls reading data from the data file by calling TrnData
//   functions.
//
static int ReadData( FILE *file, TrnData_c *trn_data )
{
   char buf[1000];
   int data_mode;
   int line_count = 0, err_mode = 0, data_lines = 0;
   LineStat_t line_stat;

   while( fgets( buf, 1000, file ) != NULL )
   {
      line_count++;
      if( err_mode )
         continue;
      line_stat = trn_data->ProcessLine( buf );
      if( line_stat.line_type == DATA_LINE )
      {
         data_lines++;
         if( data_lines == 1 )
         {
            data_mode = line_stat.data_mode;
         }
         else
         {
            if( data_mode != line_stat.data_mode )
               line_stat.error = ERR_MIXED_FORMATS;
         }
         if( data_lines != line_stat.item_no )
            line_stat.error = ERR_OUT_OF_SEQ;
      }
      if( (err_mode = HandleDataError( line_stat, data_lines )) != 0 )
         continue;
      if( line_stat.line_type == BEGIN_LINE && data_lines )
         break;
   }
   return 0;
}


static int HandleDataError( LineStat_t& line_stat, int line_no )
{
   int err_mode = 0;

   switch( line_stat.error )
   {
      case ERR_UNKNOWN_PARAM:
         PrintError( line_no, WARNING_MSG,"unknown parameter" );
         break;
      case ERR_DUPL_STARTNO:
         PrintError( line_no, ERROR_MSG,"duplicate start no" );
         err_mode = 1;
         break;
      case ERR_UNKNOWN_TOKEN:
         PrintError( line_no, ERROR_MSG,"unknown token" );
         err_mode = 1;
         break;
      case ERR_OUT_OF_SEQ:
         PrintError( line_no, ERROR_MSG,"data out of sequence" );
         err_mode = 1;
         break;
      case ERR_MIXED_FORMATS:
         PrintError( line_no, ERROR_MSG,"different data formats in one test" );
         err_mode = 1;
         break;
   }
   return err_mode;
}

static void PrintError( int line_no, int msg_type, char *text )
{
   if( msg_type == ERROR_MSG )
   {
      printf( "Error in line %d: %s\n", line_no, text );
      printf( "Skipping this test\n" );
   }
   else
   {
      printf( "Warning line %d: %s\n", line_no, text );
   }
}

//////////////////////////////////////////////////////////////////////
//
//   Prints rating results.
//
static void PrintRtgSummary( unsigned int no, RtgSummary_t *rtg_table )
{
   RtgSummary_t summ;
   int sco_int, sco_fra;

   if( no == 0 )
   {
      printf( "\nRATING SUMMARY\n" );
   }
   else
   {
      summ = rtg_table[ no-1 ];
      sco_int = summ.score / 2;
      sco_fra = ( summ.score & 0x01 ) * 5;
      printf("No:%-4d score=%d.%d/%d\n ", no, sco_int, sco_fra, summ.games );
      printf("       R=%4d, Rav=%-5d ", summ.own_rtg, summ.aver_rtg );
      sco_int = summ.exp_score / 100;
      sco_fra = summ.exp_score % 100;
      if( summ.rated )
         printf("Exp=%2d.%02d ", sco_int, sco_fra );
      else
         printf("          " );
      printf("Rp=%4d ", summ.perf_rtg );
      if( summ.rated )
         printf( "Chg*10=%-3d", summ.gain );
      printf( "\n" );
   }
}

//////////////////////////////////////////////////////////////////////
//
//   Prints usage information 
//
static void PrintUsage()
{
   printf( SPRTG_VERSION );
   printf("\nUsage: sprtg -f filename");
   printf("\n\n");
}
