/*****************************************************************
*
* PROJECT:        Rating Module Tester
*
*                 Copyright (C) 1996 Robert Rozycki
*                 SwissPerfect General Public Licence
*
* FILE:           trn_data.cpp
*
* DESCRIPTION:    Stores tournament data.
*                 Parses input lines to extract data and allows
*                 data retrieval.
*
* AUTHOR:         Robert Rozycki   rozycki@perth.dialix.oz.au
*
* PORTABILITY:    ANSI C++
*
* $Revision: 1.1 $
*
* $Date: 1996/03/30 11:08:11 $
*
*******************************************************************/

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include "sp3util.h"
#include "sp3sctyp.h"
#include "trn_data.h"

#define ALLOC_SEGM  100

enum { TOK_NONE, TOK_TEST_BEGIN, TOK_NUMBER, TOK_GAME_RES, TOK_TOT_SCORE,
       TOK_RTG_AVER, TOK_RTG_OWN };

//////////////////////////////////////////////////////////////////////
//
// CONSTRUCTOR
//
// Initial memory allocation.
// Data initialisation.
//
TrnData_c::TrnData_c()
{
   Allocated = ALLOC_SEGM;
   CrossTab = (PlayerData_t *)malloc( (Allocated+1) * sizeof(PlayerData_t) );
   ByeEquiv = 2;
   PlayersCount = 0;
   RoundsPlayed = 0;
   DataMode = RESULTS_DATA;
   TrnType  = TRNTYP_SWISS;
   MultiGames = 1;
   memset( Params, 0, sizeof( Params ) );
}


/*************************** INTERFACE *****************************/

///////////////////////////////////////////////////////////////////
//
// Returns tournament type (Swiss or Round-Robin)
//
int TrnData_c::GetTrnType()
{
   int  int_val;
   char str_val[ MAX_PARAM_NAME + 1 ];
   char trn_type_str[] = "TrnType";
   char *str;

   str = str_val;
   if( GetParamValue( trn_type_str, &int_val, &str ) == 0 )
   {
      if( !strcmp( "ROUNDROBIN", uUpperString(str_val) ) )
         TrnType = TRNTYP_ROUND_ROBIN;
      if( !strcmp( "SWISS", uUpperString(str_val) ) )
         TrnType = TRNTYP_SWISS;
   }
   return TrnType;
};

///////////////////////////////////////////////////////////////////
//
// Returns tournament type (Swiss or Round-Robin)
//
int TrnData_c::GetMultiGames()
{
   int  int_val;
   char trn_type_str[] = "MultiGames";

   if( GetParamValue( trn_type_str, &int_val, NULL ) == 0 )
   {
      MultiGames = int_val;
   }
   return MultiGames;
};

///////////////////////////////////////////////////////////////////
//
// Returns player's rating
// 'sys' is a number of rating system (starts from 1)
//
int TrnData_c::GetPlayerRtg( int player, int sys )
{
   if( player > PlayersCount || player < 1 ||
                     sys > MAX_RTG_SYSTEMS || sys < 1 )
      return 0;
   return CrossTab[ player ].rtg[ sys-1 ];
}

///////////////////////////////////////////////////////////////////
//
// Returns player's average rating.
// Second parameter not used.
//
int TrnData_c::GetAverRtg( int player, int )
{
   if( player > PlayersCount || player < 1 )
      return 0;
   return CrossTab[ player ].aver_rtg;
}

///////////////////////////////////////////////////////////////////
//
// Returns player's total score
//
int TrnData_c::GetPlayerTotScore( int player )
{
   return CrossTab[ player ].score;
}

///////////////////////////////////////////////////////////////////
//
// Returns number of games played by 'player'
//
int TrnData_c::GetPlayerGames( int player )
{
   return CrossTab[ player ].rounds;
}

////////////////////////////////////////////////////////////////////
//
//  Returns players result in a given round
//
Res_t TrnData_c::GetPlayerResult( int player, int round )
{
   Res_t res;

   memset( &res, 0, sizeof( Res_t ) );
   if( player > PlayersCount || player < 1 )
      return res;
   if( CrossTab[ player ].rounds < round )
      return res;

   res = CrossTab[ player ].results[ round ];
   return res;
}

//////////////////////////////////////////////////////////////////////////
//
//  Obtains values of a parameter if it was read in from the data file.
//  int_val and str_val are loaded with integer and string values
//  of the parameter.
//  If parameter not found does not change int_val and str_val
//
//  Function returns 0 on if parameter found or -1 otherwise
//
int TrnData_c::GetParamValue( char *param_name, int *int_val, char **str_val )
{
   int i;

   for( i = 0; i < MAX_PARAMS; i++ )
   {
      if(!strcmp( uUpperString(param_name), uUpperString(Params[i].name) ) ) 
      {
         if( str_val != NULL && *str_val != NULL )
            strncpy( *str_val, Params[i].string, MAX_PARAM_STRLEN );
         if( int_val != NULL )
            *int_val = Params[i].value;
         return 0;
      }
   }
   return -1;
}

//////////////////////////////////////////////////////////
//
//  Processes one input line
//
LineStat_t TrnData_c::ProcessLine( char *line )
{
   int line_type;
   LineStat_t line_stat;

   memset( &line_stat, 0, sizeof( line_stat ) );
   if( CrossTab == NULL )
   {
      line_stat.error = ERR_NO_MEMORY;
      return line_stat;  // System Error
   }
   while( isspace( *line ) )
      line++;

   line_type = GetLineType( line );
   switch( line_type )
   {
      case COMMENT_LINE:
         break;
      case CONFIG_LINE:
         line_stat = ParseConfigLine( line );
         break;
      case DATA_LINE:
         line_stat = ParseDataLine( line, CrossTab[ PlayersCount+1 ] );
         break;
   }
   line_stat.line_type = line_type;

   if( line_type == DATA_LINE )
   {
      // load data
      PlayersCount++;
      if( PlayersCount >= Allocated )
      {
         Allocated += ALLOC_SEGM;
         realloc( CrossTab, Allocated * sizeof( PlayerData_t ) );
         if( CrossTab == NULL )
            line_stat.error = ERR_NO_MEMORY;
      }
   }
   return line_stat;
}


/**************************  PRIVATE *********************************/

////////////////////////////////////////////////////////////////////
//
//  Pre-examines an input line to establish the type of data in it
//
int TrnData_c::GetLineType( char *line )
{
   char *ptr;
   int line_type;

   ptr = line;
   line_type = UNKNOWN_LINE;
   switch( line[0] )
   {
      case '#':
      case '*':
         line_type = COMMENT_LINE;
         break;
      case '%':
         line_type = CONFIG_LINE;
         break;
   }
   if( line_type != UNKNOWN_LINE )
      return line_type;

  // check if it is a beginning of a section ie [Test]
   if( ( ptr = strchr( ptr, '\[' ) ) != NULL )     
      if( ( ptr = strstr( uUpperString(ptr), "TEST" ) ) != NULL )
         if( ( ptr = strchr( ptr, ']' ) ) != NULL )     
            line_type = BEGIN_LINE;
   
   if( isdigit( line[0] ) )
      line_type = DATA_LINE;

   return line_type;
}

//////////////////////////////////////////////////////////////////
//
//  Parses one input line
//
LineStat_t TrnData_c::ParseDataLine( char *line, PlayerData_t& player )
{
   char *ptr, *tok_ptr;
   int i;
   LineStat_t line_stat;

   memset( &player, 0, sizeof( PlayerData_t ) );
   player.results = (Res_t *)malloc( (RoundsPlayed+1) * sizeof(Res_t) );
   memset( &line_stat, 0, sizeof( line_stat ) );

   ptr = line;
   while( ( tok_ptr = strtok( ptr, " \t\n\r" ) ) != NULL )
   {
      ptr = NULL;  // for next calls
      switch( IdentifyToken( tok_ptr ) )
      {
         case TOK_NUMBER:
            if( player.start_no == 0 )  // in case it's a spurious no
            {
               player.start_no = ReadStartNo( tok_ptr );
               line_stat.item_no = player.start_no;
            } 
            else
            {
               line_stat.error = ERR_DUPL_STARTNO;
            }
            break;
         case TOK_GAME_RES:
            if( player.results == NULL )
               break;
            player.rounds++;
            if( player.rounds > RoundsPlayed )
            {
               RoundsPlayed = player.rounds;
               player.results =
                  (Res_t *)realloc( player.results, (RoundsPlayed+1) * sizeof(Res_t) );
            }
            player.results[player.rounds] = ReadResult( tok_ptr );
            break;
         case TOK_TOT_SCORE:
            free( player.results );
            player.results = NULL;
            DataMode = TOT_SCORES_DATA;
            player.score = ReadTotScore( tok_ptr, player.rounds );
            if( player.rounds > RoundsPlayed )
               RoundsPlayed = player.rounds;
            break;
         case TOK_RTG_OWN:
            for( i = 0; i < MAX_RTG_SYSTEMS; i++ )
            {
               if( player.rtg[i] == 0 )
               {
                  player.rtg[i] = ReadIntParam( tok_ptr );
                  break;
               }
            }
            break;
         case TOK_RTG_AVER:
            player.aver_rtg = ReadIntParam( tok_ptr );
            break;
         default:
            line_stat.error = ERR_UNKNOWN_TOKEN;
            break;
      }
   }
   return line_stat;
}

//////////////////////////////////////////////////////////////////
//
//  Parses a line with configuration parameters
//
LineStat_t TrnData_c::ParseConfigLine( char *line )
{
   char *ptr, *tok_ptr, *eq_pos;
   char buf[ MAX_PARAM_NAME + MAX_PARAM_STRLEN + 2 ];
   int i, free_pos;
   LineStat_t line_stat;

   memset( &line_stat, 0, sizeof( line_stat ) );
   ptr = line;
   while( ( tok_ptr = strtok( ptr, " \t\n\r" ) ) != NULL )
   {
      ptr = NULL;  // for subsequent calls
      strncpy( buf, tok_ptr, MAX_PARAM_NAME + MAX_PARAM_STRLEN + 1 );
      eq_pos = strchr( buf, '=' );
      if( eq_pos == NULL )
         continue;
      *eq_pos = 0;
      eq_pos++;

      free_pos = -1;
      for( i = 0; i < MAX_PARAMS; i++ )
      {
         if( Params[i].name[0] == '\0' )
         {
            free_pos = i;
            break;
         }
      }
      if( free_pos < 0 )
      {
         line_stat.error = ERR_UNKNOWN_PARAM;
         continue;
      }
      strncpy( Params[i].name, buf, MAX_PARAM_NAME );
      if( isdigit( *eq_pos ) )
      {
         Params[i].value  = atoi( eq_pos );
         strcpy( Params[i].string, "" );
      }
      else
      {
         Params[i].value = 0;
         strncpy( Params[i].string, eq_pos, MAX_PARAM_STRLEN );
      }
   }
   return line_stat;
}

/////////////////////////////////////////////////////////////////
//
// Identifies token type.
// First checks for a known identifier and if not found checks
// if it has ':' in it or is "bye" which means it's a result.
// If it consists of digits only it's a start no.
//
int TrnData_c::IdentifyToken( char *token )
{
   char *ptr;

   ptr = token;
   if( strstr( uUpperString(token), "TEST" ) != NULL )
      return TOK_TEST_BEGIN;
   if( strstr( uUpperString(token), "RES=" ) != NULL )
      return TOK_TOT_SCORE;
   if( strstr( uUpperString(token), "RAV=" ) != NULL )
      return TOK_RTG_AVER;
   if( strstr( uUpperString(token), "R=" ) != NULL )
      return TOK_RTG_OWN;
   if( strchr( token,':' ) != NULL || !strcmp( uUpperString( token ),"BYE" ))
      return TOK_GAME_RES;
   while( isdigit( *ptr ) )
   {
      ptr++;
      if( *ptr == '\0' )
         return TOK_NUMBER;
   }
   return TOK_NONE;
}


///////////////////////////////////////////////////////////////////
//
// Converts a string token to an int.
//
int TrnData_c::ReadStartNo( char *token )
{
   return atoi( token );
}


///////////////////////////////////////////////////////////////////
//
// Converts a string token to an int.
//
int TrnData_c::ReadRtg( char *token )
{
   return atoi( token );
}

///////////////////////////////////////////////////////////////////
//
// Converts a parameter token to an int.
// Parametr token is in a form like: Param=20.
//
int TrnData_c::ReadIntParam( char *token )
{
   char *pos;
   int int_val;

   pos = strchr( token, '=' );
   if( pos == NULL )
      int_val = 0;
   else
      int_val = atoi( pos+1 );
   return int_val;
}

/////////////////////////////////////////////////////////////////
//
// Parses a result token and stores data in a Res_t type.
//
Res_t TrnData_c::ReadResult( char *token )
{
   Res_t result;
   char score_buf[ MAX_SCORE_STR + 1 ];
   char *p;
   int i, score_len;

   memset( &result, 0, sizeof( Res_t ) );
   if( !strcmp( uUpperString( token ), "BYE" ) )
   {
      result.oppon = 0;
      result.score = ByeEquiv;
      result.type  = BYE;
      return result;
   }

   if( ( p = strchr( token, ':' ) ) == NULL )
      return result;

   score_len = p - token;
   if( score_len > MAX_SCORE_STR )
      return result;

   for( i = 0; i < score_len; i++ )
      score_buf[ i ] = token[ i ];
   score_buf[ score_len ] = '\0';

   result.score = StrToScore( score_buf, result.type );
   p = strchr( token, ':' );
   p++;
   result.oppon = atoi( p );

   return result;
}

//////////////////////////////////////////////////////////////
//
//  Reads in total score given in a form like:  5.5/10
//
int TrnData_c::ReadTotScore( char *token, int& games )
{
   char score_buf[ MAX_SCORE_STR + 1 ];
   int i, score, game_type;
   char *p;

   games = 0;
   if( ( p = strchr( token, '=' ) ) == NULL )
      return 0;

   p++;
   i = 0;
   while( ( isdigit( *p ) || *p == '.' ) && i <= MAX_SCORE_STR )
   {
      score_buf[ i ] = *p;
      p++;
      i++;
   }
   score_buf[ i ] = '\0';

   score = StrToScore( score_buf, game_type );

// p should be pointing to '/' now
   if( *p != '/' )
      return 0;
   p++;
   games = atoi( p );
   return score;
}

////////////////////////////////////////////////////////////////
//
// Converts string to a score.
// Returns points scored and result type (&type).
//
int TrnData_c::StrToScore( char *buf, int& type, char draw_symb )
{
   int points;
   char *p;

   if( buf == NULL )
      return 0;
   else
      p = buf;
   type = NORMAL_PLAYED;
   switch( *p )
   {
      case 'w':
      case 'W':
         points = 2;
         type = NORMAL_PLAYED;
         break;
      case 'l':
      case 'L':
         points = 0;
         type = NORMAL_PLAYED;
         break;
      case 'd':
      case 'D':
         points = 1;
         type = NORMAL_PLAYED;
         break;
      case '-':
         points = 0;
         type = FORFEIT_FLAG;
         break;
      case '+':
         points = 0;
         type = FORFEIT_FLAG;
         break;
      case '\0':
         points = 0;
         type = NOT_PLAYED;
         break;
      default:
         if( *p == draw_symb )
         {
            points = 1;
            type = NORMAL_PLAYED;
            break;
         }
         if( isdigit( *p ) )
         {
            points = atoi( buf );
            points <<= 1;
            type = NORMAL_PLAYED;
            p = strchr( buf, '.' );
            if( (p != NULL) && (*(p+1) == '5' ) )
               points++;
         }
         else
         {
            type = NOT_PLAYED;
            points = 0;
         }
         break;
   }
   return points;
}
