/*
#####################################
#            PMDemo.c               #
#   Presentation Manager Demo for   #
#             PrjMgr                #
#####################################
#   Written 1995 by Roland Knospe   #
#####################################
*/

/* Only Win & Process functions from os2.h */
#define INCL_WIN
#define INCL_DOSPROCESS
#include <os2.h>

/* Standard library for _beginthread */
#include <stdlib.h>

/* Math library for the simulation */
#include <math.h>

/* Own Header */
#include "PMDemo.h"

/*
##############################
#        Main Function       #
##############################
 */
INT main( void)
{

   HAB  hab;          /* Anchor-Block Handle */
   HMQ  hmq;          /* Message-Queue Handle */
   QMSG qmsg;         /* Message-Queue */
   HWND MainHwnd;     /* Dialog-Box Handle */

   /* Get HAB */
   hab=WinInitialize(0);
   /* Create Message Queue */
   hmq=WinCreateMsgQueue(hab,0);

   /* Load Dialog-Box defined in PMDEmo.rc */
   MainHwnd= WinLoadDlg( HWND_DESKTOP, HWND_DESKTOP,
                         MainDialog, NULLHANDLE, ID_MAINDIALOG, NULL );

   /* Main Message Loop */
   while( WinGetMsg( hab, &qmsg, 0L, 0, 0))
      WinDispatchMsg( hab, &qmsg);

   /* Bye Bye Window */
   WinDestroyWindow( MainHwnd );
   /* Bye Bye Messagequeue */
   WinDestroyMsgQueue( hmq);
   /* Bye Bye HAB */
   WinTerminate(hab);

   return 0;
}


/*
##############################
#      Dialog Procedure      #
##############################
 */
MRESULT EXPENTRY MainDialog (HWND hwnd,ULONG msg,MPARAM mp1,MPARAM mp2)
{
   static LONG xmax,xmin,     /* Area for the Satellite */
               ymax,ymin,
               posx,posy,     /* Satellite position */
               midx,midy;     /* The middle */
   static THREADARGS Args;    /* Thread arguments */
   static TID SimulationID;   /* Simulation thread ID */
   PTIB pTib;                 /* Thread info Block */
   PPIB pPib;                 /* Process info Block */

   /* Check Message */
   switch(msg)
   {
      /* First -> Init */
      case WM_INITDLG:
         /* Get TIB and PIB */
         DosGetInfoBlocks ( &pTib, &pPib);
         /* Set priority */
         DosSetPriority (PRTYS_THREAD,PRTYC_FOREGROUNDSERVER,PRTYD_MINIMUM,
                         pTib->tib_ptib2->tib2_ultid);

         /* Calculate new Window Position & satellite area */
         InitDialog( hwnd, &xmin, &xmax, &ymin, &ymax );

         /* Middle offest */
         midx=xmin+((xmax-xmin)>>1);
         midy=ymin+((ymax-ymin)>>1);

         /* Set Ball position -> the middle */
         WinSetWindowPos ( WinWindowFromID ( hwnd, ID_BALL), NULLHANDLE,
                           midx-(WinQuerySysValue(HWND_DESKTOP, SV_CXICON)>>1),
                           midy-(WinQuerySysValue(HWND_DESKTOP, SV_CYICON)>>1),
                           0, 0,SWP_MOVE);

         /* Slider position 100 */
         WinSendDlgItemMsg( hwnd, ID_GRAVITY, SLM_SETSLIDERINFO,
                            MPFROM2SHORT( SMA_SLIDERARMPOSITION, SMA_INCREMENTVALUE),
                            MPFROMSHORT( 100 ));

         Args.Gravity=100* GRAV_FACTOR;   /* Start gravity */
         Args.xmax=(xmax-xmin)>>1;        /* Simulation X-Range */
         Args.ymax=(ymax-ymin)>>1;        /* Simulation Y-Range */
         Args.Termination=FALSE;          /* It's the start, not the end */
         /* Ball area */
         Args.BallArea=(WinQuerySysValue (HWND_DESKTOP, SV_CXICON)>>1)*
                       (WinQuerySysValue (HWND_DESKTOP, SV_CYICON)>>1)+
                       SAT_RADIUS*SAT_RADIUS;
         /* Start simulation thread */
         SimulationID=_beginthread( Simulation, NULL, 2048, &Args);

         /* Start Timer for Satellite move */
         WinStartTimer( WinQueryAnchorBlock( hwnd), hwnd, ID_TIMER,200);
      break;

      /* Commands from Buttons */
      case WM_COMMAND:
         switch( SHORT1FROMMP(mp1))
         {
            /* Bye Bye .... */
            case ID_CANCEL:
              WinPostMsg( hwnd, WM_QUIT, (MPARAM)0, (MPARAM)0);
            break;
         }
      break;

      /* Slider notification -> Gravity has changed */
      case WM_CONTROL:
         switch( SHORT2FROMMP( mp1))
         {
            case SLN_CHANGE:
               DosEnterCritSec(); /* To avoid conflicts between Dialog & Thread */
               Args.Gravity=GRAV_FACTOR*((double)(LONG)(mp2));
               DosExitCritSec();
            break;
         }

      case WM_TIMER:
         switch( SHORT1FROMMP( mp1))
         {
            case ID_TIMER:
               /* Stop timer to avoid message overflow */
               WinStopTimer( WinQueryAnchorBlock( hwnd), hwnd, ID_TIMER);

               DosEnterCritSec();
               posx=Args.xpos;    /* New Satellite position */
               posy=Args.ypos;
               DosExitCritSec();

               /* Move Satellite */
               WinSetWindowPos ( WinWindowFromID ( hwnd, ID_SAT), NULLHANDLE,
                                 posx+midx-SAT_RADIUS, posy+midy-SAT_RADIUS,
                                 0, 0, SWP_MOVE);

               WinStartTimer( WinQueryAnchorBlock( hwnd), hwnd, ID_TIMER,30);
            break;
         }
      break;

      /* Bye Bye .....*/
      case WM_CLOSE:
         WinStopTimer( WinQueryAnchorBlock( hwnd), hwnd, ID_TIMER);
         WinPostMsg( hwnd, WM_QUIT, (MPARAM)0, (MPARAM)0);
      break;

      case WM_DESTROY:
         /* Terminate thread */
         Args.Termination=TRUE;
         DosWaitThread( &SimulationID, DCWW_WAIT );
      break;

      /* An other will do the work !! */
      default:
         return WinDefDlgProc(hwnd,msg,mp1,mp2);
      break;

   }

   return (MRESULT)0;
}



/*
##############################
#         InitDialog         #
#   Calculate new Window     #
# Position & Satellite area  #
##############################
*/
void InitDialog(HWND hwnd, LONG *xmin, LONG *xmax, LONG *ymin, LONG *ymax)
{
   SWP Pos;          /* Structure for Window Position */
   POINTL coords[2]; /* Structure for CharSize to Point conversion */

   /* Get Window Position */
   WinQueryWindowPos( hwnd, &Pos);

   /* Calculate the new Position -> The middle of the Desktop */
   Pos.x= (WinQuerySysValue (HWND_DESKTOP, SV_CXSCREEN)>>1)-(Pos.cx>>1);
   Pos.y= (WinQuerySysValue (HWND_DESKTOP, SV_CYSCREEN)>>1)-(Pos.cy>>1);

   /* Move & Show Window */
   WinSetWindowPos( hwnd, HWND_TOP, Pos.x, Pos.y, 0, 0,
                    SWP_SHOW|SWP_MOVE|SWP_ZORDER);

   /* Charsize -> Points */
   coords[0].x=15;
   coords[0].y=50;
   coords[1].x=220;
   coords[1].y=111;
   WinMapDlgPoints(HWND_DESKTOP,coords,2,TRUE);

   /* Calculate satellite area */
   *xmin=WinQuerySysValue(HWND_DESKTOP, SV_CXDLGFRAME)+coords[0].x+5;
   *ymin=WinQuerySysValue(HWND_DESKTOP, SV_CYDLGFRAME)+coords[0].y+5;
   *xmax=*xmin+coords[1].x-10;
   *ymax=*ymin+coords[1].y-10;
}


/*
##############################
#         Simulation         #
##############################
*/
void Simulation( void *ArgPtr )
{
   VECTOR Pos,          /* Position */
          Vel;          /* Velocity */
   double Angle1,Angle2,/* Temp vars */
          Value1,Value2,
          Grav;         /* Gravity */
   LONG   xmax,ymax;    /* Local simulation range */
   THREADARGS *Arg;     /* Argument pointer */
   PTIB pTib;           /* Thread info Block */
   PPIB pPib;           /* Process info Block */

   /* Initialisation */
   /* Get TIB and PIB */
   DosGetInfoBlocks ( &pTib, &pPib);
   /* Set priority */
   DosSetPriority (PRTYS_THREAD,PRTYC_TIMECRITICAL,PRTYD_MINIMUM,
                   pTib->tib_ptib2->tib2_ultid);
   Arg=(THREADARGS*)ArgPtr;   /* Get arguments */
   xmax=Arg->xmax;            /* Copy range to local vars */
   ymax=Arg->ymax;
   Vel.x=START_V_X;           /* Init start velocity */
   Vel.y=START_V_Y;
   Pos.x=-(xmax>>1);          /* Start position */
   Pos.y=-(ymax>>1);

   while( !Arg->Termination )
   {
      DosEnterCritSec();      /* To avoid conflicts between Dialog & Thread */
      Grav=Arg->Gravity;      /* Get gravity */
      DosExitCritSec();

      /* Calculate next position */
      GravSimulation( &Pos, &Vel, Grav);

      /* right x border reached  */
      if( Pos.x > xmax)
      {
         Vel.x*=-DAMPING;
         Pos.x=xmax;
         GravSimulation( &Pos, &Vel, Grav);
      }

      /* left x border reached  */
      if( Pos.x < -xmax)
      {
         Vel.x*=-DAMPING;
         Pos.x=-xmax;
         GravSimulation( &Pos, &Vel, Grav);
      }

      /* right y border reached  */
      if( Pos.y > ymax)
      {
         Vel.y*=-DAMPING;
         Pos.y=ymax;
         GravSimulation( &Pos, &Vel, Grav);
      }

      /* left y border reached  */
      if( Pos.y < -ymax)
      {
         Vel.y*=-DAMPING;
         Pos.y=-ymax;
         GravSimulation( &Pos, &Vel, Grav);
      }

      /* Satellite has reached Ball ? */
      if( (Pos.y*Pos.y+Pos.x*Pos.x)<Arg->BallArea )
      {
         Angle1=atan2(Pos.y,Pos.x);                      /* Position angle */
         Angle2=atan2(Vel.y,Vel.x);                      /* Velocity angle */
         Value1=1+sqrt(Arg->BallArea);                   /* Ball radius */
         Value2=DAMPING*sqrt(Vel.x*Vel.x+Vel.y*Vel.y);   /* New Velocity value*/
         Pos.x=cos(Angle1)*Value1;                       /* Correct position */
         Pos.y=sin(Angle1)*Value1;
         Vel.x=-cos(2*Angle1-Angle2)*Value2;             /* New Velocity vector */
         Vel.y=-sin(2*Angle1-Angle2)*Value2;
      }


      DosEnterCritSec();      /* The Dialog shoud know it ! */
      Arg->xpos=(LONG)Pos.x;
      Arg->ypos=(LONG)Pos.y;
      DosExitCritSec();

      DosSleep(10);           /* Wait a moment */
   }
}



/* GravSimulation :
ͻ
                                                                 
    dX(t)                       X(t)                             
   ------ = Vx0 - Const * ----------------- *t                   
     dt                      2      2   1.5                      
                          ( X(t) + Y(t) )                        
                                                                 
                                                                 
    dY(t)                       Y(t)                             
   ------ = Vy0 - Const * ----------------- *t                   
     dt                      2      2   1.5                      
                          ( X(t) + Y(t) )                        
                                                                 
                                                                 
          dX(t)                                                  
   X(t) =------ * t                                              
           dt                                                    
                                                                 
          dY(t)                                                  
   Y(t) =------ * t                                              
           dt                                                    
͹
Time discreet:                                                   
                                                                 
  Vx[0] = Vx0                                                    
  Vy[0] = Vy0                                                    
                                                                 
                                                                 
  Vy[K*T] - Vy[(K-1)*T]                     X[K*T]               
  --------------------- = - Const * --------------------- * K*T  
            T                          2        2     1.5        
                                    ( X[K*T] + Y[K*T] )          
                                                                 
  Vx[K*T] - Vx[(K-1)*T]                     Y[K*T]               
  --------------------- = - Const * --------------------- * K*T  
            T                          2        2     1.5        
                                    ( X[K*T] + Y[K*T] )          
                                                                 
  X[K*T]= Vx[K*T] * T*K                                          
                                                                 
  Y[K*T]= Vy[K*T] * T*K                                          
                                                                 
                                                                 
͹
 With T=1:                                                       
ͼ
*/

void GravSimulation( VECTOR *Pos, VECTOR *Vel, double Grav )
{
   double Temp;

   Temp=Grav / pow((Pos->x*Pos->x)+(Pos->y*Pos->y),1.5);

   Vel->y-= Temp * Pos->y;
   Vel->x-= Temp * Pos->x;

   Pos->x+=Vel->x;
   Pos->y+=Vel->y;

}
