/* tube.c -- test program for libkb and graphic libraries
 * Copyright (C) Markus F.X.J. Oberhumer and James Johnson
 * For conditions of distribution and use, see copyright notice in kb.h 
 */

/* note: I've made many changes to this program, orignal info follows */

/*************************************************************************
	Program	: Tubular
	File	: tube.cpp
	Notes	: Set tab spaces to 4.
	
	Programmer	: James Johnson <plexus@stein.u.washington.edu>
	Date		: 1-14-94
	Version		: 1.0
 ************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <math.h>
#include <signal.h>
#include <assert.h>

#include <kb.h>
#include "intro.h"


#if !defined(CLK_TCK) && defined(CLOCKS_PER_SEC)
#  define CLK_TCK			CLOCKS_PER_SEC 
#endif


/***********************************************************************
// Linux svgalib wrapper
************************************************************************/

#if defined(__KB_LINUX)

#include <vga.h>
#include <vgagl.h>

GraphicsContext physicalscreen;

int setmode13(int x)
{
	int modes[] = { G320x200x256, G320x240x256, G720x348x2, G640x480x256 };
	int m;

	if (x < 0 || x > 3)
		x = 0;
	m = modes[x];

	vga_init();
	if (vga_setmode(m) != 0)		/* <- this does the real init of svgalib */
		return -1;
	if (gl_setcontextvga(m) != 0)
		return -1;
	gl_getcontext(&physicalscreen);
	/* gl_enableclipping(); */		/* not needed */
	return 0;
}

void setmode03(void)
{
	vga_setmode(TEXT);
}

void setcolor(int c, int r, int g, int b)
{
	vga_setpalette(c,r,g,b);
}

__inline__ void setpixel(int x, int y, int c)
{
	vga_setcolor(c);
	vga_drawpixel(x,y);
}

void waitvrt(void)
{
	vga_waitretrace();
}

#endif
    

/***********************************************************************
// MSDOS graphics wrapper 
************************************************************************/

#if defined(__KB_MSDOS)

#define WIDTH	320
#define HEIGHT	200

#include "_kb.h"		/* KB_INP8, KB_OUTP8 */

#if defined(__KB_MSDOS16) && defined(__BORLANDC__)
#define VIDMEM	((unsigned char far *) 0xa0000000l)
#pragma argsused
int setmode13(int dummy)
{
	asm	mov		ax,13h
	asm	int		10h
	return 0;
}
void setmode03(void)
{
	asm	mov		ax,03h
	asm	int		10h
}
#endif

#if defined(__GO32__) && !defined(__DJGPP__)
#define VIDMEM	((unsigned char *) 0xd0000000)
int setmode13(int dummy)
{
	__asm__ __volatile__ ("movw  $0x0013,%%ax":::"%ax");
	__asm__ __volatile__ ("int   $0x10");
	return 0;
}
void setmode03(void)
{
	__asm__ __volatile__ ("movw  $0x0003,%%ax":::"%ax");
	__asm__ __volatile__ ("int   $0x10");
}
#endif

#if defined(__KB_MSDOS32) && defined(__WATCOMC__)
#define VIDMEM	((unsigned char *) 0xa0000)
void my_setmode(short mode);
#pragma aux my_setmode = "int 10h" parm [ax] modify[ax bx cx dx];
int setmode13(int dummy)
{
	my_setmode(0x13);
	return 0;
}
void setmode03(void)
{
	my_setmode(0x03);
}
#endif

void setpixel(int x, int y, int c)
{
	*(VIDMEM + y*WIDTH + x) = c;
}

static void kb_delay(void)
{
	volatile int k = 10;
	while (k > 0)
		k--;
}

void setcolor(int c, int r, int g, int b)
{
	/* fprintf(stderr,"%3d %3d %3d %3d\n",c,r,g,b); */
	KB_OUTP8(0x3c8, c);	
	kb_delay();
	KB_OUTP8(0x3c9, r);
	kb_delay();
	KB_OUTP8(0x3c9, g);
	kb_delay();
	KB_OUTP8(0x3c9, b);
}

/* wait for the vertical retrace trailing edge */
void waitvrt(void)
{
	while (!(KB_INP8(0x3da) & 0x8))
		;
	while ((KB_INP8(0x3da) & 0x8))
		;
}

#endif /* __KB_MSDOS */


/***********************************************************************
//
************************************************************************/

#ifndef M_PI
#define M_PI        3.14159265358979323846
#endif

#define Min_Z        (1  << 3)				/* color  1 */
#define Max_Z       ((64 << 3) - 1)			/* color 63 */
#define Min_Dz      2
#define Max_Dz      64
#define Def_DZ		3
#define Step	    20
#define Max_Rand	256
#define Max_Active	((Max_Z / 2) * Step)

typedef struct
{
	int x;
	int y;
	int z;
	int Old_x;
	int Old_y;
	int xc;
	int yc;
}
Star;

#if defined(__KB_MSDOS16)
typedef Star huge *PStar;
static huge Star star[Max_Active];
#else
typedef Star *PStar;
static Star star[Max_Active];
#endif

static int Xtable[Max_Rand];
static int Ytable[Max_Rand];
static int Index[Max_Rand];


/* setup colors - lowest z value is nearest star (should be brightest) */
/* play around - this really changes the way it looks */
void initpal(void)
{
	int i;

	setcolor(0,0,0,15);
	for (i = 1; i < 64; i++)
	{
#if 1
		int tmp = (int) (pow(64.0,(48-i)/48.0) + 0.5);
		if (tmp > 63)
			tmp = 63;
		else if (tmp < 1)
			tmp = 1;
#else
		unsigned tmp = (unsigned) (pow(64,i/48.0) + 0.5);
		tmp = (tmp > 63) ? 63 : tmp;
		tmp = 63 - tmp;
#endif
		setcolor(i, tmp, tmp, (tmp < 15) ? 15 : tmp);
	}
}		
						

/***********************************************************************
//
************************************************************************/

static void init(void)
{
	int i, k, tmp;
	unsigned Randnum;

	for (i = 0; i < Max_Active; i++)
		star[i].z = 0;

	for (i = 0; i < Max_Rand; i++)
	{
		Xtable[i] = (int)(128 * cos(i * 2*M_PI / Max_Rand) + 0.5); 
		Ytable[i] = (int)(128 * sin(i * 2*M_PI / Max_Rand) + 0.5);
		Index[i] = i;
	}
	
	/* shuffle Index[] */
	for (k = 0; k < Max_Rand/2; k++)
		for (i = 0; i < Max_Rand; i++)
		{
			Randnum = rand() % Max_Rand;
			tmp = Index[Randnum];
			Index[Randnum] = Index[i];
			Index[i] = tmp;
		}
}
				

static unsigned long doit(void)
{
	int i, k;
	int dz				= Def_DZ;
	unsigned Tube_Xr	= 60;
	unsigned Tube_Yr	= 60;
	unsigned X_choice	= 0;
	unsigned Y_choice	= 0;
	unsigned X_count	= 64;
	unsigned Y_count	= 0;
	unsigned D_tube_xr	= 64;
	unsigned D_tube_yr	= 0;
	int Num_Active		= 0;
	int First_Free      = 0;
	unsigned count		= 0;
	unsigned long frames = 0;

	for (;;)
	{
		/* fill in new stars */
		i = First_Free;
		for (k = 0; k < Step && Num_Active < Max_Active; k++)
		{
			while (star[i].z != 0)
				i++;
			star[i].x = Tube_Xr * Xtable[Index[X_choice]];
			star[i].y = Tube_Yr * Ytable[Index[Y_choice]];
			star[i].z = Max_Z;
			star[i].xc = WIDTH/2  + (120 * Xtable[X_count]) / Max_Rand;
			star[i].yc = HEIGHT/2 + ( 80 * Ytable[Y_count]) / Max_Rand;
				
			if (++X_choice >= Max_Rand)
				X_choice = 0;
			if (++Y_choice >= Max_Rand)
				Y_choice = 0;
				
			Num_Active++;
			i++;
		}
			
		/* move stars */
		for (i = 0; i < Max_Active; i++)
		{
			PStar s = &star[i];

			if (s->z == 0)	/* star is not active */
				continue;

 			if (s->z > Min_Z)
			{
				/* calculate the new one */
				int X_tmp = s->x / s->z + s->xc;
				
				/* if x lies within boundaries */
				if (X_tmp >= 0 && X_tmp < WIDTH)
				{
					int Y_tmp = s->y / s->z + s->yc;
					
					/* if y lies within boundaries.*/
					if (Y_tmp >= 0 && Y_tmp < HEIGHT)
					{
						if (X_tmp != s->Old_x || Y_tmp != s->Old_y)
						{
							/* erase the old star */
							setpixel(s->Old_x,s->Old_y,0);
							/* draw new star */
							s->Old_x = X_tmp;
							s->Old_y = Y_tmp;
							setpixel(X_tmp,Y_tmp,s->z >> 3);
						}
						s->z -= dz;

						continue;		/* <- done */
					}
				}
			}

			/* erase the old star */
			setpixel(s->Old_x,s->Old_y,0);

			/* remove the star */
			s->z = 0;
			Num_Active--;
			if (i < First_Free)
				First_Free = i;
		}
		
		X_count += 2;
		if (X_count >= Max_Rand) X_count = 0;
		Y_count++;
		if (Y_count >= Max_Rand) Y_count = 0;
			
		/* this section increase the X and Y radii, giving ellipses */
		if (++count >= 5)
		{
			count = 0;
			Tube_Xr = 20 * Xtable[D_tube_xr] / Max_Rand + 60;
			Tube_Yr = 20 * Ytable[D_tube_yr] / Max_Rand + 60;
			if(++D_tube_xr >= Max_Rand) D_tube_xr = 0;
			if(++D_tube_yr >= Max_Rand) D_tube_yr = 0;
		}

		frames++;

		/* handle keys */
		while (kb_kbhit())
		{
			int Factor;
			unsigned key;

			Factor = dz >= 10 ? 2 : 1;
			key = kb_getkey();
			switch(key)
			{
			case 0x1b:
				return frames;
			case '+':
				if (dz+Factor <= Max_Dz) dz += Factor;
				break;
			case '-':
				if (dz-Factor >= Min_Dz) dz -= Factor;
				break;
			}
		}

		/* could wait for retrace here... */
		/* waitvrt(); */
	}
}


/***********************************************************************
//
************************************************************************/

static int in_graphics = 0;

void my_textmode(void)
{
	if (in_graphics)
	{
		in_graphics = 0;
		setmode03();
	}
}

void my_cleanup(void)
{
	my_textmode();
}

void my_shutdown(void)
{
	my_cleanup();
	exit(255);
}


/***********************************************************************
//
************************************************************************/

int main(int argc, char *argv[])
{   
	unsigned long frames;
	clock_t t1, t2;
	float secs, fps;
	char s[80+1];
	int mode = 0;

	/* init graphics */
	if (argc > 1)
		mode = atoi(argv[1]);
	if (mode < 0)				/* allow '-1' */
		mode = -mode;
	if (setmode13(mode) != 0)
	{
		printf("could not enter graphics mode %d\n",mode);
		exit(1);
	}
	in_graphics = 1;
	atexit(my_textmode);
	initpal();

	/* init keyboard after graphics (svgalib) */
	if (!(argc > 2))
	{
#if 0
		signal(SIGINT,SIG_IGN);		/* ignore SIGINT */
#endif
		if (kb_install(KB_FLAG_EMERGENCY_EXIT) == 0)
		{
			kb_set_cleanup(NULL,my_cleanup);
			kb_set_emergency_cleanup(NULL,my_shutdown);
		}
	}


	srand(0);
	srand((unsigned)time(NULL));
	init();
	t1 = clock();
	frames = doit();
	t2 = clock();


#if 0
	/* now let's test the libkb signal handler */
	/* !!! DON'T TRY THIS WITH Borland C v3.1, see documentation !!! */
	/* (works with Borland C v4.0, though) */
#if !(defined(__KB_MSDOS16) && defined(__BORLANDC__))
	assert(0 == 1);
#endif
#endif
	

	my_textmode();

	fputs("\n",stdout);
	fputs(_kb_intro_text(s),stdout);
	fputs("\n\n",stdout);

	secs = (t2-t1) / (float)(CLK_TCK);
	fps = (secs > 0.001) ? frames / secs : 0;
	printf("%lu frames, %.2f secs, %.2f frames/sec\n", frames, secs, fps);
	fputs("\n",stdout);

	return 0;
}


/*
vi:ts=4
*/

