/****************************************************************************

    Module  : Dialogs.c

	All functions to handle dialog boxes used by FreeDock

	Changes:
	3.0 beta 3 - Created this module
	3.0 beta 4 - Modified way manual filling out of empty slot was handling
	default Icon for aoolications with no icons
    3.0 beta 5 - Added NT specific shutdown / reboot / logoff options to 
    ExitWindows built-in
*****************************************************************************/

#include <windows.h>
#include <commdlg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <direct.h>
#include <ctype.h>
#include <shellapi.h>
#include "freedock.h"
#include "dock_reg.h"
#include "titles.h"
//#include "hooks.h"

/**********************************************************
	Global Data for the various system metrics we need
**********************************************************/
extern GLOBAL_METRICS gMetrics;
extern GLOBAL_OPTIONS gOptions;


BOOL FAR PASCAL	AppOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
	static SLOT_ENTRY TmpSlot;		// Temporary work space for slot structure
	static SLOT_ENTRY *Slot;		// Pointer to actual Slot
	static DOCK_STRUCT  *Dock, *SubDock;		// Pointer to dock containing above slot.
												// Pointer to sub-dock

    static BOOL     Grabbing, bWasEmpty;
    static int      NumIcons;
    RECT            rect;
    static HWND     hwndGrabber, hwndFocus, hwndOldFocus, hwndTmp;
	static HDC		hdcFocus;
    POINT           pt;
    char            TmpBuffer[MAX_FPATH_LEN]; //, *TmpCharPtr;
    int             Status;
    short int       TmpX, TmpY;
    static HCURSOR  hGrabberCur;
    static FARPROC  lpfnChooseIconDlgProc, lpfnPreviewerDlgProc;
    DRAWITEMSTRUCT *ItemStruct;
           RECT     rcItem;
           HWND     hwndItem;
           HDC      hdcGrabber;
    static HBITMAP  hbmGrabber, hbmGrabbing;

    switch (iMessage) {
        case WM_INITDIALOG:

			Slot = (SLOT_ENTRY *)lParam; // Retrieve pointer to slot from parameter
			Dock = Slot->Dock;			 // Retrieve pointer to parent dock

            SendDlgItemMessage(hDlg, QX_FILE, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);
            SendDlgItemMessage(hDlg, QX_CMDLINE, EM_LIMITTEXT, MAX_CMDLINE_LEN, 0L);
            SendDlgItemMessage(hDlg, QX_RUNTIMEDIR, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);
            SendDlgItemMessage(hDlg, QX_ICONFILE, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);
            SendDlgItemMessage(hDlg, QX_TITLE_TEXT, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);

            lpfnChooseIconDlgProc = MakeProcInstance((FARPROC) ChooseIconDlgProc, gOptions.hAppInst);
            lpfnPreviewerDlgProc  = MakeProcInstance((FARPROC) PreviewerDlgProc,  gOptions.hAppInst);

			// Copy Selected Slot into Workspace
            memcpy(&TmpSlot, Slot, sizeof(SLOT_ENTRY));

            // Initialise TmpSlots Scratch Space
            TmpSlot.IconIndex = (gMetrics.IconWidth * (gOptions.MaxDockSize));
            // Load the current Iocn into the Scratch Space
            UtilLoadIcon( Dock, &TmpSlot );

            CheckDlgButton(hDlg, QX_HIDE_ON_GRAB, gOptions.HideDlgOnGrab);

			// Set the Start in Separate Memory check (NT only)
#ifdef WIN32
            CheckDlgButton(hDlg, QX_START_SEPARATE, TmpSlot.StartSeparate);
#else
            // Disable "WIN 32 only" Options
            EnableWindow( GetDlgItem(hDlg, QX_START_SEPARATE), FALSE );
#endif

            switch (TmpSlot.StartState) {

                case START_MINIMUM:
                    CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_MINIMUM);
                    break;

                case START_MAXIMUM:
                    CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_MAXIMUM);
                    break;

                case START_STORE:
                    CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_STORE);
                    break;

                default:
                    CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_NORMAL);
                    break;
            }

            CheckDlgButton(hDlg, QX_START_TOP, TmpSlot.StartOnTop);

            if ( TmpSlot.SlotType == SLOT_USED ){
                NumIcons = TmpSlot.IconTotal;
            }
            else{
                NumIcons = 0;
            }

            SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
            SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);

            /* Set starting values. */
            SetDlgItemText(hDlg, QX_FILE, TmpSlot.AppName);
            SetDlgItemText(hDlg, QX_CMDLINE, TmpSlot.CmdLine);
            SetDlgItemText(hDlg, QX_RUNTIMEDIR, TmpSlot.RunTimeDir);
            SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
			sprintf( TmpBuffer, "%s.%d", Dock->DockID, TmpSlot.ID );
            SetDlgItemText(hDlg, QX_SLOT_ID, TmpBuffer);
            SetDlgItemText(hDlg, QX_TITLE_TEXT, TmpSlot.Title);
            Grabbing = FALSE;

			/********************************************************
				There is a special condition when an empty slot is
				left clicked and the App name specified
			********************************************************/
			bWasEmpty = (TmpSlot.AppName[0] == '\0');

            return (TRUE);

        case WM_PAINT:
            /** Paint the Grabber bitmap **/
            hwndTmp = GetDlgItem( hDlg, QX_GRABWIN );
            hdcGrabber = GetDC( hwndTmp );
            hbmGrabber = LoadBitmap(gOptions.hAppInst, "GRABBER");
            hbmGrabbing = LoadBitmap(gOptions.hAppInst, "GRABBER2");
			if(Grabbing){
            	DrawBitmap(hdcGrabber, hbmGrabbing, 0, 0);
			}
			else{
            	DrawBitmap(hdcGrabber, hbmGrabber, 0, 0);
			}
            ReleaseDC( hwndTmp, hdcGrabber );
            DeleteObject( hbmGrabber );
            DeleteObject( hbmGrabbing );
            /** return false to force the default message handler to paint **/
            /** the rest of the diaog box **/
            return(FALSE);

        case WM_DRAWITEM :  // Draw the Current Icon
            ItemStruct = (DRAWITEMSTRUCT *)lParam;
            UtilDrawIcon( ItemStruct->hDC, Dock, &TmpSlot, 0, 0, FALSE );
            return (TRUE);

		case WM_MOUSEMOVE :
			if (Grabbing){
				// manually convert client coords to screen coords to handle
				// correctly the event when the client coords are negative.
	            GetWindowRect(hDlg, &rect);
    	        pt.x = rect.left + (short int)LOWORD(lParam) + gMetrics.DlgFrameWidth;
        	    pt.y = rect.top + (short int)HIWORD(lParam) + gMetrics.DlgFrameHeight + gMetrics.DlgTitleHeight;

 				hwndOldFocus = hwndFocus;
				hwndFocus = WindowFromPoint(pt);

	            /* Climb back up to the top of a parent-child list */
    	        hwndTmp = GetParent(hwndFocus);
        	    while (hwndTmp != NULL) {
            	    hwndFocus = hwndTmp;
                	hwndTmp = GetParent(hwndFocus);
	            }

				// If the window under the cursor is different from the one which
				// was under the cursor last time...
				if( hwndOldFocus != hwndFocus ){
					// Erase old focus rect if this is not the initial move
					if( hwndOldFocus != NULL ){
	  			        GetWindowRect( hwndOldFocus, &rect);
						rect.left = 0; // convert screen coords to window relative
						rect.top = 0;
						hdcFocus = GetWindowDC( hwndOldFocus );
						DrawFocusRect( hdcFocus, &rect );
						ReleaseDC( hwndOldFocus, hdcFocus );
					}
					// draw new focus rect
  			        GetWindowRect( hwndFocus, &rect);
					rect.left = 0; // convert screen coords to window relative
					rect.top = 0;
					hdcFocus = GetWindowDC( hwndFocus );
					DrawFocusRect( hdcFocus, &rect );
					ReleaseDC( hwndFocus, hdcFocus );
				}
				return (TRUE);
			}
			break;

		case WM_MOVE : // If the dialog box is moved, we need to repaint the
                       // grabber icon
			GetWindowRect(hDlg, &rect);
			InvalidateRect(hDlg, &rect, FALSE);
			UpdateWindow(hDlg);
			return (TRUE);

        case WM_HSCROLL:
            if (wParam == SB_LINEDOWN)
                if (NumIcons > 0)
                    TmpSlot.IconPos = min(NumIcons - 1, TmpSlot.IconPos + 1);

            if (wParam == SB_LINEUP)
                if (NumIcons > 0)
                    if (TmpSlot.IconPos > 0)
                        TmpSlot.IconPos = TmpSlot.IconPos - 1;

            // Update icon number
            SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
            // Load the Icon into the slot's own bitmap  & update the IconTotal Field
            UtilLoadIcon( Dock, &TmpSlot );
            // ReDraw Icon by invalidating the button area
            hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
            GetClientRect( hwndItem, &rcItem );
            InvalidateRect( hwndItem, &rcItem, TRUE );

            return (TRUE);

        case WM_COMMAND:
            switch (wParam) {

				case QX_MAKE_SUB_DOCK:
				    {
					char SubDockID[MAX_DOCK_ID_LEN];
				    // Create ID for Sub-Dock from ID of this Dock + ID of this slot
					sprintf( SubDockID, "%s.%d", Dock->DockID, TmpSlot.ID );
					// Create the SubDock Window
					TmpSlot.hwndSubDock = CreateDock( SUB_DOCK, Dock->hwndDock, SubDockID );
					SubDock = GetDockStruct( TmpSlot.hwndSubDock );
					// Setup the SubDock's parent slot
					SubDock->ParentSlot = Slot;
					}

					// Copy the original slot into the first location in the new sub-dock;
					SwapSlots( &SubDock->Slot[0], Slot );

					// Load the Icon into sub-dock's icon cache
                    UtilLoadIcon( SubDock, &SubDock->Slot[0] );

					// basic setup of new sub-doc
					SubDock->SlotCount = 10;

					// Mark this slot into a sub-dock
					TmpSlot.SlotType = SLOT_SPECIAL_SUB;

					// get the default sub-dock icon
                    Status = (UINT)FindExecutable("progman.exe", NULL, TmpSlot.IconFile);
       		        if( Status <= 31 ){
       		               DockError( hDlg, "Program Manager not found, cannot select default Icon." );
		            }
                    TmpSlot.IconPos = 0;
                    // Load the default sub-dock Icon into the slot's own bitmap
                    UtilLoadIcon( Dock, &TmpSlot );

					// Restore Icon Index to it's original value
					// Currently points to scratch area in icon cache
		            TmpSlot.IconIndex = Slot->IconIndex;

					// Copy modified WorkSpace(now a sub-dock) into Selected Slot
                    memcpy(Slot, &TmpSlot, sizeof(SLOT_ENTRY));

					SetSubDockWinPos( SubDock, FALSE );
					WriteDockOptions( SubDock );

					// Reload the Icon incase it was changed
                    UtilLoadIcon( Dock, Slot );

                    RePaintSlot(Dock, Slot, FALSE);
                    FreeProcInstance(lpfnChooseIconDlgProc);
                    EndDialog(hDlg, TRUE);
					return(TRUE);

			    case QX_START_SEPARATE:
#ifndef WIN32		// This option is not valid in 16 bit windows
                    CheckDlgButton(hDlg, QX_START_SEPARATE, FALSE);
#endif
					return(TRUE);

				case QX_HIDE_ON_GRAB:
                    gOptions.HideDlgOnGrab = IsDlgButtonChecked(hDlg, QX_HIDE_ON_GRAB);
					return(TRUE);

                case QX_OK:
                    if (SendDlgItemMessage(hDlg, QX_ICONFILE, EM_GETMODIFY, 0, 0L)) {
                        SendDlgItemMessage(hDlg, QX_ICONFILE, EM_SETMODIFY, FALSE, 0L);
                        GetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile, MAX_FPATH_LEN);
                        // Load the Icon into the slot's own bitmap & Initialise IconTotal Field
                        TmpSlot.IconPos = 0;
                        UtilLoadIcon( Dock, &TmpSlot );
                        if (TmpSlot.IconTotal == 0) {
                            DockError (hDlg, "This file contains no icons" );
                            TmpSlot.IconPos = Slot->IconPos;
                            strcpy(TmpSlot.IconFile, Slot->IconFile);
                            SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
	                        UtilLoadIcon( Dock, &TmpSlot );
                        } 

                        NumIcons = TmpSlot.IconTotal;

                        SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
                        SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
                        // ReDraw Icon by invalidating the button area
                        hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
                        GetClientRect( hwndItem, &rcItem );
                        InvalidateRect( hwndItem, &rcItem, TRUE );
                        return (TRUE);
                    }

                    /*****************************************
                        Try to fully qualify the pathname 
                    *****************************************/
                    GetDlgItemText(hDlg, QX_FILE, TmpBuffer, MAX_FPATH_LEN);

                    Status = (UINT)FindExecutable(TmpBuffer,
                                                  TmpSlot.RunTimeDir,
                                                  TmpSlot.AppName);
                    if( Status <= 31 ){
                            DockError(hDlg, "Error, File or Path Not Found");
                            return(TRUE);
                    }
                    else{
                        SetDlgItemText(hDlg, QX_FILE, TmpSlot.AppName);
                    }
                    
                    GetDlgItemText(hDlg, QX_FILE, TmpSlot.AppName, MAX_FPATH_LEN);
                    GetDlgItemText(hDlg, QX_CMDLINE, TmpSlot.CmdLine, MAX_CMDLINE_LEN);
                    GetDlgItemText(hDlg, QX_RUNTIMEDIR, TmpSlot.RunTimeDir, MAX_FPATH_LEN);
                    GetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile, MAX_FPATH_LEN);
                    GetDlgItemText(hDlg, QX_TITLE_TEXT, TmpSlot.Title, MAX_FPATH_LEN);

					/*****************************************************************
						If this is an empty slot which is being filled out by hand
					*****************************************************************/
					if(bWasEmpty){
                        TmpSlot.SlotType = SLOT_USED;
                        TmpSlot.StartState = START_NORMAL;
                        TmpSlot.WinX = DEF_STORE_X;
                        TmpSlot.WinY = DEF_STORE_Y;
                        TmpSlot.WinWidth = DEF_STORE_W;
                        TmpSlot.WinHeight = DEF_STORE_H;
					}

                    /*************************************************************************
                        If the user has just entered an application name into an empty slot
                        then mark this slot as being in use.
                    *************************************************************************/
                    if (TmpSlot.AppName[0] != '\0'){
                        TmpSlot.SlotType = SLOT_USED;
                    }

                    if (IsDlgButtonChecked(hDlg, QX_MINIMUM))
                        TmpSlot.StartState = START_MINIMUM;
                    else if (IsDlgButtonChecked(hDlg, QX_MAXIMUM))
                        TmpSlot.StartState = START_MAXIMUM;
                    else if (IsDlgButtonChecked(hDlg, QX_STORE))
                        TmpSlot.StartState = START_STORE;
                    else
                        TmpSlot.StartState = START_NORMAL;

                    if (IsDlgButtonChecked(hDlg, QX_START_TOP))
                        TmpSlot.StartOnTop = TRUE;
                    else
                        TmpSlot.StartOnTop = FALSE;

#ifdef WIN32		// is the Start in Separate memory space option set ?
                    TmpSlot.StartSeparate = IsDlgButtonChecked(hDlg, QX_START_SEPARATE);
#endif
					// Restore Icon Index to it's original value
		            TmpSlot.IconIndex = Slot->IconIndex;

					// Copy modified WorkSpace into Selected Slot
                    memcpy(Slot, &TmpSlot, sizeof(SLOT_ENTRY));

					// Reload the Icon incase it was changed
                    UtilLoadIcon( Dock, Slot );

                    RePaintSlot(Dock, Slot, FALSE);
                    FreeProcInstance(lpfnChooseIconDlgProc);
                  	WriteSlotOptions(Dock, Slot);				// Save the Slot Options
                    EndDialog(hDlg, TRUE);
                    return(TRUE);

                case QX_REMOVE:
					DeleteSlotOptions( Dock, Slot );	// Delete entry in ini file
                	UtilEmptySlot( Slot );
                    RePaintSlot(Dock, Slot, TRUE);
                    EndDialog(hDlg, TRUE);
                    return (TRUE);

                case QX_BROWSEICON :
                    Status = BrowseIconFile( hDlg, TmpSlot.IconFile );
                    if(Status){
                        TmpSlot.IconPos = 0;
                        // Load the Icon into the slot's own bitmap  & update the IconTotal Field
                        UtilLoadIcon( Dock, &TmpSlot );
                        if (TmpSlot.IconTotal == 0) {
                            DockError(hDlg, "This file contains no icons");
                            TmpSlot.IconPos = Slot->IconPos;
                            strcpy(TmpSlot.IconFile, Slot->IconFile);
                            SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
	                        UtilLoadIcon( Dock, &TmpSlot );
                        } 
                        NumIcons = TmpSlot.IconTotal;
                        SetDlgItemInt(hDlg, QX_ICONNUM, (TmpSlot.IconPos+1), TRUE);
                        SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
                        SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
                        // ReDraw Icon by invalidating the button area
                        hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
                        GetClientRect( hwndItem, &rcItem );
                        InvalidateRect( hwndItem, &rcItem, TRUE );
                    }
                    return (TRUE);

                case QX_BROWSEFILE :
                    Status = BrowseFileName( hDlg, TmpSlot.AppName );
                    if(Status){
                        FillOutSlot( Dock, &TmpSlot, TmpSlot.AppName );
                        // Update the Dialog Box
                        NumIcons = TmpSlot.IconTotal;
                        SetDlgItemText(hDlg, QX_FILE, TmpSlot.AppName);
                        SetDlgItemText(hDlg, QX_CMDLINE, TmpSlot.CmdLine);
                        SetDlgItemText(hDlg, QX_RUNTIMEDIR, TmpSlot.RunTimeDir);
                        SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
                        SetDlgItemInt(hDlg, QX_ICONNUM, (TmpSlot.IconPos+1), TRUE);
                        SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
                        SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
                        // ReDraw Icon by invalidating the button area
                        hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
                        GetClientRect( hwndItem, &rcItem );
                        InvalidateRect( hwndItem, &rcItem, TRUE );
                    }
                    return (TRUE);

                case QX_CANCEL:
                    FreeProcInstance(lpfnChooseIconDlgProc);
                    EndDialog(hDlg, FALSE);
                    return (TRUE);

				case QX_PREVIEW :
                    Status = DialogBoxParam(gOptions.hAppInst, "PREVIEWERDLG", hDlg, lpfnPreviewerDlgProc, (LPARAM)&TmpSlot);
					if(Status){
						TmpSlot.StartState = START_STORE;
    	    	        CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_STORE);
					}
					return (TRUE);

                case QX_CHOOSEICON :
                    /*************************************************************
                    	If we are showing a dialog on an empty slot, then don't
                    	allow an icon to be chosen
                    *************************************************************/
                    if ( NumIcons == 0 ) return(TRUE);

                    Status = DialogBoxParam(gOptions.hAppInst, "CHOOSEICON", hDlg, lpfnChooseIconDlgProc, (LPARAM)&TmpSlot);
                    if(Status){
                        UtilLoadIcon( Dock, &TmpSlot );
                        hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
                        GetClientRect( hwndItem, &rcItem );
                        InvalidateRect( hwndItem, &rcItem, TRUE );
                        SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
                    }
                    return (TRUE);
            }
            return (FALSE);
 
  		case WM_LBUTTONDOWN:
                GetWindowRect( hDlg, &rect );
                /** Correct for position of dialogbox       **/
				/** and correct for dialog border & frame **/
                TmpX = (short int)LOWORD(lParam) + (short int)rect.left + gMetrics.DlgFrameWidth;
                TmpY = (short int)HIWORD(lParam) + (short int)rect.top + gMetrics.DlgFrameHeight + gMetrics.DlgTitleHeight;
                /** Get the location of the Grabber Item **/
                hwndTmp = GetDlgItem( hDlg, QX_GRABWIN );
                GetWindowRect( hwndTmp, &rect );
                /** Next see if the button was pressed here **/
                if( (TmpX <= rect.right && TmpX >= rect.left ) &&
                    (TmpY <= rect.bottom && TmpY >= rect.top ) ){
                    /** Then click hit Grabber Item **/
                        SetCapture(hDlg);
                        hGrabberCur = LoadCursor(gOptions.hAppInst, "GrabberCur");
                        SetCursor(hGrabberCur);
                        Grabbing = TRUE;
						hwndFocus = NULL;		// No window under cursor until mouse moves
						GetWindowRect(hDlg, &rect);
						InvalidateRect(hDlg, &rect, FALSE);
						UpdateWindow(hDlg);
						/** Hide the Application Options Dialog window from the grabber **/
						/** If this option is selected **/
						if(gOptions.HideDlgOnGrab){
    	                    SetWindowPos( hDlg, 0, 0,0,0,0,
	                                      SWP_HIDEWINDOW|SWP_NOSIZE|SWP_NOMOVE|SWP_NOZORDER );
						}
  			    }
                return (TRUE);
  
        case WM_LBUTTONUP:
            if (!Grabbing)
                return TRUE;
            /********************************************************
               Button click positions are reported relative to the
               Dialog window, so correct for this by adding the x,y
               coordinates of the top left corner of the dialog to
               the click position. Also correct for the dialog
			   frame and title bar sizes.
            ********************************************************/
//            GetWindowRect(hDlg, &rect);
//            pt.x = rect.left + (short int)LOWORD(lParam) + gMetrics.DlgFrameWidth;
//            pt.y = rect.top + (short int)HIWORD(lParam) + gMetrics.DlgFrameHeight + gMetrics.DlgTitleHeight;

//            hwndFocus = WindowFromPoint(pt);

			// Erase the last focus rect drawn
	        GetWindowRect( hwndFocus, &rect);
			rect.left = 0; // convert screen coords to window relative
			rect.top = 0;
			hdcFocus = GetWindowDC( hwndFocus );
			DrawFocusRect( hdcFocus, &rect );
			ReleaseDC( hwndFocus, hdcFocus );

			/** Show the the dialog window again if it was hidden **/
			if(gOptions.HideDlgOnGrab){
	            SetWindowPos( hDlg, 0, 0,0,0,0,
    	                      SWP_SHOWWINDOW|SWP_NOSIZE|SWP_NOMOVE|SWP_NOZORDER );
			}

            /* Climb back up to the top of a parent-child list */
            hwndTmp = GetParent(hwndFocus);
            while (hwndTmp != NULL) {
                hwndFocus = hwndTmp;
                hwndTmp = GetParent(hwndFocus);
            }

            GetWindowRect(hwndFocus, &rect);
            TmpSlot.WinX = rect.left;
            TmpSlot.WinY = rect.top;
            TmpSlot.WinWidth = rect.right - rect.left;
            TmpSlot.WinHeight = rect.bottom - rect.top;

            if( hwndFocus == GetDesktopWindow() ){
                DockInfo(hDlg, "Oops, that's the Desktop window, please try again.");
            }
            else if( IsDockWindow( hwndFocus ) ){
                DockInfo(hDlg, "Oops, that's a FreeDock window, please try again.");
            }
            else{
                DockInfo(hDlg, "Window Grabbed Successfully.");
				TmpSlot.StartState = START_STORE;
                CheckRadioButton(hDlg, QX_NORMAL, QX_STORE, QX_STORE);
            }

            Grabbing = FALSE;
            ReleaseCapture();
            SetCursor(LoadCursor(NULL, IDC_ARROW));
            DestroyCursor(hGrabberCur);

			GetWindowRect(hDlg, &rect);
			InvalidateRect(hDlg, &rect, FALSE);
			UpdateWindow(hDlg);

            return (TRUE);
    }

    return FALSE;
}


BOOL FAR PASCAL	SubDockOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
	/*********************************************************
		Static (local) slot structure
	*********************************************************/
	static SLOT_ENTRY TmpSlot;
	static DOCK_STRUCT  *Dock;
	static SLOT_ENTRY   *Slot;
           HWND     hwndItem;
           RECT     rcItem;
    static int      NumIcons;
    RECT            rect;
    static int OrigSlotCount;
    static FARPROC  lpfnChooseIconDlgProc;
    DRAWITEMSTRUCT *ItemStruct;
    int             Status;

    switch (iMessage) {

        case WM_INITDIALOG:
			Slot = (SLOT_ENTRY *)lParam;
			Dock = GetDockStruct(Slot->hwndSubDock);			 // Retrieve pointer to sub-dock

			// Take a working copy of the Slot data struct
            memcpy(&TmpSlot, Slot, sizeof(SLOT_ENTRY));

            OrigSlotCount = Dock->SlotCount;
            lpfnChooseIconDlgProc = MakeProcInstance((FARPROC) ChooseIconDlgProc, gOptions.hAppInst);
            // Initialise TmpSlots Scratch Space
            TmpSlot.IconIndex = (gMetrics.IconWidth * (gOptions.MaxDockSize));
            NumIcons = TmpSlot.IconTotal;
            // Load the current Iocn into the Scratch Space
            UtilLoadIcon( Slot->Dock, &TmpSlot );

            SendDlgItemMessage(hDlg, QX_ICONFILE, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);
            SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
            SetDlgItemInt(hDlg, QX_SLOT_COUNT, Dock->SlotCount, TRUE);
            SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
            SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
            SendDlgItemMessage(hDlg, QX_ICONFILE, EM_SETMODIFY, FALSE, 0L);
            SetDlgItemText(hDlg, QX_DOCK_ID, Dock->DockID);
            SendDlgItemMessage(hDlg, QX_TITLE_TEXT, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);
            SetDlgItemText(hDlg, QX_TITLE_TEXT, TmpSlot.Title);

			SetSubDockWinPos( Dock, TRUE);		// Show the sub-dock

		return (TRUE);

        case WM_DRAWITEM :  // Draw the Current Icon
            ItemStruct = (DRAWITEMSTRUCT *)lParam;
            UtilDrawIcon( ItemStruct->hDC, Slot->Dock, &TmpSlot, 0, 0, FALSE );
            return (TRUE);

        case WM_VSCROLL:		// Slot Count Scroll Bar
            if (wParam == SB_LINEUP){
	            if ( Dock->SlotCount < gOptions.MaxDockSize){
 		            Dock->SlotCount++;
       		        SetDlgItemInt(hDlg, QX_SLOT_COUNT, Dock->SlotCount, TRUE);
           		}
			}
	
   	        if (wParam == SB_LINEDOWN){
    	        if ( Dock->SlotCount > 1){
   	    	        Dock->SlotCount--;
       	    	    SetDlgItemInt(hDlg, QX_SLOT_COUNT, Dock->SlotCount, TRUE);
           	    }
			}
			SetSubDockWinPos( Dock, TRUE);
   	    return (TRUE);


        case WM_HSCROLL:
            if (wParam == SB_LINEDOWN)
                if (NumIcons > 0)
                    TmpSlot.IconPos = min(NumIcons - 1, TmpSlot.IconPos + 1);

            if (wParam == SB_LINEUP)
                if (NumIcons > 0)
                    if (TmpSlot.IconPos > 0)
                        TmpSlot.IconPos = TmpSlot.IconPos - 1;

            // Update icon number
            SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
            // Load the Icon into the slot's own bitmap  & update the IconTotal Field
            UtilLoadIcon( Slot->Dock, &TmpSlot );
            // ReDraw Icon by invalidating the button area
            hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
            GetClientRect( hwndItem, &rcItem );
            InvalidateRect( hwndItem, &rcItem, TRUE );

            return (TRUE);

        case WM_COMMAND:
            switch (wParam) {
                case QX_OK:
                    if (SendDlgItemMessage(hDlg, QX_ICONFILE, EM_GETMODIFY, 0, 0L)) {
                        SendDlgItemMessage(hDlg, QX_ICONFILE, EM_SETMODIFY, FALSE, 0L);
                        GetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile, MAX_FPATH_LEN);
                        // Load the Icon into the slot's own bitmap & Initialise IconTotal Field
                        TmpSlot.IconPos = 0;
                        UtilLoadIcon( Slot->Dock, &TmpSlot );
                        if (TmpSlot.IconTotal == 0) {
                            DockError(hDlg, "This file contains no icons");
                            TmpSlot.IconPos = Slot->IconPos;
                            strcpy(TmpSlot.IconFile, Slot->IconFile);
                            SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
	                        UtilLoadIcon( Slot->Dock, &TmpSlot );
                        } 

                        NumIcons = TmpSlot.IconTotal;

                        SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
                        SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
                        // ReDraw Icon by invalidating the button area
                        hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
                        GetClientRect( hwndItem, &rcItem );
                        InvalidateRect( hwndItem, &rcItem, TRUE );
                        return (TRUE);
                    }

                    // Save Title
                    GetDlgItemText(hDlg, QX_TITLE_TEXT, TmpSlot.Title, MAX_FPATH_LEN);

					// Restore Icon Index to it's original value
		            TmpSlot.IconIndex = Slot->IconIndex;

					// Copy modified WorkSpace into Selected Slot
                    memcpy(Slot, &TmpSlot, sizeof(SLOT_ENTRY));

					// Reload the Icon incase it was changed
                    UtilLoadIcon( Slot->Dock, Slot );

                    RePaintSlot(Slot->Dock, Slot, FALSE);
                    FreeProcInstance(lpfnChooseIconDlgProc);
					SetSubDockWinPos( Dock, FALSE);		// Hide the sub-dock
					WriteDockOptions( Dock );
					WriteSlotOptions( Slot->Dock, Slot );
                    EndDialog(hDlg, TRUE);
                    return(TRUE);

                case QX_CANCEL:
                    Dock->SlotCount = OrigSlotCount ;
					SetSubDockWinPos( Dock, TRUE ); // restore original position
            		GetClientRect( Dock->hwndDock, &rect );
           		    InvalidateRect( Dock->hwndDock , &rect, TRUE );
                    FreeProcInstance(lpfnChooseIconDlgProc);
					SetSubDockWinPos( Dock, FALSE);		// Hide the sub-dock
                    EndDialog(hDlg, FALSE);
                    break;

                case QX_REMOVE:
					DestroyDock( Slot->hwndSubDock );
                	UtilEmptySlot( Slot );
                    RePaintSlot( Slot->Dock, Slot, TRUE);
                    EndDialog(hDlg, TRUE);
                    return (TRUE);

                case QX_CHOOSEICON :
                    /*************************************************************
                    	If we are showing a dialog on an empty slot, then don't
                    	allow an icon to be chosen
                    *************************************************************/
                    if ( NumIcons == 0 ) return(TRUE);

                    Status = DialogBoxParam(gOptions.hAppInst, "CHOOSEICON", hDlg, lpfnChooseIconDlgProc, (LPARAM)&TmpSlot);
                    if(Status){
                        UtilLoadIcon( Slot->Dock, &TmpSlot );
                        hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
                        GetClientRect( hwndItem, &rcItem );
                        InvalidateRect( hwndItem, &rcItem, TRUE );
                        SetDlgItemInt(hDlg, QX_ICONNUM, TmpSlot.IconPos + 1, TRUE);
                    }
                    return (TRUE);

                case QX_BROWSEICON :
                    Status = BrowseIconFile( hDlg, TmpSlot.IconFile );
                    if(Status){
                        TmpSlot.IconPos = 0;
                        // Load the Icon into the slot's own bitmap  & update the IconTotal Field
                        UtilLoadIcon( Slot->Dock, &TmpSlot );
                        if (TmpSlot.IconTotal == 0) {
                            DockError(hDlg, "This file contains no icons");
                            TmpSlot.IconPos = Slot->IconPos;
                            strcpy(TmpSlot.IconFile, Slot->IconFile);
                            SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
	                        UtilLoadIcon( Slot->Dock, &TmpSlot );
                        } 
                        NumIcons = TmpSlot.IconTotal;
                        SetDlgItemInt(hDlg, QX_ICONNUM, (TmpSlot.IconPos+1), TRUE);
                        SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
                        SetDlgItemText(hDlg, QX_ICONFILE, TmpSlot.IconFile);
                        // ReDraw Icon by invalidating the button area
                        hwndItem = GetDlgItem( hDlg, QX_CHOOSEICON );
                        GetClientRect( hwndItem, &rcItem );
                        InvalidateRect( hwndItem, &rcItem, TRUE );
                    }
                    return (TRUE);
    	        }
	}

    return (FALSE);
}


BOOL FAR PASCAL MainOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
	static DOCK_STRUCT  *Dock;
    static int 		   OrigPosn;
    static int 		   OrigSlotCount, OrigSlotScale;
	extern FARPROC     lpfnAboutDockDlgProc;	// Defined in dockwin.c
	RECT			   rect;

    switch (iMessage) {

        case WM_INITDIALOG:
			Dock = (DOCK_STRUCT *)lParam;

            OrigPosn = GetDockOrientation( Dock->hwndDock ) + QX_HORZ;  // Save original position incase user cancels
            OrigSlotCount = Dock->SlotCount;
            OrigSlotScale = gOptions.SlotScale;

            CheckDlgButton(hDlg, QX_POPINONEXEC, gOptions.PopInOnExec);
            CheckDlgButton(hDlg, QX_DOCK_LOCKED, gOptions.DockLocked);
            CheckDlgButton(hDlg, QX_MAILCHECK, gOptions.MailActive);
            CheckDlgButton(hDlg, QX_WINEXIT, gOptions.WinExitActive);
            CheckDlgButton(hDlg, QX_CLOCK, gOptions.ClockActive);
            CheckDlgButton(hDlg, QX_ON_TOP, gOptions.AlwaysOnTop);
            CheckDlgButton(hDlg, QX_MAX_VIEW, gOptions.MaxView);
            CheckDlgButton(hDlg, QX_SINGLECLICK, gOptions.SingleClickStart);
            CheckDlgButton(hDlg, QX_TITLES, gOptions.TitlesActive);
            CheckRadioButton(hDlg, QX_HORZ, QX_VERT, OrigPosn);
            SetDlgItemInt(hDlg, QX_SLOT_COUNT, Dock->SlotCount, TRUE);
            SetDlgItemInt(hDlg, QX_SLOT_SCALE, (int)(((float)gOptions.SlotScale/38.0)*100), TRUE);
            SetDlgItemText(hDlg, QX_DOCK_ID, Dock->DockID);
            return (TRUE);

        case WM_HSCROLL:		// Scale Scroll Bar
            if (wParam == SB_LINEDOWN)
	            if ( gOptions.SlotScale < MAX_SLOT_SCALE){
     		         gOptions.SlotScale++;
            		 SetDlgItemInt(hDlg, QX_SLOT_SCALE, (int)(((float)gOptions.SlotScale/38.0)*100), TRUE);
           		}
	
   	        if (wParam == SB_LINEUP)
     	        if ( gOptions.SlotScale > MIN_SLOT_SCALE){
   	    	        gOptions.SlotScale--;
       	    	    SetDlgItemInt(hDlg, QX_SLOT_SCALE, (int)(((float)gOptions.SlotScale/38.0)*100), TRUE);
           	    }
			gOptions.SlotWidth = gOptions.SlotHeight = gOptions.SlotScale;

			SetRootDockPos( Dock, GetDockOrientation( Dock->hwndDock ) + QX_HORZ );
            GetClientRect( Dock->hwndDock, &rect );
            InvalidateRect( Dock->hwndDock , &rect, TRUE );
    	    return (TRUE);

        case WM_VSCROLL:		// Slot Count Scroll Bar
            if (wParam == SB_LINEUP)
	            if ( Dock->SlotCount < gOptions.MaxDockSize){
 		            Dock->SlotCount++;
       		        SetDlgItemInt(hDlg, QX_SLOT_COUNT, Dock->SlotCount, TRUE);
           		}
	
   	        if (wParam == SB_LINEDOWN)
    	        if ( Dock->SlotCount > 1){
   	    	        Dock->SlotCount--;
       	    	    SetDlgItemInt(hDlg, QX_SLOT_COUNT, Dock->SlotCount, TRUE);
           	    }
			SetRootDockPos( Dock, GetDockOrientation( Dock->hwndDock ) + QX_HORZ );
   	    return (TRUE);


        case WM_COMMAND:
            switch (wParam) {
                case QX_OK:
                    gOptions.MailActive = IsDlgButtonChecked(hDlg, QX_MAILCHECK);
                    gOptions.WinExitActive = IsDlgButtonChecked(hDlg, QX_WINEXIT);
                    gOptions.ClockActive = IsDlgButtonChecked(hDlg, QX_CLOCK);
                    gOptions.AlwaysOnTop = IsDlgButtonChecked(hDlg, QX_ON_TOP);
                    gOptions.MaxView = IsDlgButtonChecked(hDlg, QX_MAX_VIEW);
                    gOptions.SingleClickStart = IsDlgButtonChecked(hDlg, QX_SINGLECLICK);
                    gOptions.DockLocked = IsDlgButtonChecked(hDlg, QX_DOCK_LOCKED);
                    gOptions.PopInOnExec = IsDlgButtonChecked(hDlg, QX_POPINONEXEC);
                    gOptions.TitlesActive = IsDlgButtonChecked(hDlg, QX_TITLES);
                    if( gOptions.TitlesActive ){
                        EnableTitles();
                    }
                    else{
                        DisableTitles();
                    }

                    /**************************************************************
                        Handle Always On Top Option
                    **************************************************************/
                    if (gOptions.AlwaysOnTop) {
                        SetWindowPos(Dock->hwndDock, HWND_TOPMOST, 0, 0, 0, 0,
                            SWP_NOMOVE | SWP_NOSIZE | SWP_NOACTIVATE);
                    } 
                    else {
                        SetWindowPos(Dock->hwndDock, HWND_BOTTOM, 0, 0, 0, 0,
                            SWP_NOMOVE | SWP_NOSIZE | SWP_NOACTIVATE);
                    }

//                    // Handle the MaxView Hook.
//                   if( gOptions.MaxView ){
//                        InstallHook();
//                    }
//                    else{
//                        RemoveHook();
//                    }

                   	WriteGlobalOptions();
					WriteDockOptions ( Dock );
                    EndDialog(hDlg, TRUE);
                    break;

                    /****************************************************************
                        Handle Dock Position Options, for updating dock pos as the
                        user choses the radio button.
                    ****************************************************************/
                case QX_HORZ:
                case QX_VERT:
                    SetRootDockPos( Dock, wParam );
                    break;

                case QX_CANCEL:
                    Dock->SlotCount = OrigSlotCount ;
                    gOptions.SlotScale = OrigSlotScale ;
					gOptions.SlotWidth = gOptions.SlotHeight = gOptions.SlotScale;
					SetRootDockPos( Dock, OrigPosn ); // restore original position
            		GetClientRect( Dock->hwndDock, &rect );
           		    InvalidateRect( Dock->hwndDock , &rect, TRUE );
                    EndDialog(hDlg, FALSE);
                    break;

				case QX_ABOUT:
                    DialogBoxParam( gOptions.hAppInst, "ABOUTDOCKDLG", hDlg, lpfnAboutDockDlgProc, 
                                    (LPARAM)Dock->hwndDock);
					break;
            }
            break;
    }

    return FALSE;
}


BOOL FAR PASCAL WinExitOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{

    switch (iMessage) {
        case WM_INITDIALOG:
#ifdef WIN32
             CheckDlgButton(hDlg, QX_LOGOFF,  (gOptions.ShutdownAction == FD_LOGOFF)   );
             CheckDlgButton(hDlg, QX_SHUTDOWN,(gOptions.ShutdownAction == FD_SHUTDOWN) );
             CheckDlgButton(hDlg, QX_REBOOT,  (gOptions.ShutdownAction == FD_REBOOT)   );
             CheckDlgButton(hDlg, QX_LOCKUP,  (gOptions.ShutdownAction == FD_LOCKUP)   );
#else
             // Disable "WIN 32 only" Options
             EnableWindow( GetDlgItem(hDlg, QX_LOGOFF  ), FALSE );
             EnableWindow( GetDlgItem(hDlg, QX_SHUTDOWN), FALSE );
             EnableWindow( GetDlgItem(hDlg, QX_REBOOT  ), FALSE );
             EnableWindow( GetDlgItem(hDlg, QX_LOCKUP  ), FALSE );
#endif
             CheckDlgButton(hDlg, QX_WINEXITCONFIRM, gOptions.WinExitConfirm);
             return (TRUE);

        case WM_COMMAND:
            switch (wParam) {
                case QX_OK:
                    gOptions.WinExitConfirm = IsDlgButtonChecked(hDlg, QX_WINEXITCONFIRM);
                    if( IsDlgButtonChecked( hDlg, QX_LOGOFF ) ){
                        gOptions.ShutdownAction = FD_LOGOFF;
                    }
                    else if( IsDlgButtonChecked( hDlg, QX_SHUTDOWN ) ){
                        gOptions.ShutdownAction = FD_SHUTDOWN;
                    }
                    else if( IsDlgButtonChecked( hDlg, QX_REBOOT ) ){
                        gOptions.ShutdownAction = FD_REBOOT;
                    }
                    else{
                        gOptions.ShutdownAction = FD_LOCKUP;
                    }
                    WriteGlobalOptions();
                    EndDialog(hDlg, TRUE);
                    break;

                case QX_CANCEL:
                    EndDialog(hDlg, FALSE);
                    break;
            }
            break;
    }
    return FALSE;
}


BOOL FAR PASCAL MailOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
    BOOL   				bTxlated;
	static DOCK_STRUCT *Dock;

    switch (iMessage) {
        case WM_INITDIALOG:
		    Dock = (DOCK_STRUCT *)lParam;
            SendDlgItemMessage(hDlg, QX_MAILPATH, EM_LIMITTEXT, MAX_FPATH_LEN, 0L);
            SetDlgItemText(hDlg, QX_MAILPATH, gOptions.MailPath);
            SetDlgItemInt(hDlg, QX_MAILFREQ, gOptions.MailFreq, TRUE);
            CheckDlgButton(hDlg, QX_MAILSOUND, gOptions.MailSound);
            return (TRUE);

        case WM_COMMAND:
            switch (wParam) {
                case QX_OK:
                    GetDlgItemText(hDlg, QX_MAILPATH, gOptions.MailPath, MAX_FPATH_LEN);
                    gOptions.MailFreq = GetDlgItemInt(hDlg, QX_MAILFREQ, &bTxlated, FALSE);
                    if (!bTxlated) {
                        DockError(hDlg, "Not a valid integer!");
                        SendDlgItemMessage(hDlg, QX_MAILFREQ, EM_SETSEL, 0, MAKELONG(0, -1));
                        return (TRUE);
                    }
                    gOptions.MailSound = IsDlgButtonChecked(hDlg, QX_MAILSOUND);
                    KillTimer(Dock->hwndDock, MAIL_TIMER);
                    SetTimer(Dock->hwndDock, MAIL_TIMER, gOptions.MailFreq * 1000 * 60, NULL);
                    WriteGlobalOptions();
                    EndDialog(hDlg, TRUE);
                    break;

                case QX_CANCEL:
                    EndDialog(hDlg, FALSE);
                    break;
            }
            break;
    }
    return FALSE;
}


BOOL FAR PASCAL ClockOptionsDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
    switch (iMessage) {
        case WM_INITDIALOG:
            return (TRUE);

        case WM_COMMAND:
            if (wParam == QX_OK)
                WriteGlobalOptions();
                EndDialog(hDlg, TRUE);
            break;
    }
    return FALSE;
}


BOOL FAR PASCAL AboutDockDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
	static HICON hIcon;
    static HWND  hwndRoot;

    switch (iMessage) {
        case WM_INITDIALOG:
		    hwndRoot = (HWND)lParam;
            SetDlgItemText(hDlg, QX_VERSION, VersionText);
            SetDlgItemText(hDlg, QX_OS_BITS, OS_Bits);
#ifdef WIN32
			hIcon = LoadIcon(gOptions.hAppInst, "FREEDOCK32");
#else
			hIcon = LoadIcon(gOptions.hAppInst, "FREEDOCK16");
#endif
            SendDlgItemMessage(hDlg, QX_ICON1, STM_SETICON, (WPARAM)hIcon, 0L);
			{
				char Instance[6];	// 1000000 instances of Freedock is a bit unlikely :-)
				sprintf ( Instance, "%d", gOptions.RootDockID );
            	SetDlgItemText(hDlg, QX_INSTANCE_ID, Instance);
			}
            return (TRUE);

        case WM_COMMAND:
            switch (wParam) {

                case QX_OK:
                    EndDialog(hDlg, TRUE);
                    break;

				case QX_EXIT:
                    //            nItem = MessageBox(hDlg, "Do you really want to exit FreeDock ?",
                    //                               "FreeDock", MB_OKCANCEL | MB_ICONSTOP);

                    //            if (nItem == IDOK) {
                    WriteGlobalOptions( );
                    DestroyWindow( hwndRoot );
                    //            }
                    return (TRUE);
            }
    }
    return FALSE;
}


BOOL FAR PASCAL BrowseIconFile( HWND hwnd, char *FileName )
{
    OPENFILENAME ofn;
    char DirName[256], FileTitle[256], TmpFileName[MAX_FPATH_LEN];
    char Filter[] = "Executables (*.EXE)\0*.EXE\0Icon Files (*.ICO)\0*.ICO\0DLL's (*.DLL)\0*.DLL\0\0";
    BOOL Status;
    char *TmpPtr;

    memset( &ofn, 0, sizeof(OPENFILENAME) );

    strcpy( TmpFileName, FileName );
    strcpy( DirName, FileName );
    TmpPtr = strrchr(DirName, '\\');
    if(TmpPtr != NULL){
        *TmpPtr = '\0';
    }

    ofn.lpstrTitle   = "Select an Icon File";
    ofn.lStructSize  = sizeof(OPENFILENAME);
    ofn.hwndOwner    = hwnd;
    ofn.lpstrFilter  = Filter;
    ofn.nFilterIndex = 1;
    ofn.lpstrFile = TmpFileName;
    ofn.nMaxFile = MAX_FPATH_LEN;
    ofn.lpstrFileTitle = FileTitle;
    ofn.nMaxFileTitle = sizeof(FileTitle);
    ofn.lpstrInitialDir = DirName;
    ofn.Flags = OFN_HIDEREADONLY | OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

    Status = GetOpenFileName(&ofn);

    // If no change made, this is the same as cancel
    if(!stricmp( FileName, TmpFileName ))
        return(FALSE);

    if(Status){
        strcpy( FileName, TmpFileName );
    }

    return(Status);
}


BOOL FAR PASCAL BrowseFileName( HWND hwnd, char *FileName )
{
    OPENFILENAME ofn;
    char DirName[256], FileTitle[256], TmpFileName[MAX_FPATH_LEN];
    char Filter[] = "Executables (*.EXE)\0*.EXE\0Batch Files (*.BAT)\0*.BAT\0PIF Files (*.PIF)\0*.PIF\0DOS .COM Files (*.COM)\0*.COM\0\0";
    BOOL Status;
    char *TmpPtr;

    memset( &ofn, 0, sizeof(OPENFILENAME) );

    strcpy( TmpFileName, FileName );
    strcpy( DirName, FileName );
    TmpPtr = strrchr(DirName, '\\');
    if(TmpPtr != NULL){
        *TmpPtr = '\0';
    }

    ofn.lpstrTitle   = "Select a File";
    ofn.lStructSize  = sizeof(OPENFILENAME);
    ofn.hwndOwner    = hwnd;
    ofn.lpstrFilter  = Filter;
    ofn.nFilterIndex = 1;
    ofn.lpstrFile = TmpFileName;
    ofn.nMaxFile = MAX_FPATH_LEN;
    ofn.lpstrFileTitle = FileTitle;
    ofn.nMaxFileTitle = sizeof(FileTitle);
    ofn.lpstrInitialDir = DirName;
    ofn.Flags = OFN_HIDEREADONLY | OFN_PATHMUSTEXIST | OFN_FILEMUSTEXIST;

    Status = GetOpenFileName(&ofn);

    // If no change made, this is the same as cancel
    if(!stricmp( FileName, TmpFileName ))
        return(FALSE);

    if(Status){
        strcpy( FileName, TmpFileName );
    }

    return(Status);
}


BOOL FAR PASCAL ChooseIconDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{
    // Slot info passed in via global TmpSlot struct
    // CurIcon and CurPage are 0 relative
    static int NumIcons, CurIcon, NumPages, CurPage;
    static HANDLE *Icon;            // pointer to array of icon handles
	static SLOT_ENTRY *Slot;		// Pointer to Slot
	static DOCK_STRUCT *Dock;		// Pointer to parent dock
    RECT rect;
    DRAWITEMSTRUCT *ItemStruct;
    int i, ItemNum;

    switch (iMessage) {
        case WM_INITDIALOG:

		    // Recover the pointer to the Slot from lParam
			Slot = (SLOT_ENTRY *)lParam;

            // find number of icons & create array to hold ALL handles
            NumIcons = Slot->IconTotal;
            Icon = (HANDLE *)malloc( NumIcons * sizeof(HANDLE) );

            // Read all Icon handles
            for( i=0; i< NumIcons ; i++){
                Icon[i] = ExtractIcon(gOptions.hAppInst, Slot->IconFile, i);
            }

            CurIcon = Slot->IconPos;
            NumPages = (NumIcons / 50) + 1;
            CurPage = (CurPage / 50);

            SetDlgItemInt(hDlg, QX_ICONNUM, (CurIcon+1), TRUE);
            SetDlgItemInt(hDlg, QX_ICONTOTAL, NumIcons, TRUE);
            SetDlgItemInt(hDlg, QX_PAGENUM, (CurPage+1), TRUE);
            SetDlgItemInt(hDlg, QX_PAGETOTAL, NumPages, TRUE);
            SendDlgItemMessage(hDlg, QX_ICONBOX, STM_SETICON, (WPARAM)Icon[CurIcon], 0L);
            return (TRUE);

        case WM_DRAWITEM :
            ItemStruct = (DRAWITEMSTRUCT *)lParam;
            ItemNum = (ItemStruct->CtlID-QX_ICON1) + (CurPage*50);
            if(ItemNum < NumIcons){
                DrawIcon( ItemStruct->hDC, 0, 0, Icon[ItemNum]);
            }
            return (TRUE);

        case WM_HSCROLL:
            if (wParam == SB_LINEDOWN)
                if (CurPage < (NumPages-1)){
                    CurPage = CurPage + 1;
                    GetClientRect( hDlg, &rect );
                    for(i = QX_ICON1; i < QX_ICON1+50; i++){
                        InvalidateRect( GetDlgItem( hDlg, i) , &rect, TRUE );
                    }
                }

            if (wParam == SB_LINEUP)
                if (CurPage > 0){
                    CurPage = CurPage - 1;
                    GetClientRect( hDlg, &rect );
                    for(i = QX_ICON1; i < QX_ICON1+50; i++){
                        InvalidateRect( GetDlgItem( hDlg, i) , &rect, TRUE );
                    }
                }

            SetDlgItemInt(hDlg, QX_PAGENUM, CurPage + 1, TRUE);

            return (TRUE);

        case WM_COMMAND:
            if (wParam == QX_OK){
                Slot->IconPos = CurIcon;
                free( Icon );
                EndDialog(hDlg, TRUE);
            }
            else if (wParam == QX_CANCEL){
                free( Icon );
                EndDialog(hDlg, FALSE);
            } 
            else if( (wParam >= QX_ICON1) && (wParam <= QX_ICON1+50) ){
                CurIcon = min( (int)(NumIcons-1), (int)((wParam - QX_ICON1) + (CurPage*50)));
                SetDlgItemInt(hDlg, QX_ICONNUM, (CurIcon+1), TRUE);
                SendDlgItemMessage(hDlg, QX_ICONBOX, STM_SETICON, (WPARAM)Icon[CurIcon], 0L);
            }
            break;
    }
    return FALSE;
}


BOOL FAR PASCAL PreviewerDlgProc(HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM lParam)
{

	static RECT rect;
	static SLOT_ENTRY *Slot;

    switch (iMessage) {
        case WM_INITDIALOG:

		    Slot = (SLOT_ENTRY *)lParam;

			SetWindowPos(hDlg, 0L, 
						 Slot->WinX,
						 Slot->WinY,
						 Slot->WinWidth,
						 Slot->WinHeight,
						 SWP_SHOWWINDOW | SWP_NOZORDER);
            return (TRUE);

        case WM_COMMAND:
            switch (wParam) {
                case QX_OK:
		            GetWindowRect(hDlg, &rect);
    	        	Slot->WinX = rect.left;
        		    Slot->WinY = rect.top;
        	    	Slot->WinWidth = rect.right - rect.left;
		            Slot->WinHeight = rect.bottom - rect.top;
                    EndDialog(hDlg, TRUE);
                    break;

                case QX_CANCEL:
                    EndDialog(hDlg, FALSE);
                    break;
            }
            break;
    }
    return FALSE;
}

