/****************************************************************************
    Module Created for 3.0 beta 1
    =============================
    The Dock registry module, this module is responsible for maintaining
    a register of all dock and sub-dock windows, including parents, children
    and the current state of each window (shown / Hidden) also maps the
    window handle to the memory area allocated to store the docks information.

    3.0 beta 2
    ----------
    Changed window caption to reflect Dock ID, except for special case :
    Root Dock.
    Added GetRootDockID call to support multiple instances of FreeDock.
    (WIN32)

    3.0 beta 5
    ----------
    Change to fix error in DeRegisterDock, code to handle non-last window
    in list was incorrect
    Moved error dialog box out of IsDockRegistered function.

    3.0 beta 7
    ----------
    Fixed error in DeRegister Dock which resulted in the memory allocated for
    the DockID field not being freed
*****************************************************************************/

#include <windows.h>
#include <stdio.h>        // for sprintf
#include <stdlib.h>       // for atoi
#include <ctype.h>
#include <string.h>
#include "freedock.h"
#include "dock_reg.h"

extern GLOBAL_OPTIONS gOptions;

typedef struct _DOCK_REG{
    HWND         hwndDock;
    HWND         hwndParent;
    DOCK_STRUCT *Dock;
    BOOL         PoppedOut;
    char        *DockID;
} DOCK_REG;

/***************************************
    Define local registry variables
***************************************/
static DOCK_REG gDockReg[MAX_SUB_DOCKS+1];
static int      gRegIndex = 0;

HWND CreateDock( DOCK_TYPE DockType, HWND hwndDockParent, char *DockID ){

    HWND         hwndDock;
    DOCK_STRUCT *Dock;
    char         WindowCaption[MAX_FPATH_LEN];

    if( IsDockRegistered (DockID) ){
        char MessageText[MAX_FPATH_LEN];
        sprintf( MessageText,
                 "Error, attempt to register already registered Dock : ID = %s",
                 DockID );
        DockError( NULL, MessageText, "FreeDock Error");
        exit(-4);
    }
    /********************************************
        malloc memory for the Dock Structure
    ********************************************/
        Dock = (DOCK_STRUCT *)calloc(1, sizeof(DOCK_STRUCT));
        if(Dock == NULL){
            DockError( NULL, "Cannot allocate memory for dock "); 
            exit(-3);
    }

    /******************************************************************
         malloc the memory for the maximum number of slots
         No need to initialise all slot types to being free 
         since SLOT_FREE is 0 and we used calloc to get the memory
    *******************************************************************/
    Dock->Slot = (SLOT_ENTRY *)calloc( gOptions.MaxDockSize,
                                       sizeof(SLOT_ENTRY));
    if(Dock->Slot == NULL){
        DockError( NULL, "Cannot allocate memory for dock slots"); 
        exit(-3);
    }

    Dock->DockType   = DockType;
    strncpy( Dock->DockID, DockID, MAX_DOCK_ID_LEN );

    /***************************
        Read Dock Options
    ***************************/
    ReadDockOptions( Dock );

    /***********************************************************************
        Set up window caption, dock windows are named with their Dock ID's
    ***********************************************************************/
    if(gOptions.FixedTitle){    // Create a fixed window title, same for all instances
        sprintf( WindowCaption, "%s", gOptions.szAppName );
    }
    else{ // Create different window title for each instance
        sprintf( WindowCaption, "%s [%s]", gOptions.szAppName, Dock->DockID );
    }

    hwndDock = CreateWindow(
                            gOptions.szAppName,   /* window class name    */
                            WindowCaption,        /* window caption       */
                            WS_POPUP | WS_BORDER,  /* style  */
                            0,                    /* initial x position   */
                            0,                    /* initial y position   */
                            1,                    /* initial x size       */
                            1,                    /* initial y size       */
                            hwndDockParent,       /* parent window handle */
                            NULL,                 /* window menu handle   */
                            gOptions.hAppInst,    /* program inst handle  */
                            (LPSTR)Dock           /* creation parameters  */
                            );

   if( hwndDock == NULL ){    
        if ( DockType == ROOT_DOCK ){
             DockError( NULL, "Error, creating Root Dock Window" ); 
        }
        else{
             DockError( NULL, "Error, creating Sub-Dock Window"); 
        }
        exit(-4);
    }

    return (hwndDock);
}

/*************************************************
    Destroys a sub dock and all it's children
*************************************************/
BOOL DestroyDock( HWND hwnd ){
    DOCK_STRUCT *Dock;
    int             i;

    Dock = GetDockStruct( hwnd );
    if ( Dock == NULL ){
        DockError( NULL, "Error, attempt to destroy non-registered dock"); 
        return( FALSE );
    }

    // Find each child
    for( i = 0; i < gRegIndex; i++ ){
         if ( gDockReg[i].hwndParent == hwnd ){
            // Found a child of a child
            DestroyDock (gDockReg[i].hwndDock);
            free( gDockReg[i].DockID );
        }
    }
    // Delete Parent Slot
    // Delete entry in ini file
    DeleteSlotOptions( Dock->ParentSlot->Dock, Dock->ParentSlot );
    // Ensure Deleted Parent Slot is marked FREE
    UtilEmptySlot( Dock->ParentSlot );

    SetSubDockWinPos( Dock, FALSE );

    free(Dock->Slot);
    free(Dock);

    DeRegisterDock( hwnd );

    return (TRUE);
}


/**********************************************************
    Returns a pointer to a windows dock structure
**********************************************************/
DOCK_STRUCT * GetDockStruct( HWND hwnd ){

    int i;

    for( i = 0; i < gRegIndex; i++ ){
         if ( gDockReg[i].hwndDock == hwnd ){
            return gDockReg[i].Dock;
        }
    }
    return ((DOCK_STRUCT *)NULL);
}

HWND  GetParentDock  ( HWND hwnd ){
    int i;

    for( i = 0; i < gRegIndex; i++ ){
         if ( gDockReg[i].hwndDock == hwnd ){
            return gDockReg[i].hwndParent;
        }
    }
    return ((HWND)NULL);
}

BOOL  IsDockWindow  ( HWND hwnd ){
    int i;

    for( i = 0; i < gRegIndex; i++ ){
         if ( gDockReg[i].hwndDock == hwnd ){
            return (TRUE);
        }
    }
    return (FALSE);
}

void PopInChildren ( HWND hwnd ){

    int i;
    
    if( !IsDockWindow(hwnd) ){
           DockError( NULL, "Error, attempt to popdown non-owned window" ); 
        return;
    }

    // Find each child
    for( i = 0; i < gRegIndex; i++ ){
         if ( gDockReg[i].hwndParent == hwnd ){
            // Found a child of a child
            PopInChildren (gDockReg[i].hwndDock);
        }
    }

    // Check we are not being called by the RootDock, if so, don't close it
    if( (GetParentDock( hwnd ) != NULL) && (IsDockPoppedOut( hwnd )) ){
        SetSubDockWinPos( GetDockStruct(hwnd), FALSE );
    }

    return;
}


BOOL RegisterDock ( HWND hwnd, HWND hwndParent, DOCK_STRUCT *DockStruct ){
    DOCK_STRUCT *Dock;

    /******************************************************
        Check that Dock is not already in the registry
    ******************************************************/
    Dock = GetDockStruct( hwnd );
    if ( Dock != NULL ){
           DockError( NULL, "Error, attempt register already registered dock");
        return (FALSE);
    }

    /* Dock not already registered, so add it to the end of the list */
    gDockReg[gRegIndex].hwndDock   = hwnd;
    gDockReg[gRegIndex].hwndParent = hwndParent;
    gDockReg[gRegIndex].Dock        = DockStruct;
    gDockReg[gRegIndex].PoppedOut  = FALSE;
    gDockReg[gRegIndex].DockID     = strdup( DockStruct->DockID );

    gRegIndex ++;

    return (TRUE);
}

BOOL DeRegisterDock ( HWND hwnd ){

    int             i;

    for( i = 0; i < gRegIndex; i++ ){
         if ( gDockReg[i].hwndDock == hwnd ){
            // Found the dock
            // Delete this entry by moving last entry in list into it's place &
            // shorten the list (keeps list contiguous after deletion)
            // Check that this is not the only entry in the list, or the last
            free( gDockReg[i].DockID ); // delete storage for old Dock ID
            if ( i != gRegIndex-1  ){
                 // Nope, copy last entry over this one
                gDockReg[i].hwndDock   = gDockReg[gRegIndex-1].hwndDock;
                gDockReg[i].hwndParent = gDockReg[gRegIndex-1].hwndParent;
                gDockReg[i].Dock       = gDockReg[gRegIndex-1].Dock;
                gDockReg[i].DockID     = gDockReg[gRegIndex-1].DockID;
            }
            // Delete last entry in list and return success
            gRegIndex --;
            return (TRUE);
        }
    }

    // Ooops, no dock found;
       DockError( NULL, "Error, attempt de-register non-registered dock");
    return (FALSE);
}

BOOL ReRegisterDock ( HWND hwnd, HWND hwndParent ){

    int             i;

    for( i = 0; i < gRegIndex; i++ ){
         if ( gDockReg[i].hwndDock == hwnd ){
            // Found the dock
            gDockReg[i].hwndParent = hwndParent;
            return (TRUE);
        }
    }

    // Ooops, no dock found;
       DockError( NULL, "Error, attempt re-register non-registered dock");
    return (FALSE);
}

BOOL  SetDockPoppedOut  ( HWND hwnd ){
    int i;

    for( i = 0; i < gRegIndex; i++ ){
         if ( gDockReg[i].hwndDock == hwnd ){
            gDockReg[i].PoppedOut = TRUE;
            return (TRUE);
        }
    }
       DockError( NULL, "Error, attempt pop-out non-registered dock");
    return (FALSE);
}

BOOL  SetDockPoppedIn  ( HWND hwnd ){
    int i;

    for( i = 0; i < gRegIndex; i++ ){
         if ( gDockReg[i].hwndDock == hwnd ){
            gDockReg[i].PoppedOut = FALSE;
            return (TRUE);
        }
    }
    DockError( NULL, "Error, attempt pop-out non-registered dock");
    return (FALSE);
}

/***************************************************************************
    Determine if Window is a parent (or grand parent etc) of a child window
***************************************************************************/
BOOL  IsAParentDock    ( HWND hwndTestParent, SLOT_ENTRY *ChildSlot ){

    HWND     hwndParent, hwndCur;

    hwndCur = ChildSlot->Dock->hwndDock;

    // If the slot is actually in the parent supplied, we can spot this here
    if( hwndCur == hwndTestParent ){
        return (TRUE);
    }
    
    // Climb back up the child's family tree, looking for the supplied
    // parent, until we reach the root dock.
    while ( (hwndParent = GetParentDock( hwndCur )) != NULL ){
        if( hwndTestParent == hwndParent ){
            // Yup, it's a parent of this child
            return (TRUE);
        }
        hwndCur = hwndParent;
    }
    // Can't find a link in the family tree
    return(FALSE);
}


BOOL  IsDockRegistered ( char *DockID ){
    int i;

    for( i = 0; i < gRegIndex; i++ ){
         if ( !stricmp(gDockReg[i].DockID, DockID) ){
            return (TRUE);
        }
    }
    return (FALSE);
}

BOOL  IsDockPoppedOut  ( HWND hwnd ){
    int i;

    for( i = 0; i < gRegIndex; i++ ){
         if ( gDockReg[i].hwndDock == hwnd ){
             if ( gDockReg[i].PoppedOut == TRUE ){
                return (TRUE);
            }
            else{
                return (FALSE);
            }
        }
    }
    DockError( NULL,
        "Error, attempt to determine pop-out state of non-registered dock");
    return (FALSE);
}

/****************************************************************************
  We can derive a sub-dock's orientation by how far is is away from the root
  dock in child-parent steps, each    step reverses the orientation relative
  to the root.
  => an even (or 0) number of steps means the orientation is the same as the
     root dock an odd number of steps means the orientation is the opposite
     to the root dock    
****************************************************************************/
DOCK_POS GetDockOrientation( HWND hwnd ){

    int      i = 0;
    HWND     hwndParent, hwndCur;

    hwndCur = hwnd;

    while ( (hwndParent = GetParentDock( hwndCur )) != NULL ){
        hwndCur = hwndParent;
        i++;
    }

    if( (i%2) == 0 ){
         return (gOptions.RootOrientation);
    }
    else{
        if( gOptions.RootOrientation == DOCK_HORZ ){
             return (DOCK_VERT);
        }
        else{
            return (DOCK_HORZ);
        }
    }

}


void UpdateSubDockID ( DOCK_STRUCT *SubDock, int SlotID ){

    HWND         hwndParentDock;

    if ( SubDock->DockType == ROOT_DOCK ){
        DockError( NULL, "Error, attempt to UpdateSubDockID of RootDock");
        return;
    }

    if( (hwndParentDock = GetParentDock( SubDock->hwndDock )) == NULL ){
        DockError( NULL,
            "Error, attempt to UpdateSubDockID of SubDock with no parent");
        return;
    }

    sprintf( SubDock->DockID, "%s.%d",
             SubDock->ParentSlot->Dock->DockID, SlotID );

    return;
 }

#ifdef WIN32

int GetRootDockID(void){

    HANDLE     hSem;
    LONG     FreeDockInstanceCount;

    hSem = OpenSemaphore(SEMAPHORE_MODIFY_STATE, FALSE, FREEDOCK_SEMAPHORE);

    if( hSem == NULL ){
        // First instance of FreeDock, so create mutex
        hSem = CreateSemaphore( NULL,
                                1,
                                MAX_DOCKS,
                                FREEDOCK_SEMAPHORE );
        if( hSem == NULL ){
            DockError( NULL, "Error, creating FreeDock semaphore");
            exit( -5 );
        }
        // If we created the semaphore, don't close it, otherwise the system
        // automatically destroys the semaphore.
        ReleaseSemaphore(hSem, 1L, &FreeDockInstanceCount);
     }
    else{
        // We now increment the Instance Count.
        ReleaseSemaphore(hSem, 1L, &FreeDockInstanceCount);
        CloseHandle(hSem);
    }

    return ((int)FreeDockInstanceCount);
}

#endif
