/*
 *  DATE.C
 *
 *  Written on 30-Jul-90 by jim nutt.  Changes on 10-Jul-94 by John Dennis.
 *  Released to the public domain.
 *
 *  Parse various string date formats into a UNIX style timestamp.
 */

#include "msged.h"
#include "date.h"
#include "strextra.h"
#include "memextra.h"

static char *month[] =
{
    "Jan", "Feb", "Mar", "Apr", "May", "Jun",
    "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};

static char *day[] =
{
    "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"
};

static char *attr_tokens[] =
{
    "yms",                      /* year msg */
    "yno",                      /* year now */
    "mms",                      /* month msg */
    "mno",                      /* month now */
    "dms",                      /* day msg */
    "dno",                      /* day now */
    "wms",                      /* weekday msg */
    "wno",                      /* weekday now */
    "tnm",                      /* time msg normal */
    "tnn",                      /* time now normal */
    "tam",                      /* time msg atime */
    "tan",                      /* time now atime */

    /* ------------- */

    "ofn",                      /* orginal from name */
    "off",                      /* original from first name */
    "otn",                      /* original to name */
    "otf",                      /* original to first name */
    "osu",                      /* original subject */
    "ooa",                      /* original origin address */
    "oda",                      /* orginal destination address */

    /* ------------- */

    "fna",                      /* from name */
    "ffn",                      /* from first name */
    "fad",                      /* from address */
    "tna",                      /* to name */
    "tfn",                      /* to first name */
    "tad",                      /* to address */
    "sub",                      /* subject */

    /* ------------- */

    "una",                      /* user name */
    "ufn",                      /* user first name */
    "uad",                      /* user address */
    "ceh",                      /* current echo tag */
    "oeh",                      /* old echo tag */

    /* ------------- */

    "ims",                      /* iso date of msg */
    "ino",                      /* iso date now */
    "cms",                      /* 4-digit year of msg */
    "cno",                      /* 4-digit year of now */

    NULL
};

static int valid_date(struct tm *tms);

time_t parsedate(char *ds)
{
    int t;
    struct tm tm;
    char work[80], *s;

    if (ds == NULL || strlen(ds) == 0)
    {
        return 0;
    }

    memset(&tm, 0, sizeof tm);
    strcpy(work, ds);

    if (strchr(ds, '-') != NULL)
    {                           /* quickbbs style date */
        s = strtok(work, "-");
        if (s != NULL)
        {
            tm.tm_mon = atoi(s) - 1;
        }
        s = strtok(NULL, "-");
        if (s != NULL)
        {
            tm.tm_mday = atoi(s);
        }
        s = strtok(NULL, " ");
        if (s != NULL)
        {
            tm.tm_year = atoi(s);
        }
        s = strtok(NULL, ":");
        if (s != NULL)
        {
            while (isspace(*s))
                s++;
            tm.tm_hour = atoi(s);
        }
        s = strtok(NULL, " ");
        if (s != NULL)
        {
            tm.tm_min = atoi(s);
        }
        tm.tm_sec = 0;
    }
    else
    {                           /* fido style date */
        s = strtok(work, " ");

        if (s == NULL)
        {
            return 0;
        }

        if ((t = atoi(s)) == 0)
        {                       /* a usenet date */
            s = strtok(NULL, " ");
            if (s == NULL)
            {
                return 0;
            }
            t = atoi(s);
        }
        tm.tm_mday = t;
        s = strtok(NULL, " ");
        if (s == NULL)
        {
            return 0;
        }
        for (t = 0; t < 12; t++)
        {
            if (stricmp(s, month[t]) == 0)
            {
                break;
            }
        }
        if (t == 12)
        {
            t = 1;              /* WRA */
        }
        tm.tm_mon = t;
        s = strtok(NULL, " ");
        if (s == NULL)
        {
            return 0;
        }
        tm.tm_year = atoi(s);
        s = strtok(NULL, ":");
        if (s == NULL)
        {
            return 0;
        }
        while (isspace(*s))
        {
            s++;
        }
        tm.tm_hour = atoi(s);
        s = strtok(NULL, ": \0");
        if (s == NULL)
        {
            return 0;
        }
        tm.tm_min = atoi(s);
        s = strtok(NULL, " ");
        if (s != NULL)
        {
            tm.tm_sec = atoi(s);
        }
        tm.tm_isdst = -1;
    }
#ifdef NEVER
    tm.tm_hour += 5;
#endif
    return mktime(&tm);
}

char *itime(time_t now)
{
    struct tm *tm;
    static char tmp[40];

    tm = localtime(&now);

    if (!tm || !valid_date((tm)))
    {
        sprintf(tmp, "%04d-%02d-%02d %02d:%02d:%02d",
                1970, 1, 1, 0, 0, 0);
    }
    else
    {
        sprintf(tmp, "%04d-%02d-%02d %02d:%02d:%02d",
                tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday,
                tm->tm_hour, tm->tm_min, tm->tm_sec);
    }
    return tmp;
}

char *atime(time_t now)
{
    struct tm *tm;
    static char tmp[40];

    tm = localtime(&now);

    if (!tm || !valid_date((tm)))
    {
        sprintf(tmp, "%s %s %02d %04d %02d:%02d:%02d",
                day[4], month[0], 1, 1970, 0, 0, 0);
    }
    else
    {
        sprintf(tmp, "%s %s %02d %04d %02d:%02d:%02d",
                day[tm->tm_wday], month[tm->tm_mon], tm->tm_mday,
                tm->tm_year + 1900, tm->tm_hour, tm->tm_min, tm->tm_sec);
    }
    return tmp;
}

char *mtime(time_t now)
{
    struct tm *tm;
    static char tmp[21];

    tm = localtime(&now);

    if (!tm || !valid_date((tm)))
    {
        sprintf(tmp, "%02d %s %02d  %02d:%02d:%02d",
                1, month[0], 70, 0, 0, 0);
    }
    else
    {
        sprintf(tmp, "%02d %s %02d  %02d:%02d:%02d",
                tm->tm_mday, month[tm->tm_mon], tm->tm_year,
                tm->tm_hour, tm->tm_min, tm->tm_sec);
    }
    return tmp;
}

char *qtime(time_t now)

{
    struct tm *tm;
    static char tmp[20];

    tm = localtime(&now);

    if (!tm || !valid_date((tm)))
    {
        sprintf(tmp, "%s %02d %02d:%02d",
                month[0], 1, 0, 0);
    }
    else
    {
        sprintf(tmp, "%s %02d %02d:%02d",
                month[tm->tm_mon], tm->tm_mday, tm->tm_hour, tm->tm_min);
    }
    return tmp;
}


/* find_token - returns the token number or -1 if not found */

int find_token(char *token)
{
    int i;

    i = 0;
    while (attr_tokens[i] != NULL)
    {
        if (stricmp(attr_tokens[i], token) == 0)
        {
            return i;
        }
        i++;
    }
    return -1;
}


/* Returns a pointer to the first name. Note: uses static memory. */

char *firstname(char *name)
{
    char *s;
    static char work[40];

    memset(work, 0, sizeof work);
    s = strchr(name, ' ');
    if (s == NULL)
    {
        strcpy(work, name);
    }
    else
    {
        *s = '\0';
        strcpy(work, name);
        *s = ' ';
    }
    return work;
}

/* attrib_line - builds an attribution line */

char *attrib_line(msg * m, msg * old, int olda, char *format)
{
    struct tm now, *tm;
    char work[256], token[5], *t;
    time_t n;
    int num;

    if (format == NULL)
    {
        return NULL;
    }

    memset(work, 0, sizeof work);
    t = work;
    n = time(NULL);
    tm = localtime(&n);
    now = *tm;

    if (old)
    {
        tm = localtime(&(old->timestamp));
    }

    while (*format)
    {
        if (*format == '%')
        {
            format++;
            switch (*format)
            {
            case '%':
                *t = *format;
                break;

            case '_':
                *t = ' ';
                break;

            default:
                memset(token, 0, sizeof token);
                strncpy(token, format, 3);
                num = find_token(token);

                switch (num)
                {
                case 0:
                    if (old)
                    {
                        sprintf(t, "%02d", tm->tm_year);
                    }
                    break;

                case 1:
                    sprintf(t, "%02d", now.tm_year);
                    break;

                case 2:
                    if (old)
                    {
                        strcpy(t, month[tm->tm_mon]);
                    }
                    break;

                case 3:
                    strcpy(t, month[now.tm_mon]);
                    break;

                case 4:
                    if (old)
                    {
                        sprintf(t, "%02d", tm->tm_mday);
                    }
                    break;

                case 5:
                    sprintf(t, "%02d", now.tm_mday);
                    break;

                case 6:
                    if (old)
                    {
                        strcpy(t, day[tm->tm_wday]);
                    }
                    break;

                case 7:
                    strcpy(t, day[now.tm_wday]);
                    break;

                case 8:
                    if (old)
                    {
                        sprintf(t, "%02d:%02d", tm->tm_hour, tm->tm_min);
                    }
                    break;

                case 9:
                    sprintf(t, "%02d:%02d", now.tm_hour, now.tm_min);
                    break;

                case 10:
                    if (old)
                    {
                        strcpy(t, atime(old->timestamp));
                    }
                    break;

                case 11:
                    strcpy(t, atime(n));
                    break;

                case 12:
                    if (old && old->isfrom)
                    {
                        strcpy(t, old->isfrom);
                    }
                    break;

                case 13:
                    if (old && old->isfrom)
                    {
                        strcpy(t, firstname(old->isfrom));
                    }
                    break;

                case 14:
                    if (old && old->isto)
                    {
                        strcpy(t, old->isto);
                    }
                    break;

                case 15:
                    if (old && old->isto)
                    {
                        strcpy(t, firstname(old->isto));
                    }
                    break;

                case 16:
                    if (old && old->subj)
                    {
                        strcpy(t, old->subj);
                    }
                    break;

                case 17:
                    if (old)
                    {
                        strcpy(t, show_address(&old->from));
                    }
                    break;

                case 18:
                    if (old)
                    {
                        strcpy(t, show_address(&old->to));
                    }
                    break;

                case 19:
                    if (m->isfrom)
                    {
                        strcpy(t, m->isfrom);
                    }
                    break;

                case 20:
                    if (m->isfrom)
                    {
                        strcpy(t, firstname(m->isfrom));
                    }
                    break;

                case 21:
                    strcpy(t, show_address(&m->from));
                    break;

                case 22:
                    if (m->isto)
                    {
                        strcpy(t, m->isto);
                    }
                    break;

                case 23:
                    if (m->isto)
                    {
                        strcpy(t, firstname(m->isto));
                    }
                    break;

                case 24:
                    strcpy(t, show_address(&m->to));
                    break;

                case 25:
                    strcpy(t, m->subj);
                    break;

                case 26:
                    if (ST->username)
                    {
                        strcpy(t, ST->username);
                    }
                    break;

                case 27:
                    if (ST->username)
                    {
                        strcpy(t, firstname(ST->username));
                    }
                    break;

                case 28:
                    strcpy(t, show_address(&CurArea.addr));
                    break;

                case 29:
                    strcpy(t, CurArea.tag);
                    break;

                case 30:
                    if (olda != -1)
                    {
                        strcpy(t, arealist[olda].tag);
                    }
                    break;

                case 31:
                    if (old != NULL)
                    {
                        sprintf(t, "%04d-%02d-%02d",
                        tm->tm_year + 1900, tm->tm_mon + 1, tm->tm_mday);
                    }
                    break;
                case 32:
                    sprintf(t, "%04d-%02d-%02d",
                            now.tm_year + 1900, now.tm_mon + 1,
                            now.tm_mday);
                    break;

                case 33:
                    if (old != NULL)
                    {
                        sprintf(t, "%04d", tm->tm_year + 1900);
                    }
                    break;

                case 34:
                    sprintf(t, "%04d", now.tm_year + 1900);
                    break;

                default:
                    break;
                }
                break;
            }
            t = work + strlen(work);
            format += 3;
        }
        else if (*format == '\\')
        {
            if (*(++format) == 'n')
            {
                *t++ = '\n';
            }
            format++;
        }
        else
        {
            *t++ = *format++;
        }
    }
    return xstrdup(work);
}

static int valid_date(struct tm *tms)
{
    int ret;

    ret = (tms->tm_wday > 6 || tms->tm_wday < 0 ||
           tms->tm_mon > 11 || tms->tm_mon < 0 ||
           tms->tm_mday > 31 || tms->tm_mday < 0 ||
           tms->tm_year > 99 || tms->tm_year < 0 ||
           tms->tm_hour > 23 || tms->tm_hour < 0 ||
           tms->tm_min > 59 || tms->tm_min < 0 ||
           tms->tm_sec > 59 || tms->tm_sec < 0);
    return (!ret);
}
