{ TaisDBComboBox Component
	Copyright (c) 1996, Apogee Information Systems, Inc. }

unit Ais_cb;

interface

uses SysUtils, WinTypes, WinProcs, Messages, Classes, Controls, Forms,
  Graphics, Menus, StdCtrls, ExtCtrls, DB, DBTables, Mask, Buttons,DsgnIntf;

type
  TaisDBComboBox = class(TCustomComboBox)
  private
    FDataLink: TFieldDataLink;
    FValues: TStrings;
    FAbout1, FAbout2: String;
    procedure SetValues(Value: TStrings);
    procedure DataChange(Sender: TObject);
    procedure EditingChange(Sender: TObject);
    function GetComboText: string;
    function GetDataField: string;
    function GetDataSource: TDataSource;
    function GetField: TField;
    function GetReadOnly: Boolean;
    procedure SetComboText(const Value: string);
    procedure SetDataField(const Value: string);
    procedure SetDataSource(Value: TDataSource);
    procedure SetEditReadOnly;
    procedure SetItems(Value: TStrings);
    procedure SetReadOnly(Value: Boolean);
    procedure UpdateData(Sender: TObject);
    procedure CMExit(var Message: TCMExit); message CM_EXIT;
  protected
    procedure Change; override;
    procedure Click; override;
    procedure ComboWndProc(var Message: TMessage; ComboWnd: HWnd;
      ComboProc: Pointer); override;
    procedure CreateWnd; override;
    procedure DropDown; override;
    procedure KeyDown(var Key: Word; Shift: TShiftState); override;
    procedure KeyPress(var Key: Char); override;
    procedure Notification(AComponent: TComponent;
      Operation: TOperation); override;
    procedure WndProc(var Message: TMessage); override;
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    property Field: TField read GetField;
    property Text;
  published
    property About: string read FAbout1 write FAbout2;
    property Color;
    property Ctl3D;
    property DataField: string read GetDataField write SetDataField;
    property DataSource: TDataSource read GetDataSource write SetDataSource;
    property Values: TStrings read FValues write SetValues;
    property DragMode;
    property DragCursor;
    property DropDownCount;
    property Enabled;
    property Font;
    property ItemHeight;
    property Items write SetItems;
    property ParentColor;
    property ParentCtl3D;
    property ParentFont;
    property ParentShowHint;
    property PopupMenu;
    property ReadOnly: Boolean read GetReadOnly write SetReadOnly default False;
    property ShowHint;
    property TabOrder;
    property TabStop;
    property Visible;
    property OnChange;
    property OnClick;
    property OnDblClick;
    property OnDragDrop;
    property OnDragOver;
    property OnDrawItem;
    property OnDropDown;
    property OnEndDrag;
    property OnEnter;
    property OnExit;
    property OnKeyDown;
    property OnKeyPress;
    property OnKeyUp;
    property OnMeasureItem;
  end;

TAboutProperty = class(TPropertyEditor)
  private
    { Private declarations }
  protected
    function GetAttributes: TPropertyAttributes; override;
    procedure Edit; override;
    { Protected declarations }
  public
    { Public declarations }
  end;

  TaisComboBoxEditor = class(TComponentEditor)
  private
    { Private declarations }
  protected
    procedure Edit; override;
    { Protected declarations }
  public
    { Public declarations }
  end;

procedure Register;

implementation

uses AboutDlg;

procedure Register;
begin
	{ Register the component on the Apogee Page }
  RegisterComponents('Apogee', [TaisDBComboBox]);
	{Register the About property editor}
  RegisterPropertyEditor(TypeInfo(String), TaisDBComboBox, 'About',
                         TAboutProperty);
  RegisterComponentEditor(TaisDBComboBox, TaisComboBoxEditor);
end;

{This method is called when the user double-clicks on the component.
 Here is where we display the Dialog}
procedure TaisComboBoxEditor.Edit;
var
  AboutForm: TAboutForm;
begin
  try
    AboutForm:=TAboutForm.Create(nil); {Create About component}
    AboutForm.ShowModal; {Show About dialog }
  finally
    AboutForm.Free; {Free resources from About Dialog}
  end;
end;

{This procedure returns the attributes for our property editor. In this case
we want to be able to have our property pop-up a dialog}
function TAboutProperty.GetAttributes: TPropertyAttributes;
begin
  Result:=[ paDialog];
end;

{This is where the property editor dialog gets displayed.
The routine is called when the button in the property field on
the Object Inspector is click or the field in the Inspector is
double-clicked}
procedure TAboutProperty.Edit;
var
  AboutForm: TAboutForm;
begin
  try
    AboutForm:=TAboutForm.Create(nil); {Create Database Name form}
    AboutForm.ShowModal; {Display Database Name form}
  finally
    AboutForm.Free; {Free resources from form}
  end;
end;

constructor TaisDBComboBox.Create(AOwner: TComponent);
begin
  inherited Create(AOwner);
  FDataLink := TFieldDataLink.Create;
  FDataLink.Control := Self;
  FDataLink.OnDataChange := DataChange;
  FDataLink.OnUpdateData := UpdateData;
  FDataLink.OnEditingChange := EditingChange;
  FValues := TStringList.Create;
  Style := csDropDownList;
end;

destructor TaisDBComboBox.Destroy;
begin
  FDataLink.Free;
  FDataLink := nil;
	FValues.Free;
  FValues := nil;
  inherited Destroy;
end;

procedure TaisDBComboBox.Notification(AComponent: TComponent;
  Operation: TOperation);
begin
  inherited Notification(AComponent, Operation);
  if (Operation = opRemove) and (FDataLink <> nil) and
    (AComponent = DataSource) then DataSource := nil;
end;

procedure TaisDBComboBox.CreateWnd;
begin
  inherited CreateWnd;
  SetEditReadOnly;
end;

procedure TaisDBComboBox.DataChange(Sender: TObject);
begin
  if FDataLink.Field <> nil then
    SetComboText(FDataLink.Field.Text)
  else
    if csDesigning in ComponentState then
      SetComboText(Name)
    else
      SetComboText('');
end;

procedure TaisDBComboBox.UpdateData(Sender: TObject);
begin
  FDataLink.Field.Text := GetComboText;
end;

procedure TaisDBComboBox.SetComboText(const Value: string);
{ Set the itemindex property by locating the code value
  in the Values property. This will automatically (from the
  base class) display the cooresponding description for the
  code (in Items). }
begin
  if Value = '' then
  	ItemIndex := -1
  else
   	ItemIndex := Values.IndexOf(Value);
end;

function TaisDBComboBox.GetComboText: string;
{ Get the code value of the item based on the itemindex
  of the description list.}
begin
  if ItemIndex < 0 then
   	Result := ''
  else
   	Result := Values[ItemIndex];
end;

procedure TaisDBComboBox.Change;
begin
  FDataLink.Edit;
  inherited Change;
  FDataLink.Modified;
end;

procedure TaisDBComboBox.Click;
begin
  FDataLink.Edit;
  inherited Click;
  FDataLink.Modified;
end;

procedure TaisDBComboBox.DropDown;
begin
  FDataLink.Edit;
  inherited DropDown;
end;

function TaisDBComboBox.GetDataSource: TDataSource;
begin
  Result := FDataLink.DataSource;
end;

procedure TaisDBComboBox.SetDataSource(Value: TDataSource);
begin
  FDataLink.DataSource := Value;
end;

function TaisDBComboBox.GetDataField: string;
begin
  Result := FDataLink.FieldName;
end;

procedure TaisDBComboBox.SetDataField(const Value: string);
begin
  FDataLink.FieldName := Value;
end;

function TaisDBComboBox.GetReadOnly: Boolean;
begin
  Result := FDataLink.ReadOnly;
end;

procedure TaisDBComboBox.SetReadOnly(Value: Boolean);
begin
  FDataLink.ReadOnly := Value;
end;

function TaisDBComboBox.GetField: TField;
begin
  Result := FDataLink.Field;
end;

procedure TaisDBComboBox.KeyDown(var Key: Word; Shift: TShiftState);
begin
  inherited KeyDown(Key, Shift);
  if Key in [VK_BACK, VK_DELETE, VK_UP, VK_DOWN, 32..255] then
  begin
    if not FDataLink.Edit and (Key in [VK_UP, VK_DOWN]) then
      Key := 0;
  end;
end;

procedure TaisDBComboBox.KeyPress(var Key: Char);
begin
  inherited KeyPress(Key);
  if (Key in [#32..#255]) and (FDataLink.Field <> nil) and
    not FDataLink.Field.IsValidChar(Key) then
  begin
    MessageBeep(0);
    Key := #0;
  end;
  case Key of
    ^H, ^V, ^X, #32..#255:
      FDataLink.Edit;
    #27:
      begin
        FDataLink.Reset;
        SelectAll;
        Key := #0;
      end;
  end;
end;

procedure TaisDBComboBox.EditingChange(Sender: TObject);
begin
  SetEditReadOnly;
end;

procedure TaisDBComboBox.SetEditReadOnly;
begin
  if (Style in [csDropDown, csSimple]) and HandleAllocated then
    SendMessage(FEditHandle, EM_SETREADONLY, Ord(not FDataLink.Editing), 0);
end;

procedure TaisDBComboBox.WndProc(var Message: TMessage);
begin
  if not (csDesigning in ComponentState) then
    case Message.Msg of
      WM_COMMAND:
        if TWMCommand(Message).NotifyCode = CBN_SELCHANGE then
          if not FDataLink.Edit then
          begin
            if Style <> csSimple then
              PostMessage(Handle, CB_SHOWDROPDOWN, 0, 0);
            Exit;
          end;
      CB_SHOWDROPDOWN:
        if Message.WParam <> 0 then FDataLink.Edit else
          if not FDataLink.Editing then DataChange(Self); {Restore text}
    end;
  inherited WndProc (Message);
end;

procedure TaisDBComboBox.ComboWndProc(var Message: TMessage; ComboWnd: HWnd;
  ComboProc: Pointer);
begin
  if not (csDesigning in ComponentState) then
    case Message.Msg of
      WM_LBUTTONDOWN:
        if (Style = csSimple) and (ComboWnd <> FEditHandle) then
          if not FDataLink.Edit then Exit;
    end;
  inherited ComboWndProc (Message, ComboWnd, ComboProc);
end;

procedure TaisDBComboBox.CMExit(var Message: TCMExit);
begin
  try
    FDataLink.UpdateRecord;
  except
    SelectAll;
    SetFocus;
    raise;
  end;
  inherited;
end;

procedure TaisDBComboBox.SetItems(Value: TStrings);
begin
  Items.Assign(Value);
  DataChange(Self);
end;

procedure TaisDBComboBox.SetValues(Value: TStrings);
begin
	FValues.Assign(Value);
end;

end.
