unit Xlcls;
{
// XLCLS.PAS (C) 1995 W. Raike
//              ALL RIGHTS RESERVED.
}
interface
{*******************************************************************}
{
  Classes to "mirror" (some of) the properties and methods
  exposed by Microsoft Excel as an OLE Automation server.

  Mirror classes are derived from SafeOLE unit's TSafeOleObject
  instead of the basic TOLEObject so that nested objects
  created "on the fly" by the ConnectInterface constructor can
  be freed later.

}    
{*******************************************************************}

uses
  SysUtils, OleAuto, SafeOle;

type

  TExcelCells = class (TSafeOleObject)
  private
    procedure SetValue (ValText : String);
  public
    property Value : String write SetValue;
  end;

  TExcelSheet = class (TSafeOleObject)
  private
    fCell : TExcelCells;
    function GetCell (i,j: Integer): TExcelCells;
  public
    procedure FileSave (FileSaveName : string);
    property Cells[X,Y: Integer] : TExcelCells read GetCell;
  end;

  TXLWorkBook = class(TSafeOleObject)
  private
    function GetActiveSheet : TExcelSheet;
    function GetSaved : Boolean;
    procedure SetSaved(IsSaved : Boolean);
  public
    property ActiveSheet : TExcelSheet read GetActiveSheet;
    property Saved : Boolean read GetSaved write SetSaved;
  end;

  TXLWorkBooks = class(TSafeOleObject)
  public
    function Add : TXLWorkBook;
  end;

  TExcelApp = class (TSafeOleObject)
  private
    function GetVisible : Boolean;
    procedure SetVisible(vis : Boolean);
    function GetWorkBooks : TXLWorkBooks;
  public
    property Visible : Boolean read GetVisible write SetVisible;
    property WorkBooks : TXLWorkBooks read GetWorkBooks;
    procedure Quit;
  end;

implementation

procedure TExcelCells.SetValue(ValText : String);
var
  pszText : PChar;
begin
  pszText := StrAlloc(256);
  StrPCopy(pszText, ValText);
  SetOleProperty ('Value', 'PChar', pszText);
  StrDispose(pszText);
end;

function TExcelSheet.GetCell (i,j: Integer): TExcelCells;
var
  piCell : PInterface;
begin
  SetOLEMethodArg ('Integer',i);
  SetOLEMethodArg ('Integer',j);
  CallOLEFunction ('Cells', 'pInterface', piCell);
  fCell := TExcelCells.ConnectInterface (piCell);
  Result := fCell;
end;

procedure TExcelSheet.FileSave (FileSaveName : string);
var
  arName : array[0..256] of Char;
  pName  : PChar;
begin
  StrPCopy (arName, FileSaveName);
  pName := @arName;
  SetOleMethodArg ('PChar',pName);
  CallOleProc ('SaveAs');
end;

function TXLWorkBook.GetActiveSheet : TExcelSheet;
var
  piActiveSheet : pInterface;
begin
  GetOleProperty('ActiveSheet', 'pInterface', piActiveSheet);
  if Assigned(piActiveSheet) then
    Result := TExcelSheet.ConnectInterface(piActiveSheet);
end;

function TXLWorkBook.GetSaved : Boolean;
var
  intSaved : Integer;
begin
  GetOleProperty('Saved', 'Integer', intSaved);
  Result := Boolean(intSaved);
end;

procedure TXLWorkBook.SetSaved(IsSaved : Boolean);
var
  intSaved : Integer;
begin
  intSaved := Integer(IsSaved);
  SetOleProperty('Saved', 'Integer', intSaved);
end;

function TXLWorkBooks.Add : TXLWorkBook;
var
  piNewBook : pInterface;
begin
  CallOleFunction('Add', 'pInterface', piNewBook);
  if Assigned(piNewBook) then
    Result := TXLWorkBook.ConnectInterface(piNewBook);
end;

function TExcelApp.GetVisible : Boolean;
var
  intvis : Integer;
begin
  GetOleProperty('Visible', 'Integer', intvis);
  Result := Boolean(intvis);
end;

function TExcelApp.GetWorkBooks : TXLWorkBooks;
var
  piWorkBooks : pInterface;
begin
  GetOleProperty('WorkBooks', 'pInterface', piWorkBooks);
  if Assigned(piWorkBooks) then
    Result := TXLWorkBooks.ConnectInterface(piWorkBooks);
end;

procedure TExcelApp.SetVisible(vis : Boolean);
var
  intvis : Integer;
begin
  intvis := Integer(vis);
  SetOleProperty('Visible', 'Integer', intvis);
end;

procedure TExcelApp.Quit;
begin
  CallOleProc('Quit');
end;

end.
